﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecutePrintAction.h"

#include <string>
#include <cstdlib>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
#include "../../../../../Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"
#endif

namespace album {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

namespace {

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    void PrintHexBytes(const char* pBuffer, size_t size) NN_NOEXCEPT
    {
        const char* p = reinterpret_cast<const char*>(pBuffer);
        for (int i = 0; i<size; ++i)
        {
            NN_LOG("%02X", p[i]);
        }
    }

    bool ParseHexBytes(char* pOut, const char* pIn, int byteCount) NN_NOEXCEPT
    {
        if (pIn[0] =='0' && (pIn[1] == 'x' || pIn[1] == 'X'))
        {
            pIn += 2;
        }

        if (std::strlen(pIn) != byteCount * 2)
        {
            return false;
        }

        for (int i=0; i<byteCount; ++i)
        {
            auto cUpper = pIn[i * 2 + 0];
            auto cLower = pIn[i * 2 + 1];

            if (!(std::isxdigit(cUpper) && std::isxdigit(cLower)))
            {
                return false;
            }

            char buf[3] = { cUpper, cLower, '\0' };
            auto value = std::strtol(buf, NULL, 16);
            *pOut++ = value;
        }
        *pOut = '\0';
        return true;
    }

    //-------------------------------------------------------------------------
    // 指定された ApplicationId を難読化して表示する
    bool ShowEncryptedApplicationId(const char* pApplicationId, const ProgramOption& opts) NN_NOEXCEPT
    {
        bool isExtra = opts.IsExtraAlbumFileOnly();

        char buf[9] = {};
        if (!ParseHexBytes(buf, pApplicationId, 8))
        {
            NN_LOG("Input:  ApplicationId        = Invalid applicationId\n");
            return false;
        }

        nn::ncm::ApplicationId applicationId = {};
        applicationId.value = std::strtoll(pApplicationId, NULL, 16);
        NN_LOG("Input:  ApplicationId        = 0x%016llx\n", applicationId.value);
        NN_LOG("        IsExtra flag         = %s\n", isExtra ? "true" : "false");

        auto encryptedApplicationId = nn::capsrv::server::detail::EncryptApplicationId(applicationId, isExtra);
        NN_LOG("Output: EncodedApplicationId = ");
        PrintHexBytes(encryptedApplicationId.value, sizeof(encryptedApplicationId.value));
        NN_LOG("\n");
        return true;
    }

    //-------------------------------------------------------------------------
    // 指定された EncryptedApplicationId の難読化を解除して表示する
    bool ShowDecryptedApplicationId(const char* pEncryptedApplicationId, const ProgramOption& opts) NN_NOEXCEPT
    {
        char buf[17] = {};
        if (!ParseHexBytes(buf, pEncryptedApplicationId, 16))
        {
            NN_LOG("Input:  EncodedApplicationId = Invalid encoded-applicationId\n");
            return false;
        }
        NN_LOG("Input:  EncodedApplicationId = %s\n", pEncryptedApplicationId);

        nn::capsrv::server::detail::EncryptedApplicationId encryptedApplicationId;
        std::memcpy(encryptedApplicationId.value, buf, sizeof(encryptedApplicationId.value));
        nn::ncm::ApplicationId applicationId = {};
        bool isExtra = false;
        if (!nn::capsrv::server::detail::TryDecryptApplicationId(&applicationId, &isExtra, encryptedApplicationId).IsSuccess())
        {
            NN_LOG("Output: ApplicationId        = Failed to decode.\n");
            return false;
        }

        NN_LOG("Output: ApplicationId        = 0x%016llx\n", applicationId.value);
        NN_LOG("        IsExtra flag         = %s\n", isExtra ? "true" : "false");
        return true;
    }

#endif // defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

}   // namespace

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
bool ExecutePrintEncryptedApplicationIdAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Album Print\n\n");

    switch (opts.GetArgumentCount())
    {
        case 0: // 引数なし
        {
            NN_LOG("Album Print ... missing <ApplicationId> argument(s)\n");
            break;
        }
        case 1: // 引数１つ（ApplicationId）
        {
            auto pApplicationId = opts.GetLastArgument();
            if (!ShowEncryptedApplicationId(pApplicationId, opts))
            {
                return false;
            }
            break;
        }
        default:
        {
            NN_LOG("Album Print ... too many argument(s)\n");
            return false;
        }
    }

    NN_LOG("\n");
    NN_LOG("Album Print ... completed.\n");
    return true;
}

bool ExecutePrintDecryptedApplicationIdAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Album Print\n\n");

    switch (opts.GetArgumentCount())
    {
        case 0: // 引数なし
        {
            NN_LOG("Album Print ... missing <EncodedApplicationId> argument(s)\n");
            break;
        }
        case 1: // 引数１つ（EncryptedApplicationId）
        {
            auto pEncryptedApplicationId = opts.GetLastArgument();
            if (!ShowDecryptedApplicationId(pEncryptedApplicationId, opts))
            {
                return false;
            }
            break;
        }
        default:
        {
            NN_LOG("Album Print ... too many argument(s)\n");
            return false;
        }
    }

    NN_LOG("\n");
    NN_LOG("Album Print ... completed.\n");
    return true;
}
#endif

#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

}   // namespace album
