﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommandAlbum_ExecuteInfoAction.h"

#include <vector>
#include <string>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>
#include <nn/capsrv/capsrv_AlbumStorage.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/capsrv/capsrv_AlbumAccessForDevMenuCommand.h>

#include "../DevMenuCommand_PrintUtil.h"
#include "DevMenuCommandAlbum_StorageUtility.h"
#include "DevMenuCommandAlbum_ParseFilename.h"
#include "../../../../../Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_EncryptApplicationId.h"
#include "../../../../../Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_MakerNoteInfo.h"

namespace album {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

namespace {

    const char* GetStorageString(int storage, const char* pDefault) NN_NOEXCEPT
    {
        switch (storage)
        {
            case nn::capsrv::AlbumStorage_Nand:
            {
                return "NAND";
            }
            case nn::capsrv::AlbumStorage_Sd:
            {
                return "SD-Card";
            }
            default: return pDefault;
        }
    }

    const char* GetContentsString(int contentsType, const char* pDefault) NN_NOEXCEPT
    {
        switch (contentsType)
        {
            case nn::capsrv::AlbumFileContents_ScreenShot:
            {
                return "ScreenShot";
            }
            case nn::capsrv::AlbumFileContents_Movie:
            {
                return "Movie";
            }
            case nn::capsrv::AlbumFileContents_ExtraScreenShot:
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
            {
                return "ExtraScreenShot";
            }
            case nn::capsrv::AlbumFileContents_ExtraMovie:
            {
                return "ExtraMovie";
            }
            case -1:
            {
                return "Unknown";
            }
#endif
            default: return pDefault;
        }
    }

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    NN_ALIGNAS(4096) char g_WorkBuffer[nn::capsrv::AlbumFileSizeLimit_ScreenShot];
    nn::capsrv::server::detail::MakerNoteInfo g_MakerNoteInfo;

    const char* GetAlbumFileDescription(nn::capsrv::AlbumFileDescriptionType type) NN_NOEXCEPT
    {
        switch (type)
        {
            case nn::capsrv::AlbumFileDescription_ScreenShot:
            {
                return "AlbumFileDescription_ScreenShot";
            }
            case nn::capsrv::AlbumFileDescription_ScreenShotCaptured:
            {
                return "AlbumFileDescription_ScreenShotCaptured";
            }
            case nn::capsrv::AlbumFileDescription_ScreenShotEdited:
            {
                return "AlbumFileDescription_ScreenShotEdited";
            }
            case nn::capsrv::AlbumFileDescription_ScreenShotSaved:
            {
                return "AlbumFileDescription_ScreenShotSaved";
            }
            case nn::capsrv::AlbumFileDescription_Movie:
            {
                return "AlbumFileDescription_Movie";
            }
            case nn::capsrv::AlbumFileDescription_MovieContinuous:
            {
                return "AlbumFileDescription_MovieContinuous";
            }
            case nn::capsrv::AlbumFileDescription_MovieTrimmed:
            {
                return "AlbumFileDescription_MovieTrimmed";
            }
            case nn::capsrv::AlbumFileDescription_MovieMakerSaved:
            {
                return "AlbumFileDescription_MovieMakerSaved";
            }
            default:
            {
                return "Unknown";
            }
        }
    }

    void PrintHexBytes(const void* pBuffer, size_t size) NN_NOEXCEPT
    {
        if (size == 0)
        {
            return;
        }

        const char* p = reinterpret_cast<const char*>(pBuffer);
        for (int i = 0; i<size - 1; ++i)
        {
            NN_LOG("%02X,", *p++);
        }
        NN_LOG("%02X\n", *p);
    }

    void PrintScreenShotInfo(const char* path, nn::capsrv::AlbumFileId& fileId, uint64_t fileSize, const nn::capsrv::server::detail::MakerNoteInfo& info) NN_NOEXCEPT
    {
        const auto horizon = "-----------------------------------------------------------------------------\n";

        NN_LOG(horizon);
        NN_LOG("PATH=\x22%s\x22\n", path);
        NN_LOG("File Size = %llu\n", fileSize);
        NN_LOG("\n");

        NN_LOG(horizon);
        NN_LOG("AlbumFileId.ApplicationId = 0x%016llx\n", fileId.applicationId);
        NN_LOG("AlbumFileId.time          = %04d/%02d/%02d %02d:%02d:%02d (id=%02d)\n", fileId.time.year, fileId.time.month, fileId.time.day, fileId.time.hour, fileId.time.minute, fileId.time.second, fileId.time.id);
        NN_LOG("AlbumFileId.storage       = %d  (%s)\n", fileId.storage, GetStorageString(fileId.storage, "???"));
        NN_LOG("AlbumFileId.contents      = %d  (%s)\n", fileId.contents, GetContentsString(fileId.contents, "???"));
        NN_LOG("\n");

        NN_LOG(horizon);
        NN_LOG("MakerNoteInfo:\n\n");
        NN_LOG("MakerNoteVersion          : %d\n", info.version);
        NN_LOG("Signature                 : ");
        PrintHexBytes(info.signature.value, sizeof(info.signature.value));
        NN_LOG("EncryptedApplicationId    : ");
        PrintHexBytes(info.encryptedApplicationId.value, sizeof(info.encryptedApplicationId.value));
        NN_LOG("DateTime                  : %04d/%02d/%02d %02d:%02d:%02d (id=%02d)\n", info.dateTime.year, info.dateTime.month, info.dateTime.day, info.dateTime.hour, info.dateTime.minute, info.dateTime.second, info.dateTime.id);
        NN_LOG("FileDescriptionType       : 0x%04x (%s)\n", info.fileDescription, GetAlbumFileDescription(info.fileDescription));
        NN_LOG("ScreenShot Width x Height : %d x %d\n", info.dataWidth, info.dataHeight);
        NN_LOG("     Movie Width x Height : %d x %d\n", info.movieDataWidth, info.movieDataHeight);
        NN_LOG("DataOrientationType       : %u      (Rotate %u degrees)\n", info.dataOrientation, info.dataOrientation * 90);
        NN_LOG("Movie frame count         : %u\n", info.frameCount);
        NN_LOG("Movie frame rate NUME/DENO: %u / %u\n", info.frameRateNumerator, info.frameRateDenominator);
        NN_LOG("Movie data duration       : %u.%03u (sec)\n", info.dataDurationMilliseconds / 1000, info.dataDurationMilliseconds % 1000);
        NN_LOG("Movie key frame interval  : %u (frame)\n", info.keyFrameInterval);
        NN_LOG("IsCopyrightComposted      : %u      (%s)\n", info.isCopyrightComposited, info.isCopyrightComposited ? "true" : "false");
        NN_LOG("HasUneditableArea         : %u      (%s)\n", info.hasUneditableArea, info.hasUneditableArea ? "true" : "false");
        NN_LOG("Uneditable area (Left,Top): (%u, %u)\n", info.uneditableAreaCoordinateX, info.uneditableAreaCoordinateY);
        NN_LOG("Uneditable area size      :  %u x %u\n", info.uneditableAreaWidth, info.uneditableAreaHeight);

        NN_LOG("\n");

        // AppletData
        {
            auto size = sizeof(info.appletData);
            NN_LOG("AppletData: size=%zu\n", size);
            devmenuUtil::PrintHexDump(info.appletData.value, size);
            NN_LOG("\n");
        }

        // ApplicationData
        {
            auto size = info.applicationData.size;
            NN_LOG("ApplicationData: size=%u\n", size);
            devmenuUtil::PrintHexDump(info.applicationData.value, size);
            NN_LOG("\n");
        }

        // SystemReservedInfo.uidList
        {
            const auto& uidList = info.systemReservedInfo.uidList;
            NN_LOG("UserIdtList: count=%u\n", uidList.uidCount);

            for (int i=0; i<uidList.uidCount; ++i)
            {
                auto p = &uidList.uid[i];
                NN_LOG("  Uid[%d] = { 0x%016llx, 0x%016llx }\n", i, p->_data[0], p->_data[1]);
            }
            NN_LOG("\n");
        }

        // SystemReservedInfo._reserved
        {
            auto size = sizeof(info.systemReservedInfo._reserved);
            NN_LOG("SystemReservedInfo._reserved[]: size=%u\n", size);
            devmenuUtil::PrintHexDump(info.systemReservedInfo._reserved, size);
            NN_LOG("\n");
        }

        NN_LOG(horizon);

    }   // NOLINT(impl/function_size)

#endif // defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

    // ストレージ情報の表示
    void PrintAlbumContentsUsage(int storage, int contentsType, const nn::capsrv::AlbumContentsUsage* p) NN_NOEXCEPT
    {
        auto pStorageStr = GetStorageString(storage, NULL);
        if (!pStorageStr)
        {
            return;
        }
        auto pContentsStr = GetContentsString(contentsType, NULL);
        if (!pContentsStr)
        {
            return;
        }
        NN_LOG(" %7s   %15s   %10llu   %14llu\n", pStorageStr, pContentsStr, p->count, p->size);
    }

    //-------------------------------------------------------------------------
    // アルバムの全ストレージの状況を表示する
    nn::Result ShowAlbumSystemInfo(const ProgramOption& opts) NN_NOEXCEPT
    {
        NN_UNUSED(opts);

        NN_LOG(" Storage      ContentsType    FileCount        TotalSize\n");
        NN_LOG("---------+-----------------+------------+----------------\n");
        for (nn::capsrv::AlbumStorageType storage :
            {
                nn::capsrv::AlbumStorage_Nand,
                nn::capsrv::AlbumStorage_Sd
            })
        {
            // 最初にリフレッシュ
            NN_RESULT_DO(nn::capsrv::RefreshAlbumCache(storage));

            // ストレージ情報を取得
            nn::capsrv::AlbumFileContentsFlag mask = {};
            mask.Set();
            nn::capsrv::AlbumUsage albumUsage;
            NN_RESULT_DO(nn::capsrv::GetAlbumUsage(&albumUsage, storage, mask));

            // ストレージ情報の表示
            for (int i=0; i<nn::capsrv::AlbumFileContentsCount; ++i)
            {
                PrintAlbumContentsUsage(storage, i, albumUsage.GetContentsUsage(i));
            }
            PrintAlbumContentsUsage(storage, -1, albumUsage.GetUnknownUsage());
            NN_LOG("---------+-----------------+------------+----------------\n");
        }

        NN_LOG("\n");
        NN_RESULT_SUCCESS;
    }


#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
    #define NN_DMC_MOUNT_POINT_NAND "NAND:/Nintendo/Album/"
    #define NN_DMC_MOUNT_POINT_SD   "SD:/Nintendo/Album/"

    bool TryParseMountPoint(nn::capsrv::AlbumStorageType* pOutStorage, const char** pOutNext, const char* str, const ProgramOption& opts) NN_NOEXCEPT
    {
        // SD カードか？
        if (std::strncmp(str, NN_DMC_MOUNT_POINT_SD, sizeof(NN_DMC_MOUNT_POINT_SD) - 1) == 0)
        {
            if (!MountTargetStorage(nn::fs::ImageDirectoryId::SdCard))
            {
                NN_LOG("List files ... cannot mount SD-Card storage.\n");
                return false;
            }
            *pOutStorage = nn::capsrv::AlbumStorage_Sd;
            *pOutNext = str + sizeof(NN_DMC_MOUNT_POINT_SD) - 1;
            return true;
        }
        // NAND か？
        if (std::strncmp(str, NN_DMC_MOUNT_POINT_NAND, sizeof(NN_DMC_MOUNT_POINT_NAND) - 1) == 0)
        {
            if (!MountTargetStorage(nn::fs::ImageDirectoryId::Nand))
            {
                NN_LOG("List files ... cannot mount NAND storage.\n");
                return false;
            }
            *pOutStorage = nn::capsrv::AlbumStorage_Nand;
            *pOutNext = str + sizeof(NN_DMC_MOUNT_POINT_NAND) - 1;
            return true;
        }
        return false;
    }

    struct AlbumFileContentsInfo
    {
        const char* pSuffix;
        nn::capsrv::AlbumFileContents contents;
    };

    const AlbumFileContentsInfo albumFileContentsList[] =
    {
        {   "X.mp4",    nn::capsrv::AlbumFileContents_ExtraMovie        },
        {   "X.jpg",    nn::capsrv::AlbumFileContents_ExtraScreenShot   },
        {   ".mp4",     nn::capsrv::AlbumFileContents_Movie             },
        {   ".jpg",     nn::capsrv::AlbumFileContents_ScreenShot        },
    };
    NN_STATIC_ASSERT(sizeof(albumFileContentsList) / sizeof(AlbumFileContentsInfo) == nn::capsrv::AlbumFileContentsCount);

    bool ConvertToAlbumFileId(nn::capsrv::AlbumFileId* pOut, const char* pFilename, nn::capsrv::AlbumStorageType storage) NN_NOEXCEPT
    {
        nn::capsrv::AlbumFileId fileId = {};
        fileId.storage = storage;
        auto p = pFilename;
        auto pEnd = pFilename + std::strlen(pFilename);

        // "YYYYMMDDHHMMSSNN"
        if (!TryParseDateTime(&p, &fileId.time, p, pEnd - p))
        {
            return false;
        }
        // "-"
        if (*p++ != '-')
        {
            return false;
        }
        // "EncryptedApplicationId"
        nn::capsrv::server::detail::EncryptedApplicationId encryptedId;
        for (int i=0; i<sizeof(encryptedId); ++i)
        {
            if (!TryParseHexaByte(&p, &encryptedId.value[i], p, pEnd - p))
            {
                return false;
            }
        }
        bool isExtra;
        if (!nn::capsrv::server::detail::TryDecryptApplicationId(&fileId.applicationId, &isExtra, encryptedId).IsSuccess())
        {
            return false;
        }

        // Suffix
        for (int i=0; i<nn::capsrv::AlbumFileContentsCount; ++i)
        {
            auto& info = albumFileContentsList[i];
            if (std::strcmp(p, info.pSuffix) == 0)
            {
                fileId.contents = info.contents;
                *pOut = fileId;
                return true;
            }
        }
        return false;
    }


    //-------------------------------------------------------------------------
    // 指定された path のファイル情報を表示する
    bool ShowFileInfo(const char* path, const ProgramOption& opts) NN_NOEXCEPT
    {
        NN_UNUSED(opts);

        // マウントポイントのチェック
        nn::capsrv::AlbumStorageType storage;
        const char* pNext;
        if (!TryParseMountPoint(&storage, &pNext, path, opts))
        {
            NN_LOG("Album Information ... Invalid file rootpath.\n");
            return false;
        }

        // ファイル名のチェック
        auto pFilename = std::strrchr(path, '/');
        if (pFilename == nullptr)
        {
            NN_LOG("Album Information ... Invalid path.\n");
            return false;
        }
        ++pFilename;

        // ファイルが実在するかチェック
        nn::fs::FileHandle hFile = {};
        std::string filePath = std::string("TARG") + ":/" + pNext;
        if (!nn::fs::OpenFile(&hFile, filePath.c_str(), nn::fs::OpenMode_Read).IsSuccess())
        {
            NN_LOG("Album Information ... File not found.\n");
            return false;
        }
        nn::fs::CloseFile(hFile);

        // AlbumFileId の構築
        nn::capsrv::AlbumFileId fileId = {};
        if (!ConvertToAlbumFileId(&fileId, pFilename, storage))
        {
            NN_LOG("Album Information ... Invalid filename.\n");
            return false;
        }

        // ファイルの読込み
        uint64_t fileSize = 0;
        if (!nn::capsrv::LoadMakerNoteInfoForDebug(
            &fileSize,
            &g_MakerNoteInfo,
            sizeof(g_MakerNoteInfo),
            fileId,
            g_WorkBuffer,
            sizeof(g_WorkBuffer)
            ).IsSuccess())
        {
            NN_LOG("Album Information ... Failed to load album file.\n");
            return false;
        }

        // 情報の出力
        PrintScreenShotInfo(path, fileId, fileSize, g_MakerNoteInfo);
        return true;

    }   // NOLINT(impl/function_size)

#endif // defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)

}   // namespace

bool ExecuteInfoAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Album Information\n\n");

    switch (opts.GetArgumentCount())
    {
        case 0: // 引数なし
        {
            if (!ShowAlbumSystemInfo(opts).IsSuccess())
            {
                return false;
            }
            break;
        }
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        case 1: // 引数１つ（ファイル名）
        {
            auto path = opts.GetLastArgument();
            if (!ShowFileInfo(path, opts))
            {
                return false;
            }
            break;
        }
#endif
        default:
        {
            NN_LOG("Album Information ... Invalid argument(s)\n");
            return false;
        }
    }

    NN_LOG("Album Information ... completed.\n");
    return true;
}

#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

}   // namespace album
