﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/ec/system/ec_DeviceLinkApi.h>
#include <nn/ec/system/ec_NotificationApi.h>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/es.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/nifm.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/ns/ns_DeviceLinkApi.h>
#include <nn/ns/ns_TicketApi.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/nsd/nsd_ApiForNasService.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenuCommand_ErrorContextUtil.h"
#include "DevMenuCommand_HttpConnection.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ShopCommand.h"
#include "DevMenuCommand_ShopUtil.h"

using namespace nn;
using namespace nne;
using namespace devmenuUtil;

namespace
{
    util::optional<TimeSpan> GetCancelAfterTimeSpanOption(const Option& option)
    {
        const char CancelOption[] = "--cancel-after";
        return option.HasKey(CancelOption) ?
            util::optional<TimeSpan>(TimeSpan::FromMilliSeconds(std::strtoll(option.GetValue(CancelOption), nullptr, 10))) :
            util::nullopt;
    }

    template<typename AsyncResultT>
    Result Wait(AsyncResultT* async, const Option& option)
    {
        auto timeout = GetCancelAfterTimeSpanOption(option);
        if (timeout && !async->GetEvent().TimedWait(*timeout))
        {
            async->Cancel();
        }

        NN_RESULT_DO(Get(async));
        NN_RESULT_SUCCESS;
    }

    template<typename T>
    Result Wait(T* outValue, ec::system::AsyncValue<T>* async, const Option& option)
    {
        auto timeout = GetCancelAfterTimeSpanOption(option);
        if (timeout && !async->GetEvent().TimedWait(*timeout))
        {
            async->Cancel();
        }

        NN_RESULT_DO(Get(outValue, async));
        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " shop start <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop link-device <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop unlink-device <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop unlink-device-all\n"
        "       " DEVMENUCOMMAND_NAME " shop device-link-status <user_account_index>\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " shop start-without-link-device\n"
        "       " DEVMENUCOMMAND_NAME " shop register-device-account\n"
        "       " DEVMENUCOMMAND_NAME " shop unregister-device-account\n"
        "       " DEVMENUCOMMAND_NAME " shop transfer-device-account\n"
        "       " DEVMENUCOMMAND_NAME " shop sync-registration\n"
        "       " DEVMENUCOMMAND_NAME " shop device-registration-info\n"
        "       " DEVMENUCOMMAND_NAME " shop device-account-status\n"
        "       " DEVMENUCOMMAND_NAME " shop device-account-info\n"
        "       " DEVMENUCOMMAND_NAME " shop device-auth-token\n"
        "       " DEVMENUCOMMAND_NAME " shop download-ticket <rights_id>\n"
        "       " DEVMENUCOMMAND_NAME " shop download-ticket --application-id <application_id> [--key-id <key_id>]\n"
        "       " DEVMENUCOMMAND_NAME " shop shop-account-status <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop has-device-link <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop delete-all-rights <user_account_index>\n"
        "       " DEVMENUCOMMAND_NAME " shop delete-all-rights-and-ticket\n"
        "       " DEVMENUCOMMAND_NAME " shop prepurchase <user_account_index> --ns-uid <ns_uid> [--type <aocs|bundles|contents|demos|tickets|titles>]\n"
        "       " DEVMENUCOMMAND_NAME " shop purchase <user_account_index> --ns-uid <ns_uid> --price-id <price_id> --amount <amount> [--type <aocs|bundles|contents|demos|tickets|titles>]\n"
        "       " DEVMENUCOMMAND_NAME " shop purchase <user_account_index> --ns-uid <ns_uid> --auto [--type <aocs|bundles|contents|demos|tickets|titles>]\n"
        "       " DEVMENUCOMMAND_NAME " shop download-demo <user_account_index> --id <id>\n"
        "       " DEVMENUCOMMAND_NAME " shop subscribe-download-task-notification\n"
        "       " DEVMENUCOMMAND_NAME " shop sync-ticket\n"
        "       " DEVMENUCOMMAND_NAME " shop search [--aoc <nsuid>]\n"
        "       " DEVMENUCOMMAND_NAME " shop sync-rights <user_account_index>\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result Start(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Registering as active console of the account user index %d ...\n", userIndex);
        {
            DEVMENUCOMMANDSYSTEM_LOG("Request device authentication token...\n");
            ec::system::AsyncDeviceAuthenticationToken async;
            // デバイス認証トークンを取得する
            // 3時間以上の有効期限が残っているデバイス認証トークンをキャッシュしていたらキャッシュを利用する
            NN_RESULT_DO(ec::system::RequestCachedDeviceAuthenticationToken(&async, TimeSpan::FromHours(3)));
            ec::system::DeviceAuthenticationToken token;
            NN_RESULT_DO(Get(&token, &async));
            DEVMENUCOMMANDSYSTEM_LOG("    token: %s\n", token.data);
        }

        util::optional<ec::system::DeviceAccountInfo> deviceAccountInfo;
        {
            DEVMENUCOMMANDSYSTEM_LOG("Check device account is saved\n");
            deviceAccountInfo.emplace();;
            NN_RESULT_TRY(ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)))
                NN_RESULT_CATCH(ec::ResultDeviceAccountNotRegistered) {}
                {
                    deviceAccountInfo = util::nullopt;
                }
            NN_RESULT_END_TRY
        }

        if (!deviceAccountInfo)
        {
            DEVMENUCOMMANDSYSTEM_LOG("Device account is not saved\n");
            DEVMENUCOMMANDSYSTEM_LOG("Request device registration info...\n");

            util::optional<ec::system::DeviceRegistrationInfo> deviceRegistrationInfo;
            {
                ec::system::AsyncDeviceRegistrationInfo async;
                NN_RESULT_DO(ec::system::RequestDeviceRegistrationInfo(&async));
                deviceRegistrationInfo.emplace();
                NN_RESULT_TRY(async.Get(&(*deviceRegistrationInfo)))
                    NN_RESULT_CATCH(ec::ResultDeviceNotRegistered)
                    {
                        deviceRegistrationInfo = util::nullopt;
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        ReportErrorContext(&async, NN_RESULT_CURRENT_RESULT);
                    }
                NN_RESULT_END_TRY
            }

            if(deviceRegistrationInfo)
            {
                DEVMENUCOMMANDSYSTEM_LOG("    deviceAccountId: %s\n", deviceRegistrationInfo->accountInfo.id);
                DEVMENUCOMMANDSYSTEM_LOG("    deviceAccountStatus: %s\n", deviceRegistrationInfo->accountStatus.data);

                DEVMENUCOMMANDSYSTEM_LOG("Registered device is found even device account is not saved\n");
                DEVMENUCOMMANDSYSTEM_LOG("Seems this device is initialized\n");
                DEVMENUCOMMANDSYSTEM_LOG("Request unlink device all...\n");

                {
                    nn::ec::system::AsyncResult async;
                    NN_RESULT_DO(nn::ec::system::RequestUnlinkDeviceAll(&async));
                    NN_RESULT_DO(Wait(&async, option));
                }

                DEVMENUCOMMANDSYSTEM_LOG("Request unregister device account...\n");

                {
                    ec::system::AsyncResult async;
                    NN_RESULT_DO(ec::system::RequestUnregisterDeviceAccount(&async));
                    NN_RESULT_DO(Get(&async));
                }
            }

            {
                DEVMENUCOMMANDSYSTEM_LOG("Request register device account...\n");
                ec::system::AsyncResult async;
                NN_RESULT_DO(ec::system::RequestRegisterDeviceAccount(&async));
                NN_RESULT_DO(Get(&async));
            }

            {
                DEVMENUCOMMANDSYSTEM_LOG("Check device account is saved");
                deviceAccountInfo.emplace();
                NN_RESULT_DO(ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)));
            }
        }

        DEVMENUCOMMANDSYSTEM_LOG("    deviceId: %s\n", deviceAccountInfo->id);

        {
            DEVMENUCOMMANDSYSTEM_LOG("Check shop account status...\n");

            nn::ec::system::AsyncResult async;
            NN_RESULT_DO(nn::ec::system::RequestCreateVirtualAccount(&async, *uid));
            NN_RESULT_DO(Wait(&async, option));
        }

        {
            DEVMENUCOMMANDSYSTEM_LOG("Check device link status...\n");

            ec::system::DeviceLinkStatus status;
            {
                ec::system::AsyncDeviceLinkStatus async;
                NN_RESULT_DO(ec::system::RequestDeviceLinkStatus(&async, *uid));
                NN_RESULT_DO(Get(&status, &async));
            }

            if (!status.hasDeviceLink)
            {
                DEVMENUCOMMANDSYSTEM_LOG("No device is linked to this user\n");
                DEVMENUCOMMANDSYSTEM_LOG("Request link device...\n");

                ec::system::AsyncResult async;
                NN_RESULT_DO(ec::system::RequestLinkDevice(&async, *uid));
                NN_RESULT_DO(Get(&async));
            }
            else
            {
                DEVMENUCOMMANDSYSTEM_LOG("Device already linked to this user\n");
                DEVMENUCOMMANDSYSTEM_LOG("Skip link device\n");
            }
        }

        {
            DEVMENUCOMMANDSYSTEM_LOG("Check device account status...\n");

            ec::system::DeviceAccountStatus status;
            {
                Result result;
                err::ErrorContext errorContext = {};
                {
                    ec::system::AsyncDeviceAccountStatus async;
                    NN_RESULT_DO(ec::system::RequestDeviceAccountStatus(&async));
                    result = async.Get(&status);
                    if (result.IsFailure())
                    {
                        async.GetErrorContext(&errorContext);
                    }
                }

                NN_RESULT_TRY(result)
                    // デバイストークンが失効していた場合
                    NN_RESULT_CATCH(ec::ResultInvalidDeviceAccountToken)
                    {
                        DEVMENUCOMMANDSYSTEM_LOG("Device token is expired\n");
                        DEVMENUCOMMANDSYSTEM_LOG("Request synchronize registration...\n");
                        // デバイストークンを復旧する
                        {
                            ec::system::AsyncResult async;
                            NN_RESULT_DO(ec::system::RequestSyncRegistration(&async));
                            NN_RESULT_DO(Get(&async));
                        }

                        DEVMENUCOMMANDSYSTEM_LOG("Check device account status again...\n");
                        {
                            ec::system::AsyncDeviceAccountStatus async;
                            NN_RESULT_DO(ec::system::RequestDeviceAccountStatus(&async));
                            NN_RESULT_DO(Get(&status, &async));
                        }
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        std::unique_ptr<char[]> buffer(new char[1024]);
                        if( buffer )
                        {
                            nn::err::ReportError(devmenuUtil::GetSelfApplicationId(), NN_RESULT_CURRENT_RESULT, errorContext, buffer.get(), 1024);
                        }
                        else
                        {
                            NN_LOG("Failed to allocate work buffer to report error.\n");
                        }
                    }
                NN_RESULT_END_TRY
            }

            DEVMENUCOMMANDSYSTEM_LOG("    status: %s\n", status.data);
            DEVMENUCOMMANDSYSTEM_LOG("    needsTicketSync %s\n", status.needsTicketSync ? "true" : "false");

            if (status.needsTicketSync)
            {
                DEVMENUCOMMANDSYSTEM_LOG("This device account needs to sync ticket\n");
                DEVMENUCOMMANDSYSTEM_LOG("Request sync ticket...\n");
                ec::system::AsyncProgressResult async;
                NN_RESULT_DO(ec::system::RequestSyncTicket(&async));
                NN_RESULT_DO(Get(&async));
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result LinkDevice(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Registering as active console of the account user index %d ...\n", userIndex);
        DEVMENUCOMMANDSYSTEM_LOG("Request link device for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);
        nn::ec::system::AsyncResult async;
        NN_RESULT_DO(nn::ec::system::RequestLinkDevice(&async, *uid));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnlinkDevice(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (nn::ns::IsAnyApplicationRunning())
        {
            DEVMENUCOMMAND_LOG("To unlink console, please close application first.\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Unregistering as active console of the account user index %d ...\n", userIndex);
        DEVMENUCOMMANDSYSTEM_LOG("Request unlink device for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);
        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestUnlinkDevice(&async, *uid));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnlinkDeviceAll(bool* outValue, const Option& option)
    {
        if (nn::ns::IsAnyApplicationRunning())
        {
            DEVMENUCOMMAND_LOG("To unlink console, please close application first.\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::ec::system::AsyncResult async;
        NN_RESULT_DO(nn::ec::system::RequestUnlinkDeviceAll(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisplayLinkedDeviceStatus(bool* outValue, const Option& option)
    {
        char messageStatus[128];
        NN_RESULT_DO(DisplayLinkedDeviceStatus(outValue, messageStatus, sizeof(messageStatus), option));
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM || defined NN_DEVMENULOTCHECK_DOWNLOADER
    Result StartWithoutLinkDevice(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // check device account already exists
        util::optional<ec::system::DeviceAccountInfo> deviceAccountInfo;
        {
            deviceAccountInfo.emplace();
            NN_RESULT_TRY(ec::system::GetDeviceAccountInfo(&(*deviceAccountInfo)))
                NN_RESULT_CATCH(ec::ResultDeviceAccountNotRegistered) {}
                {
                    deviceAccountInfo = util::nullopt;
                }
            NN_RESULT_END_TRY
        }
        if (deviceAccountInfo)
        {
            DEVMENUCOMMANDSYSTEM_LOG("Account ID: %s\n", deviceAccountInfo->id);
        }
        else
        {
            // clean up device registration
            util::optional<ec::system::DeviceRegistrationInfo> deviceRegistrationInfo;
            {
                ec::system::AsyncDeviceRegistrationInfo async;
                NN_RESULT_DO(ec::system::RequestDeviceRegistrationInfo(&async));
                deviceRegistrationInfo.emplace();
                NN_RESULT_TRY(async.Get(&(*deviceRegistrationInfo)))
                    NN_RESULT_CATCH(ec::ResultDeviceNotRegistered)
                    {
                        deviceRegistrationInfo = util::nullopt;
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        ReportErrorContext(&async, NN_RESULT_CURRENT_RESULT);
                    }
                NN_RESULT_END_TRY
            }
            if(deviceRegistrationInfo)
            {
                {
                    ec::system::AsyncResult async;
                    NN_RESULT_DO(nn::ec::system::RequestUnlinkDeviceAll(&async));
                    NN_RESULT_DO(Wait(&async, option));
                }

                {
                    ec::system::AsyncResult async;
                    NN_RESULT_DO(ec::system::RequestUnregisterDeviceAccount(&async));
                    NN_RESULT_DO(Get(&async));
                }
            }

            // register device account
            ec::system::AsyncResult async;
            NN_RESULT_DO(ec::system::RequestRegisterDeviceAccount(&async));
            NN_RESULT_DO(Wait(&async, option));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RegisterDeviceAccount(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestRegisterDeviceAccount(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnregisterDeviceAccount(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestUnregisterDeviceAccount(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TransferDeviceAccount(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            NN_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestTransferDeviceAccount(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SyncRegistration(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            NN_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestSyncRegistration(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeviceRegistrationInfo(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncDeviceRegistrationInfo async;
        NN_RESULT_DO(ec::system::RequestDeviceRegistrationInfo(&async));

        ec::system::DeviceRegistrationInfo regInfo;
        NN_RESULT_TRY(Wait(&regInfo, &async, option))
            NN_RESULT_CATCH(ec::ResultDeviceNotRegistered)
            {
                DEVMENUCOMMAND_LOG("Device not registered\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        DEVMENUCOMMAND_LOG("Account ID: %s\n", regInfo.accountInfo.id);
        DEVMENUCOMMAND_LOG("Account status: %s\n", regInfo.accountStatus.data);
        DEVMENUCOMMAND_LOG("Device token expired: %s\n", regInfo.accountStatus.isDeviceTokenExpired ? "true" : "false");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeviceAccountStatus(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncDeviceAccountStatus asyncStatus;
        NN_RESULT_DO(ec::system::RequestDeviceAccountStatus(&asyncStatus));

        ec::system::DeviceAccountStatus status;
        NN_RESULT_DO(Wait(&status, &asyncStatus, option));

        DEVMENUCOMMAND_LOG("%s\n", status.data);
        if (status.needsTicketSync)
        {
            DEVMENUCOMMAND_LOG("Needs ticket sync\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeviceAccountInfo(bool* outValue, const Option&)
    {
        ec::system::DeviceAccountInfo info;
        NN_RESULT_TRY(ec::system::GetDeviceAccountInfo(&info))
            NN_RESULT_CATCH(ec::ResultDeviceAccountNotRegistered)
            {
                DEVMENUCOMMAND_LOG("Device account is not registered\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        DEVMENUCOMMAND_LOG("Account ID: %s\n", info.id);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeviceAuthenticationToken(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ec::system::AsyncDeviceAuthenticationToken async;
        NN_RESULT_DO(ec::system::RequestDeviceAuthenticationToken(&async));

        ec::system::DeviceAuthenticationToken token;
        NN_RESULT_DO(Wait(&token, &async, option));

        DEVMENUCOMMAND_LOG("%s\n", token.data);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result NaIdToken(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NintendoAccountAuthorizationCode authCode;
        NintendoAccountIdToken idToken;
        NN_RESULT_DO(GetNintendoAccountAuthorization(&authCode, &idToken, *uid));

        DEVMENUCOMMAND_LOG("%s\n", idToken.data);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DownloadTicket(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        std::string rightsIdString;

        if (option.HasKey("--application-id"))
        {
            if (option.HasKey("--key-id"))
            {
                rightsIdString = std::string(option.GetValue("--application-id")) + option.GetValue("--key-id");
            }
            else
            {
                const std::string DefaultKeyId = "0000000000000000";
                rightsIdString = std::string(option.GetValue("--application-id")) + DefaultKeyId;
            }
        }
        else
        {
            rightsIdString = option.GetTarget();
            if (rightsIdString.length() != sizeof(es::RightsIdIncludingKeyId) * 2)
            {
                DEVMENUCOMMAND_LOG("Rights ID requires 32 hex characters %s\n", rightsIdString.c_str());
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        es::RightsIdIncludingKeyId rightsId = {};
        for (int i = 0; i < sizeof(es::RightsIdIncludingKeyId); i++)
        {
            rightsId._data[i] = static_cast<uint8_t>(std::stoi(rightsIdString.substr(i * 2, 2), 0, 16));
        }

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestDownloadTicket(&async, rightsId));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ShopAccountStatus(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Request shop account status for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

        char readBuffer[2048];
        auto result = GetShopAccountStatusResponse(readBuffer, sizeof(readBuffer), *uid);
        DEVMENUCOMMAND_LOG("%s\n", readBuffer);
        NN_RESULT_DO(result);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result HasDeviceLink(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));
        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        rapidjson::Document document(ec::system::HasDeviceLink(*uid) ? rapidjson::kTrueType : rapidjson::kFalseType);

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteAllRights(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Request delete all rights for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

        char readBuffer[2048];
        auto result = GetDeleteAllRightsResponse(readBuffer, sizeof(readBuffer), *uid);
        DEVMENUCOMMAND_LOG("%s\n", readBuffer);
        NN_RESULT_DO(result);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteAllRightsAndTicket(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        for (int i = 0; i < account::UserCountMax; i++)
        {
            util::optional<account::Uid> uid;
            NN_RESULT_DO(GetUidFromIndex(&uid, i));

            if (!uid || !ec::system::HasDeviceLink(*uid))
            {
                continue;
            }

            DEVMENUCOMMAND_LOG("Request delete all rights for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

            char readBuffer[2048];
            auto result = GetDeleteAllRightsResponse(readBuffer, sizeof(readBuffer), *uid);
            DEVMENUCOMMAND_LOG("%s\n", readBuffer);
            NN_RESULT_DO(result);
        }

        DEVMENUCOMMAND_LOG("Delete all ticket.\n");
        es::DeleteAllCommonTicket();
        es::DeleteAllPersonalizedTicket();
        es::DeleteAllPrepurchaseRecord();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Prepurchase(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto nsuid = std::strtoll(option.GetValue("--ns-uid"), nullptr, 10);
        DEVMENUCOMMAND_LOG("Request prepurchase ns-uid %lld for uid %016llx-%016llx\n", nsuid, uid->_data[0], uid->_data[1]);

        auto typeOption = option.GetValue("--type");
        char readBuffer[2048];
        auto result = GetPrepurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsuid, typeOption ? typeOption : "titles");
        DEVMENUCOMMAND_LOG("%s\n", readBuffer);
        NN_RESULT_DO(result);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool ParsePrepurchaseResponse(int64_t* outPriceId, int64_t* outAmount, char* buffer)
    {
        rapidjson::Document document;
        if (document.ParseInsitu(buffer).HasParseError())
        {
            return false;
        }

        auto contentsObj = document.FindMember("contents");
        if (contentsObj == document.MemberEnd())
        {
            return false;
        }
        auto& item = contentsObj->value[0];
        auto paymentObj = item.FindMember("payment_amount");
        if (paymentObj == item.MemberEnd())
        {
            return false;
        }
        auto priceObj = paymentObj->value.FindMember("price");
        if (priceObj == paymentObj->value.MemberEnd())
        {
            return false;
        }
        auto regularPriceObj = priceObj->value.FindMember("regular_price");
        if (regularPriceObj == priceObj->value.MemberEnd())
        {
            return false;
        }
        auto priceIdObj = regularPriceObj->value.FindMember("id");
        if (priceIdObj == regularPriceObj->value.MemberEnd())
        {
            return false;
        }
        auto amountObj = regularPriceObj->value.FindMember("raw_value");
        if (amountObj == regularPriceObj->value.MemberEnd())
        {
            return false;
        }
        if (!priceIdObj->value.IsInt64() || !amountObj->value.IsString())
        {
            return false;
        }

        *outPriceId = priceIdObj->value.GetInt64();
        *outAmount = std::strtoll(amountObj->value.GetString(), nullptr, 10);

        return true;
    }

    Result Purchase(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto typeOption = option.GetValue("--type");
        auto nsUid = std::strtoll(option.GetValue("--ns-uid"), nullptr, 10);

        int64_t priceId;
        int64_t amount;
        char readBuffer[2048];

        if (option.HasKey("--auto"))
        {
            auto result = GetPrepurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsUid, typeOption ? typeOption : "titles");
            DEVMENUCOMMAND_LOG("%s\n", readBuffer);
            NN_RESULT_DO(result);

            if (!ParsePrepurchaseResponse(&priceId, &amount, readBuffer))
            {
                DEVMENUCOMMAND_LOG("Unexpected prepurchase response\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            priceId = std::strtoll(option.GetValue("--price-id"), nullptr, 10);
            amount = std::strtoll(option.GetValue("--amount"), nullptr, 10);
        }

        DEVMENUCOMMAND_LOG("Request purchase ns-uid %lld price-id %lld amount %lld for uid %016llx-%016llx\n", nsUid, priceId, amount, uid->_data[0], uid->_data[1]);
        auto result = GetPurchaseResponse(readBuffer, sizeof(readBuffer), *uid, nsUid, priceId, amount, typeOption ? typeOption : "titles");
        DEVMENUCOMMAND_LOG("%s\n", readBuffer);
        NN_RESULT_DO(result);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DownloadDemo(bool* outValue, const Option& option)
    {
        auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
        util::optional<account::Uid> uid;
        NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

        if (!uid)
        {
            DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto idString = option.GetValue("--id");
        if (!idString)
        {
            DEVMENUCOMMAND_LOG("--id option is necessary.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto metaId = std::strtoull(idString, nullptr, 16);
        int64_t nsuid;
        char readBuffer[2048];

        NN_RESULT_DO(GetNsuidByContentMetaId(
            &nsuid, readBuffer, sizeof(readBuffer), metaId, ShopIdType::Demo));

        auto result = GetDownloadDemoResponse(readBuffer, sizeof(readBuffer), *uid, nsuid);
        DEVMENUCOMMAND_LOG("%s\n", readBuffer);
        NN_RESULT_DO(result);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SubscribeDownloadTaskNotification(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // TODO: とりあえず ns の代わりに作成・登録
        ApplicationId appId = { 0x010000000000001f };
        npns::NotificationToken token;
        account::Uid uid = {};
        NN_RESULT_DO(npns::CreateToken(&token, uid, appId));

        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestRegisterNotificationToken(&async, token));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SyncTicket(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto timeout = GetCancelAfterTimeSpanOption(option);
        os::TimerEvent timer(os::EventClearMode_ManualClear);
        if (timeout)
        {
            timer.StartOneShot(*timeout);
        }

        ec::system::AsyncProgressResult asyncProgressResult;
        NN_RESULT_DO(ec::system::RequestSyncTicket(&asyncProgressResult));

        while (!asyncProgressResult.TryWait())
        {
            if (timer.TryWait())
            {
                asyncProgressResult.Cancel();
                break;
            }

            auto progress = asyncProgressResult.GetProgress();
            DEVMENUCOMMAND_LOG("%lld / %lld\n", progress.done, progress.total);
            asyncProgressResult.GetEvent().TimedWait(TimeSpan::FromSeconds(1));
        }
        NN_RESULT_DO(Get(&asyncProgressResult));
        auto progress = asyncProgressResult.GetProgress();
        DEVMENUCOMMAND_LOG("%lld / %lld\n", progress.done, progress.total);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }


    Result Search(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        static const int BufferSize = 4096 * 1024;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);

        auto aocOption = option.GetValue("--aoc");
        if (aocOption)
        {
            auto nsuid = std::strtoll(aocOption, nullptr, 10);
            NN_RESULT_DO(GetSearchAddOnContentResponse(buffer.get(), BufferSize, nsuid));
        }
        else
        {
            NN_RESULT_DO(GetSearchResponse(buffer.get(), BufferSize));
        }

        nne::rapidjson::Document document;
        if (document.ParseInsitu(buffer.get()).HasParseError())
        {
            DEVMENUCOMMAND_LOG("Invalid download task list json\n");
        }

        rapidjson::StringBuffer stringBuffer;
        rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(stringBuffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", stringBuffer.GetString());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SyncRights(bool* outValue, const Option& option)
    {
        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::ns::AsyncResult async;
        NN_RESULT_DO(nn::ns::RequestSyncRights(&async));
        NN_RESULT_DO(Wait(&async, option));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const SubCommand g_SubCommands[] =
    {
        { "start", Start },
        { "link-device", LinkDevice },
        { "unlink-device", UnlinkDevice },
        { "unlink-device-all", UnlinkDeviceAll },
        { "device-link-status", DisplayLinkedDeviceStatus },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "start-without-link-device", StartWithoutLinkDevice },
        { "register-device-account", RegisterDeviceAccount },
        { "unregister-device-account", UnregisterDeviceAccount },
        { "transfer-device-account", TransferDeviceAccount },
        { "sync-registration", SyncRegistration },
        { "device-registration-info", DeviceRegistrationInfo },
        { "device-account-status", DeviceAccountStatus },
        { "device-account-info", DeviceAccountInfo },
        { "device-auth-token", DeviceAuthenticationToken },
        { "na-id-token", NaIdToken },
        { "download-ticket", DownloadTicket },
        { "shop-account-status", ShopAccountStatus },
        { "has-device-link", HasDeviceLink },
        { "delete-all-rights", DeleteAllRights },
        { "delete-all-rights-and-ticket", DeleteAllRightsAndTicket },
        { "prepurchase", Prepurchase },
        { "purchase", Purchase },
        { "download-demo", DownloadDemo },
        { "subscribe-download-task-notification", SubscribeDownloadTaskNotification },
        { "sync-ticket", SyncTicket },
        { "search", Search },
        { "sync-rights", SyncRights },
#endif
    };
}

Result DisplayLinkedDeviceStatus(bool* outValue, char* outMessage, size_t messageSize, const Option& option)
{
    auto userIndex = std::strtol(option.GetTarget(), nullptr, 10);
    util::optional<account::Uid> uid;
    NN_RESULT_DO(GetUidFromIndex(&uid, userIndex));

    if (!uid)
    {
        DEVMENUCOMMAND_LOG("Not found user index %d\n", userIndex);

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    nifm::SubmitNetworkRequestAndWait();
    if (!nifm::IsNetworkAvailable())
    {
        DEVMENUCOMMAND_LOG("Network is not available.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    DEVMENUCOMMANDSYSTEM_LOG("Request console link status for uid %016llx-%016llx\n", uid->_data[0], uid->_data[1]);

    ec::system::DeviceLinkStatus status;
    ec::system::AsyncDeviceLinkStatus async;
    NN_RESULT_DO(ec::system::RequestDeviceLinkStatus(&async, *uid));
    NN_RESULT_DO(Get(&status, &async));


    if (status.hasDeviceLink)
    {
        // このデバイスが機器認証されていた場合
        if (ec::system::IsOwnDeviceId(status.linkedDeviceId))
        {
            nn::util::SNPrintf(outMessage, messageSize, "This console is registered as an active console of the account user index %d\n", userIndex);
        }
        // 他のデバイスが機器認証されていた場合
        else
        {
            nn::util::SNPrintf(outMessage, messageSize, "Another console is registered as an active console of the account user index %d\n", userIndex);
        }
    }
    else
    {
        nn::util::SNPrintf(outMessage, messageSize, "Unregistered\n");
    }
    DEVMENUCOMMAND_LOG(outMessage);

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ShopCommand(bool* outValue, const Option& option)
{
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
    NN_UNUSED(HelpMessage);
#endif
#if !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
    account::InitializeForAdministrator();
    NN_ABORT_UNLESS_RESULT_SUCCESS(npns::InitializeForSystem());
#endif

    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a " DEVMENUCOMMAND_NAME " shop command. See '" DEVMENUCOMMAND_NAME " shop --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
