﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_SdCardCommand.h"

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SdCardSystemApi.h>

using namespace nn;

namespace
{
    Result Status(bool* outValue, const Option&) NN_NOEXCEPT
    {
        auto result = nn::ns::CheckSdCardMountStatus();

        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::ns::ResultSdCardDatabaseCorrupted) { NN_LOG("Database of sdcard is corrupted. Please cleanup sdcard and reboot the system.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardFileSystemCorrupted) { NN_LOG("FileSystem of sdcard is corrupted. Please format sdcard on PC.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardNeedsSystemUpdate) { NN_LOG("Sdcard is formated with exFAT. Please update the sytem.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardNoOwnership) { NN_LOG("SdCard is initialized on the diffrent machine. Please cleanup sdcard and reboot the system.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardNotInserted) { NN_LOG("Sdcard is not inserted.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardNotMounted) { NN_LOG("Sdcard can be used, if the system reboots.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardNotEnoughSpace) { NN_LOG("There is not enough space in sdcard.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardAccessFailed) { NN_LOG("Sdcard cannot be accessed.\n"); }
            NN_RESULT_CATCH(nn::ns::ResultSdCardMountUnexpected) { NN_LOG("Unexpected error occurs while mounting sdcard. result=0x%08x\n", nn::ns::GetLastSdCardMountUnexpectedResult().GetInnerValueForDebug()); }
        NN_RESULT_END_TRY

        if (result.IsSuccess())
        {
            NN_LOG("Sdcard can be used.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Cleanup(bool* outValue, const Option&) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::ns::CleanupSdCard());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ShowFormatUnexpectedResult(Result result) NN_NOEXCEPT
    {
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::ns::ResultSdCardFormatUnexpected)
            {
                auto detailResult = nn::ns::GetLastSdCardFormatUnexpectedResult();
                NN_LOG("Unexpected error: 0x%08x\n", detailResult.GetInnerValueForDebug());
                NN_RESULT_THROW(detailResult);
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result Format(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        if (option.HasKey("--dry-run"))
        {
            bool needsUpdate;
            NN_RESULT_DO(ShowFormatUnexpectedResult(nn::ns::NeedsSystemUpdateToFormatSdCard(&needsUpdate)));
            if (needsUpdate)
            {
                NN_LOG("Needs system update to format sdcard\n");
            }
        }
        else
        {
            NN_RESULT_DO(ShowFormatUnexpectedResult(nn::ns::FormatSdCard()));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // 既存のコマンドを残すため、エイリアスとする
    Result Detach(bool* outValue, const Option&) NN_NOEXCEPT
    {
        nn::ns::RemoveSdCard();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result Insert(bool* outValue, const Option&) NN_NOEXCEPT
    {
        nn::ns::InsertSdCard();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Remove(bool* outValue, const Option&) NN_NOEXCEPT
    {
        nn::ns::RemoveSdCard();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " sdcard status\n"
        "usage: " DEVMENUCOMMAND_NAME " sdcard cleanup\n"
        "usage: " DEVMENUCOMMAND_NAME " sdcard format [--dry-run]\n"
        "usage: " DEVMENUCOMMAND_NAME " sdcard detach\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "usage: " DEVMENUCOMMAND_NAME " sdcard insert\n"
        "usage: " DEVMENUCOMMAND_NAME " sdcard remove\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "status", Status},
        { "cleanup", Cleanup},
        { "format", Format},
        { "detach", Detach },
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "insert", Insert },
        { "remove", Remove },
#endif
    };
}

Result SdCardCommand(bool* outValue, const Option& option) NN_NOEXCEPT
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu sdcard command. See '" DEVMENUCOMMAND_NAME " sdcard --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
