﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/rid.h>
#include <nn/time.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_DeviceLinkApi.h>
#include <nn/fs/fs_Debug.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Drm.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_StandardSteadyClock.h>
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_RidCommand.h"

using namespace nn;

#if defined (NN_BUILD_CONFIG_OS_HORIZON)
namespace
{
    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " rid setup [--region <ncl|noa|noe>]\n"
        "       " DEVMENUCOMMAND_NAME " rid systemupdate [--timeout <s>]\n"
        "       " DEVMENUCOMMAND_NAME " rid menuupdate [--timeout <s>]\n"
        "       " DEVMENUCOMMAND_NAME " rid applicationupdate [--timeout <s>]\n"
        "       " DEVMENUCOMMAND_NAME " rid controllerupdate [--timeout <s>]\n"
        "       " DEVMENUCOMMAND_NAME " rid prepare\n"
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    // 試遊台メニューの ApplicationId
    const ncm::ApplicationId RidMenuId = { NN_TOOL_DEVMENUCOMMAND_PROGRAM_ID_OF_RIDMENU };

    size_t GetSampleUserAccountImage(void* outBuffer, size_t bufferSize)
    {
        static const char DefaultUserIconFilePath[] = "Contents:/accounts/defaultIcon.jpg";

        // ROMマウント
        size_t mountRomCacheBufferSize = 0;
        NN_RESULT_DO(fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
        std::unique_ptr<uint8_t> mountRomBuffer(new uint8_t[mountRomCacheBufferSize]);
        NN_RESULT_DO(fs::MountRom("Contents", mountRomBuffer.get(), mountRomCacheBufferSize));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount("Contents"); };

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, DefaultUserIconFilePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t size;
        NN_RESULT_DO(fs::GetFileSize(&size, file));
        NN_SDK_ASSERT(static_cast<size_t>(size) <= bufferSize);
        NN_RESULT_DO(fs::ReadFile(file, 0, outBuffer, static_cast<size_t>(size)));

        return static_cast<size_t>(size);
    }

    Result Setup(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        enum Region
        {
            Region_Ncl,
            Region_Noa,
            Region_Noe,
        } region;

        bool hasRegion = option.HasKey("--region");
        if (hasRegion)
        {
            if (util::Strncmp(option.GetValue("--region"), "ncl", 3) == 0)
            {
                region = Region_Ncl;
            }
            else if (util::Strncmp(option.GetValue("--region"), "noa", 3) == 0)
            {
                region = Region_Noa;
            }
            else if (util::Strncmp(option.GetValue("--region"), "noe", 3) == 0)
            {
                region = Region_Noe;
            }
            else
            {
                DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " rid setup [--region <ncl|noa|noe>]\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // DevMenuCommandSystem が持っている権限の関係上、time::InitializeForMenu() を呼ぶ
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ NN_ABORT_UNLESS_RESULT_SUCCESS(time::Finalize()); };

        // ユーザアカウントの作成
        account::InitializeForAdministrator();
        int userCount;
        NN_RESULT_DO(account::GetUserCount(&userCount));

        for (int i = 0; i < account::UserCountMax - userCount; i++)
        {
            account::Nickname nickname = { "User" };
            std::unique_ptr<uint8_t> image(new uint8_t[128 * 1024]);
            size_t imageSize = GetSampleUserAccountImage(image.get(), 128 * 1024);

            NN_RESULT_DO(ns::CreateUserAccount(nickname, image.get(), imageSize));
        }

        // デバイスアカウントの作成
        {
            // 機器認証解除しておく
            ec::system::AsyncResult async;
            NN_RESULT_DO(ec::system::RequestUnlinkDeviceAll(&async));
            NN_RESULT_DO(async.Get());
        }
        {
            // デバイス登録解除しておく
            ec::system::AsyncResult async;
            NN_RESULT_DO(ec::system::RequestUnregisterDeviceAccount(&async));
            NN_RESULT_DO(async.Get());
        }
        {
            ec::system::AsyncResult async;
            NN_RESULT_DO(ec::system::RequestRegisterDeviceAccount(&async));
            NN_RESULT_DO(async.Get());
        }

        // シリアル番号取得
        settings::system::SerialNumber serial;
        settings::system::GetSerialNumber(&serial);
        DEVMENUCOMMAND_LOG("SerialNumber: %s\n", serial);

        // 自動スリープ OFF
        settings::system::SleepFlagSet flag = {};
        settings::system::SleepSettings settings = { flag, nn::settings::system::HandheldSleepPlan_Never, nn::settings::system::ConsoleSleepPlan_Never };
        settings::system::SetSleepSettings(settings);

        // 画面焼け軽減 OFF
        settings::system::TvSettings tvSettings;
        settings::system::GetTvSettings(&tvSettings);
        tvSettings.flags.Reset<settings::system::TvFlag::PreventsScreenBurnIn>();
        settings::system::SetTvSettings(tvSettings);

        //ソフトの自動更新 OFF
        settings::system::SetAutoUpdateEnabled(false);

        //初回起動処理
        settings::system::InitialLaunchSettings value;
        settings::system::GetInitialLaunchSettings(&value);

        if (!value.flags.Test<settings::system::InitialLaunchFlag::HasTimeStamp>())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&value.timeStamp));
            value.flags.Set<settings::system::InitialLaunchFlag::HasTimeStamp>();
        }
        value.flags.Set<settings::system::InitialLaunchFlag::IsCompleted>();
        settings::system::SetInitialLaunchSettings(value);

        // リージョン・言語・タイムゾーン設定
        if (hasRegion)
        {
            time::LocationName name = {};
            switch (region)
            {
            case Region_Ncl:
                settings::system::SetRegionCode(nn::settings::system::RegionCode_Japan);
                settings::system::SetLanguageCode(settings::LanguageCode::Make(settings::Language_Japanese));
                util::SNPrintf(name._value, sizeof(name._value), "Asia/Tokyo");
                time::SetDeviceLocationName(name);
                break;
            case Region_Noa:
                settings::system::SetRegionCode(nn::settings::system::RegionCode_Usa);
                settings::system::SetLanguageCode(settings::LanguageCode::Make(settings::Language_AmericanEnglish));
                util::SNPrintf(name._value, sizeof(name._value), "America/Los_Angeles");
                time::SetDeviceLocationName(name);
                break;
            case Region_Noe:
                settings::system::SetRegionCode(nn::settings::system::RegionCode_Europe);
                settings::system::SetLanguageCode(settings::LanguageCode::Make(settings::Language_BritishEnglish));
                util::SNPrintf(name._value, sizeof(name._value), "Europe/Berlin");
                time::SetDeviceLocationName(name);
                break;
            default:
                break;
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    struct SystemUpdater
    {
        rid::SystemUpdater updater;
        Result result;
        bool isFinished = false;
    };

    const char* ToString(rid::SystemUpdateProgress::State state)
    {
        switch (state)
        {
        case rid::SystemUpdateProgress::State::Applying:
            return "Applying";
        case rid::SystemUpdateProgress::State::Checking:
            return "Checking";
        case rid::SystemUpdateProgress::State::Completed:
            return "Completed";
        case rid::SystemUpdateProgress::State::DoNothing:
            return "DoNothing";
        case rid::SystemUpdateProgress::State::Downloading:
            return "Downloading";
        case rid::SystemUpdateProgress::State::Failed:
            return "Failed";
        case rid::SystemUpdateProgress::State::NeedNoUpdate:
            return "NeedNoUpdate";
        case rid::SystemUpdateProgress::State::Cancelled:
            return "Cancelled";
        default:
            break;
        }
    }

    Result SystemUpdate(bool* outValue, const Option& option)
    {
        os::TimerEvent timer(os::EventClearMode_AutoClear);
        auto timeoutSeconds = option.GetValue("--timeout");
        if (timeoutSeconds)
        {
            timer.StartOneShot(TimeSpan::FromSeconds(std::strtoull(timeoutSeconds, nullptr, 10)));
        }

        SystemUpdater updater;

        os::ThreadType thread;
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&thread, [](void * p)
        {
            SystemUpdater* updater = reinterpret_cast<SystemUpdater*>(p);
            updater->result = updater->updater.Execute();
            updater->isFinished = true;
        }
        , &updater, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority + 1));

        os::StartThread(&thread);

        auto PrintProgress = [&]() {
            rid::SystemUpdateProgress progress = updater.updater.GetProgress();
            if (progress.state == rid::SystemUpdateProgress::State::Downloading)
            {
                DEVMENUCOMMAND_LOG("[SystemUpdate] Downloading: %lld / %lld\n", progress.loaded, progress.total);
            }
            else
            {
                DEVMENUCOMMAND_LOG("[SystemUpdate] %s\n", ToString(progress.state));
            };
        };

        while (!updater.isFinished)
        {
            if (timer.TryWait())
            {
                updater.updater.Cancel();
                DEVMENUCOMMAND_LOG("[SystemUpdate] Timeout. Trigging Cancel System Update.\n");
            }

            PrintProgress();
            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        PrintProgress();
        if (updater.result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("[SystemUpdate] SystemUpdate has been succeeded. Last state is \"%s\".\n", ToString(updater.updater.GetProgress().state));
        }
        else
        {
            DEVMENUCOMMAND_LOG("[SystemUpdate] SystemUpdate is failed as result 0x%08x. Last state is \"%s\".\n", updater.result.GetInnerValueForDebug(), ToString(updater.updater.GetProgress().state));
        }

        os::WaitThread(&thread);
        os::DestroyThread(&thread);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    struct MenuUpdater
    {
        rid::MenuUpdater updater;
        Result result;
        bool isFinished = false;
    };

    const char* ToString(rid::MenuUpdateState state)
    {
        switch (state)
        {
        case rid::MenuUpdateState::Completed:
            return "Completed";
        case rid::MenuUpdateState::DoNothing:
            return "DoNotihing";
        case rid::MenuUpdateState::Downloading:
            return "Downloading";
        case rid::MenuUpdateState::Failed:
            return "Failed";
        case rid::MenuUpdateState::NeedNoUpdate:
            return "NeedNoUpdate";
        case rid::MenuUpdateState::Cancelled:
            return "Cancelled";
        default:
            break;
        }
    }

    Result MenuUpdate(bool* outValue, const Option& option)
    {
        os::TimerEvent timer(os::EventClearMode_AutoClear);
        auto timeoutSeconds = option.GetValue("--timeout");
        if (timeoutSeconds)
        {
            timer.StartOneShot(TimeSpan::FromSeconds(std::strtoull(timeoutSeconds, nullptr, 10)));
        }

        MenuUpdater updater;

        os::ThreadType thread;
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&thread, [](void * p)
        {
            MenuUpdater* updater = reinterpret_cast<MenuUpdater*>(p);
            updater->result = updater->updater.Execute();
            updater->isFinished = true;
        }
        , &updater, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority + 1));

        os::StartThread(&thread);

        auto PrintProgress = [&]() {
            rid::MenuUpdateState state = updater.updater.GetState();
            DEVMENUCOMMAND_LOG("[MenuUpdate] %s\n", ToString(state));
        };

        while (!updater.isFinished)
        {
            if (timer.TryWait())
            {
                updater.updater.Cancel();
                DEVMENUCOMMAND_LOG("[MenuUpdate] Timeout. Trigging Cancel Menu Update.\n");
            }

            PrintProgress();
            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        PrintProgress();
        if (updater.result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("[MenuUpdate] MenuUpdate has been succeeded. Last state is \"%s\".\n", ToString(updater.updater.GetState()));
        }
        else
        {
            DEVMENUCOMMAND_LOG("[MenuUpdate] MenuUpdate is failed as result 0x%08x. Last state is \"%s\".\n", updater.result.GetInnerValueForDebug(), ToString(updater.updater.GetState()));
        }

        os::WaitThread(&thread);
        os::DestroyThread(&thread);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    bool ReadUpdateApplicationListFile(char* outBuffer, size_t bufferSize, const char* fileName)
    {
        fs::FileHandle handle;
        NN_RESULT_TRY(fs::OpenFile(&handle, fileName, fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] Target file is not found.\n");
                return false;
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        int64_t size;
        NN_RESULT_DO(fs::GetFileSize(&size, handle));
        NN_SDK_ASSERT(static_cast<size_t>(size) <= bufferSize);
        NN_RESULT_DO(fs::ReadFile(handle, 0, outBuffer, static_cast<size_t>(size)));
        outBuffer[size] = '\0';

        return true;
    }

    bool ReadUpdateApplicationListFileFromSd(char* outBuffer, size_t bufferSize)
    {
        NN_RESULT_TRY(fs::MountSdCardForDebug("sd"))
            NN_RESULT_CATCH(fs::ResultSdCardAccessFailed)
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] SD card is not found. Please insert SD card and set /NintendoSDK/RetailInteractiveDisplay/ContentList.json in SD card.\n");
                return false;
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::Unmount("sd"); };

        fs::FileHandle handle;
        NN_RESULT_TRY(fs::OpenFile(&handle, "sd:/NintendoSDK/RetailInteractiveDisplay/ContentList.json", fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] /NintendoSDK/RetailInteractiveDisplay/ContentList.json is not found in SD card.\n");
                return false;
            }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        int64_t size;
        NN_RESULT_DO(fs::GetFileSize(&size, handle));
        NN_SDK_ASSERT(static_cast<size_t>(size) <= bufferSize);
        NN_RESULT_DO(fs::ReadFile(handle, 0, outBuffer, static_cast<size_t>(size)));
        outBuffer[size] = '\0';

        return true;
    }

    bool ParseUpdateApplicationListAndCreateApplicationAsset(int* outAssetCount, rid::ApplicationAsset* outAssetList, int assetListLength, char* buffer)
    {
        nne::rapidjson::Document document;
        document.Parse(buffer);

        if (document.HasParseError())
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] Rapidjson could not parse ContentList.json as json. error code: %d\n", document.GetParseError());
            return false;
        }

        if (document.HasMember("content"))
        {
            auto toAsset = [](rid::ApplicationAsset* pAsset, const nne::rapidjson::Value& value)->void
            {
                std::string usage = value["usage"].GetString();

                if (usage == "index")
                {
                    pAsset->index = std::stoi(value["id"].GetString());
                }
                else if (usage == "demoProgramID")
                {
                    if (util::Strncmp(value["id"].GetString(), "", 1) == 0)
                    {
                        pAsset->demoApplicationId = rid::InvalidApplicationId;
                    }
                    else
                    {
                        pAsset->demoApplicationId = { std::stoull(value["id"].GetString(), nullptr, 16) };
                    }
                }
                else if (usage == "menuProgramID")
                {
                    pAsset->aocApplicationId = { std::stoull(value["id"].GetString(), nullptr, 16) };
                }
            };

            const nne::rapidjson::Value& contentValue = document["content"];
            size_t contentValueSize = contentValue.Size();

            for (int i = 0; i < contentValueSize && i < assetListLength; i++)
            {
                // "content" 内オブジェクトの参照
                for (nne::rapidjson::Value::ConstMemberIterator contentIter = contentValue[i].MemberBegin();
                    contentIter != contentValue[i].MemberEnd(); contentIter++)
                {
                    if (contentIter->name == "id")
                    {
                        // "id" オブジェクトの参照
                        const nne::rapidjson::Value& idValue = contentIter->value;
                        for (nne::rapidjson::Value::ConstMemberIterator idIter = idValue.MemberBegin();
                            idIter != idValue.MemberEnd(); idIter++)
                        {
                            if (idIter->name == "idForDeviceMultiple")
                            {
                                // "idForDeviceMultiple" 配列の参照
                                const int idCountOfApplicationAsset = 3;
                                const nne::rapidjson::Value& idForDeviceMultipleValue = idIter->value;
                                for (int j = 0; j < idCountOfApplicationAsset; j++)
                                {
                                    toAsset(&outAssetList[i], idForDeviceMultipleValue[j]);
                                }
                            }
                        }
                    }
                }
            }

            *outAssetCount = static_cast<int>(contentValueSize) < assetListLength ? static_cast<int>(contentValueSize) : assetListLength;
            return true;
        }
        else
        {
            return false;
        }
    }

    struct ApplicationUpdater
    {
        static const int MaxApplicationAsset = 128;
        static const int MaxResultList = MaxApplicationAsset + 1;
        rid::ApplicationAsset assetList[MaxApplicationAsset];
        int assetCount = 0;
        int outResultListCount;
        rid::DownloadResult resultList[MaxResultList];
        rid::ApplicationUpdater updater;
        Result result;
        bool isFinished = false;
    };

    const char* ToString(rid::ApplicationUpdateProgress::State state)
    {
        switch (state)
        {
        case nn::rid::ApplicationUpdateProgress::State::DoNothing:
            return "DoNothing";
        case nn::rid::ApplicationUpdateProgress::State::Downloading:
            return "Downloading";
        case nn::rid::ApplicationUpdateProgress::State::Completed:
            return "Completed";
        case nn::rid::ApplicationUpdateProgress::State::Failed:
            return "Failed";
        case nn::rid::ApplicationUpdateProgress::State::Cancelled:
            return "Cancelled";
        default:
            break;
        }
    }

    Result ApplicationUpdate(bool* outValue, const Option& option)
    {
        ApplicationUpdater updater;

        std::unique_ptr<char[]> buffer(new char[16 * 1024]);
        bool isRead;

        if (option.HasTarget())
        {
            isRead = ReadUpdateApplicationListFile(buffer.get(), 16 * 1024, option.GetTarget());
        }
        else
        {
            isRead = ReadUpdateApplicationListFileFromSd(buffer.get(), 16 * 1024);
        }

        if (!isRead)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isParsed = ParseUpdateApplicationListAndCreateApplicationAsset(&updater.assetCount, updater.assetList, updater.MaxApplicationAsset, buffer.get());
        if (!isParsed || updater.assetCount == 0)
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] Json file is not correct format.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        os::TimerEvent timer(os::EventClearMode_AutoClear);
        auto timeoutSeconds = option.GetValue("--timeout");
        if (timeoutSeconds)
        {
            timer.StartOneShot(TimeSpan::FromSeconds(std::strtoull(timeoutSeconds, nullptr, 10)));
        }

        os::ThreadType thread;
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&thread, [](void * p)
        {
            ApplicationUpdater* updater = reinterpret_cast<ApplicationUpdater*>(p);
            updater->result = updater->updater.Execute(&updater->outResultListCount, updater->resultList, updater->MaxResultList, updater->assetList, updater->assetCount, true);
            updater->isFinished = true;
        }
        , &updater, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority + 1));

        os::StartThread(&thread);

        auto PrintProgress = [&]() {
            rid::ApplicationUpdateProgress progress = updater.updater.GetProgress();
            if (progress.state == rid::ApplicationUpdateProgress::State::Downloading)
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] Downloading: %lld / %lld\n", progress.done, progress.total);
            }
            else
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] %s\n", ToString(progress.state));
            }
        };

        while (!updater.isFinished)
        {
            if (timer.TryWait())
            {
                updater.updater.Cancel();
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] Timeout. Trigging Cancel Application Update.\n");
            }

            PrintProgress();
            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        PrintProgress();
        DEVMENUCOMMAND_LOG("[ApplicationUpdate] Application Update Result:\n");
        for (int i = 0; i < updater.outResultListCount; i++)
        {
            if (updater.resultList[i].result.IsSuccess())
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] ApplicationId: %llx, Result: Success\n", updater.resultList[i].applicationId);
            }
            else
            {
                DEVMENUCOMMAND_LOG("[ApplicationUpdate] ApplicationId: %llx, Result: Failed(0x%08x)\n", updater.resultList[i].applicationId, updater.resultList[i].result.GetInnerValueForDebug());
            }
        }

        if (updater.result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] ApplicationUpdate has been succeeded. Last state is \"%s\".\n", ToString(updater.updater.GetProgress().state));
        }
        else
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] ApplicationUpdate is failed as result 0x%08x. Last state is \"%s\".\n", updater.result.GetInnerValueForDebug(), ToString(updater.updater.GetProgress().state));
        }

        if (updater.updater.IsCommitRequired())
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] Commit is required.\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("[ApplicationUpdate] Commit is not required.\n");
        }

        os::WaitThread(&thread);
        os::DestroyThread(&thread);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char* ToString(rid::ControllerUpdateProgress::State state)
    {
        switch (state)
        {
        case rid::ControllerUpdateProgress::State::DoNothing:
            return "DoNothing";
        case rid::ControllerUpdateProgress::State::Applying:
            return "Applying";
        case rid::ControllerUpdateProgress::State::Completed:
            return "Completed";
        case rid::ControllerUpdateProgress::State::Failed:
            return "Failed";
        case rid::ControllerUpdateProgress::State::NeedNoUpdate:
            return "NeedNoUpdate";
        case rid::ControllerUpdateProgress::State::Cancelled:
            return "Cancelled";
        default:
            break;
        }
    }

    struct ControllerUpdater
    {
        rid::ControllerUpdater updater;
        Result result;
        bool isFinished = false;
    };

    Result ControllerUpdate(bool* outValue, const Option& option)
    {
        os::TimerEvent timer(os::EventClearMode_AutoClear);
        auto timeoutSeconds = option.GetValue("--timeout");
        if (timeoutSeconds)
        {
            timer.StartOneShot(TimeSpan::FromSeconds(std::strtoull(timeoutSeconds, nullptr, 10)));
        }

        ControllerUpdater updater;

        os::ThreadType thread;
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&thread, [](void * p)
        {
            ControllerUpdater* updater = reinterpret_cast<ControllerUpdater*>(p);
            updater->result = updater->updater.Execute();
            updater->isFinished = true;
        }
        , &updater, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority + 1));

        os::StartThread(&thread);

        auto PrintProgress = [&]() {
            rid::ControllerUpdateProgress progress = updater.updater.GetProgress();
            if (progress.state == rid::ControllerUpdateProgress::State::Applying)
            {
                DEVMENUCOMMAND_LOG("[ControllerUpdate] Applying: %lld / %lld\n", progress.done, progress.total);
            }
            else
            {
                DEVMENUCOMMAND_LOG("[ControllerUpdate] %s\n", ToString(progress.state));
            }
        };

        while (!updater.isFinished)
        {
            if (timer.TryWait())
            {
                updater.updater.Cancel();
                DEVMENUCOMMAND_LOG("[ControllerUpdate] Timeout. Trigging Cancel Controller Update.\n");
            }

            PrintProgress();
            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        PrintProgress();
        if (updater.result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("[ControllerUpdate] ControllerUpdate has been succeeded. Last state is \"%s\".\n", ToString(updater.updater.GetProgress().state));
        }
        else
        {
            DEVMENUCOMMAND_LOG("[ControllerUpdate] ControllerUpdate is failed as result 0x%08x. Last state is \"%s\".\n", updater.result.GetInnerValueForDebug(), ToString(updater.updater.GetProgress().state));
        }

        os::WaitThread(&thread);
        os::DestroyThread(&thread);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Prepare(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        Result result = rid::CheckContentsIntegrity();
        if (result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("CheckContentsIntegrity has been succeeded.\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("CheckContentsIntegrity is failed as result 0x%08x.\n", result.GetInnerValueForDebug());
        }

        result = rid::DeleteAllMii();
        if (result.IsSuccess())
        {
            DEVMENUCOMMAND_LOG("DeleteAllMii has been succeeded.\n");
        }
        else
        {
            DEVMENUCOMMAND_LOG("DeleteAllMii is failed as result 0x%08x.\n", result.GetInnerValueForDebug());
        }

        // ImageManager 権限がないためとりあえず無効化
        //result = rid::DeleteAllAlbums();
        //if (result.IsSuccess())
        //{
        //    DEVMENUCOMMAND_LOG("DeleteAllAlbums has been succeeded.\n");
        //}
        //else
        //{
        //    DEVMENUCOMMAND_LOG("DeleteAllAlbums is failed as result 0x%08x.\n", result.GetInnerValueForDebug());
        //}

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "setup", Setup },
        { "systemupdate", SystemUpdate },
        { "menuupdate", MenuUpdate },
        { "applicationupdate", ApplicationUpdate },
        { "controllerupdate", ControllerUpdate },
        { "prepare", Prepare },
    };
}
#endif

Result RidCommand(bool* outValue, const Option& option)
{
#if defined (NN_BUILD_CONFIG_OS_HORIZON)
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    rid::Initialize(RidMenuId);

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }
#endif

    DEVMENUCOMMAND_LOG("'%s' is not a " DEVMENUCOMMAND_NAME " rid command. See '" DEVMENUCOMMAND_NAME " rid --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
