﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include "DevMenuCommand_ProgramIdNameMap.h"
#include "DevMenuCommand_StrToUll.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#define RAPIDJSON_NO_INT64DEFINE
#define RAPIDJSON_NAMESPACE             nne::rapidjson
#define RAPIDJSON_NAMESPACE_BEGIN       namespace nne { namespace rapidjson {
#define RAPIDJSON_NAMESPACE_END         }}
#define RAPIDJSON_ASSERT(x)             NN_SDK_ASSERT(x)
#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(readability/define)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(readability/define)
#include <rapidjson/document.h>
#include <rapidjson/memorystream.h>
#include <rapidjson/encodedstream.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

using namespace nn;
using namespace nne::rapidjson;

class ProgramIdNameMap::Impl : public nne::rapidjson::Document
{
public:
    const char* GetNameById(Bit64 id)
    {
        for (Value::ConstMemberIterator i = MemberBegin(); i != MemberEnd(); i++)
        {
            if (STR_TO_ULL(i->value.GetString(), nullptr, 16) == id)
            {
                return i->name.GetString();
            }
        }

        return nullptr;
    }
};

ProgramIdNameMap::ProgramIdNameMap() : m_Impl(new Impl()) {}
ProgramIdNameMap::~ProgramIdNameMap() {}

Result ProgramIdNameMap::Initialize()
{
    size_t mountRomCacheBufferSize = 0;
    NN_RESULT_DO(fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    std::vector<char> mountRomCacheBuffer(mountRomCacheBufferSize);
    NN_RESULT_DO(fs::MountRom("rom", mountRomCacheBuffer.data(), mountRomCacheBufferSize));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount("rom"); };

    kvdb::BoundedString<32> path;
    path.Assign("rom:/" NN_TOOL_DEVMENUCOMMANDSYSTEM_SDKPROGRAMIDSJSON_NAME);

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    int64_t size;
    NN_RESULT_DO(fs::GetFileSize(&size, file));

    size_t dataSize = static_cast<size_t>(size);
    std::vector<char> buffer(dataSize + 1);
    NN_RESULT_DO(fs::ReadFile(file, 0, buffer.data(), dataSize));
    buffer[dataSize] = '\0';

    // BOM 付き JSON 読み込みはエンコードが必要
    MemoryStream memoryStream(buffer.data(), buffer.size());
    EncodedInputStream<UTF8<>, MemoryStream> encodedInputStream(memoryStream);
    NN_ABORT_UNLESS(!m_Impl->ParseStream(encodedInputStream).HasParseError());

    NN_RESULT_SUCCESS;
}

bool ProgramIdNameMap::HasName(Bit64 id)
{
    return m_Impl->GetNameById(id) != nullptr;
}

const char* ProgramIdNameMap::GetName(Bit64 id)
{
    auto name = m_Impl->GetNameById(id);

    return name ? name : "Unknown";
}

util::optional<Bit64> ProgramIdNameMap::GetId(const char* name)
{
    auto obj = m_Impl->FindMember(name);
    if (obj == m_Impl->MemberEnd())
    {
        return util::nullopt;
    }

    return STR_TO_ULL(obj->value.GetString(), nullptr, 16);
}
