﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_ApiConfig.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_PrepoCommand.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    Result SetDevMenuPrepoPageView(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is invalid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("prepo", "devmenu_prepo_page_view", &isEnabled, sizeof (isEnabled));

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetBackgroundProcessingEnabled(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is invalid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("prepo", "background_processing", &isEnabled, sizeof (isEnabled));

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetUserAgreementCheck(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is invalid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(isEnabled));

        // 設定の反映に再起動は不要。

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetIntervalImpl(bool* outValue, const Option& option, const char* key)
    {
        auto intervalString = option.GetTarget();
        if (std::string(intervalString) == "")
        {
            NN_LOG("You must specify a interval. Set <interval_seconds>.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        char* pEnd;
        auto interval = static_cast<int32_t>(std::strtoll(intervalString, &pEnd, 10));
        if (pEnd == nullptr || *pEnd != '\0')
        {
            NN_LOG("%s is invalid interval. Set <interval_seconds> with number.\n", intervalString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::settings::fwdbg::SetSettingsItemValue("prepo", key, &interval, sizeof (interval));

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetTransmissionIntervalMin(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "transmission_interval_min");
    }

    Result SetTransmissionRetryIntervalMin(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "transmission_retry_interval_min");
    }

    Result SetTransmissionRetryIntervalMax(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "transmission_retry_interval_max");
    }

    Result SetTransmissionIntervalInSleep(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "transmission_interval_in_sleep");
    }

    Result SetStatisticsSaveIntervalMin(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "statistics_save_interval_min");
    }

    Result SetStatisticsPostInterval(bool* outValue, const Option& option)
    {
        return SetIntervalImpl(outValue, option, "statistics_post_interval");
    }

    Result SetOutputSaveLogEnabled(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is invalid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("prepo", "output_save_log", &isEnabled, sizeof (isEnabled));

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    template <typename T>
    void ReadDebugSettings(const char* key, T* out) NN_NOEXCEPT
    {
        const size_t readSize = sizeof (T);
        NN_ABORT_UNLESS(nn::settings::fwdbg::GetSettingsItemValue(out, readSize, "prepo", key) == readSize);
    }

    Result GetSettings(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = false;

        bool devmenuPrepoPageView;
        bool userAgreementCheck;
        bool backgroundProcessing;
        int32_t transmissionIntervalMin;
        int32_t transmissionRetryIntervalMin;
        int32_t transmissionRetryIntervalMax;
        int32_t transmissionIntervalInSleep;
        int32_t statisticsSaveIntervalMin;
        int32_t statisticsPostInterval;
        bool outputSaveLog;

        ReadDebugSettings("devmenu_prepo_page_view", &devmenuPrepoPageView);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::IsUserAgreementCheckEnabled(&userAgreementCheck));
        ReadDebugSettings("background_processing", &backgroundProcessing);
        ReadDebugSettings("transmission_interval_min", &transmissionIntervalMin);
        ReadDebugSettings("transmission_retry_interval_min", &transmissionRetryIntervalMin);
        ReadDebugSettings("transmission_retry_interval_max", &transmissionRetryIntervalMax);
        ReadDebugSettings("transmission_interval_in_sleep", &transmissionIntervalInSleep);
        ReadDebugSettings("statistics_save_interval_min", &statisticsSaveIntervalMin);
        ReadDebugSettings("statistics_post_interval", &statisticsPostInterval);
        ReadDebugSettings("output_save_log", &outputSaveLog);

        NN_LOG(
            "devmenu_prepo_page_view:           %s\n"
            "user_agreement_check:              %s\n"
            "background_processing:             %s\n"
            "transmission_interval_min:         %d\n"
            "transmission_retry_interval_min:   %d\n"
            "transmission_retry_interval_max:   %d\n"
            "transmission_interval_in_sleep:    %d\n"
            "statistics_save_interval_min:      %d\n"
            "statistics_post_interval:          %d\n"
            "output_save_log:                   %s\n"
            , devmenuPrepoPageView ? "true" : "false"
            , userAgreementCheck ? "true" : "false"
            , backgroundProcessing ? "true" : "false"
            , transmissionIntervalMin
            , transmissionRetryIntervalMin
            , transmissionRetryIntervalMax
            , transmissionIntervalInSleep
            , statisticsSaveIntervalMin
            , statisticsPostInterval
            , outputSaveLog ? "true" : "false");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearStorage(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = false;

        NN_RESULT_DO(nn::prepo::ClearStorage());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearStatistics(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        *outValue = false;

        NN_RESULT_DO(nn::prepo::ClearStatistics());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " prepo set-devmenu-page-view <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-user-agreement-check <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-background-processing <enable|disable>\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM) // ヘルプには表示しない。
        "       " DEVMENUCOMMAND_NAME " prepo set-transmission-interval-min <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-transmission-retry-interval-min <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-transmission-retry-interval-max <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-transmission-interval-in-sleep <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-statistics-save-interval-min <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-statistics-post-interval <interval_seconds>\n"
        "       " DEVMENUCOMMAND_NAME " prepo set-output-save-log <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " prepo get-settings\n"
#endif
        "       " DEVMENUCOMMAND_NAME " prepo clear-storage\n"
        "       " DEVMENUCOMMAND_NAME " prepo clear-statistics\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"set-devmenu-page-view",               SetDevMenuPrepoPageView},
        {"set-user-agreement-check",            SetUserAgreementCheck},
        {"set-background-processing",           SetBackgroundProcessingEnabled},
        {"set-transmission-interval-min",       SetTransmissionIntervalMin},
        {"set-transmission-retry-interval-min", SetTransmissionRetryIntervalMin},
        {"set-transmission-retry-interval-max", SetTransmissionRetryIntervalMax},
        {"set-transmission-interval-in-sleep",  SetTransmissionIntervalInSleep},
        {"set-statistics-save-interval-min",    SetStatisticsSaveIntervalMin},
        {"set-statistics-post-interval",        SetStatisticsPostInterval},
        {"get-settings",                        GetSettings},
        {"set-output-save-log",                 SetOutputSaveLogEnabled},
        {"clear-storage",                       ClearStorage},
        {"clear-statistics",                    ClearStatistics},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result PrepoCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu prepo command. See '" DEVMENUCOMMAND_NAME " prepo --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
