﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/nifm.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForDebug.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_PctlCommand.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    ////////////////////////////////////////////////////////////////////////////
    // utility for common

    bool ParseInteger(int* outValue, const char* pString)
    {
        if (pString == nullptr)
            return false;
        char* pEnd;
        *outValue = static_cast<int>(std::strtoll(pString, &pEnd, 10));
        return pEnd != nullptr && *pEnd == '\0';
    }

    bool ParseApplicationId(ncm::ApplicationId* outValue, const char* pString)
    {
        if (pString == nullptr)
            return false;
        char* pEnd;
        if (pString[0] == '0' && (pString[1] == 'X' || pString[1] == 'x'))
        {
            pString += 2;
        }
        outValue->value = std::strtoull(pString, &pEnd, 16);
        return pEnd != nullptr && *pEnd == '\0';
    }

    bool ParseFlag(bool* outFlag, const char* flagString, size_t stringLen)
    {
        if (flagString == nullptr)
            return false;
        if (stringLen == 4 && nn::util::Strnicmp(flagString, "true", static_cast<int>(stringLen)) == 0)
        {
            *outFlag = true;
        }
        else if (stringLen == 5 && nn::util::Strnicmp(flagString, "false", static_cast<int>(stringLen)) == 0)
        {
            *outFlag = false;
        }
        else
        {
            return false;
        }

        return true;
    }

    bool ParseFlag(bool* outFlag, const char* flagString)
    {
        if (flagString == nullptr)
            return false;
        return ParseFlag(outFlag, flagString, std::strlen(flagString));
    }

    bool ParseEnumString(int* outValue, const char* const* stringValues, int valueCount, const char* valueString)
    {
        if (valueString == nullptr)
            return false;
        int intVal;
        if (ParseInteger(&intVal, valueString))
        {
            if (intVal < 0 || intVal >= valueCount)
                return false;
            *outValue = intVal;
            return true;
        }
        else
        {
            auto len = static_cast<int>(std::strlen(valueString) + 1);
            for (int i = 0; i < valueCount; ++i)
            {
                if (nn::util::Strnicmp(stringValues[i], valueString, len) == 0)
                {
                    *outValue = i;
                    return true;
                }
            }
            return false;
        }
    }

    template <class T, int Count>
    bool ParseEnumString(T* outValue, const char* const (& stringValues)[Count], const char* valueString)
    {
        int val;
        if (!ParseEnumString(&val, stringValues, Count, valueString))
            return false;
        *outValue = static_cast<T>(val);
        return true;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result InitializeTime()
    {
        return nn::time::InitializeForSystem();
    }

    Result FinalizeTime()
    {
        return nn::time::Finalize();
    }
#endif

    ////////////////////////////////////////////////////////////////////////////
    // utility for fwdbg APIs

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    template <typename T>
    bool GetSettingsFwdbgValue(T* pOut, const char* keyString)
    {
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(pOut, sizeof(T), "pctl", keyString);
        bool bResult = (bytes == sizeof(T));
        if (!bResult)
        {
            DEVMENUCOMMAND_LOG("Failed to read settings: %s\n", keyString);
        }
        return bResult;
    }

    template <typename T>
    void SetSettingsFwdbgValue(T value, const char* keyString)
    {
        nn::settings::fwdbg::SetSettingsItemValue("pctl", keyString, &value, sizeof(T));
    }

    Result ConfigSettingInteger(bool* outValue, const Option& option, const char* keyString)
    {
        auto modeString = option.GetTarget();
        int value;
        if (std::string(modeString).empty())
        {
            if ((*outValue = GetSettingsFwdbgValue<int32_t>(&value, keyString)) == true)
            {
                DEVMENUCOMMAND_LOG("%s: %d\n", keyString, value);
            }
        }
        else
        {
            if (ParseInteger(&value, modeString))
            {
                SetSettingsFwdbgValue<int32_t>(static_cast<int32_t>(value), keyString);
                DEVMENUCOMMAND_LOG("%s: %d\n", keyString, value);
                *outValue = true;
            }
            else
            {
                *outValue = false;
            }
        }
        NN_RESULT_SUCCESS;
    }

    char* GetEventNameFromEventType(nn::pctl::detail::service::watcher::EventType e)
    {
        NN_FUNCTION_LOCAL_STATIC(char, name[32], = "");

        switch (e)
        {
        case nn::pctl::detail::service::watcher::EventType::Invalid:
            nn::util::Strlcpy(name, "Invalid", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidDeviceLaunch:
            nn::util::Strlcpy(name, "DidDeviceLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidWakeup:
            nn::util::Strlcpy(name, "DidWakeup", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidSleep:
            nn::util::Strlcpy(name, "DidSleep", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationLaunch:
            nn::util::Strlcpy(name, "DidApplicationLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationTerminate:
            nn::util::Strlcpy(name, "DidApplicationTerminate", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::Idle:
            nn::util::Strlcpy(name, "Idle", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUnlock:
            nn::util::Strlcpy(name, "DidUnlock", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLock:
            nn::util::Strlcpy(name, "DidLock", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidWrongUnlockCode:
            nn::util::Strlcpy(name, "DidWrongUnlockCode", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidComeOnline:
            nn::util::Strlcpy(name, "DidComeOnline", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidGoOffline:
            nn::util::Strlcpy(name, "DidGoOffline", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAddNewManagedApplication:
            nn::util::Strlcpy(name, "DidAddNewManagedApplication", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidRemoveManagedApplication:
            nn::util::Strlcpy(name, "DidRemoveManagedApplication", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidInterruptPlaying:
            nn::util::Strlcpy(name, "DidInterruptPlaying", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationPlay:
            nn::util::Strlcpy(name, "DidApplicationPlay", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAlarmMakeInvisible:
            nn::util::Strlcpy(name, "DidAlarmMakeInvisible", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAlarmMakeVisible:
            nn::util::Strlcpy(name, "DidAlarmMakeVisible", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationDownloadStart:
            nn::util::Strlcpy(name, "DidApplicationDownloadStart", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidReachLimitTime:
            nn::util::Strlcpy(name, "DidReachLimitTime", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationSuspend:
            nn::util::Strlcpy(name, "DidApplicationSuspend", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationResume:
            nn::util::Strlcpy(name, "DidApplicationResume", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUserOpen:
            nn::util::Strlcpy(name, "DidUserOpen", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUserClose:
            nn::util::Strlcpy(name, "DidUserClose", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidShutdown:
            nn::util::Strlcpy(name, "DidShutdown", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidPlayTimerStart:
            nn::util::Strlcpy(name, "DidPlayTimerStart", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidPlayTimerStop:
            nn::util::Strlcpy(name, "DidPlayTimerStop", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidDeviceActivate:
            nn::util::Strlcpy(name, "DidDeviceActivate", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLocationNameChange:
            nn::util::Strlcpy(name, "DidLocationNameChange", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUnexpectedShutdownOccur:
            nn::util::Strlcpy(name, "DidUnexpectedShutdownOccur", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLimitedApplicationLaunch:
            nn::util::Strlcpy(name, "DidLimitedApplicationLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::EventTypeMaxCount:
            nn::util::Strlcpy(name, "EventTypeMaxCount", sizeof(name));
            break;
        default:
            nn::util::Strlcpy(name, "Unknown", sizeof(name));
            break;
        }
        return name;
    } // NOLINT(impl/function_size)
#endif

    ////////////////////////////////////////////////////////////////////////////
    // utility for pctl APIs

    const char* const SafetyLevelNames[] = {
        "None",
        "Custom",
        "Children",
        "YoungTeens",
        "OlderTeens"
    };
    NN_STATIC_ASSERT(static_cast<int>(std::extent<decltype(SafetyLevelNames)>::value) == static_cast<int>(nn::pctl::SafetyLevel_Max));

    const char* const RatingOrganizations[] = {
        "CERO",
        "GRACGCRB",
        "GSRMR",
        "ESRB",
        "ClassInd",
        "USK",
        "PEGI",
        "PEGIPortug",
        "PEGIBBFC",
        "Russian",
        "ACB",
        "OFLC"
    };
    NN_STATIC_ASSERT(static_cast<int>(nn::ns::RatingOrganization::OFLC) < std::extent<decltype(RatingOrganizations)>::value);

    const char* const TimerModeNames[] = {
        "Alarm",
        "Suspend"
    };
    NN_STATIC_ASSERT(0 == static_cast<int>(nn::pctl::PlayTimerMode_Alarm));
    NN_STATIC_ASSERT(1 == static_cast<int>(nn::pctl::PlayTimerMode_Suspend));

    const char* const WeekNames[] = {
        "Sunday",
        "Monday",
        "Tuesday",
        "Wednesday",
        "Thursday",
        "Friday",
        "Saturday",
        "Daily" // コマンドラインで指定できる値として追加
    };
    NN_STATIC_ASSERT(static_cast<int>(std::extent<decltype(WeekNames)>::value) == static_cast<int>(nn::pctl::Week_TotalCount) + 1);
    const int Week_Daily = static_cast<int>(nn::pctl::Week_TotalCount);

    bool IsValidPinCode(const char* string)
    {
        auto codeLen = std::strlen(string);
        if (codeLen < 4 || codeLen > 8)
        {
            return false;
        }
        // 数字4桁～8桁なので便宜上整数に変換できるかどうかで確認可能
        int tempValue;
        if (!ParseInteger(&tempValue, string))
        {
            return false;
        }
        return true;
    }

    void DumpSafetyLevelSettings(const nn::pctl::SafetyLevelSettings& settings)
    {
        DEVMENUCOMMAND_LOG("Rating age: %d\n", static_cast<int>(settings.ratingAge));
        DEVMENUCOMMAND_LOG("SNS post restriction: %s\n", settings.snsPostRestriction ? "true" : "false");
        DEVMENUCOMMAND_LOG("Free communication restriction: %s\n", settings.freeCommunicationRestriction ? "true" : "false");
    }

    void PrintPairingActive()
    {
        DEVMENUCOMMAND_LOG("Cannot change settings because pairing is active.\n");
    }

    void DumpPlayTimerDaySettings(const nn::pctl::PlayTimerDaySettings& settings)
    {
        DEVMENUCOMMAND_LOG("Bedtime enabled: %s\n", settings.isBedtimeEnabled ? "true" : "false");
        DEVMENUCOMMAND_LOG("Bedtime: %02d:%02d\n",
            static_cast<int>(settings.bedtimeHour), static_cast<int>(settings.bedtimeMinute));
        DEVMENUCOMMAND_LOG("Limit time enabled: %s\n", settings.isLimitTimeEnabled ? "true" : "false");
        DEVMENUCOMMAND_LOG("Limit time: %d [min.]\n",
            static_cast<int>(settings.limitTime));
    }

    // param の先頭に「true,」「false,」がある場合はそれを *outValue に書き込む
    // *outNextPosition には上記を読み取った場合次の位置、それ以外は param を返す
    // (「true」「false」のみの場合は nullptr、「,」のみで始まる場合はその次の文字の位置)
    // 戻り値は上記が読み取れれば true、「true」「false」以外の文字+「,」があれば false
    bool ReadFlagParameter(const char** outNextPosition, bool* outValue, const char* param)
    {
        auto commaPosition = std::strchr(param, ',');
        if (commaPosition != nullptr)
        {
            size_t flagLen = (commaPosition - param);
            // (「,」がいきなり先頭に来るケース(flagLen == 0)は許容する)
            if (flagLen > 0 && !ParseFlag(outValue, param, flagLen))
            {
                return false;
            }
            *outNextPosition = commaPosition + 1;
        }
        else if (ParseFlag(outValue, param))
        {
            *outNextPosition = nullptr;
        }
        else
        {
            *outNextPosition = param;
        }
        return true;
    }

    bool ParseBedtimeParameter(nn::pctl::PlayTimerDaySettings* pDaySettings, const char* param)
    {
        const char* timeValue;
        bool isFailure = false;
        if (!ReadFlagParameter(&timeValue, &pDaySettings->isBedtimeEnabled, param))
        {
            isFailure = true;
        }
        else if (timeValue != nullptr)
        {
            int hour = 0;
            int minute = 0;
            if (std::sscanf(timeValue, "%02d:%02d", &hour, &minute) != 2)
            {
                isFailure = true;
            }
            else
            {
                if (hour < 0 || hour >= 24 || minute < 0 || minute >= 60)
                {
                    isFailure = true;
                }
                if (!isFailure)
                {
                    pDaySettings->bedtimeHour = static_cast<uint8_t>(hour);
                    pDaySettings->bedtimeMinute = static_cast<uint8_t>(minute);
                }
            }
        }
        return !isFailure;
    }

    bool ParseLimitTimeParameter(nn::pctl::PlayTimerDaySettings* pDaySettings, const char* param)
    {
        const char* timeValue;
        bool isFailure = false;
        if (!ReadFlagParameter(&timeValue, &pDaySettings->isLimitTimeEnabled, param))
        {
            isFailure = true;
        }
        else if (timeValue != nullptr)
        {
            int minutes = 0;
            if (!ParseInteger(&minutes, timeValue))
            {
                isFailure = true;
            }
            else if (minutes < 0 || minutes > 360)
            {
                isFailure = true;
            }
            else
            {
                pDaySettings->limitTime = static_cast<uint16_t>(minutes);
            }
        }
        return !isFailure;
    }

    void DumpTimerStatus()
    {
        DEVMENUCOMMAND_LOG("--- Timer Status ---\n");
        DEVMENUCOMMAND_LOG("Timer enabled: %s\n",
            nn::pctl::IsPlayTimerEnabled() ? "true" : "false");
        DEVMENUCOMMAND_LOG("Restricted by timer: %s\n",
            nn::pctl::IsRestrictedByPlayTimer() ? "true" : "false");
        DEVMENUCOMMAND_LOG("Remaining time: %lld [sec.]\n",
            nn::pctl::GetPlayTimerRemainingTime().GetSeconds());

        nn::pctl::PlayTimerSettings settings;
        nn::pctl::GetPlayTimerSettings(&settings);
        DEVMENUCOMMAND_LOG("--- Timer Settings ---\n");
        DEVMENUCOMMAND_LOG("Timer mode: %s\n",
            settings.playTimerMode == nn::pctl::PlayTimerMode_Suspend ? "Suspend" : "Alarm");
        DEVMENUCOMMAND_LOG("Using week settings: %s\n",
            settings.isWeekSettingsUsed ? "true" : "false");
        DEVMENUCOMMAND_LOG("* Daily settings\n");
        DumpPlayTimerDaySettings(settings.dailySettings);
        for (int i = 0; i < static_cast<int>(nn::pctl::Week_TotalCount); ++i)
        {
            DEVMENUCOMMAND_LOG("* '%s' settings\n", WeekNames[i]);
            DumpPlayTimerDaySettings(settings.weekSettings[i]);
        }
    }

    void PrintUsageTimerSet()
    {
        DEVMENUCOMMAND_LOG(
            "usage: " DEVMENUCOMMAND_NAME " pctl timer-set [--timer-mode <mode>] [--using-week-settings <true|false>] [--week <week>] [--bedtime [<true|false>][,HH:mm]] [--limit-time [<true|false>][,<minutes>]]\n"
            "    --timer-mode <mode> : set timer mode\n"
            "      <mode> : Alarm (0), Suspend (1)\n"
            "    --using-week-settings <true|false> : specify whether the week settings is used\n"
            "    --week <week> : specify the week (or daily-settings) which to change settings\n"
            "      <week> : Sunday (0), Monday (1), Tuesday (2), Wednesday (3),\n"
            "               Thursday (4), Friday (5), Saturday (6), Daily (7)\n"
            "      * if omitted, Daily settings will be changed\n"
            "    --bedtime [<true|false>][,HH:mm] : set bedtime\n"
            "      <true|false> : if true, the bedtime will be used (if omitted, the flag will not be changed)\n"
            "      HH:mm : the bedtime to set (24h format)\n"
            "      * example: --bedtime false / --bedtime 22:00 / --bedtime true,23:45\n"
            "    --limit-time [<true|false>][,<minutes>] : set limit time\n"
            "      <true|false> : if true, the limit-time will be used (if omitted, the flag will not be changed)\n"
            "      <minutes> : the limit-time to set (minutes)\n"
            "      * example: --limit-time false / --limit-time 60 / --limit-time true,90\n"
            "  * At least one option except --week must be specified to change settings.\n"
            "  * The omitted settings will not be changed.\n"
        );
    }

    ////////////////////////////////////////////////////////////////////////////

    Result SetPin(bool* outValue, const Option& option)
    {
        auto pin = option.GetTarget();
        if (std::string(pin).empty())
        {
            DEVMENUCOMMAND_LOG("<pin-code> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!IsValidPinCode(pin))
        {
            DEVMENUCOMMAND_LOG("<pin-code> must have 4 to 8 digits. (specified: %s)\n", pin);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::SetPinCode(pin);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnsetPin(bool* outValue, const Option&)
    {
        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::SetPinCode(nullptr);

        // 強制中断から抜けられなくなるのを防止するため初期値(オール 0)を設定してクリアする
        nn::pctl::PlayTimerSettings settings = {};
        nn::pctl::SetPlayTimerSettingsForDebug(settings);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckPin(bool* outValue, const Option& option)
    {
        auto pin = option.GetTarget();
        if (std::string(pin).empty())
        {
            DEVMENUCOMMAND_LOG("<pin-code> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!IsValidPinCode(pin))
        {
            DEVMENUCOMMAND_LOG("<pin-code> must have 4 to 8 digits. (specified: %s)\n", pin);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("Parental control restriction is not enabled.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // この関数は特に内部状態を変更しないのでPINコードチェックに使用可能
        NN_RESULT_TRY(nn::pctl::UnlockSystemSettingsRestriction(pin))
            NN_RESULT_CATCH(nn::pctl::ResultWrongPinCode)
            {
                DEVMENUCOMMAND_LOG("Incorrect pin-code. (specified: %s)\n", pin);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetSafetyLevel(bool* outValue, const Option&)
    {
        auto level = nn::pctl::GetSafetyLevel();
        DEVMENUCOMMAND_LOG("level = %d (%s)\n",
            static_cast<int>(level),
            SafetyLevelNames[level]);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSafetyLevel(bool* outValue, const Option& option)
    {
        auto levelString = option.GetTarget();
        bool isFailure = false;
        auto level = nn::pctl::SafetyLevel_None;
        if (std::string(levelString).empty())
        {
            DEVMENUCOMMAND_LOG("<level> is required.\n");
            isFailure = true;
        }
        else if (!ParseEnumString(&level, SafetyLevelNames, levelString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified for <level>. (specified: %s)\n", levelString);
            isFailure = true;
        }
        if (isFailure)
        {
            DEVMENUCOMMAND_LOG("Following values are valid for <level>:\n");
            int i = 0;
            for (auto value : SafetyLevelNames)
            {
                DEVMENUCOMMAND_LOG("  %s (%d)\n", value, i);
                ++i;
            }
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::SetSafetyLevel(level);

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("*** To apply restrictions, please set pin code (use 'set-pin').\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpCurrentSettings(bool* outValue, const Option&)
    {
        nn::pctl::SafetyLevelSettings settings = {0};
        nn::pctl::GetCurrentSettings(&settings);

        DEVMENUCOMMAND_LOG("--- Current Settings ---\n");
        DEVMENUCOMMAND_LOG("Restriction enabled: %s\n",
            nn::pctl::IsRestrictionEnabled() ? "true" : "false");
        DumpSafetyLevelSettings(settings);
        {
            auto rating = nn::pctl::GetDefaultRatingOrganization();
            DEVMENUCOMMAND_LOG("Rating organization: %s (%d)\n",
                RatingOrganizations[static_cast<int>(rating)], static_cast<int>(rating));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpCustomSettings(bool* outValue, const Option&)
    {
        nn::pctl::SafetyLevelSettings settings = {0};
        nn::pctl::GetSafetyLevelSettings(&settings, nn::pctl::SafetyLevel_Custom);

        DEVMENUCOMMAND_LOG("--- Custom Settings ---\n");
        DumpSafetyLevelSettings(settings);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetCustomSettings(bool* outValue, const Option& option)
    {
        nn::pctl::SafetyLevelSettings settings = {0};
        nn::pctl::GetSafetyLevelSettings(&settings, nn::pctl::SafetyLevel_Custom);

        if (option.HasKey("--rating-age"))
        {
            auto ageString = option.GetValue("--rating-age");
            int val = 0;
            if (std::string(ageString).empty() ||
                !ParseInteger(&val, ageString) ||
                val < 0 || val > 255)
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for <age>. (specified: %s)\n", ageString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            settings.ratingAge = static_cast<uint8_t>(val);
        }
        if (option.HasKey("--sns-post"))
        {
            auto flagString = option.GetValue("--sns-post");
            bool val = false;
            if (std::string(flagString).empty() ||
                !ParseFlag(&val, flagString))
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for --sns-post. (specified: %s)\n", flagString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            settings.snsPostRestriction = val;
        }
        if (option.HasKey("--free-communication"))
        {
            auto flagString = option.GetValue("--free-communication");
            bool val = false;
            if (std::string(flagString).empty() ||
                !ParseFlag(&val, flagString))
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for --free-communication. (specified: %s)\n", flagString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            settings.freeCommunicationRestriction = val;
        }

        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::SetCustomSafetyLevelSettings(settings);

        DEVMENUCOMMAND_LOG("--- Custom Settings ---\n");
        DumpSafetyLevelSettings(settings);

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("*** To apply restrictions, please set pin code (use 'set-pin').\n");
        }
        if (nn::pctl::GetSafetyLevel() != nn::pctl::SafetyLevel_Custom)
        {
            DEVMENUCOMMAND_LOG("*** To use custom settings, please set safety-level to 'Custom' (use 'set-safety-level').\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetRatingOrganization(bool* outValue, const Option& option)
    {
        auto valueString = option.GetTarget();
        bool isFailure = false;
        auto organization = nn::ns::RatingOrganization::CERO;
        if (std::string(valueString).empty())
        {
            DEVMENUCOMMAND_LOG("<organization> is required.\n");
            isFailure = true;
        }
        else if (!ParseEnumString(&organization, RatingOrganizations, valueString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified for <organization>. (specified: %s)\n", valueString);
            isFailure = true;
        }
        if (isFailure)
        {
            DEVMENUCOMMAND_LOG("Following values are valid for <organization>:\n");
            int i = 0;
            for (auto value : RatingOrganizations)
            {
                DEVMENUCOMMAND_LOG("  %s (%d)\n", value, i);
                ++i;
            }
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::SetDefaultRatingOrganization(organization);

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("*** To apply restrictions, please set pin code (use 'set-pin').\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpFreeCommunicationAppList(bool* outValue, const Option&)
    {
        auto totalCount = nn::pctl::GetFreeCommunicationApplicationListCount();
        static const int InfoCount = 50;
        nn::pctl::FreeCommunicationApplicationInfo infoArray[InfoCount];

        DEVMENUCOMMAND_LOG("--- Free Communication Application List ---\n");
        DEVMENUCOMMAND_LOG("total count: %d\n", totalCount);
        for (int i = 0; i < totalCount; )
        {
            int c = (i + InfoCount > totalCount ? totalCount - i : InfoCount);
            c = nn::pctl::GetFreeCommunicationApplicationList(infoArray, i, c);
            if (c == 0)
            {
                break;
            }
            for (int j = 0; j < c; ++j)
            {
                auto& info = infoArray[i + j];
                DEVMENUCOMMAND_LOG("[% 4d] 0x%016llX: %s\n",
                    i + j,
                    info.applicationId.value,
                    info.isRestricted ? "true" : "false");
            }
            i += c;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetFreeCommunicationRestriction(bool* outValue, const Option& option)
    {
        nn::pctl::FreeCommunicationApplicationInfo info;

        if (!option.HasKey("--application-id"))
        {
            DEVMENUCOMMAND_LOG("--application-id <id> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto appIdString = option.GetValue("--application-id");
        if (std::string(appIdString).empty())
        {
            DEVMENUCOMMAND_LOG("--application-id <id> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!ParseApplicationId(&info.applicationId, appIdString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified for --application-id <id>. (specified: %s)\n", appIdString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto flagString = option.GetTarget();
        if (std::string(appIdString).empty())
        {
            DEVMENUCOMMAND_LOG("Boolean value is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!ParseFlag(&info.isRestricted, flagString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified. (specified: %s)\n", flagString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto totalCount = nn::pctl::GetFreeCommunicationApplicationListCount();
        static const int InfoCount = 50;
        nn::pctl::FreeCommunicationApplicationInfo infoArray[InfoCount];
        bool found = false;

        for (int i = 0; i < totalCount; )
        {
            int c = (i + InfoCount > totalCount ? totalCount - i : InfoCount);
            c = nn::pctl::GetFreeCommunicationApplicationList(infoArray, i, c);
            if (c == 0)
            {
                break;
            }
            for (int j = 0; j < c; ++j)
            {
                if (infoArray[i + j].applicationId == info.applicationId)
                {
                    found = true;
                    break;
                }
            }
            if (found)
            {
                break;
            }
            i += c;
        }
        if (!found)
        {
            DEVMENUCOMMAND_LOG("Application [0x%016llX] is not registered.\n",
                info.applicationId.value);
            DEVMENUCOMMAND_LOG("Before changing settings, please launch the application with Free Communication flag specified to the .nmeta file.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::UpdateFreeCommunicationApplicationList(&info, 1);

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("*** To apply restrictions, please set pin code (use 'set-pin').\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteSettings(bool* outValue, const Option&)
    {
        if (nn::pctl::IsPairingActive())
        {
            PrintPairingActive();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::DeleteSettings();

        // 強制中断から抜けられなくなるのを防止するため初期値(オール 0)を設定してクリアする
        nn::pctl::PlayTimerSettings settings = {};
        nn::pctl::SetPlayTimerSettingsForDebug(settings);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    ////////////////////////////////////////////////////////////////////////////

    Result TimerStart(bool* outValue, const Option&)
    {
        nn::pctl::StartPlayTimer();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TimerStop(bool* outValue, const Option&)
    {
        nn::pctl::StopPlayTimer();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TimerStatus(bool* outValue, const Option&)
    {
        DumpTimerStatus();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TimerSet(bool* outValue, const Option& option)
    {
        nn::pctl::PlayTimerSettings settings;
        nn::pctl::PlayTimerDaySettings* pDaySettings;
        nn::pctl::GetPlayTimerSettings(&settings);

        bool isChanged = false;

        if (option.HasKey("--timer-mode"))
        {
            // --timer-mode <mode>
            auto modeString = option.GetValue("--timer-mode");
            bool isFailure = false;
            if (std::string(modeString).empty())
            {
                DEVMENUCOMMAND_LOG("<mode> is required for --timer-mode.\n");
                isFailure = true;
            }
            else if (!ParseEnumString(&settings.playTimerMode, TimerModeNames, modeString))
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for --timer-mode. (specified: %s)\n", modeString);
                isFailure = true;
            }
            if (isFailure)
            {
                DEVMENUCOMMAND_LOG("Following values are valid for <mode>:\n");
                for (int i = 0; i < static_cast<int>(std::extent<decltype(TimerModeNames)>::value); ++i)
                {
                    DEVMENUCOMMAND_LOG("  %s (%d)\n", TimerModeNames[i], i);
                }
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            isChanged = true;
        }

        if (option.HasKey("--using-week-settings"))
        {
            // --using-week-settings <true|false>
            auto valueString = option.GetValue("--using-week-settings");
            bool isFailure = false;
            if (std::string(valueString).empty())
            {
                DEVMENUCOMMAND_LOG("true or false is required for --using-week-settings.\n");
                isFailure = true;
            }
            else if (!ParseFlag(&settings.isWeekSettingsUsed, valueString))
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for --using-week-settings.\n");
                isFailure = true;
            }
            if (isFailure)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            isChanged = true;
        }

        if (option.HasKey("--week"))
        {
            // --week <week>
            auto weekString = option.GetValue("--week");
            int week = 0;
            bool isFailure = false;
            if (std::string(weekString).empty())
            {
                DEVMENUCOMMAND_LOG("<week> is required for --week.\n");
                isFailure = true;
            }
            else if (!ParseEnumString(&week, WeekNames, weekString))
            {
                DEVMENUCOMMAND_LOG("Invalid value specified for --week.\n");
                isFailure = true;
            }
            if (isFailure)
            {
                DEVMENUCOMMAND_LOG("Following values are valid for <week>:\n");
                for (int i = 0; i < static_cast<int>(std::extent<decltype(WeekNames)>::value); ++i)
                {
                    DEVMENUCOMMAND_LOG("  %s (%d)\n", WeekNames[i], i);
                }
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            if (week == Week_Daily)
            {
                pDaySettings = &settings.dailySettings;
            }
            else
            {
                pDaySettings = &settings.weekSettings[week];
            }
        }
        else
        {
            pDaySettings = &settings.dailySettings;
        }

        if (option.HasKey("--bedtime"))
        {
            // --bedtime [<true|false>][,HH:mm]
            //   ('--bedtime true', '--bedtime true,23:45', '--bedtime 23:45' are valid)
            auto valueString = option.GetValue("--bedtime");
            if (std::string(valueString).empty())
            {
                DEVMENUCOMMAND_LOG("Empty value for --bedtime is not valid.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            if (!ParseBedtimeParameter(pDaySettings, valueString))
            {
                DEVMENUCOMMAND_LOG("The value '%s' for --bedtime is not valid.\n", valueString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            isChanged = true;
        }

        if (option.HasKey("--limit-time"))
        {
            // --limit-time [<true|false>][,<minute>]
            //   ('--limit-time true', '--limit-time true,60', '--limit-time 120' are valid)
            auto valueString = option.GetValue("--limit-time");
            if (std::string(valueString).empty())
            {
                DEVMENUCOMMAND_LOG("Empty value for --limit-time is not valid.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            if (!ParseLimitTimeParameter(pDaySettings, valueString))
            {
                DEVMENUCOMMAND_LOG("The value '%s' for --limit-time is not valid.\n", valueString);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            isChanged = true;
        }

        if (!isChanged)
        {
            PrintUsageTimerSet();
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!nn::pctl::IsRestrictionEnabled())
        {
            DEVMENUCOMMAND_LOG("Restriction is not enabled. Please set pin code first.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("*** Warning: The settings will be lost when synchronized to server.\n");
        }

        nn::pctl::SetPlayTimerSettingsForDebug(settings);

        // 設定反映後の状態を出力
        DumpTimerStatus();

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    ////////////////////////////////////////////////////////////////////////////
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    ////////////////////////////////////////////////////////////////////////////

    Result Pairing(bool* outValue, const Option& option)
    {
        auto code = option.GetTarget();
        if (std::string(code).empty())
        {
            DEVMENUCOMMAND_LOG("<pairing-code> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        int val = 0;
        if (!ParseInteger(&val, code) || val < 0)
        {
            DEVMENUCOMMAND_LOG("<pairing-code> must be an unsigned number. (specified: %s)\n", code);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("Pairing is already active.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Connecting...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::PairingInfo info;
        DEVMENUCOMMAND_LOG("Request pairing with code: '%s'...\n", code);
        NN_RESULT_DO(nn::pctl::RequestPairing(&info, code));

        nn::pctl::PairingAccountInfo accountInfo;
        info.GetAccountInfo(&accountInfo);
        char nickname[nn::account::NicknameBytesMax + 1] = {0};
        accountInfo.GetNickname(nickname, std::extent<decltype(nickname)>::value);

        DEVMENUCOMMAND_LOG("Pairing to account '%s'...\n", nickname);
        NN_RESULT_DO(nn::pctl::AuthorizePairing(&info));

        DEVMENUCOMMAND_LOG("Done.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpPairingStatus(bool* outValue, const Option&)
    {
        NN_LOG("IsPairingActive: %s\n", nn::pctl::IsPairingActive() ? "true" : "false");
        nn::time::PosixTime lastUpdated;
        if (!nn::pctl::GetSettingsLastUpdated(&lastUpdated))
        {
            DEVMENUCOMMAND_LOG("GetSettingsLastUpdated: not synchronized or invalid\n");
        }
        else
        {
            nn::time::CalendarTime calTime;
            NN_RESULT_DO(InitializeTime());
            auto result = nn::time::ToCalendarTime(&calTime, nullptr, lastUpdated);
            if (result.IsSuccess())
            {
                DEVMENUCOMMAND_LOG("GetSettingsLastUpdated: %04d/%02d/%02d %02d:%02d:%02d [yyyy/MM/dd hh:mm:ss]\n",
                    static_cast<int>(calTime.year), static_cast<int>(calTime.month), static_cast<int>(calTime.day),
                    static_cast<int>(calTime.hour), static_cast<int>(calTime.minute), static_cast<int>(calTime.second));
            }
            else
            {
                DEVMENUCOMMAND_LOG("Warning: nn::time::ToCalendarTime failed: result = 0x%08lX\n", result.GetInnerValueForDebug());
                DEVMENUCOMMAND_LOG("GetSettingsLastUpdated: %lld [PosixTime]\n", lastUpdated.value);
            }
            FinalizeTime();
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetOwner(bool* outValue, const Option&)
    {
        if (!nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("Pairing is not active.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Connecting...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::PairingInfo info;
        DEVMENUCOMMAND_LOG("Retrieve pairing info...\n");
        NN_RESULT_DO(nn::pctl::RetrievePairingInfo(&info));

        nn::pctl::PairingAccountInfo accountInfo;
        info.GetAccountInfo(&accountInfo);
        {
            char nickname[nn::account::NicknameBytesMax + 1];
            nickname[0] = 0;
            accountInfo.GetNickname(nickname, std::extent<decltype(nickname)>::value);

            DEVMENUCOMMAND_LOG("Owner's nickname: '%s'\n", nickname);
        }
        {
            size_t length = 0;
            NN_RESULT_DO(accountInfo.GetMiiImageContentType(&length, nullptr, 0));

            ++length;
            char* buffer = static_cast<char*>(malloc(sizeof(char) * (length)));
            accountInfo.GetMiiImageContentType(&length, buffer, length);
            buffer[length] = 0;
            DEVMENUCOMMAND_LOG("Owner's Mii image: content type = '%s'\n", buffer);
            free(buffer);
        }

        DEVMENUCOMMAND_LOG("Done.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SyncSettings(bool* outValue, const Option&)
    {
        if (!nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("Pairing is not active.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Connecting...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Synchronizing parental control settings...\n");
        nn::time::PosixTime lastUpdated;
        NN_RESULT_DO(nn::pctl::SynchronizeParentalControlSettings(&lastUpdated));

        auto result = InitializeTime();
        if (result.IsSuccess())
        {
            nn::time::CalendarTime calTime;
            result = nn::time::ToCalendarTime(&calTime, nullptr, lastUpdated);
            if (result.IsSuccess())
            {
                DEVMENUCOMMAND_LOG("Done. (last updated = %04d/%02d/%02d %02d:%02d:%02d [yyyy/MM/dd hh:mm:ss])\n",
                    static_cast<int>(calTime.year), static_cast<int>(calTime.month), static_cast<int>(calTime.day),
                    static_cast<int>(calTime.hour), static_cast<int>(calTime.minute), static_cast<int>(calTime.second));
            }
            else
            {
                DEVMENUCOMMAND_LOG("Warning: nn::time::ToCalendarTime failed: result = 0x%08lX\n", result.GetInnerValueForDebug());
            }
            FinalizeTime();
        }
        else
        {
            DEVMENUCOMMAND_LOG("Warning: nn::time::Initialize failed: result = 0x%08lX\n", result.GetInnerValueForDebug());
        }
        if (result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("Done. (last updated = %lld [PosixTime])\n", lastUpdated.value);
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Unlink(bool* outValue, const Option& option)
    {
        bool isForce = option.HasKey("--force");
        if (!nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("Pairing is not active; do nothing.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Connecting...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Unlinking...\n");
        NN_RESULT_DO(nn::pctl::UnlinkPairing(isForce));

        DEVMENUCOMMAND_LOG("Done.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeletePairing(bool* outValue, const Option&)
    {
        if (!nn::pctl::IsPairingActive())
        {
            DEVMENUCOMMAND_LOG("Pairing is not active; do nothing.\n");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::DeletePairing();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    ////////////////////////////////////////////////////////////////////////////

    Result ConfigGetIntermittentInterval(bool* outValue, const Option&)
    {
        int32_t intermittent_task_interval;
        GetSettingsFwdbgValue(&intermittent_task_interval, "intermittent_task_interval_seconds");

        DEVMENUCOMMAND_LOG("intermittent_interval = %d\n", intermittent_task_interval);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ConfigSetIntermittentInterval(bool* outValue, const Option& option)
    {
        return ConfigSettingInteger(outValue, option, "intermittent_task_interval_seconds");
    }

    Result ConfigGetPostEventInterval(bool* outValue, const Option&)
    {
        int32_t post_event_interval;
        GetSettingsFwdbgValue(&post_event_interval, "post_event_interval_seconds");

        DEVMENUCOMMAND_LOG("post_event_interval = %d\n", post_event_interval);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ConfigSetPostEventInterval(bool* outValue, const Option& option)
    {
        return ConfigSettingInteger(outValue, option, "post_event_interval_seconds");
    }

    Result DumpExemptAppList(bool* outValue, const Option&)
    {
        auto totalCount = nn::pctl::GetExemptApplicationListCountForDebug();
        static const int InfoCount = 50;
        nn::pctl::ExemptApplicationInfo infoArray[InfoCount];

        DEVMENUCOMMAND_LOG("--- Exempt Application List ---\n");
        DEVMENUCOMMAND_LOG("total count: %d\n", totalCount);
        for (int i = 0; i < totalCount; )
        {
            int c = (i + InfoCount > totalCount ? totalCount - i : InfoCount);
            c = nn::pctl::GetExemptApplicationListForDebug(infoArray, i, c);
            if (c == 0)
            {
                break;
            }
            for (int j = 0; j < c; ++j)
            {
                auto& info = infoArray[i + j];
                DEVMENUCOMMAND_LOG("[% 4d] 0x%016llX: %s\n",
                    i + j,
                    info.applicationId.value,
                    info.isExempted ? "true" : "false");
            }
            i += c;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetExemption(bool* outValue, const Option& option)
    {
        nn::pctl::ExemptApplicationInfo info;

        if (!option.HasKey("--application-id"))
        {
            DEVMENUCOMMAND_LOG("--application-id <id> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto appIdString = option.GetValue("--application-id");
        if (std::string(appIdString).empty())
        {
            DEVMENUCOMMAND_LOG("--application-id <id> is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!ParseApplicationId(&info.applicationId, appIdString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified for --application-id <id>. (specified: %s)\n", appIdString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto flagString = option.GetTarget();
        if (std::string(appIdString).empty())
        {
            DEVMENUCOMMAND_LOG("Boolean value is required.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (!ParseFlag(&info.isExempted, flagString))
        {
            DEVMENUCOMMAND_LOG("Invalid value specified. (specified: %s)\n", flagString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::pctl::AddToExemptApplicationListForDebug(info.applicationId);
        nn::pctl::UpdateExemptApplicationListForDebug(&info, 1);

        DEVMENUCOMMAND_LOG("Note: The local exempt application list has been updated. This will be overwritten when synchronizing parental control settings.");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result PostEvents(bool* outValue, const Option&)
    {
        nn::nifm::SubmitNetworkRequestAndWait();

        int outCount = 0;
        nn::pctl::detail::service::watcher::EventData eventData[200];

        NN_RESULT_TRY(nn::pctl::RequestPostEvent(&outCount, eventData, NN_ARRAY_SIZE(eventData)))
            NN_RESULT_CATCH(nn::pctl::ResultPairingNotActive)
            {
                DEVMENUCOMMAND_LOG("Need pairing.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        NN_LOG("Event Count = %d\n", outCount);
        auto eventCount = std::min(outCount, static_cast<int>(NN_ARRAY_SIZE(eventData)));
        for (int u = 0; u < eventCount; ++u)
        {
            auto& e = eventData[u];
            if (e.eventType == nn::pctl::detail::service::watcher::EventType::DidUserOpen ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidUserClose)
            {
                auto& uid = e.payload.openedUser.uid;
                NN_LOG("  [% 3lu] type = %s, timestamp = %lld, uid = %08x_%08x_%08x_%08x\n",
                    u, GetEventNameFromEventType(e.eventType), e.timestamp.value,
                    static_cast<uint32_t>(uid._data[0] >> 32),
                    static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                    static_cast<uint32_t>(uid._data[1] >> 32),
                    static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
            }
            else if (
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationLaunch ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationTerminate ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidAddNewManagedApplication ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidRemoveManagedApplication ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationPlay ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationDownloadStart ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationSuspend ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationResume ||
                e.eventType == nn::pctl::detail::service::watcher::EventType::DidLimitedApplicationLaunch
                )
            {
                NN_LOG("  [% 3lu] type = %s, timestamp = %lld, appId = 0x%016llX\n",
                    u, GetEventNameFromEventType(e.eventType), e.timestamp.value,
                    e.payload.applicationLaunch.applicationId.value);
            }
            else
            {
                NN_LOG("  [% 3lu] type = %s, timestamp = %lld\n",
                    u, GetEventNameFromEventType(e.eventType), e.timestamp.value);
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    ////////////////////////////////////////////////////////////////////////////
#endif
    ////////////////////////////////////////////////////////////////////////////

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " pctl set-pin <pin-code>\n"
        "       " DEVMENUCOMMAND_NAME " pctl unset-pin\n"
        "       " DEVMENUCOMMAND_NAME " pctl check-pin <pin-code>\n"
        "       " DEVMENUCOMMAND_NAME " pctl get-safety-level\n"
        "       " DEVMENUCOMMAND_NAME " pctl set-safety-level <level>\n"
        "       " DEVMENUCOMMAND_NAME " pctl dump-current-settings\n"
        "       " DEVMENUCOMMAND_NAME " pctl dump-custom-settings\n"
        "       " DEVMENUCOMMAND_NAME " pctl set-custom-settings [--rating-age <age>] [--sns-post <true|false>] [--free-communication <true|false>]\n"
        "       " DEVMENUCOMMAND_NAME " pctl set-rating-organization <organization>\n"
        "       " DEVMENUCOMMAND_NAME " pctl dump-free-communication-app-list\n"
        "       " DEVMENUCOMMAND_NAME " pctl set-free-communication-restriction <true|false> --application-id <id>\n"
        "       " DEVMENUCOMMAND_NAME " pctl delete-settings\n"
        "       " DEVMENUCOMMAND_NAME " pctl timer-start\n"
        "       " DEVMENUCOMMAND_NAME " pctl timer-stop\n"
        "       " DEVMENUCOMMAND_NAME " pctl timer-status\n"
        "       " DEVMENUCOMMAND_NAME " pctl timer-set [--timer-mode <mode>] [--using-week-settings <true|false>] [--week <week>] [--bedtime [<true|false>][,HH:mm]] [--limit-time [<true|false>][,<minutes>]]\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " pctl pairing <pairing-code>\n"
        "       " DEVMENUCOMMAND_NAME " pctl dump-pairing-status\n"
        "       " DEVMENUCOMMAND_NAME " pctl get-owner\n"
        "       " DEVMENUCOMMAND_NAME " pctl sync-settings\n"
        "       " DEVMENUCOMMAND_NAME " pctl unlink [--force]\n"
        "       " DEVMENUCOMMAND_NAME " pctl delete-pairing\n"
        "       " DEVMENUCOMMAND_NAME " pctl config-get-intermittent-interval\n"
        "       " DEVMENUCOMMAND_NAME " pctl config-set-intermittent-interval [interval]\n"
        "       " DEVMENUCOMMAND_NAME " pctl config-get-post-event-interval\n"
        "       " DEVMENUCOMMAND_NAME " pctl config-set-post-event-interval [interval]\n"
        "       " DEVMENUCOMMAND_NAME " pctl dump-exempt-app-list-offline\n"
        "       " DEVMENUCOMMAND_NAME " pctl set-exemption-offline <true|false> --application-id <id>\n"
        "       " DEVMENUCOMMAND_NAME " pctl post-events\n"
#endif
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"set-pin",                            SetPin},
        {"unset-pin",                          UnsetPin},
        {"check-pin",                          CheckPin},
        {"get-safety-level",                   GetSafetyLevel},
        {"set-safety-level",                   SetSafetyLevel},
        {"dump-current-settings",              DumpCurrentSettings},
        {"dump-custom-settings",               DumpCustomSettings},
        {"set-custom-settings",                SetCustomSettings},
        {"set-rating-organization",            SetRatingOrganization},
        {"dump-free-communication-app-list",   DumpFreeCommunicationAppList},
        {"set-free-communication-restriction", SetFreeCommunicationRestriction},
        {"delete-settings",                    DeleteSettings},
        {"timer-start",                        TimerStart},
        {"timer-stop",                         TimerStop},
        {"timer-status",                       TimerStatus},
        {"timer-set",                          TimerSet},
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        {"pairing",                            Pairing},
        {"dump-pairing-status",                DumpPairingStatus},
        {"get-owner",                          GetOwner},
        {"sync-settings",                      SyncSettings},
        {"unlink",                             Unlink},
        {"delete-pairing",                     DeletePairing},
        {"config-get-intermittent-interval",   ConfigGetIntermittentInterval},
        {"config-set-intermittent-interval",   ConfigSetIntermittentInterval},
        {"config-get-post-event-interval",     ConfigGetPostEventInterval},
        {"config-set-post-event-interval",     ConfigSetPostEventInterval},
        {"dump-exempt-app-list-offline",       DumpExemptAppList},
        {"set-exemption-offline",              SetExemption},
        {"post-events",                        PostEvents},
#endif
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result PctlCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu pctl command. See '" DEVMENUCOMMAND_NAME " pctl --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
