﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <string>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_PatchCommand.h"
#include "DevMenuCommand_InstallUtil.h"

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StorageId.h"
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Stopwatch.h"

#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Mount.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ns/ns_Result.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

using namespace nn;

namespace
{
const char* ControlMountName = "devmenuControl";

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
Result FindPatchFromDelta(util::optional<ncm::ContentMetaKey>* patch, ncm::ApplicationId* applicationId, const ncm::StorageContentMetaKey& delta)
{
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, delta.storageId));

    const int bufferSize = 16 << 10; // 16KiB
    std::unique_ptr<Bit8[]> data(new Bit8[bufferSize]);

    size_t size;
    NN_RESULT_DO(db.Get(&size, data.get(), bufferSize, delta.key));
    ncm::ContentMetaReader reader(data.get(), size);
    ncm::ContentMetaKey deltaKey = delta.key; // GetApplicationId が const を受け付けないためコピー
    auto appId = reader.GetApplicationId(deltaKey);

    ncm::ContentMetaKey key;
    auto listcount = db.ListContentMeta(&key, 1, ncm::ContentMetaType::Patch, *appId);
    if (listcount.listed == 0)
    {
        *patch = nullptr;
        NN_RESULT_SUCCESS;
    }
    // TODO 複数見つかった場合、とりあえず先頭に対して適用する
    *applicationId = *appId;
    *patch = key;
    NN_RESULT_SUCCESS;
}

Result GetApplicationIdOfPatch(util::optional<ncm::ApplicationId>* applicationId, const ncm::StorageContentMetaKey& patch)
{
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, patch.storageId));

    const int bufferSize = 16 << 10; // 16KiB
    std::unique_ptr<Bit8[]> data(new Bit8[bufferSize]);

    size_t size;
    NN_RESULT_DO(db.Get(&size, data.get(), bufferSize, patch.key));
    ncm::ContentMetaReader reader(data.get(), size);
    ncm::ContentMetaKey patchKey = patch.key; // GetApplicationId が const を受け付けないためコピー
    *applicationId = reader.GetApplicationId(patchKey);

    NN_RESULT_SUCCESS;
}
#endif


nn::Result GetControlPath(char* outPath, size_t size, const ncm::StorageContentMetaKey& key)
{
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, key.storageId));

    ncm::ContentId contentId;
    NN_RESULT_DO(db.GetContentIdByType(&contentId, key.key, ncm::ContentType::Control));

    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, key.storageId));

    ncm::Path path;
    storage.GetPath(&path, contentId);

    util::Strlcpy(outPath, path.string, static_cast<int>(size));

    NN_RESULT_SUCCESS;
}


// Control を読み込む。ns を介する場合、メインアプリケーションがないと取得できないため、その回避
// とりあえず今回の用途ではアイコンはいらないので除外
nn::Result ReadApplicationControl(size_t* outSize, void* buffer, size_t bufferSize, const nn::ncm::StorageContentMetaKey& key, nn::ncm::ApplicationId applicationId)
{
    NN_UNUSED(bufferSize);

    // 参考にするのは ns_ApplicationControlFileUtil.cpp
    const size_t propertySize = sizeof(ns::ApplicationControlProperty);

    char contentPath[256];
    NN_RESULT_DO(GetControlPath(contentPath, sizeof(contentPath), key));
    NN_RESULT_DO(fs::MountContent(ControlMountName, contentPath, applicationId, fs::ContentType_Control));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(ControlMountName); };

    char filePath[256];
    util::SNPrintf(filePath, sizeof(filePath), "%s:/control.nacp", ControlMountName);
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
    NN_RESULT_DO(fs::ReadFile(file, 0, buffer, propertySize));
    *outSize = propertySize;

    NN_RESULT_SUCCESS;
}

//! Use in ListCommandCommon(bool* outValue, const Option& option, ncm::ContentMetaType type)
struct ListPatches
{
    Bit64               ownerId;            //! owner application identifier.
    uint16_t            version;            //! release-version.
    ncm::StorageId      installedStorageId; //! installed-storage identifer.
    Bit8                padding;
    char                displayVersion[16]; //! display-version.

    class Collection : public std::vector<ListPatches>
    {
    public:
        explicit Collection(size_type initialCapacity = 16)
            : std::vector<ListPatches>()
        {
            std::vector<ListPatches>::reserve(initialCapacity);
        }

        Result Search(const ncm::StorageId storageId, const ncm::ContentMetaType type, const int bufferSize = 16 * 1024, const int maxKeys = 512)
        {
            if (ncm::StorageId::Card == storageId)
            {
                NN_RESULT_DO(ns::EnsureGameCardAccess());
            }
            else if (ncm::StorageId::SdCard == storageId)
            {
                NN_RESULT_DO(ns::CheckSdCardMountStatus());
            }
            std::unique_ptr<ncm::ContentMetaKey[]> keys(new ncm::ContentMetaKey[maxKeys]);
            std::unique_ptr<Bit8[]> data(new Bit8[bufferSize]);

            ncm::ContentMetaDatabase db;
            NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
            auto listcount = db.ListContentMeta(keys.get(), maxKeys, type, ncm::ContentMetaDatabase::AnyApplicationId);

            for (int i = 0; i < listcount.listed; ++i)
            {
                // ApplicationId の取得
                size_t size;
                NN_RESULT_DO(db.Get(&size, data.get(), bufferSize, keys[i]));
                ncm::ContentMetaReader reader(data.get(), size);
                auto appId = reader.GetApplicationId(keys[i]);

                ListPatches item;
                item.ownerId = appId->value;
                item.installedStorageId = storageId;
                item.version = static_cast<uint16_t>(keys[i].version >> 16);
                item.displayVersion[0] = '\0';

                if (ncm::ContentMetaType::Delta != type)
                {
                    ncm::StorageContentMetaKey skey = {keys[i], storageId};
                    std::unique_ptr<Bit8> control(new Bit8[sizeof(ns::ApplicationControlProperty)]);
                    size_t controlSize;
                    NN_RESULT_DO(ReadApplicationControl(&controlSize, control.get(),
                        sizeof(ns::ApplicationControlProperty), skey,
                        appId.value()
                    ));

                    ns::ApplicationControlDataAccessor accessor(control.get(), controlSize);
                    nn::util::Strlcpy(item.displayVersion, accessor.GetProperty().displayVersion,
                        sizeof(item.displayVersion) / sizeof(item.displayVersion[0])
                    );
                }
                push_back(item);
            }
            NN_RESULT_SUCCESS;
        }
    };
};

Result ListCommandCommon(bool* outValue, const Option& option, ncm::ContentMetaType type)
{
    ListPatches::Collection collection;
    const auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : util::nullopt;

    if (storage)
    {
        // 従来
        NN_RESULT_DO(collection.Search(*storage, type));
    }
    else
    {
        // ビルトインは常時検索。
        NN_RESULT_DO(collection.Search(ncm::StorageId::BuiltInUser, type));

        // SdCard は未挿入状態、データベース無効時はスキップする。
        if (ns::CheckSdCardMountStatus().IsSuccess())
        {
            auto result = collection.Search(ncm::StorageId::SdCard, type);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ncm::ResultContentMetaDatabaseNotActive)
                NN_RESULT_CATCH_ALL
                {
                    // 想定外はログ出力して通知。但しコマンド自体は FAILURE にはしません。
                    NN_LOG("SdCard acccess failed, result => 0x%08x.\n", result.GetInnerValueForDebug());
                }
            NN_RESULT_END_TRY
        }

        // GameCard は未挿入状態、データベース無効時はスキップする。
        if (fs::IsGameCardInserted())
        {
            auto result = collection.Search(ncm::StorageId::Card, type);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ncm::ResultContentMetaDatabaseNotActive)
                NN_RESULT_CATCH_ALL
                {
                    // 想定外はログ出力して通知。但しコマンド自体は FAILURE にはしません。
                    NN_LOG("GameCard acccess failed, result => 0x%08x.\n", result.GetInnerValueForDebug());
                }
            NN_RESULT_END_TRY
        }
    }

    const auto listedCount = static_cast<int>(collection.size());
    NN_LOG("%d patch(es) found.\n", listedCount);
    NN_LOG("id                  display-version   release-version   installed-storage\n");
    for (int i = 0; i < listedCount; ++i)
    {
        const auto& data = collection[i];
        NN_LOG("0x%016llx  %15s               %3u        %12s\n",
            data.ownerId,
            data.displayVersion,
            data.version,
            ToStringFromStorageId(data.installedStorageId)
        );
    }
    *outValue = true;
    NN_RESULT_SUCCESS;
}

    Result InstallCommand(bool* outValue, const Option& option)
    {
        ncm::StorageContentMetaKey key;
        int outCount;
        NN_RESULT_DO(devmenuUtil::InstallCommandCommon(&key, &outCount, 1, outValue, option, ncm::ContentMetaType::Patch));
        NN_RESULT_SUCCESS;
    }
    Result UninstallCommand(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }
        auto idString = option.GetTarget();
        ncm::ApplicationId appId = { STR_TO_ULL(idString, NULL, 16) };

        ns::DeleteRedundantApplicationEntity();
        NN_RESULT_DO(devmenuUtil::DeleteBasedOnDatabase(ncm::ContentMetaType::Patch, appId, *storage));
        NN_RESULT_DO(devmenuUtil::PushApplicationRecordBasedOnDatabase(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result ListCommand(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ListCommandCommon(outValue, option, ncm::ContentMetaType::Patch));
        NN_RESULT_SUCCESS;
    }
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result InstallDeltaCommand(bool* outValue, const Option& option)
    {
        ncm::StorageContentMetaKey key;
        int outCount;
        NN_RESULT_DO(devmenuUtil::InstallCommandCommon(&key, &outCount, 1, outValue, option, ncm::ContentMetaType::Delta));
        NN_RESULT_SUCCESS;
    }
    Result UninstallDeltaCommand(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }
        auto idString = option.GetTarget();
        ncm::ApplicationId appId = { STR_TO_ULL(idString, NULL, 16) };

        ns::DeleteRedundantApplicationEntity();
        NN_RESULT_DO(devmenuUtil::DeleteBasedOnDatabase(ncm::ContentMetaType::Delta, appId, *storage));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result ApplyDeltaCommand(bool* outValue, const Option& option)
    {
        // まずは普通にインストール
        ncm::StorageContentMetaKey key;
        int outCount;
        NN_RESULT_DO(devmenuUtil::InstallCommandCommon(&key, &outCount, 1, outValue, option, ncm::ContentMetaType::Delta));

        if(*outValue == false)
        {
            // 成功していない場合は終わり
            NN_RESULT_SUCCESS;
        }
        *outValue = false;

        // Delta から ApplicationId を取得 -> Patch の key を取得して、Apply
        util::optional<ncm::ContentMetaKey> patch;
        ncm::ApplicationId appId;
        NN_RESULT_DO(FindPatchFromDelta(&patch, &appId, key));
        if(!patch)
        {
            // 適用対象が見つからなかった
            NN_RESULT_SUCCESS;
        }
        ncm::ApplyDeltaTask task;
        ncm::StorageContentMetaKey patch2 = {*patch, key.storageId};
        ncm::ApplyDeltaTaskBase::TaskState taskState;
        const int ApplyBufferSize = 256 << 10; // 256KB
        std::unique_ptr<Bit8[]> buffer(new Bit8[ApplyBufferSize]);

        NN_RESULT_DO(task.SetBuffer(buffer.get(), ApplyBufferSize));
        NN_RESULT_DO(task.Initialize(patch2, key, &taskState));
        NN_RESULT_DO(task.Prepare());
        NN_RESULT_DO(devmenuUtil::PushApplicationRecordBasedOnDatabase(appId)); // prepare 時点で一度パッチが見えなくなる

        NN_RESULT_DO(devmenuUtil::WaitInstallTaskExecute(&task));
        NN_RESULT_DO(task.Commit());

        NN_RESULT_DO(devmenuUtil::PushApplicationRecordBasedOnDatabase(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ListDeltaCommand(bool* outValue, const Option& option)
    {
        NN_RESULT_DO(ListCommandCommon(outValue, option, ncm::ContentMetaType::Delta));
        NN_RESULT_SUCCESS;
    }
    Result ListContents(const nn::ncm::ContentMetaKey& key, ncm::StorageId storageId)
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

        ncm::ContentInfo buffer[16];
        int offset = 0;
        for (;;)
        {
            int outCount;
            const int count = sizeof(buffer) / sizeof(ncm::ContentInfo);
            NN_RESULT_DO(db.ListContentInfo(&outCount, buffer, count, key, offset));
            for (int i = 0; i < outCount; i++)
            {
                bool has;
                storage.Has(&has, buffer[i].id);
                NN_LOG(" - %02x%02x%02x%02x (%s)\n", buffer[i].id.data[0], buffer[i].id.data[1], buffer[i].id.data[2], buffer[i].id.data[3], has ? "installed" : "not installed");
            }
            if (outCount < count)
            {
                break;
            }
            offset += outCount;
        }
        NN_RESULT_SUCCESS;
    }

    Result ListUpdateRelatesCommand(bool* outValue, const Option& option)
    {
        // 更新に関係するものを一覧する
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, *storage));
        const int maxKeys = 1024;
        std::unique_ptr<nn::ncm::ContentMetaKey[]> keys(new nn::ncm::ContentMetaKey[maxKeys]);

        auto count = db.ListContentMeta(keys.get(), maxKeys, nn::ncm::ContentMetaType::Delta);
        for(int i = 0; i < count.listed; ++i)
        {
            NN_LOG("Delta, %016llx, %d\n", keys[i].id, keys[i].version);
            ListContents(keys[i], *storage);
        }
        count = db.ListContentMeta(keys.get(), maxKeys, nn::ncm::ContentMetaType::Patch, nn::ncm::ContentMetaDatabase::AnyApplicationId, 0, 0xffffffffffffffff, nn::ncm::ContentInstallType::Unknown);
        for(int i = 0; i < count.listed; ++i)
        {
            NN_LOG("Patch, %016llx, %d (%s)\n", keys[i].id, keys[i].version,
                   keys[i].installType == nn::ncm::ContentInstallType::FragmentOnly ? "FragmentOnly" : "Full");
            ListContents(keys[i], *storage);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result RemoveUpdateRelatesCommand(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        if (!storage)
        {
            NN_RESULT_THROW(ncm::ResultUnknownStorage());
        }
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInUser));

        const int maxKeys = 1024;
        std::unique_ptr<nn::ncm::ContentMetaKey[]> keys(new nn::ncm::ContentMetaKey[maxKeys]);

        auto count = db.ListContentMeta(keys.get(), maxKeys, nn::ncm::ContentMetaType::Delta);
        for(int i = 0; i < count.listed; ++i)
        {
            NN_LOG("Remove Delta, %016llx, %d\n", keys[i].id, keys[i].version);
            NN_RESULT_DO(devmenuUtil::Delete(keys[i], *storage));
        }
        count = db.ListContentMeta(keys.get(), maxKeys, nn::ncm::ContentMetaType::Patch, nn::ncm::ContentMetaDatabase::AnyApplicationId, 0, 0xffffffffffffffff, nn::ncm::ContentInstallType::Unknown);
        for(int i = 0; i < count.listed; ++i)
        {
            NN_LOG("Remove Patch, %016llx, %d (%s)\n", keys[i].id, keys[i].version,
                   keys[i].installType == nn::ncm::ContentInstallType::FragmentOnly ? "FragmentOnly" : "Full");
            util::optional<ncm::ApplicationId> appId;
            ncm::StorageContentMetaKey skey = {keys[i], *storage};
            NN_RESULT_DO(GetApplicationIdOfPatch(&appId, skey));
            NN_RESULT_DO(devmenuUtil::Delete(keys[i], *storage));
            if (appId)
            {
                NN_RESULT_DO(devmenuUtil::PushApplicationRecordBasedOnDatabase(*appId));
            }
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result SetPreferDeltaCommand(bool* outValue, const Option& option)
    {
        *outValue = false;

        auto t = option.GetTarget();
        bool flag;
        if (std::string(t) == "true")
        {
            flag = true;
        }
        else if (std::string(t) == "false")
        {
            flag = false;
        }
        else
        {
            NN_LOG("true or false should be specified\n");
            NN_RESULT_SUCCESS;
        }
        if (nn::settings::fwdbg::GetSettingsItemValueSize( "nim.install", "prefer_delta_evenif_inefficient" ) > 0)
        {
            settings::fwdbg::SetSettingsItemValue("nim.install", "prefer_delta_evenif_inefficient", &flag, sizeof(flag));
            *outValue = true;
        }
        else
        {
            NN_LOG("fwdbg doesn't seem to have the parameter\n");
        }

        NN_RESULT_SUCCESS;
    }

#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " patch install <absolute_nsp_path> [--force] [--skip-patch-combination-check]\n"
        "       " DEVMENUCOMMAND_NAME " patch list\n"
        "       " DEVMENUCOMMAND_NAME " patch uninstall <application_id>\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " patch install-delta <absolute_nsp_path>\n"
        "       " DEVMENUCOMMAND_NAME " patch list-delta\n"
        "       " DEVMENUCOMMAND_NAME " patch uninstall-delta <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " patch list-update-relates\n"
        "       " DEVMENUCOMMAND_NAME " patch remove-update-relates\n"
        "       " DEVMENUCOMMAND_NAME " patch set-prefer-delta <true|false>\n"
#endif
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "install", InstallCommand },
        { "list", ListCommand },
        { "uninstall", UninstallCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "install-delta", InstallDeltaCommand },
        { "list-delta", ListDeltaCommand },
        { "apply-delta", ApplyDeltaCommand },
        { "uninstall-delta", UninstallDeltaCommand },
        { "list-update-relates", ListUpdateRelatesCommand },
        { "remove-update-relates", RemoveUpdateRelatesCommand },
        { "set-prefer-delta", SetPreferDeltaCommand },
#endif
    };
}

Result PatchCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    // TORIAEZU : ns::InitializeForSystemUpdate() を呼び出しただけでは接続に成功して待たないため、
    // 動作上無意味な API を読んで初期化が終わるまで待つようにする
    ns::GetBackgroundNetworkUpdateState();

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu patch command. See '" DEVMENUCOMMAND_NAME " patch --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
