﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ProgramIdNameMap.h"
#include "DevMenuCommand_NpnsCommand.h"
#include "DevMenuCommand_HttpConnection.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    typedef kvdb::BoundedString<8192> LongUrl;
    typedef kvdb::BoundedString<8192> LongHeader;
    typedef kvdb::BoundedString<8192> PostData;

    const char* GetNdidAccessToken()
    {
        return "4a2427b0fcb0b0fcd15bbd27df7818eb";
    }

    bool ReadNotificationData(char* outBuffer, size_t bufferSize, const char* fileName)
    {
        NN_UNUSED(bufferSize);

        fs::FileHandle handle;
        NN_RESULT_TRY(fs::OpenFile(&handle, fileName, fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            return false;
        }
        NN_RESULT_END_TRY
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };

        int64_t size;
        NN_RESULT_DO(fs::GetFileSize(&size, handle));
        NN_SDK_ASSERT(static_cast<size_t>(size) <= bufferSize);
        NN_RESULT_DO(fs::ReadFile(handle, 0, outBuffer, static_cast<size_t>(size)));
        outBuffer[size] = '\0';

        return true;
    }
#endif

    template <typename T>
    bool GetSettingsFwdbgValue(T* pOut, const char* keyString)
    {
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(pOut, sizeof(T), "npns", keyString);
        bool bResult = (bytes == sizeof(T));
        if (!bResult)
        {
            NN_LOG("Failed to read settings: %s\n", keyString);
            *pOut = -1;
        }
        return bResult;
    }

    template <typename T>
    void SetSettingsFwdbgValue(T value, const char* keyString)
    {
        nn::settings::fwdbg::SetSettingsItemValue("npns", keyString, &value, sizeof(T));
    }

    bool ParseInteger(int32_t* pOut, const char* pString)
    {
        char* pEnd;
        *pOut = static_cast<int32_t>(std::strtoll(pString, &pEnd, 10));
        return pEnd != nullptr && *pEnd == '\0';
    }

    Result WriteIntegerValue(bool* outValue, const Option& option, const char* pKey)
    {
        int32_t value;
        if (!ParseInteger(&value, option.GetTarget())|| value <= 0)
        {
            NN_LOG("%s is not valid value.\n", option.GetTarget());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        SetSettingsFwdbgValue(value, pKey);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetSleepConfig(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        int32_t sleep_processing_timeout, sleep_periodic_interval;
        GetSettingsFwdbgValue(&sleep_processing_timeout, "sleep_processing_timeout");
        GetSettingsFwdbgValue(&sleep_periodic_interval, "sleep_periodic_interval");

        NN_LOG("sleep_processing_timeout = %d, sleep_periodic_interval = %d\n", sleep_processing_timeout, sleep_periodic_interval);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSleepProcessingTimeout(bool* outValue, const Option& option)
    {
        return WriteIntegerValue(outValue, option, "sleep_processing_timeout");
    }

    Result SetSleepPeriodicInterval(bool* outValue, const Option& option)
    {
        return WriteIntegerValue(outValue, option, "sleep_periodic_interval");
    }

    Result GetBackgroundProcessingStatus(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isEnabled;
        nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, 1, "npns", "background_processing");

        NN_LOG("%s\n", isEnabled ? "Enabled" : "Disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetBackgroundProcessingStatus(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is not valid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("npns", "background_processing", &isEnabled, 1);

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetLogManagerRedirectionStatus(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        bool isEnabled;
        nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, 1, "npns", "logmanager_redirection");

        NN_LOG("%s\n", isEnabled ? "Enabled" : "Disabled");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetLogManagerRedirectionStatus(bool* outValue, const Option& option)
    {
        auto modeString = option.GetTarget();
        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is not valid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("npns", "logmanager_redirection", &isEnabled, 1);

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const account::Uid TestUid = {{0x00ull, 0x00ull}};
    Result CreateToken(bool* outValue, const Option& option)
    {
        ApplicationId appId;
        if (option.HasKey("--id"))
        {
            appId.value = std::strtoll(option.GetValue("--id"), NULL, 16);
        }
        else
        {
            ProgramIdNameMap programIdNameMap;
            programIdNameMap.Initialize();
            auto idByName = programIdNameMap.GetId("DevMenuCommand");
            appId.value = *idByName;
        }
        Result result;
        nn::npns::NotificationToken token;
        *outValue = false;
        NN_RESULT_DO(nn::npns::InitializeForSystem());
        result = nn::npns::CreateToken(&token, TestUid, appId);
        if (result.IsSuccess())
        {
            NN_LOG("Test token created: %s\n", token.data);
            NN_LOG(
                "*** You can send a notification by the following command ***\n"
                "curl --request POST \\\n"
                "  --proxy http://proxy.example.com:8080 \\\n"
                "  --url https://provider-XX1.npns.srv.nintendo.net/api/v1/notifications \\\n"
                "  --header 'content-type: application/json' \\\n"
                "  --header 'X-Ndid-AccessToken: ACCESS-TOKEN' \\\n"
                "  --data '{\"requests\": [{\"to\": \"%s\", \"data\": \"test\", \"store_offline\": false }]}'\n", token.data);
            *outValue = true;
        }
        nn::npns::FinalizeForSystem();
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result Publish(bool* outValue, const Option& option)
    {
        std::unique_ptr<char[]> buffer(new char[1024]);
        if (option.HasTarget())
        {
            if (!ReadNotificationData(buffer.get(), 1024, option.GetTarget()))
            {
                DEVMENUCOMMAND_LOG("Target file is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " npns publish <notification_data_path> --id <application_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ApplicationId appId;
        if (option.HasKey("--id"))
        {
            appId.value = std::strtoll(option.GetValue("--id"), NULL, 16);
        }
        else
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " npns publish <notification_data_path> --id <application_id>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(npns::InitializeForSystem());

        npns::NotificationToken token;
        NN_RESULT_DO(npns::CreateToken(&token, {}, appId));

        auto url = std::unique_ptr<LongUrl>(new LongUrl());
        url->Assign("https://provider-%.npns.srv.nintendo.net/api/v1/notifications");

        auto accessToken = std::unique_ptr<LongHeader>(new LongHeader());
        accessToken->AssignFormat("X-Ndid-AccessToken: %s", GetNdidAccessToken());

        const char* headerList[] = { "Content-Type: application/json", accessToken->Get() };

        auto postData = std::unique_ptr<PostData>(new PostData());
        postData->AssignFormat(
            "{\n"
            "   \"requests\": [\n"
            "       {\n"
            "           \"to\": \"%s\",\n"
            "           \"data\": \"%s\",\n"
            "           \"store_offline\": %s\n"
            "       }\n"
            "   ]\n"
            "}\n",
            token.data, buffer.get(), "false");

        char readBuffer[1024];
        size_t readSize = 0;
        HttpConnection connection;
        NN_RESULT_DO(connection.Initialize());
        NN_RESULT_DO(connection.Post(url->Get(), postData->Get(),
            [&readSize, &readBuffer](const void* buffer, size_t bufferSize) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + bufferSize < sizeof(readBuffer), npns::ResultConnectionFailed());
                std::memcpy(&readBuffer[readSize], buffer, bufferSize);
                readSize += bufferSize;

                NN_RESULT_SUCCESS;
            }, headerList, sizeof(headerList) / sizeof(headerList[0])));
        readBuffer[readSize] = '\0';

        npns::FinalizeForSystem();

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
#endif

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " npns get-background-processing-status\n"
        "       " DEVMENUCOMMAND_NAME " npns set-background-processing-status <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " npns get-logmanager-redirection-status\n"
        "       " DEVMENUCOMMAND_NAME " npns set-logmanager-redirection-status <enable|disable>\n"
        "       " DEVMENUCOMMAND_NAME " npns get-sleep-config\n"
        "       " DEVMENUCOMMAND_NAME " npns set-sleep-processing-timeout <sleep_processing_timeout>\n"
        "       " DEVMENUCOMMAND_NAME " npns set-sleep-periodic-interval <sleep_periodic_interval>\n"
        "       " DEVMENUCOMMAND_NAME " npns create-token-test\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " npns publish <notification_data_path> --id <application_id>\n"
#endif
       ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"get-background-processing-status", GetBackgroundProcessingStatus},
        {"set-background-processing-status", SetBackgroundProcessingStatus},
        {"get-logmanager-redirection-status", GetLogManagerRedirectionStatus},
        {"set-logmanager-redirection-status", SetLogManagerRedirectionStatus},
        {"get-sleep-config", GetSleepConfig},
        {"set-sleep-processing-timeout", SetSleepProcessingTimeout},
        {"set-sleep-periodic-interval",  SetSleepPeriodicInterval},
        {"create-token-test",  CreateToken},
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        {"publish", Publish},
#endif
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result NpnsCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu npns command. See '" DEVMENUCOMMAND_NAME " npns --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
