﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/mii.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/mii/detail/mii_DatabaseFile.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_MiiCommand.h"
#include "DevMenuCommand_Eula.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_DebugCommand.h"

#include <nn/mii/mii_ImageDatabase.h>

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " mii get-testing-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " mii enable-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " mii disable-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " mii export-database <absolute_host_file_path>\n"
        "       " DEVMENUCOMMAND_NAME " mii import-database <absolute_host_file_path>\n"
        "       " DEVMENUCOMMAND_NAME " mii export-char-info <absolute_host_directory_path>\n"
        "       " DEVMENUCOMMAND_NAME " mii format-database\n"
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " mii get-img-testing-mode-enabled\n"
        "       " DEVMENUCOMMAND_NAME " mii enable-img-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " mii disable-img-testing-mode\n"
        "       " DEVMENUCOMMAND_NAME " mii delete-img-database\n"
        "       " DEVMENUCOMMAND_NAME " mii destroy-img-database\n"
        "       " DEVMENUCOMMAND_NAME " mii import-img-database <absolute_host_file_path>\n"
        "       " DEVMENUCOMMAND_NAME " mii export-img-database <absolute_host_file_path>\n"
#endif
        "       "
        ""; // 終端

    nn::Result CreateDirectory(const char* pFolderPath)
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(pFolderPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists){}
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    nn::Result OutputFile(const char* pPath,const void* pSrc,size_t size)
    {
        /// 既にファイルがあれば削除して再生成
        NN_RESULT_TRY(nn::fs::DeleteFile(pPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound){}
        NN_RESULT_END_TRY
        NN_RESULT_DO(nn::fs::CreateFile( pPath, 0 ));

        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, pPath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, pSrc, size,nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    template<class T>
    nn::Result OutputFile(const char* pPath,const T& obj)
    {
        return OutputFile(pPath,&obj,sizeof(obj));
    }

    class LogFile {
    public:
        NN_IMPLICIT LogFile(size_t bufferSize = 100 * 1024)
        : m_Buffer(new char[bufferSize])
        , m_BufferSize(bufferSize)
        , m_Index(0)
        {
        }
        ~LogFile()
        {
        }

        void Append(const char* pMessage)
        {
            auto length = strlen(pMessage);
            NN_ABORT_UNLESS_LESS(m_Index + length, m_BufferSize);
            strncpy(&m_Buffer[m_Index],pMessage, m_BufferSize - m_Index);
            m_Index += length;
        }

        void AppendFormat(const char* pFormat,...)
        {
            auto* pBuffer = &m_Buffer[m_Index];

            std::va_list list;
            va_start(list, pFormat);
            m_Index += nn::util::VSNPrintf(pBuffer, m_BufferSize - m_Index, pFormat, list);
            va_end(list);
        }

        /// リセットする
        void Reset()
        {
            m_Index = 0;
        }

        /// ファイルに書き込む
        nn::Result Write(const char* pPath)
        {
            return OutputFile(pPath,m_Buffer.get(),m_Index);
        }
    private:
        std::unique_ptr<char[]> m_Buffer;
        size_t m_BufferSize;
        size_t m_Index;
    };

    Result GetBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key)
    {
        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), name, key);

        if( sizeof(isEnabled) == size )
        {
            NN_LOG("%s\n", isEnabled ? "enabled" : "disabled");
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_LOG("Key is not found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }
    Result SetBooleanFirmwareDebugSetting(bool *outValue, bool isEnabled, const char* name, const char* key, bool needsReboot)
    {
        settings::fwdbg::SetSettingsItemValue(name, key, &isEnabled, sizeof(isEnabled));
        if (needsReboot)
        {
            NN_LOG("Please restart your target to apply this change.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
    Result EnableBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key, bool needsReboot = false)
    {
        return SetBooleanFirmwareDebugSetting(outValue, true, name, key, needsReboot);
    }

    Result DisableBooleanFirmwareDebugSetting(bool* outValue, const char* name, const char* key, bool needsReboot = false)
    {
        return SetBooleanFirmwareDebugSetting(outValue, false, name, key, needsReboot);
    }

    bool IsEnabledMiiTestingMode(bool* outValue)
    {
        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "mii", "is_db_test_mode_enabled");
        if(sizeof(isEnabled) == size && isEnabled)
        {
            *outValue = true;
            return true;
        }
        NN_LOG("Disabled mii-testing-mode\n");
        *outValue = false;
        return false;
    }
    Result GetMiiTestingModeEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "mii", "is_db_test_mode_enabled");
    }

    Result EnableMiiTestingMode(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "mii", "is_db_test_mode_enabled");
    }

    Result DisableMiiTestingMode(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "mii", "is_db_test_mode_enabled");
    }
    nn::Result ExportDatabaseImpl(bool* pOutValue,const char* pFilePath) NN_NOEXCEPT
    {
        *pOutValue = false;
        NN_LOG("exporting %s\n",pFilePath);

        /// データベースエクスポート
        std::unique_ptr<nn::mii::detail::ImportFile> importFile(new nn::mii::detail::ImportFile);
        nn::mii::Database database;
        NN_RESULT_DO(database.Initialize());
        NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
        database.Export(importFile.get(),sizeof(*importFile));

        /// ファイルとして出力
        NN_RESULT_DO(OutputFile(pFilePath,*importFile));

        *pOutValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result ImportDatabaseImpl(bool* pOutValue,const char* pFilePath) NN_NOEXCEPT
    {
        *pOutValue = false;
        NN_LOG("importing %s\n",pFilePath);

        /// ファイルを読み込み
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, pFilePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        std::unique_ptr<nn::mii::detail::ImportFile> importFile(new nn::mii::detail::ImportFile);
        NN_RESULT_DO(nn::fs::ReadFile(file, 0, importFile.get(), sizeof(*importFile)));

        /// 読み込んだファイルをデータベースにインポート
        nn::mii::Database database;
        NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
        NN_RESULT_DO(database.Initialize());
        NN_RESULT_DO(database.Import(importFile.get(),sizeof(*importFile)));
        *pOutValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result ExportCharInfoImpl(bool* pOutValue,const char* pFolderPath) NN_NOEXCEPT
    {
        static const auto MaxPathLength = 300;

        *pOutValue = false;

        /// 時計の初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());
        NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };

        /// パス生成用に現在の日時を取得
        nn::time::PosixTime posixTime;
        NN_RESULT_DO(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
        nn::time::CalendarTime calenderTime;
        NN_RESULT_DO(nn::time::ToCalendarTime(&calenderTime, nullptr, posixTime));

        /// 日時を使って出力パスを決める
        char path[MaxPathLength];
        sprintf(path,"%s/%04d%02d%02d%02d%02d%02d"
            ,pFolderPath
            ,calenderTime.year
            ,calenderTime.month
            ,calenderTime.day
            ,calenderTime.hour
            ,calenderTime.minute
            ,calenderTime.second
            );

        /// 出力先ディレクトリを作成
        NN_RESULT_DO(CreateDirectory(path));


        /// CharInfoを取り出す
        int miiCount;
        std::unique_ptr<nn::mii::CharInfo[]> charInfoArray(new nn::mii::CharInfo[nn::mii::DatabaseMiiCount]);
        nn::mii::Database database;
        NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
        NN_RESULT_DO(database.Initialize());
        NN_RESULT_DO(database.Get(&miiCount,charInfoArray.get(),nn::mii::DatabaseMiiCount,nn::mii::SourceFlag_Database));

        /// Miiが1体もいない
        if(miiCount <= 0)
        {
            *pOutValue = true;
            NN_RESULT_THROW(nn::mii::ResultNotFound());
        }

        ///
        NN_LOG("exporting(count : %u) %s\n",miiCount,path);

        LogFile exportLog;
        for(int i = 0; i < miiCount;++i)
        {
            nn::mii::CharInfoAccessor accessor(charInfoArray[i]);

            /// ニックネームを取得
            nn::mii::Nickname miiNameUtf16;
            accessor.GetNickname(&miiNameUtf16, nn::mii::FontRegionFlag_All);

            /// UTF-8に変換
            char miiNameUtf8[MaxPathLength];
            auto convResult = nn::util::ConvertStringUtf16NativeToUtf8(miiNameUtf8, sizeof(miiNameUtf8), miiNameUtf16.name);
            NN_ABORT_UNLESS(convResult == nn::util::CharacterEncodingResult_Success);

            sprintf(path,"%s/%04d%02d%02d%02d%02d%02d/%02d.dat"
                ,pFolderPath
                ,calenderTime.year
                ,calenderTime.month
                ,calenderTime.day
                ,calenderTime.hour
                ,calenderTime.minute
                ,calenderTime.second
                ,i
                );

            /// ファイルを出力
            NN_RESULT_DO(OutputFile(path,charInfoArray[i]));

            /// ニックネームを出力
            exportLog.AppendFormat("%02d.dat : %s\n",i,miiNameUtf8);
        }

        sprintf(path,"%s/%04d%02d%02d%02d%02d%02d/nicknames.txt"
            ,pFolderPath
            ,calenderTime.year
            ,calenderTime.month
            ,calenderTime.day
            ,calenderTime.hour
            ,calenderTime.minute
            ,calenderTime.second
            );
        NN_RESULT_DO(exportLog.Write(path));

        *pOutValue = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result FormatDatabaseImpl(bool* pOutValue) NN_NOEXCEPT
    {
        *pOutValue = false;

        /// フォーマットを実行
        nn::mii::Database database;
        NN_RESULT_DO(database.Initialize());
        NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
        database.Format();
        *pOutValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ExportMiiDatabase(bool* outValue, const Option& option)
    {
        if(!IsEnabledMiiTestingMode(outValue))
        {
            NN_RESULT_THROW(nn::mii::ResultDevelopmentOnly());
        }
        auto filePath = option.GetTarget();
        return ExportDatabaseImpl(outValue,filePath);
    }

    Result ImportMiiDatabase(bool* outValue, const Option& option)
    {
        if(!IsEnabledMiiTestingMode(outValue))
        {
            NN_RESULT_THROW(nn::mii::ResultDevelopmentOnly());
        }
        auto filePath = option.GetTarget();
        return ImportDatabaseImpl(outValue,filePath);
    }

    Result ExportMiiCharInfo(bool* outValue, const Option& option)
    {
        if(!IsEnabledMiiTestingMode(outValue))
        {
            NN_RESULT_THROW(nn::mii::ResultDevelopmentOnly());
        }
        auto folderPath = option.GetTarget();
        return ExportCharInfoImpl(outValue,folderPath);
    }

    Result FormatMiiDatabase(bool* outValue, const Option&)
    {
        if(!IsEnabledMiiTestingMode(outValue))
        {
            NN_RESULT_THROW(nn::mii::ResultDevelopmentOnly());
        }
        return FormatDatabaseImpl(outValue);
    }

    Result GetMiiImageTestingModeEnabled(bool* outValue, const Option&)
    {
        return GetBooleanFirmwareDebugSetting(outValue, "mii", "is_img_db_test_mode_enabled");
    }

    Result EnableMiiImageTestingMode(bool* outValue, const Option&)
    {
        return EnableBooleanFirmwareDebugSetting(outValue, "mii", "is_img_db_test_mode_enabled");
    }

    Result DisableMiiImageTestingMode(bool* outValue, const Option&)
    {
        return DisableBooleanFirmwareDebugSetting(outValue, "mii", "is_img_db_test_mode_enabled");
    }

    Result CheckMiiImageTestingMode()
    {
        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "mii", "is_img_db_test_mode_enabled");
        if(sizeof(isEnabled) == size && isEnabled)
        {
            NN_RESULT_SUCCESS;
        }

        NN_LOG("Disabled mii-img-testing-mode\n");
        NN_RESULT_THROW(nn::mii::ResultDevelopmentOnly());
    }

    Result DeleteImageDatabase(bool* outValue, const Option&)
    {
        auto DeleteImageDatabaseImpl = []() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(CheckMiiImageTestingMode());

            nn::mii::ImageDatabase::DeleteFile();

            NN_RESULT_SUCCESS;
        };

        auto result = DeleteImageDatabaseImpl();
        *outValue = result.IsSuccess();
        return result;
    }

    Result DestroyImageDatabase(bool* outValue, const Option&)
    {
        auto DestroyImageDatabaseImpl = []() NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(CheckMiiImageTestingMode());

            nn::mii::ImageDatabase::DestroyFile();

            NN_RESULT_SUCCESS;
        };

        auto result = DestroyImageDatabaseImpl();
        *outValue = result.IsSuccess();
        return result;
    }

    Result ImportImageDatabase(bool* outValue, const Option& option)
    {

        auto ImportImageDatabaseImpl = [](const char* filePath) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(CheckMiiImageTestingMode());

            // ファイルを読み込み
            nn::fs::FileHandle file;
            NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

            // TODO GetSize() チェック

            std::unique_ptr<char[]> pBuffer(new char[nn::mii::ImageDatabase::MaxFileSize]);
            NN_RESULT_DO(nn::fs::ReadFile(file, 0, pBuffer.get(), nn::mii::ImageDatabase::MaxFileSize));

            nn::mii::ImageDatabase::ImportFile(pBuffer.get(), nn::mii::ImageDatabase::MaxFileSize);
            NN_RESULT_SUCCESS;
        };

        auto result = ImportImageDatabaseImpl(option.GetTarget());
        *outValue = result.IsSuccess();
        return result;
    }

    Result ExportImageDatabase(bool* outValue, const Option& option)
    {
        auto ExportImageDatabaseImpl = [](const char* filePath) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(CheckMiiImageTestingMode());

            std::unique_ptr<char[]> pBuffer(new char[nn::mii::ImageDatabase::MaxFileSize]);
            size_t exportedSize = nn::mii::ImageDatabase::ExportFile(pBuffer.get(), nn::mii::ImageDatabase::MaxFileSize);

            NN_RESULT_DO(OutputFile(filePath, pBuffer.get(), exportedSize));

            NN_RESULT_SUCCESS;
        };

        auto result = ExportImageDatabaseImpl(option.GetTarget());
        *outValue = result.IsSuccess();
        return result;
    }

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "get-testing-mode-enabled",   GetMiiTestingModeEnabled                    },
        { "enable-testing-mode",        EnableMiiTestingMode                        },
        { "disable-testing-mode",       DisableMiiTestingMode                       },
        { "export-database",            ExportMiiDatabase                           },
        { "import-database",            ImportMiiDatabase                           },
        { "export-char-info",           ExportMiiCharInfo                           },
        { "format-database",            FormatMiiDatabase                           },

        // Systemでない通常の DevMenuCommand でも利用できるようにしておく(help表示はなし)
        { "get-img-testing-mode-enabled",   GetMiiImageTestingModeEnabled           },
        { "enable-img-testing-mode",        EnableMiiImageTestingMode               },
        { "disable-img-testing-mode",       DisableMiiImageTestingMode              },

        { "delete-img-database",        DeleteImageDatabase                         },
        { "destroy-img-database",       DestroyImageDatabase                        },
        { "import-img-database",        ImportImageDatabase                         },
        { "export-img-database",        ExportImageDatabase                         },
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result MiiCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu mii command. See '" DEVMENUCOMMAND_NAME " mii --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
