﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <curl/curl.h>
#include <cstdlib>
#include <string>
#include <vector>
#include <unordered_map>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/apm/apm_Lib.h>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/nifm.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_VulnerabilityApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/socket.h>
#include <nn/util/util_FormatString.h>
#include <nn/idle/idle_SystemApi.h>

#include "DevMenuCommand_AccountCommand.h"
#include "DevMenuCommand_AddOnContentCommand.h"
#include "DevMenuCommand_AlbumCommand.h"
#include "DevMenuCommand_AppletCommand.h"
#include "DevMenuCommand_ApplicationCommand.h"
#include "DevMenuCommand_AudioCommand.h"
#include "DevMenuCommand_BatchCommand.h"
#include "DevMenuCommand_BcatCommand.h"
#include "DevMenuCommand_BgtcCommand.h"
#include "DevMenuCommand_ControllerCommand.h"
#include "DevMenuCommand_DebugCommand.h"
#include "DevMenuCommand_DynamicRightsCommand.h"
#include "DevMenuCommand_ElicenseCommand.h"
#include "DevMenuCommand_ErrorCommand.h"
#include "DevMenuCommand_FirmwareCommand.h"
#include "DevMenuCommand_FriendsCommand.h"
#include "DevMenuCommand_GameCardCommand.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_MiiCommand.h"
#include "DevMenuCommand_NetworkCommand.h"
#include "DevMenuCommand_NpnsCommand.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_PatchCommand.h"
#include "DevMenuCommand_PctlCommand.h"
#include "DevMenuCommand_PlayDataCommand.h"
#include "DevMenuCommand_PrepoCommand.h"
#include "DevMenuCommand_PowerCommand.h"
#include "DevMenuCommand_Result.h"
#include "DevMenuCommand_SaveDataCommand.h"
#include "DevMenuCommand_ServiceDiscoveryCommand.h"
#include "DevMenuCommand_SettingsCommand.h"
#include "DevMenuCommand_SdCardCommand.h"
#include "DevMenuCommand_ShopCommand.h"
#include "DevMenuCommand_Stopwatch.h"
#include "DevMenuCommand_TimeCommand.h"
#include "DevMenuCommand_UtilityCommand.h"

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include "DevMenuCommand_FactoryResetCommand.h"
#include "DevMenuCommand_FileSystemCommand.h"
#include "DevMenuCommand_LocalContentShareCommand.h"
#include "DevMenuCommand_LogCommand.h"
#include "DevMenuCommand_NewsCommand.h"
#include "DevMenuCommand_OlscCommand.h"
#include "DevMenuCommand_RidCommand.h"
#include "DevMenuCommand_SystemProgramCommand.h"
#include "DevMenuCommand_SystemUpdateCommand.h"
#include "DevMenuCommand_TicketCommand.h"
#include "DevMenuCommand_WlanCommand.h"
#if defined NN_BUILD_CONFIG_OS_HORIZON
#include "DevMenuCommand_NeighborDetectionCommand.h"
#endif
#endif

using namespace nn;

namespace {

const size_t HeapByteSize    = 30 * 1024 * 1024;
const size_t MallocBytesSize = 30 * 1024 * 1024;

const char HelpMessage[] =
    "usage: " DEVMENUCOMMAND_NAME " <command> [<args>]\n"
    "\n"
    "The DevMenu commands are:\n"
    "   firmware        Show firmware information\n"
#if defined NN_BUILD_CONFIG_OS_HORIZON
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    "   album           Download, upload, list, info or clean album files\n"
#else
    "   album           Download, list, info or clean album files\n"
#endif
#endif
    "   application     Install, uninstall, list, or launch applications\n"
    "   audio           Operation for audio\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    "   systemprogram   Install, uninstall or list system programs\n"
    "   systemupdate    Check, download or install system update\n"
    "   applet          Configurate applet programs\n"
    "   factoryreset    Reset to factory settings\n"
    "   filesystem      Show system file or user file\n"
    "   log             Set log output destination\n"
    "   ticket          List or delete ticket\n"
    "   dynamicrights   Operation for license\n"
    "   olsc            Online storage control\n"
    "   power           Control power\n"
#endif
    "   shop            Operation for shop\n"
    "   savedata        List or delete savedata\n"
    "   controller      Count or delete controller pairing settings\n"
    "   batch           Execute a batch of command list\n"
    "   debug           Debug settings\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    "   servicediscovery    Operation for service discovery settings\n"
#endif
    "   time            Operation for time settings\n"
    "   network         Show network information\n"
    "   error           Operation for error report\n"
    "   gamecard        Write nsp to gamecard\n"
    "   sdcard          Operation for sdcard\n"
    "   mii             Operation for mii\n"
    "   util            Utilities\n"
    "   pctl            Operation for parental controls\n"
    "   playdata        Operation for play data manager.\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    "   localcontentshare   Operation for local content share\n"
    "   wlan                Operation for wireless lan system\n"
#if defined NN_BUILD_CONFIG_OS_HORIZON
    // "   nd             Operation for neighbor detection\n"
#endif
#endif
    "";

socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

struct Command
{
    std::string name;
    Result (*function)(bool* outValue, const Option&);
};

const Command g_Commands[] =
{
    // Alphabetical order
    { "account", AccountCommand },
    { "addoncontent", AddOnContentCommand },
    { "album", AlbumCommand },
    { "applet", AppletCommand },
    { "application", ApplicationCommand },
    { "audio", AudioCommand },
    { "batch", BatchCommand },
    { "bcat", BcatCommand },
    { "bgtc", BgtcCommand },
    { "controller", ControllerCommand },
    { "debug", DebugCommand },
    { "error", ErrorCommand },
    { "firmware", FirmwareCommand },
    { "friends", FriendsCommand },
    { "gamecard", GameCardCommand },
    { "network", NetworkCommand },
    { "npns", NpnsCommand },
    { "mii", MiiCommand },
    { "patch", PatchCommand },
    { "pctl", PctlCommand },
    { "prepo", PrepoCommand },
    { "savedata", SaveDataCommand },
    { "sdcard", SdCardCommand },
    { "servicediscovery", ServiceDiscoveryCommand },
    { "settings", SettingsCommand },
    { "shop", ShopCommand },
    { "time", TimeCommand },
    { "util", UtilityCommand },
    { "playdata", PlayDataCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    { "factoryreset", FactoryResetCommand },
    { "filesystem", FileSystemCommand },
    { "log", LogCommand },
#if defined NN_BUILD_CONFIG_OS_HORIZON
    { "nd", NeighborDetectionCommand },
#endif
    { "news", NewsCommand },
    { "olsc", OlscCommand },
    { "power", PowerCommand },
    { "rid", RidCommand },
    { "systemprogram", SystemProgramCommand },
    { "systemupdate", SystemUpdateCommand },
    { "ticket", TicketCommand },
    { "dynamicrights", DynamicRightsCommand },
    { "elicense", ElicenseCommand },
    { "localcontentshare", LocalContentShareCommand },
    { "wlan", WlanCommand },
#endif
};

std::string GetErrorMessage(Result result) NN_NOEXCEPT
{
    if (result <= ncm::ResultContentAlreadyExists()) return "Already exists.";
    if (result <= ncm::ResultContentMetaNotFound()) return "Not found.";
    if (result <= ns::ResultApplicationRecordNotFound()) return "Not found.";
    if (result <= ns::ResultSdCardNotInserted()) return "SD card is not inserted.";
    if (result <= ns::ResultSdCardAccessFailed()) return "Failed to access to the SD card.";
    if (result <= ns::ResultSdCardNoOwnership()) return "SD card was initialized on another device or database is corrupted.\nCleanup is required to use SD card.";
    if (result <= ns::ResultSdCardNotMounted()) return "SD card was inserted after device boot.\nDevice reboot is required to install contents to the card.";
    if (result <= ns::ResultGameCardNotInserted()) return "Game Card is not inserted.";
    if (result <= ns::ResultGameCardIsInserted()) return "Failed to execute the command. Game Card is inserted.";
    if (result <= ncm::ResultUnknownStorage()) return "Unknown storage.";
    if (result <= fs::ResultPathNotFound()) return "File not found.";
    if (result <= fs::ResultUnsupportedSdkVersion()) return "Built with an older version SDK. Must be rebuilt.";
    if (result <= devmenuUtil::ResultBadApplicationForThePatch()) return "The application is not one that is used to create the patch.";
    if (result <= devmenuUtil::ResultApplicationNotFound()) return "The application is not found.";
    if (result <= devmenuUtil::ResultPatchNotFound()) return "The patch is not found.";
    if (result <= devmenuUtil::ResultContentMetaNotFound()) return "The content meta is not found.";

    char buffer[64];
    util::SNPrintf(buffer, sizeof(buffer), "Unexpected error. result = 0x%08x.", result.GetInnerValueForDebug());
    return buffer;
}

enum class ExecuteType
{
    Anyway,
    IfSuccess,
    IfFailure
};

std::vector<std::pair<Option, ExecuteType> > SplitCommand(int argc, char** argv)
{
    std::vector<std::pair<Option, ExecuteType> > options;

    char* parseArgv[Option::TargetArgumentCountMax];
    ExecuteType commandType = ExecuteType::Anyway; // 最初のコマンドは必ず Anyway
    parseArgv[0] = argv[0];
    int parseArgvIndex = 1;

    for (int i = 1; i < argc; ++i)
    {
        // スペースなしを許容する場合
        auto f = [&](std::string splitter, ExecuteType type) -> bool
        {
            std::string str(argv[i]);
            auto pos = str.find(splitter);
            if (pos != std::string::npos)
            {
                if (pos != 0)
                {
                    if (parseArgvIndex >= Option::TargetArgumentCountMax)
                    {
                        // 見つけたけど追加できないので、インデックスだけ増やして終了する
                        parseArgvIndex++;
                        return true;
                    }
                    argv[i][pos] = '\0';
                    parseArgv[parseArgvIndex] = argv[i];
                    parseArgvIndex++;
                }
                Option option(parseArgvIndex, parseArgv);
                options.push_back(std::pair<Option, ExecuteType>(option, commandType));

                parseArgvIndex = 1;
                commandType = type;

                if (argv[i][pos + splitter.length()] != '\0')
                {
                    if (parseArgvIndex >= Option::TargetArgumentCountMax)
                    {
                        // 追加できないので、インデックスだけ増やして終了する
                        parseArgvIndex++;
                        return true;
                    }
                    parseArgv[parseArgvIndex] = &(argv[i][pos + splitter.length()]);
                    parseArgvIndex++;
                }
                return true;
            }
            else
            {
                return false;
            }
        };
        // スペースなしを許容しない場合
        auto g = [&](std::string splitter, ExecuteType type) -> bool
        {
            std::string str(argv[i]);
            if (str == splitter)
            {
                Option option(parseArgvIndex, parseArgv);
                options.push_back(std::pair<Option, ExecuteType>(option, commandType));

                parseArgvIndex = 1;
                commandType = type;

                return true;
            }
            else
            {
                return false;
            }
        };
        if (f(";",  ExecuteType::Anyway)    || g("THEN", ExecuteType::Anyway) ||
            f("&&", ExecuteType::IfSuccess) || g("AND",  ExecuteType::IfSuccess) ||
            f("||", ExecuteType::IfFailure) || g("OR",   ExecuteType::IfFailure))
        {
        }
        else
        {
            if (parseArgvIndex >= Option::TargetArgumentCountMax)
            {
                // 追加できないので、インデックスだけ増やして終了する
                parseArgvIndex++;
                break;
            }
            parseArgv[parseArgvIndex] = argv[i];
            parseArgvIndex++;
        }
    }

    if (parseArgvIndex > Option::TargetArgumentCountMax)
    {
        NN_LOG("Too many arguments.\n");
        options.clear();
        return options;
    }
    else if ((commandType == ExecuteType::Anyway && parseArgvIndex > 1) || commandType != ExecuteType::Anyway)
    {
        // 最後にセミコロンを入れられたときに、エラーにならないように
        // eg. applicaiton list; application list;
        //                                       ^ これ
        Option option(parseArgvIndex, parseArgv);
        options.push_back(std::pair<Option, ExecuteType>(option, commandType));
    }

    return options;
}


}   // namespace

bool RunCommandImpl(const Option& option) NN_NOEXCEPT
{
    if (option.HasGlobalOption("--help"))
    {
        NN_LOG(HelpMessage);
        return true;
    }
    for (const Command& command : g_Commands)
    {
        if (command.name == option.GetCommand())
        {
            Stopwatch sw = option.HasGlobalOption("--time") ? Stopwatch() : Stopwatch(false);

            bool isSuccess;
            auto result = command.function(&isSuccess, option);
            if (result.IsFailure())
            {
                NN_LOG("%s\n", GetErrorMessage(result).c_str());
                return false;
            }
            return isSuccess;
        }
    }
    NN_LOG("'%s' is not a DevMenu command. See '" DEVMENUCOMMAND_NAME " --help'.\n", option.GetCommand());
    return false;
}

bool RunCommand(int argc, char** argv) NN_NOEXCEPT
{
    if (argc < 2)
    {
        NN_LOG(HelpMessage);
        return false;
    }

    auto options = SplitCommand(argc, argv);

    bool isSuccess = false;
    for (const auto& pair : options)
    {
        if ((pair.second == ExecuteType::IfSuccess && !isSuccess) || (pair.second == ExecuteType::IfFailure && isSuccess))
        {
            break;
        }
        isSuccess = RunCommandImpl(pair.first);
    }
    return isSuccess;
}

extern "C" void nninitStartup()
{
    auto result = nn::os::SetMemoryHeapSize( HeapByteSize );
    if (result.IsFailure())
    {
        NN_LOG("Unable to allocate heap memory (%zu bytes).\nPlease restart your target and try again.\n", HeapByteSize);
        NN_ABORT("");
    }

    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocBytesSize ) );
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MallocBytesSize );
}

// TODO :コマンドラインツールフレームワークが正しい場所に置かれたら対応したい
extern "C" void nnMain()
{
    fs::InitializeWithMultiSessionForTargetTool();
    fs::SetEnabledAutoAbort(false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountHostRoot());
    NN_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(g_SocketConfigWithMemory));
    curl_global_init(CURL_GLOBAL_DEFAULT);
    es::Initialize();

#ifdef NN_BUILD_CONFIG_OS_WIN
    ns::InitializeDependenciesForDfc();
#else
    ncm::Initialize();
    nn::apm::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
    idle::InitializeForSystem();
#ifdef NN_TOOL_DEVMENUCOMMANDSYSTEM
    lr::Initialize();
    nim::InitializeForNetworkInstallManager();
#endif
#endif

    NN_ABORT_UNLESS_RESULT_SUCCESS(ns::InitializeForDevelop());
    ns::Initialize();

    std::string resultString = RunCommand(os::GetHostArgc(), os::GetHostArgv()) ? "[SUCCESS]\n" : "[FAILURE]\n";
    NN_LOG(resultString.c_str());
}
