﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringView.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Language.h"
#include "DevMenuCommand_LocalContentShareCommand.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_StorageId.h"

using namespace nn;
using namespace nne;
using namespace devmenuUtil;

#define DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(cond, ...) \
    do \
    { \
        if (!(cond)) \
        { \
            DEVMENUCOMMAND_LOG(__VA_ARGS__); \
            NN_RESULT_SUCCESS; \
        } \
    } while (NN_STATIC_CONDITION(false))

#define DEVMENUCOMMAND_RETURN_UNLESS(cond) \
    do \
    { \
        if (!(cond)) \
        { \
            NN_RESULT_SUCCESS; \
        } \
    } while (NN_STATIC_CONDITION(false))


namespace
{
    lcs::SessionInfo g_Session[lcs::ScanResultCountMax];

    Result InitializeLcs(os::SystemEventType* pNotifyEvent, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        // Settings からデバイスニックネームを取得して、LCS の config に設定します。
        settings::system::DeviceNickName deviceNickName;
        settings::system::GetDeviceNickName(&deviceNickName);
        lcs::Config config;
        config.SetName(deviceNickName.string);


        NN_RESULT_DO(ldn::InitializeSystem());
        NN_RESULT_DO(lcs::Initialize(buffer, bufferSize, config));

        // ホストになるときの設定を AcceptPolicy_AlwaysAccept に設定します。
        lcs::SetClientAcceptPolicy(lcs::AcceptPolicy_AlwaysAccept);

        // オブジェクトの破棄の管理は lcs が行うので、利用者は行う必要がない
        lcs::AttachStateChangeEvent(pNotifyEvent);

        NN_RESULT_SUCCESS;
    }

    void FinalizeLcs() NN_NOEXCEPT
    {
        lcs::Finalize();
        ldn::FinalizeSystem();
    }

    lcs::SessionInfo* FindSession(ncm::ApplicationId id, lcs::SessionInfo sessionList[], int numSession) NN_NOEXCEPT
    {
        for (int i = 0; i < numSession; i++)
        {
            const lcs::SessionInfo& info = sessionList[i];
            if (std::any_of(
                &info.contents[0], &info.contents[info.contentsCount],
                [&id](const lcs::ContentsInfo& content) NN_NOEXCEPT { return content.applicationId == id; }))
            {
                return &sessionList[i];
            }
        }
        return nullptr;
    }

    bool EqualsLcsState(lcs::State expectedState) NN_NOEXCEPT
    {
        auto state = lcs::GetState();
        return state == expectedState;
    }

    void OutputJson(rapidjson::Document* pDocument) NN_NOEXCEPT
    {
        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        pDocument->Accept(writer);
        DEVMENUCOMMAND_LOG("=== JSON ===\n", buffer.GetString());
        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());
    }

    bool IsTimeout(TimeSpan absoluteTimeout, TimeSpan current) NN_NOEXCEPT
    {
        const TimeSpan ZeroTime{};
        if (absoluteTimeout == ZeroTime)
        {
            return false;
        }
        if (current > absoluteTimeout)
        {
            DEVMENUCOMMAND_LOG("Timeout\n");
            return true;
        }
        return false;
    }

    Result OutputLcsProgress() NN_NOEXCEPT
    {
        lcs::Progress progress;
        NN_RESULT_DO(lcs::GetProgress(&progress));
        DEVMENUCOMMAND_LOG("id=0x%llx (flag=%d)\n", progress.info.applicationId, progress.info.contentsFlag);
        DEVMENUCOMMAND_LOG("0x%llx / 0x%llx\n", progress.downloadedSize, progress.size);

        NN_RESULT_SUCCESS;
    }

    Result HandleSuspend(bool* outIsSuccess, bool* outIsSuspended, const char* resumeFile, bool isVerbose) NN_NOEXCEPT
    {
        *outIsSuccess = false;
        *outIsSuspended = false;
        auto reason = lcs::GetSuspendedReason();
        switch (reason)
        {
            case lcs::SuspendedReason_EulaRequired:
                if (isVerbose)
                {
                    DEVMENUCOMMAND_LOG("EulaRequired\n");
                    NN_RESULT_DO(OutputLcsProgress());
                }

                // 送信側で進捗を表示させるため、若干待つ
                os::SleepThread(TimeSpan::FromSeconds(2));
                NN_RESULT_DO(lcs::ResumeContentsShare());
                break;
            case lcs::SuspendedReason_NeedTerminateApplication:
                if (isVerbose)
                {
                    DEVMENUCOMMAND_LOG("NeedTerminateApplication\n");
                    NN_RESULT_DO(OutputLcsProgress());
                }
                NN_RESULT_DO(lcs::ResumeContentsShare());
                break;
            case lcs::SuspendedReason_IncompatibleContentsInfo:
                if (isVerbose)
                {
                    DEVMENUCOMMAND_LOG("IncompatibleContentsInfo\n");
                    NN_RESULT_DO(OutputLcsProgress());
                }
                NN_RESULT_DO(lcs::ResumeContentsShare());
                break;
            case lcs::SuspendedReason_RebootRequired:
            case lcs::SuspendedReason_StorageSpaceNotEnough:
                {
                    lcs::SessionContext context;
                    NN_RESULT_DO(lcs::GetSessionContext(&context));

                    size_t bufferSize = sizeof(context._context) * 2;
                    std::unique_ptr<char[]> base64Buffer(new char[bufferSize]);
                    auto status = util::Base64::ToBase64String(base64Buffer.get(), bufferSize, context._context, sizeof(context._context), util::Base64::Mode_NormalNoLinefeed);
                    DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(status == util::Base64::Status_Success, "Miss to convert binary to base64 string\n");

                    NN_RESULT_DO(lcs::SuspendSession());

                    if (resumeFile)
                    {
                        if (isVerbose)
                        {
                            DEVMENUCOMMAND_LOG("Output resume file to `%s`\n", resumeFile);
                        }
                        util::string_view view(base64Buffer.get());

                        NN_RESULT_TRY(fs::CreateFile(resumeFile, view.length()));
                            NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
                            {
                                NN_RESULT_DO(fs::DeleteFile(resumeFile));
                                NN_RESULT_DO(fs::CreateFile(resumeFile, static_cast<int64_t>(view.length())));
                            }
                        NN_RESULT_END_TRY
                        fs::FileHandle file;
                        NN_RESULT_DO(fs::OpenFile(&file, resumeFile, fs::OpenMode_Write));
                        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

                        NN_RESULT_DO(fs::WriteFile(file, 0, view.data(), view.length(), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
                    }

                    {
                        rapidjson::Document document;
                        document.SetObject();
                        document.AddMember("state", rapidjson::Value(static_cast<int>(lcs::State_Suspended)), document.GetAllocator());
                        document.AddMember("reason", rapidjson::Value(static_cast<int>(reason)), document.GetAllocator());
                        document.AddMember("context", rapidjson::Value(base64Buffer.get(), document.GetAllocator()), document.GetAllocator());
                        OutputJson(&document);
                    }

                    *outIsSuspended = true;
                }
                break;
            default:
                {
                    rapidjson::Document document;
                    document.SetObject();
                    document.AddMember("state", rapidjson::Value(static_cast<int>(lcs::State_Suspended)), document.GetAllocator());
                    document.AddMember("reason", rapidjson::Value(static_cast<int>(reason)), document.GetAllocator());
                    OutputJson(&document);
                    *outIsSuspended = true;
                }
                break;
        }

        *outIsSuccess = true;
        NN_RESULT_SUCCESS;
    }

    Result ContentDeliveryStateHandler(bool* outValue, os::SystemEventType* pNotifyEvent, const char* resumeFile, TimeSpan absoluteTimeout, bool isVerbose) NN_NOEXCEPT
    {
        const TimeSpan zeroTime{};
        os::TimerEventType timerEvent;
        os::InitializeTimerEvent(&timerEvent, os::EventClearMode_AutoClear);

        auto currentTime = ConvertToTimeSpan(os::GetSystemTick());
        DEVMENUCOMMAND_RETURN_UNLESS(!IsTimeout(absoluteTimeout, currentTime));
        if (zeroTime != absoluteTimeout)
        {
            os::StartOneShotTimerEvent(&timerEvent, absoluteTimeout - currentTime);
        }

        bool isTransferring = false;
        *outValue = false;
        while (NN_STATIC_CONDITION(true))
        {
            bool isEnd = false;
            const int NotifyIndex = 0;
            const int TimeoutIndex = 1;
            auto index = os::TimedWaitAny(TimeSpan::FromSeconds(1), pNotifyEvent, &timerEvent);
            if (index == TimeoutIndex)
            {
                DEVMENUCOMMAND_LOG("Timeout\n");
                NN_RESULT_DO(lcs::LeaveSession());
                DEVMENUCOMMAND_RETURN_UNLESS(NN_STATIC_CONDITION(false));
            }
            else if (index == NotifyIndex)
            {
                os::ClearSystemEvent(pNotifyEvent);
                auto state = nn::lcs::GetState();
                switch(state)
                {
                case lcs::State_Transferring:
                    isTransferring = true;
                    break;
                case lcs::State_Suspended:
                    bool isSuccess;
                    NN_RESULT_DO(HandleSuspend(&isSuccess, &isEnd, resumeFile, isVerbose));
                    DEVMENUCOMMAND_RETURN_UNLESS(isSuccess);
                    break;
                case lcs::State_ContentsShareFailed:
                    {
                        auto reason = lcs::GetContentsShareFailureReason();

                        rapidjson::Document document;
                        document.SetObject();
                        document.AddMember("state", rapidjson::Value(static_cast<int>(state)), document.GetAllocator());
                        document.AddMember("reason", rapidjson::Value(static_cast<int>(reason)), document.GetAllocator());

                        nn::err::ErrorContext context;
                        nn::lcs::GetErrorContext(&context);

                        rapidjson::Value baseErrorContext(rapidjson::kObjectType);
                        baseErrorContext.AddMember("type", rapidjson::Value(static_cast<int>(context.type)), document.GetAllocator());
                        if (context.type == nn::err::ErrorContextType::LocalContentShare)
                        {
                            rapidjson::Value localContentShareErrorContext(rapidjson::kObjectType);
                            const auto& lcsContext = context.localContentShare;
                            char appIdStr[20];
                            util::TSNPrintf(appIdStr, sizeof(appIdStr), "0x%016llx", lcsContext.applicationId.value);
                            localContentShareErrorContext.AddMember("applicationId", rapidjson::StringRef(appIdStr), document.GetAllocator());
                            localContentShareErrorContext.AddMember("resultInnerValue", rapidjson::Value(lcsContext.resultInnerValue), document.GetAllocator());
                            localContentShareErrorContext.AddMember("ip", rapidjson::Value(lcsContext.ip), document.GetAllocator());
                            localContentShareErrorContext.AddMember("isSender", rapidjson::Value(lcsContext.isSender), document.GetAllocator());
                            localContentShareErrorContext.AddMember("isApplicationRequest", rapidjson::Value(lcsContext.isApplicationRequest), document.GetAllocator());
                            localContentShareErrorContext.AddMember("hasExFatDriver", rapidjson::Value(lcsContext.hasExFatDriver), document.GetAllocator());
                            localContentShareErrorContext.AddMember("numKey", rapidjson::Value(lcsContext.numKey), document.GetAllocator());

                            rapidjson::Value contentMetaKeyList(rapidjson::kArrayType);
                            for (auto& key : lcsContext.keyList)
                            {
                                rapidjson::Value contentMetaKey(rapidjson::kObjectType);
                                contentMetaKey.AddMember("id", rapidjson::Value(key.id), document.GetAllocator());
                                contentMetaKey.AddMember("version", rapidjson::Value(key.version), document.GetAllocator());
                                contentMetaKey.AddMember("type", rapidjson::Value(static_cast<int>(key.type)), document.GetAllocator());
                                contentMetaKey.AddMember("installType", rapidjson::Value(static_cast<int>(key.installType)), document.GetAllocator());

                                contentMetaKeyList.PushBack(contentMetaKey, document.GetAllocator());
                            }
                            localContentShareErrorContext.AddMember("keyList", contentMetaKeyList, document.GetAllocator());

                            baseErrorContext.AddMember("localContentShare", localContentShareErrorContext, document.GetAllocator());
                        }
                        document.AddMember("errorContext", baseErrorContext, document.GetAllocator());

                        OutputJson(&document);
                        isEnd = true;
                    }
                    break;
                case lcs::State_Completed:
                    {
                        rapidjson::Document document;
                        document.SetObject();
                        document.AddMember("state", rapidjson::Value(static_cast<int>(state)), document.GetAllocator());
                        OutputJson(&document);

                        isEnd = true;
                    }
                    break;
                case lcs::State_Joined:
                    break;
                default:
                    {
                        DEVMENUCOMMAND_LOG("Unexpected state: %d\n", state);
                        NN_RESULT_SUCCESS;
                    }
                }
            }

            if (isEnd)
            {
                break;
            }

            if (isVerbose)
            {
                if (isTransferring)
                {
                    NN_RESULT_DO(OutputLcsProgress());
                }
                else
                {
                    DEVMENUCOMMAND_LOG("Waiting...\n");
                }
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result ResumeSession(bool* outValue, const char* resumeFile, TimeSpan absoluteTimeout) NN_NOEXCEPT
    {
        *outValue = false;
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, resumeFile, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
        size_t bufferSize = static_cast<size_t>(fileSize) + 1;

        std::unique_ptr<char[]> base64StringBuffer(new char[bufferSize]);
        NN_RESULT_DO(fs::ReadFile(file, 0, base64StringBuffer.get(), bufferSize - 1));
        // 念のため
        base64StringBuffer[bufferSize - 1] = '\0';

        lcs::SessionContext context;
        size_t outCount;
        auto status = util::Base64::FromBase64String(&outCount, &context._context, NN_ARRAY_SIZE(context._context), base64StringBuffer.get(), util::Base64::Mode_NormalNoLinefeed);
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(status == util::Base64::Status_Success, "Miss to convert base64 string to binary\n");

        while (NN_STATIC_CONDITION(true))
        {
            DEVMENUCOMMAND_RETURN_UNLESS(!IsTimeout(absoluteTimeout, ConvertToTimeSpan(os::GetSystemTick())));
            auto result = lcs::ResumeSession(context);
            if (result.IsSuccess())
            {
                break;
            }
            else if (lcs::ResultConnectionFailed::Includes(result))
            {
                // 接続に失敗した場合はリトライ
            }
            else
            {
                NN_RESULT_THROW(result);
            }
            os::SleepThread(TimeSpan::FromSeconds(1));
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetSystemDeliveryInfo(bool* outValue, const Option&)
    {
        *outValue = false;

        ns::SystemDeliveryInfo info;
        NN_RESULT_DO(ns::GetSystemDeliveryInfo(&info));

        rapidjson::Document document;
        document.SetObject();
        document.AddMember("systemDeliveryProtocolVersion", rapidjson::Value(info.systemDeliveryProtocolVersion), document.GetAllocator());
        document.AddMember("applicationDeliveryProtocolVersion", rapidjson::Value(info.applicationDeliveryProtocolVersion), document.GetAllocator());
        document.AddMember("systemUpdateId", rapidjson::Value(devmenuUtil::GetBit64String(info.systemUpdateId).data, document.GetAllocator()), document.GetAllocator());
        document.AddMember("systemUpdateVersion", rapidjson::Value(info.systemUpdateVersion), document.GetAllocator());
        document.AddMember("attributes", rapidjson::Value(info.attributes._storage[0]), document.GetAllocator());

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetApplicationDeliveryInfo(bool* outValue, const Option& option)
    {
        *outValue = false;
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        const int ListCount = 16;
        std::unique_ptr<ns::ApplicationDeliveryInfo[]> appInfoList(new ns::ApplicationDeliveryInfo[ListCount]);

        ns::ApplicationDeliveryAttribute attributes {};
        if (option.HasKey("--app"))
        {
            attributes |= ns::ApplicationDeliveryAttribute_RequestApplication::Mask;
        }
        if (option.HasKey("--patch"))
        {
            attributes |= ns::ApplicationDeliveryAttribute_RequestPatch::Mask;
        }
        int count;
        NN_RESULT_DO(ns::GetApplicationDeliveryInfo(&count, appInfoList.get(), ListCount, appId, attributes));

        rapidjson::Document document;
        document.SetArray();

        for (int i = 0; i < count; i++)
        {
            rapidjson::Value item(rapidjson::kObjectType);
            item.AddMember("id", rapidjson::Value(devmenuUtil::GetBit64String(appInfoList[i].id.value).data, document.GetAllocator()), document.GetAllocator());
            item.AddMember("version", rapidjson::Value(appInfoList[i].version), document.GetAllocator());
            item.AddMember("requiredVersion", rapidjson::Value(appInfoList[i].requiredVersion), document.GetAllocator());
            item.AddMember("requiredSystemVersion", rapidjson::Value(appInfoList[i].requiredSystemVersion), document.GetAllocator());
            item.AddMember("attributes", rapidjson::Value(appInfoList[i].attributes._storage[0]), document.GetAllocator());
            item.AddMember("applicationDeliveryProtocolVersion", rapidjson::Value(appInfoList[i].applicationDeliveryProtocolVersion), document.GetAllocator());
            document.PushBack(item, document.GetAllocator());
        }

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ReceiveApplication(bool* outValue, const Option& option)
    {
        *outValue = false;
        const char* ResumeOption = "--resume";
        const char* OutputResumeFileOption = "--output-resume-file";
        const char* TimeOutOption = "--timeout";
        const char* VerboseOption = "--verbose";

        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        const char* outputResumeFile = option.HasKey(OutputResumeFileOption) ? option.GetValue(OutputResumeFileOption) : nullptr;
        const TimeSpan AbsoluteTimeout = option.HasKey(TimeOutOption) ? TimeSpan::FromSeconds(std::strtoull(option.GetValue(TimeOutOption), nullptr, 10)) + os::ConvertToTimeSpan(os::GetSystemTick()) : TimeSpan();
        bool isVerbose = option.HasKey(VerboseOption);


#if defined(NN_BUILD_CONFIG_OS_WIN)
        auto buffer = _aligned_malloc(lcs::RequiredBufferSize, lcs::RequiredBufferAlignment);
#else
        auto buffer = std::aligned_alloc(lcs::RequiredBufferAlignment, lcs::RequiredBufferSize);
#endif
        NN_UTIL_SCOPE_EXIT
        {
            std::free(buffer);
        };
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(buffer, "Cannot allocate buffer.\n");

        os::SystemEventType notifyEvent;
        NN_RESULT_DO(InitializeLcs(&notifyEvent, buffer, lcs::RequiredBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            FinalizeLcs();
        };

        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(EqualsLcsState(lcs::State_Initialized), "lcs state is invalid (expect=%d, actual=%d).\n", static_cast<int>(lcs::State_Initialized), static_cast<int>(lcs::GetState()));

        if (option.HasKey(ResumeOption))
        {
            bool isResumed;
            NN_RESULT_DO(ResumeSession(&isResumed, option.GetValue(ResumeOption), AbsoluteTimeout));
            DEVMENUCOMMAND_RETURN_UNLESS(isResumed);
        }
        else
        {
            while (NN_STATIC_CONDITION(true))
            {
                DEVMENUCOMMAND_RETURN_UNLESS(!IsTimeout(AbsoluteTimeout, ConvertToTimeSpan(os::GetSystemTick())));

                int count;
                auto result = lcs::Scan(g_Session, &count, static_cast<int>(NN_ARRAY_SIZE(g_Session)));
                if (result.IsSuccess() && count > 0)
                {
                    auto pSession = FindSession(appId, g_Session, count);
                    if (pSession)
                    {
                        result = lcs::JoinSession(*pSession);
                        if (result.IsSuccess())
                        {
                            break;
                        }
                        else if (lcs::ResultConnectionFailed::Includes(result))
                        {
                            // 接続に失敗した場合はリトライ
                        }
                        else
                        {
                            NN_RESULT_THROW(result);
                        }
                    }
                }
                os::SleepThread(TimeSpan::FromSeconds(1));
            }
       }

        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(EqualsLcsState(lcs::State_Joined) || EqualsLcsState(lcs::State_Transferring), "lcs state is invalid (expect=%d or %d, actual=%d).\n", static_cast<int>(lcs::State_Joined), static_cast<int>(lcs::State_Transferring), static_cast<int>(lcs::GetState()));
        bool isSuccess;
        NN_RESULT_DO(ContentDeliveryStateHandler(&isSuccess, &notifyEvent, outputResumeFile, AbsoluteTimeout, isVerbose));
        DEVMENUCOMMAND_RETURN_UNLESS(isSuccess);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SendApplication(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        const char* NodeOption = "--node";
        const char* ResumeOption = "--resume";
        const char* OutputResumeFileOption = "--output-resume-file";
        const char* TimeOutOption = "--timeout";
        const char* VerboseOption = "--verbose";

        *outValue = false;
        ncm::ApplicationId appId = devmenuUtil::GetApplicationIdTarget(option);
        const char* outputResumeFile = option.HasKey(OutputResumeFileOption) ? option.GetValue(OutputResumeFileOption) : nullptr;
        int8_t nodeCount = option.HasKey(NodeOption) ? static_cast<int8_t>(std::strtoull(option.GetValue(NodeOption), nullptr, 10)) : 2;
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(nodeCount >= 2 && nodeCount <= 8, "Node count is out of range. %d\n", nodeCount);
        const TimeSpan ZeroTime{};
        const TimeSpan BeginTime = os::ConvertToTimeSpan(os::GetSystemTick());
        const TimeSpan TimeoutOptionTime = option.HasKey(TimeOutOption) ? TimeSpan::FromSeconds(std::strtoull(option.GetValue(TimeOutOption), nullptr, 10)) : TimeSpan();
        const TimeSpan AbsoluteTimeout = (TimeoutOptionTime > ZeroTime) ? TimeoutOptionTime + BeginTime : ZeroTime;
        bool isVerbose = option.HasKey(VerboseOption);

#if defined(NN_BUILD_CONFIG_OS_WIN)
        auto buffer = _aligned_malloc(lcs::RequiredBufferSize, lcs::RequiredBufferAlignment);
#else
        auto buffer = std::aligned_alloc(lcs::RequiredBufferAlignment, lcs::RequiredBufferSize);
#endif
        NN_UTIL_SCOPE_EXIT
        {
            std::free(buffer);
        };
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(buffer, "Cannot allocate buffer.\n");

        os::SystemEventType notifyEvent;
        NN_RESULT_DO(InitializeLcs(&notifyEvent, buffer, lcs::RequiredBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            FinalizeLcs();
        };

        if (option.HasKey(ResumeOption))
        {
            bool isResumed;
            NN_RESULT_DO(ResumeSession(&isResumed, option.GetValue(ResumeOption), AbsoluteTimeout));
            DEVMENUCOMMAND_RETURN_UNLESS(isResumed);
        }
        else
        {
            lcs::SessionSettings settings {};
            settings.applications[0] = appId;
            settings.applicationCount = 1;
            settings.nodeCountMax = nodeCount;
            settings.contentsShareVersionPolicy = nn::lcs::ContentsShareVersionPolicy_Latest;
            NN_RESULT_DO(lcs::OpenSession(settings));
            DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(EqualsLcsState(lcs::State_Opened), "lcs state is invalid (expect=%d, actual=%d).\n", static_cast<int>(lcs::State_Opened), static_cast<int>(lcs::GetState()));

            while (NN_STATIC_CONDITION(true))
            {
                if (TimeoutOptionTime > ZeroTime)
                {
                    if (!os::TimedWaitSystemEvent(&notifyEvent, TimeoutOptionTime))
                    {
                        DEVMENUCOMMAND_LOG("Timeout\n");
                        DEVMENUCOMMAND_RETURN_UNLESS(NN_STATIC_CONDITION(false));
                    }
                }
                else
                {
                    os::WaitSystemEvent(&notifyEvent);
                }
                os::ClearSystemEvent(&notifyEvent);

                lcs::SessionInfo info;
                NN_RESULT_DO(lcs::GetSessionInfo(&info));
                if (info.nodeCount == nodeCount)
                {
                    break;
                }
            }

            NN_RESULT_DO(lcs::StartContentsShare());
        }
        bool isSuccess;
        NN_RESULT_DO(ContentDeliveryStateHandler(&isSuccess, &notifyEvent, outputResumeFile, AbsoluteTimeout, isVerbose));
        DEVMENUCOMMAND_RETURN_UNLESS(isSuccess);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSystemUpdateId(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        settings::fwdbg::SetSettingsItemValue("contents_delivery", "system_delivery_info_id", option.GetTarget(), std::strlen(option.GetTarget()) + 1);
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSystemDeliveryProtocolVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 10));
        settings::fwdbg::SetSettingsItemValue("contents_delivery", "system_delivery_protocol_version", &version, sizeof(version));
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetApplicationDeliveryProtocolVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 10));
        settings::fwdbg::SetSettingsItemValue("contents_delivery", "application_delivery_protocol_version", &version, sizeof(version));
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetAcceptableApplicationDeliveryProtocolVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 10));
        settings::fwdbg::SetSettingsItemValue("contents_delivery", "acceptable_application_delivery_protocol_version", &version, sizeof(version));
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetAcceptableApplicationDeliveryProtocolVersion(bool* outValue, const Option&) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version;
        auto size = settings::fwdbg::GetSettingsItemValue(&version, sizeof(version), "contents_delivery", "acceptable_application_delivery_protocol_version");
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(size == sizeof(version), "Cannot get acceptable application delivery protocol version.\n");
        DEVMENUCOMMAND_LOG("acceptable application delivery protocol version = %u\n", version);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetRequiredSystemVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 10));
        settings::fwdbg::SetSettingsItemValue("contents_delivery", "required_system_version_to_deliver_application", &version, sizeof(version));
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetRequiredSystemVersion(bool* outValue, const Option&) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t version;
        auto size = settings::fwdbg::GetSettingsItemValue(&version, sizeof(version), "contents_delivery", "required_system_version_to_deliver_application");
        DEVMENUCOMMAND_TRACE_AND_RETURN_UNLESS(size == sizeof(version), "Cannot get required system version to deliver application contents.\n");
        DEVMENUCOMMAND_LOG("required system version = %u\n", version);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetAsyncTaskResult(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;
        uint32_t resultInnerValue = static_cast<int>(std::strtoull(option.GetTarget(), NULL, 16));
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "error_localcommunication_result", &resultInnerValue, sizeof(resultInnerValue));
        DEVMENUCOMMAND_LOG("Please restart your target to apply this change.\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " localcontentshare get-system-delivery-info\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare get-application-delivery-info <application_id> [--app] [--patch]\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare receive-application <application_id> [--resume <Base64String>] [--timeout <second>] [--verbose]\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare send-application <application_id> [--node <2..8>] [--resume <Base64String>] [--timeout <second>] [--verbose]\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-system-update-id <system_update_id>\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-system-delivery-protocol-version <version>\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-application-delivery-protocol-version <version>\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-acceptable-application-delivery-protocol-version <version>\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare get-acceptable-application-delivery-protocol-version\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-required-system-version <version>\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare get-required-system-version\n"
        "       " DEVMENUCOMMAND_NAME " localcontentshare set-async-task-result\n"
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "get-system-delivery-info", GetSystemDeliveryInfo },
        { "get-application-delivery-info", GetApplicationDeliveryInfo },
        { "receive-application", ReceiveApplication },
        { "send-application", SendApplication },
        { "set-system-update-id", SetSystemUpdateId },
        { "set-system-delivery-protocol-version", SetSystemDeliveryProtocolVersion },
        { "set-application-delivery-protocol-version", SetApplicationDeliveryProtocolVersion },
        { "set-acceptable-application-delivery-protocol-version", SetAcceptableApplicationDeliveryProtocolVersion },
        { "get-acceptable-application-delivery-protocol-version", GetAcceptableApplicationDeliveryProtocolVersion },
        { "set-required-system-version", SetRequiredSystemVersion },
        { "get-required-system-version", GetRequiredSystemVersion },
        { "set-async-task-result", SetAsyncTaskResult },
    };
}

Result LocalContentShareCommand(bool* outValue, const Option& option) NN_NOEXCEPT
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMANDSYSTEM_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMANDSYSTEM_LOG("'%s' is not a DevMenu localcontentshare command. See '" DEVMENUCOMMAND_NAME " localcontentshare --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
