﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_InstallUtil.h"

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StorageId.h"
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Stopwatch.h"

// InstallCommand
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ns/ns_Result.h>
#include "DevMenuCommand_HashUtil.h"
#include "DevMenuCommand_Result.h"

// UninstallCommand

// ListCommand

using namespace nn;

namespace devmenuUtil
{
namespace {

const size_t InstallBufferSize = 16 * 1024 * 1024;

class ScopedRecordUpdater
{
public:
    enum class UpdateMode
    {
        None,
        Recreate,
        Push,
    };

    explicit ScopedRecordUpdater(ncm::ApplicationId id) : m_Id(id), m_Mode(UpdateMode::None), m_RecordChanged(false), m_InstallSucceeded(false), m_CountKeys(0) {}
    ~ScopedRecordUpdater()
    {
        Result result = ResultSuccess();
        switch(m_Mode)
        {
        case UpdateMode::Recreate:
            if (m_RecordChanged)
            {
                result = PushApplicationRecordBasedOnDatabase(m_Id);
                if (result.IsSuccess() && m_CountKeys > 0)
                {
                    // 新規にインストールしたものを改めて push しておかないと、
                    // push の順に依存してインストールしたものが消える
                    result = ns::PushApplicationRecord(m_Id, ns::ApplicationEvent::LocalInstalled, m_Keys, m_CountKeys);
                }
            }
            break;
        case UpdateMode::Push:
            if (m_InstallSucceeded)
            {
                result = ns::PushApplicationRecord(m_Id, ns::ApplicationEvent::LocalInstalled, m_Keys, m_CountKeys);
            }
            break;
        case UpdateMode::None:
        default:
            break;
        }
        if (result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("Update record failed: %08x\n", result.GetInnerValueForDebug());
        }
        result = ns::CleanupUnrecordedApplicationEntity(m_Id);
        if (result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("Cleanup failed: %08x\n", result.GetInnerValueForDebug());
        }
    }
    void SetRecordChanged()
    {
        m_RecordChanged = true;
    }
    void SetInstallSucceeded()
    {
        m_RecordChanged = true;
        m_InstallSucceeded = true;
    }
    void RegisterToRecreate()
    {
        m_Mode = UpdateMode::Recreate;
    }

    void RegisterToPush(ncm::StorageContentMetaKey* keys, int count)
    {
        m_Mode = UpdateMode::Push;
        m_Keys = keys;
        m_CountKeys = count;
    }

private:
    ncm::ApplicationId          m_Id;
    UpdateMode                  m_Mode;
    bool                        m_RecordChanged;
    bool                        m_InstallSucceeded;
    ncm::StorageContentMetaKey* m_Keys;
    int                         m_CountKeys;
};

    Result MeasureReadTime(const char* nspPath)
    {
        DEVMENUCOMMAND_LOG("Measuring to read %s...\n", nspPath);
        Stopwatch s;

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, nspPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_OS_ALIGNAS_THREAD_STACK static Bit8 s_Stack[16 * 1024];
        os::ThreadType thread;
        os::CreateThread(&thread, [](void* p){
            std::unique_ptr<char[]> buffer(new char[InstallBufferSize]);

                int64_t offset = 0;
                while (NN_STATIC_CONDITION(true))
                {
                    size_t read;
                    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(&read, *reinterpret_cast<fs::FileHandle*>(p), offset, buffer.get(), InstallBufferSize));
                    offset += read;
                    if (read < InstallBufferSize)
                    {
                        DEVMENUCOMMAND_LOG("    Read %lld bytes\n", offset);
                        break;
                    }
                }
            }, &file, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority);

        os::StartThread(&thread);
        int waitingSeconds = 0;
        while (os::TimedWaitAny(TimeSpan::FromSeconds(1), &thread) == -1)
        {
            waitingSeconds++;
            DEVMENUCOMMAND_LOG("    Waiting %d seconds\n", waitingSeconds);
        }
        os::DestroyThread(&thread);

        NN_RESULT_SUCCESS;
    }

    Result ExistsHigherVersion(bool* outExists, ncm::ContentMetaDatabase* db, const ncm::ContentMetaKey& key)
    {
        ncm::ContentMetaKey latest;
        NN_RESULT_TRY(db->GetLatest(&latest, key.id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *outExists = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outExists = (key.version < latest.version);

        NN_RESULT_SUCCESS;
    }
}

Result InstallCommonOne(bool* outValue, ncm::StorageContentMetaKey* outInstalledKey, int* outCount, int maxOutCount, const char* nspPath, ncm::StorageId storage, ncm::ContentMetaType type, bool onlyPush, bool forceUpdate, bool skipInvalidateCache, int64_t detachMillisec)
{
    ncm::ApplicationId applicationId;
    ncm::SubmissionPackageInstallTask task;
    NN_RESULT_DO(InstallCommonOne(outValue, outInstalledKey, outCount, &applicationId, maxOutCount, nspPath, storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachMillisec, &task, false));
    NN_RESULT_SUCCESS;
}

Result InstallCommonOne(bool* outValue, ncm::StorageContentMetaKey* outInstalledKey, int* outCount, int maxOutCount, const char* nspPath, ncm::StorageId storage, ncm::ContentMetaType type, bool onlyPush, bool forceUpdate, bool skipInvalidateCache, int64_t detachMillisec, ncm::SubmissionPackageInstallTask* task, bool ignoreTicket)
{
    ncm::ApplicationId applicationId;
    NN_RESULT_DO(InstallCommonOne(outValue, outInstalledKey, outCount, &applicationId, maxOutCount, nspPath, storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachMillisec, task, ignoreTicket));
    NN_RESULT_SUCCESS;
}

Result InstallCommonOne(bool* outValue, ncm::StorageContentMetaKey* outInstalledKey, int* outCount, ncm::ApplicationId* outId, int maxOutCount, const char* nspPath, ncm::StorageId storage, ncm::ContentMetaType type, bool onlyPush, bool forceUpdate, bool skipInvalidateCache, int64_t detachMillisec)
{
    ncm::SubmissionPackageInstallTask task;
    NN_RESULT_DO(InstallCommonOne(outValue, outInstalledKey, outCount, outId, maxOutCount, nspPath, storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachMillisec, &task, false));
    NN_RESULT_SUCCESS;
}

Result InstallCommonOne(bool* outValue, ncm::StorageContentMetaKey* outInstalledKey, int* outCount, ncm::ApplicationId* outId, int maxOutCount, const char* nspPath, ncm::StorageId storage, ncm::ContentMetaType type, bool onlyPush, bool forceUpdate, bool skipInvalidateCache, int64_t detachMillisec, ncm::SubmissionPackageInstallTask* task, bool ignoreTicket)
{
    NN_ABORT_UNLESS(task != nullptr);
    DEVMENUCOMMAND_LOG("Preparing to install %s...\n", nspPath);
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, nspPath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    std::unique_ptr<char[]> buffer(new char[InstallBufferSize]);
    NN_RESULT_DO(task->Initialize(file, storage, buffer.get(), InstallBufferSize, ignoreTicket));

    NN_RESULT_DO(task->Prepare());
    bool needsCleanup = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsCleanup == true)
        {
            task->Cleanup();
        }
    };

    const int MaxContentMetaCount = 2048;
    std::unique_ptr<ncm::StorageContentMetaKey[]> keys(new ncm::StorageContentMetaKey[MaxContentMetaCount]);
    ncm::ApplicationContentMetaKey appkey;

    int contentMetaCount;
    NN_RESULT_DO(task->ListContentMetaKey(&contentMetaCount, keys.get(), MaxContentMetaCount, 0));
    int outAppMetaKeyCount;
    NN_RESULT_DO(task->ListApplicationContentMetaKey(&outAppMetaKeyCount, &appkey, 1, 0));

    if (outAppMetaKeyCount == 0)
    {
        DEVMENUCOMMAND_LOG("This file doesn't contain application contents.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    // 必要な情報は取得したので、いったん cleanup
    NN_RESULT_DO(task->Cleanup());
    needsCleanup = false;
    {
        // アプリケーション記録の再生成要求管理
        ScopedRecordUpdater recordUpdater(appkey.applicationId);
        if (!onlyPush)
        {
            recordUpdater.RegisterToPush(keys.get(), contentMetaCount);
            recordUpdater.RegisterToRecreate();
        }
        else
        {
            recordUpdater.RegisterToPush(keys.get(), contentMetaCount);
        }
        for (int i = 0; i < contentMetaCount; ++i)
        {
            auto key = keys[i];
            // ContentMetaType が期待したものでなければエラー
            if (type != ncm::ContentMetaType::Unknown && key.key.type != type)
            {
                DEVMENUCOMMAND_LOG("The nsp is not expected\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            // 既にインストール済みで、そちらのバージョンが高いか同じ場合は --force オプションがない場合エラー
            bool existsHigherOrEqualVersion;
            NN_RESULT_DO(ExistsHigherOrEqualVersion(&existsHigherOrEqualVersion, appkey.applicationId, key.key));
            if (existsHigherOrEqualVersion && !forceUpdate)
            {
                DEVMENUCOMMAND_LOG("Same or higher version nsp is already installed\n");
                DEVMENUCOMMAND_LOG("To overwrite, please use --force option\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            if (existsHigherOrEqualVersion)
            {
                // 新しいものを削除し、アプリケーション記録の再生成要求をしておく
                // 記録があるけど実体がない時には見つからないので、それは無視する
                NN_RESULT_TRY(DeleteBasedOnDatabase(key.key.type, key.key.id, key.storageId))
                    NN_RESULT_CATCH(ncm::ResultContentMetaNotFound) {}
                NN_RESULT_END_TRY
                recordUpdater.SetRecordChanged();
            }
        }

        NN_RESULT_DO(task->Prepare());
        needsCleanup = true;

        NN_RESULT_DO(WaitInstallTaskExecute(task, TimeSpan::FromMilliSeconds(detachMillisec)));
        NN_RESULT_DO(task->Commit());
        recordUpdater.SetInstallSucceeded();
        needsCleanup = false;

        if (!skipInvalidateCache)
        {
            nn::ns::InvalidateApplicationControlCache(appkey.applicationId);
        }
    }
    *outCount = std::min(maxOutCount, contentMetaCount);
    std::memcpy(outInstalledKey, &keys[0], sizeof(keys[0]) * (*outCount));
    *outId = appkey.applicationId;

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result PushApplicationRecordBasedOnDatabase(ncm::ApplicationId id)
{
    // 現在インストールしたものではなく、現状のコンテントメタデータベースから Record を再生成する
    NN_RESULT_THROW_UNLESS(id != ncm::ContentMetaDatabase::AnyApplicationId, ns::ResultApplicationRecordNotFound());

    const int maxKeys = 2048;
    std::unique_ptr<ncm::ContentMetaKey[]> keys(new ncm::ContentMetaKey[maxKeys]);
    std::unique_ptr<ncm::StorageContentMetaKey[]> skeys(new ncm::StorageContentMetaKey[maxKeys * 2]);
    int countSkeys = 0;

    // Application / Patch の探索 (なくてもカードのために登録する)
    auto f = [&](ncm::StorageId storage) -> Result
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_TRY(OpenContentMetaDatabase(&db, storage))
            NN_RESULT_CATCH(ncm::ResultContentMetaDatabaseNotActive)
            {
                // Inactive な場合は、何もせずに成功扱い
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        auto listcount = db.ListContentMeta(keys.get(), maxKeys, ncm::ContentMetaType::Unknown, id);
        for (int i = 0; i < listcount.listed; ++i)
        {
            if (keys[i].type == ncm::ContentMetaType::Application
                || keys[i].type == ncm::ContentMetaType::Patch
                || keys[i].type == ncm::ContentMetaType::AddOnContent)
            {
                skeys[countSkeys] = { keys[i], storage };
                countSkeys++;
            }
        }
        NN_RESULT_SUCCESS;
    };
    // builtin
    {
        NN_RESULT_DO(f(ncm::StorageId::BuildInUser));
    }
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // card
    if (fs::IsGameCardInserted())
    {
        NN_RESULT_DO(f(ncm::StorageId::Card));
    }
#endif
    // sd
    if (ns::CheckSdCardMountStatus().IsSuccess())
    {
        NN_RESULT_DO(f(ncm::StorageId::SdCard));
    }
    NN_RESULT_TRY(ns::DeleteApplicationRecord(id))
        NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound) {} // 見つからないは無視する
    NN_RESULT_END_TRY

    if (countSkeys > 0)
    {
        NN_RESULT_DO(ns::PushApplicationRecord(id, ns::ApplicationEvent::LocalInstalled, skeys.get(), countSkeys));
    }
    NN_RESULT_SUCCESS;
}

Result ExistsHigherOrEqualVersion(bool* exists, ncm::ApplicationId id, const ncm::ContentMetaKey& key)
{
    const int MaxKeys = 16;
    ncm::StorageContentMetaKey keys[MaxKeys];
    int offset = 0;

    *exists = false;
    for(;;)
    {
        int outCount;
        NN_RESULT_TRY(ns::ListApplicationRecordInstalledContentMeta(&outCount, keys, MaxKeys, id, offset))
            NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        for (int i = 0; i < outCount; ++i)
        {
            if (ncm::IsInstallableStorage(keys[i].storageId) && keys[i].key.id == key.id && keys[i].key.type == key.type)
            {
                *exists = (keys[i].key.version >= key.version);
                NN_RESULT_SUCCESS;
            }
        }
        if (outCount != MaxKeys)
        {
            break;
        }
        offset += outCount;
    }

    NN_RESULT_SUCCESS;
}

Result Delete(const nn::ncm::ContentMetaKey& key, ncm::StorageId storageId)
{
    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, storageId));

    ncm::ContentInfo buffer[16];
    int offset = 0;
    for (;;)
    {
        int outCount;
        const int count = sizeof(buffer) / sizeof(ncm::ContentInfo);
        NN_RESULT_DO(db.ListContentInfo(&outCount, buffer, count, key, offset));
        for (int i = 0; i < outCount; i++)
        {
            NN_RESULT_TRY(storage.Delete(buffer[i].id))
                NN_RESULT_CATCH(ncm::ResultContentNotFound) {}
            NN_RESULT_END_TRY
        }
        if (outCount < count)
        {
            break;
        }
        offset += outCount;
    }
    NN_RESULT_DO(db.Remove(key));
    NN_RESULT_DO(db.Commit());

    NN_RESULT_SUCCESS;
}

Result DeleteBasedOnDatabase(ncm::ContentMetaType type, Bit64 id, ncm::StorageId storageId)
{
    const int maxKeys = 2048;
    std::unique_ptr<ncm::ContentMetaKey[]> keys(new ncm::ContentMetaKey[maxKeys]);

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, storageId));

    // install 途中の patch も削除できるようにする
    auto listcount = db.ListContentMeta(keys.get(), maxKeys, type, ncm::ContentMetaDatabase::AnyApplicationId, id, id, ncm::ContentInstallType::Unknown);
    NN_RESULT_THROW_UNLESS(listcount.listed > 0, ncm::ResultContentMetaNotFound());

    for (int i = 0; i < listcount.listed; ++i)
    {
        NN_RESULT_DO(Delete(keys[i], storageId));
    }
    NN_RESULT_SUCCESS;
}

Result DeleteBasedOnDatabase(ncm::ContentMetaType type, ncm::ApplicationId id, ncm::StorageId storageId)
{
    NN_RESULT_THROW_UNLESS(id != ncm::ContentMetaDatabase::AnyApplicationId, ncm::ResultContentMetaNotFound());

    const int maxKeys = 2048;
    std::unique_ptr<ncm::ContentMetaKey[]> keys(new ncm::ContentMetaKey[maxKeys]);

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, storageId));

    // install 途中の patch も削除できるようにする
    auto listcount = db.ListContentMeta(keys.get(), maxKeys, type, id, 0, 0xffffffffffffffff, ncm::ContentInstallType::Unknown);
    NN_RESULT_THROW_UNLESS(listcount.listed > 0, ncm::ResultContentMetaNotFound());

    for (int i = 0; i < listcount.listed; ++i)
    {
        NN_RESULT_DO(Delete(keys[i], storageId));
    }
    NN_RESULT_SUCCESS;
}

Result DeleteContent(const ncm::StorageContentMetaKey* installedKey, int contentMetaKeyCount, ncm::ApplicationId applicationId)
{
    for (int i = 0; i < contentMetaKeyCount; ++i)
    {
        auto key = installedKey[i];
        ns::DeleteRedundantApplicationEntity();
        NN_RESULT_DO(DeleteBasedOnDatabase(key.key.type, applicationId, key.storageId));
        NN_RESULT_DO(PushApplicationRecordBasedOnDatabase(applicationId));
    }

    NN_RESULT_SUCCESS;
}

Result InstallCommandCommon(ncm::StorageContentMetaKey* outInstalledKey, int* outCount, int maxOutCount, bool* outValue, const Option& option, ncm::ContentMetaType type)
{
    auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("builtin");
    auto onlyPush = option.HasKey("--only-push-record");
    NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());

    if (*storage == ncm::StorageId::SdCard)
    {
        NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
    }

    auto nspPath = option.GetTarget();
    auto isDestinationSdCard = false;
    if (!IsAbsolutePath(nspPath))
    {
        if (!IsInsertedSdCardPath(nspPath))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        const auto result = fs::MountSdCardForDebug(SdCardMountName);
        if (result.IsFailure())
        {
            if (fs::ResultPortSdCardNoDevice::Includes(result))
            {
                DEVMENUCOMMAND_LOG("SD card is not inserted.\n");
            }
            if (fs::ResultSdCardAccessFailed::Includes(result))
            {
                DEVMENUCOMMAND_LOG("Failed to access SD card.(0x%08x)\n", result.GetInnerValueForDebug());
            }
            return result;
        }
        isDestinationSdCard = true;
    }
    NN_UTIL_SCOPE_EXIT
    {
        if (isDestinationSdCard)
        {
            fs::Unmount(SdCardMountName);
        }
    };

    NN_RESULT_DO(nn::ns::DeleteRedundantApplicationEntity());

    bool forceUpdate = option.HasKey("--force");
    bool skipInvalidateCache = option.HasKey("--no-invalidate-cache");
    auto detach = option.GetValue("--detach-sdcard-after");
    int64_t detachTime = detach ? std::strtoll(detach, nullptr, 10) : 0;

    if (option.HasKey("--all"))
    {
        nn::fs::DirectoryEntryType entryType;
        NN_RESULT_DO(nn::fs::GetEntryType(&entryType, nspPath));
        if (entryType != nn::fs::DirectoryEntryType_Directory)
        {
            DEVMENUCOMMAND_LOG("%s is not directory path. --all option must used with directory.\n", nspPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int readCount = 0;
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, nspPath, nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            nn::fs::DirectoryEntry entry;
            NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, dir, 1));
            if (count == 0)
            {
                break;
            }

            auto pathString = std::string(nspPath) + "/" + entry.name;
            auto path = pathString.c_str();
            if (entry.directoryEntryType != nn::fs::DirectoryEntryType_File)
            {
                DEVMENUCOMMAND_LOG("Skip %s since it is not file.\n", path);
                continue;
            }
            if (!IsNspFile(path))
            {
                DEVMENUCOMMAND_LOG("Skip %s since it is not .nsp file.\n", path);
                continue;
            }

            int readCountOne;
            bool success;
            ncm::ApplicationId applicationId;
            NN_RESULT_DO(InstallCommonOne(&success, &outInstalledKey[readCount], &readCountOne, &applicationId, maxOutCount - readCount, path, *storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachTime));
            if(!success)
            {
                DEVMENUCOMMAND_LOG("Aborted since failed to install %s.\n", path);
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            if (!option.HasKey("--skip-patch-combination-check"))
            {
                NN_RESULT_DO(VerifyCombinationWithApplicationAtPatchInstall(&success, &outInstalledKey[readCount], readCountOne, applicationId));
                if (!success)
                {
                    NN_RESULT_DO(DeleteContent(&outInstalledKey[readCount], readCountOne, applicationId));
                    DEVMENUCOMMAND_LOG("The application is not one that is used to create the patch.\nTo not verify, please use --skip-patch-combination-check option\n");

                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            }
        }

        *outCount = readCount;
    }
    else
    {
        if (!IsNspFile(nspPath))
        {
            DEVMENUCOMMAND_LOG("Warning: The file extension is not .nsp\n");
        }

        if (option.HasKey("--measure"))
        {
            NN_RESULT_DO(MeasureReadTime(nspPath));
        }

        bool success;
        ncm::ApplicationId applicationId;
        NN_RESULT_DO(InstallCommonOne(&success, outInstalledKey, outCount, &applicationId, maxOutCount, nspPath, *storage, type, onlyPush, forceUpdate, skipInvalidateCache, detachTime));
        if (!success)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        if (!option.HasKey("--skip-patch-combination-check"))
        {
            NN_RESULT_DO(VerifyCombinationWithApplicationAtPatchInstall(&success, outInstalledKey, *outCount, applicationId));
            if (!success)
            {
                NN_RESULT_DO(DeleteContent(outInstalledKey, *outCount, applicationId));
                DEVMENUCOMMAND_LOG("The application is not one that is used to create the patch.\nTo not verify, please use --skip-patch-combination-check option\n");

                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result CheckHigherVersionAndUninstallIfNeeded(bool* exists, ncm::ContentMetaDatabase* db, ncm::ContentStorage* storage, const ncm::ContentMetaKey& key, bool doUninstall)
{
    ncm::ContentManagerAccessor accessor(db, storage);

    bool hasHigherVersion;
    NN_RESULT_DO(ExistsHigherVersion(&hasHigherVersion, db, key));
    if (hasHigherVersion && doUninstall)
    {
        NN_RESULT_DO(accessor.DeleteAll(key.id));
        NN_RESULT_DO(db->Commit());
        *exists = false;
    }
    else
    {
        *exists = hasHigherVersion;
    }

    NN_RESULT_SUCCESS;
}

Result VerifyCombinationWithApplicationAtPatchInstall(bool* outValue, const ncm::StorageContentMetaKey* installedKey, int contentMetaKeyCount, ncm::ApplicationId applicationId)
{
    for (int i = 0; i < contentMetaKeyCount; ++i)
    {
        auto key = installedKey[i];
        if (key.key.type == ncm::ContentMetaType::Patch)
        {
            // インストール済みのアプリケーションと組み合わせをチェック(アプリケーションのストレージは問わない)
            NN_RESULT_TRY(VerifyApplicationAndPatchCombination(applicationId, nn::ncm::StorageId::Any, key.storageId))
                NN_RESULT_CATCH(ResultBadApplicationForThePatch)
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(ResultApplicationNotFound)
                {
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    *outValue = false;
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

} // devmenuUtil
