﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/ncm/ncm_ContentMetaType.h>

struct GameCardWriteProgress
{
    int64_t currentSize;
    int64_t totalSize;
};

const size_t MaxFilePathLength = 512;

bool IsGameCardSizeRight(nn::fs::GameCardSize gameCardSize) NN_NOEXCEPT;
bool IsGameCardClockRateRight(nn::fs::GameCardClockRate gameCardClockRate) NN_NOEXCEPT;
nn::Result WriteFromNsp(const char (*srcRomNspPaths)[MaxFilePathLength], int srcRomNspCount, nn::fs::GameCardSize cardSize, nn::fs::GameCardClockRate cardClockRate, const char* srcUpdateNspPath, bool isAutoBoot, bool isHistoryErase, bool isRepairTool, size_t timeout, bool isFastParam) NN_NOEXCEPT;
nn::Result EraseGameCard(size_t timeout) NN_NOEXCEPT;
void GetGameCardWriteProgress(GameCardWriteProgress* pOutProgress) NN_NOEXCEPT;
nn::Result CheckContentMeta( bool* pOutValue, const char* nspFilePath, nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT;
#ifdef NN_BUILD_CONFIG_OS_WIN
nn::Result WriteFromNsp(const char (*srcRomNspPaths)[MaxFilePathLength], int srcRomNspCount, nn::fs::GameCardSize cardSize, nn::fs::GameCardClockRate cardClockRate, const char* srcUpdateNspPath, bool isAutoBoot, bool isHistoryErase, bool isRepairTool, size_t timeout, bool isFastParam, const char* dstXciPath) NN_NOEXCEPT;
#endif

struct ContentMetaInfo
{
    nn::Bit64 id;
    uint32_t version;
    // char type[16]; // Application or Patch が入るが、あえてチェックする必要はないので省略する
    char digest[64];

    bool operator==(const ContentMetaInfo& rhs) const NN_NOEXCEPT
    {
        return  id == rhs.id &&
                version == rhs.version &&
                memcmp(digest, rhs.digest, sizeof(digest)) == 0;
    }

    bool operator<(const ContentMetaInfo& rhs) const NN_NOEXCEPT
    {
        return  id < rhs.id &&
                version < rhs.version &&
                memcmp(digest, rhs.digest, sizeof(digest)) < 0;
    }
};

class MultiApplicationCardIndicator
{
public:
    explicit MultiApplicationCardIndicator(const char* nspPath) NN_NOEXCEPT;

    nn::Result Initialize();

    bool Verify(const char (*pathList)[MaxFilePathLength], int numPath);

    nn::fs::GameCardSize GetCardSize() const NN_NOEXCEPT { return m_CardSize; }
    nn::fs::GameCardClockRate GetCardClockRate() const NN_NOEXCEPT { return m_CardClockRate; }

private:
    const char* m_NspPath;

    std::vector<ContentMetaInfo> m_ApplicationList;
    std::vector<ContentMetaInfo> m_PatchList;

    nn::fs::GameCardSize m_CardSize {};
    nn::fs::GameCardClockRate m_CardClockRate {};
};
