﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>

#include <algorithm>
#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <vector>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>

#include <nn/fs_Base.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_GameCard.h>

#include <nn/nifm.h>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_GameCardApi.h>
#include <nn/ns/ns_GameCardRegistrationApi.h>
#include <nn/ns/ns_Result.h>

#include "DevMenuCommand_ApplicationCommand.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_GameCardWriterImpl.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Result.h"
#include "DevMenuCommand_ShopUtil.h"
#include "DevMenuCommand_Stopwatch.h"

using namespace nn;
using namespace nne;
using namespace nn::fs;

namespace {

    struct InputCheckItem {
        enum class Type {
            FilePath,
            GameCardSize,
            TimeoutSecond,
            GameCardClockRate,
        } type;
        void* pOutValue;
        const size_t outSize;
        const char*  inputString;
        const char*  name;
        const nn::ncm::ContentMetaType contentMetaType;
    };

    class NspListFile
    {
    public:
        Result Initialize(const char* filePath)
        {
            NN_ASSERT_NOT_NULL(filePath);

            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
            NN_ABORT_UNLESS(fileSize < SIZE_MAX);

            const size_t bufferSize = static_cast<size_t>(fileSize) + 1; // 最後のヌル文字用 に1バイト余分に確保する
            std::unique_ptr<char[]> buffer(new char[bufferSize]);
            NN_ABORT_UNLESS(buffer);

            NN_RESULT_DO(fs::ReadFile(file, 0, buffer.get(), static_cast<size_t>(fileSize)));

            m_FileSize = fileSize;
            m_Buffer = std::move(buffer);
            m_BufferSize = bufferSize;

            ParseFile();

            NN_RESULT_SUCCESS;
        }

        int Count() const NN_NOEXCEPT
        {
            return static_cast<int>(m_LineList.size());
        }

        const char* Get(int index) const
        {
            NN_ASSERT_LESS(index, Count());
            return m_LineList[index];
        }

    private:
        char* ReadLine(size_t* outSize, char** outBegin, char* src, size_t length) const NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(outSize);
            NN_ASSERT_NOT_NULL(outBegin);
            NN_ASSERT_NOT_NULL(src);

            const std::array<char, 2> SpaceChars = { { ' ', '\t' } };
            auto hasSpaceChars = [&SpaceChars](char c) NN_NOEXCEPT
            {
                return std::find(SpaceChars.begin(), SpaceChars.end(), c) != SpaceChars.end();
            };

            const std::array<char, 3> EndChars = { { '\n', '\r', '\0', } };
            auto hasEndChars = [&EndChars](char c) NN_NOEXCEPT
            {
                return std::find(EndChars.begin(), EndChars.end(), c) != EndChars.end();
            };

            const auto begin = src;
            const auto end = src + length;

            auto copyBegin = std::find_if_not(begin, end, hasSpaceChars);
            auto lineEnd = std::find_if(copyBegin, end, hasEndChars);

            const std::reverse_iterator<char*> rbegin(lineEnd);
            const std::reverse_iterator<char*> rend(copyBegin);
            auto copyEnd = (lineEnd != copyBegin) ? std::find_if_not(rbegin, rend, hasSpaceChars).base() : lineEnd;

            *outSize = std::distance(copyBegin, copyEnd);
            *outBegin = (copyBegin != copyEnd) ? copyBegin : nullptr;

            auto next = std::find_if_not(lineEnd, end, hasEndChars);
            return next != end ? next : nullptr;
        }

        void ParseFile() NN_NOEXCEPT
        {
            const auto begin = m_Buffer.get();
            char* current = m_Buffer.get();

            // BOMB を削除
            constexpr Bit8 Bom[] = { 0xEF, 0xBB, 0xBF };
            if (m_FileSize >= static_cast<int64_t>(NN_ARRAY_SIZE(Bom)) &&
                memcmp(begin, Bom, sizeof(Bom)) == 0)
            {
                current = &current[NN_ARRAY_SIZE(Bom)];
                if (current == begin + m_FileSize)
                {
                    return;
                }
            }

            while (current)
            {
                size_t length;
                char* dst;
                auto next = ReadLine(&length, &dst, current, static_cast<size_t>(m_FileSize) - (current - begin));
                if (length > 0)
                {
                    dst[length] = '\0';
                    m_LineList.push_back(dst);
                }
                current = next;
            };
        }

        std::unique_ptr<char[]> m_Buffer {};
        size_t m_BufferSize {};
        int64_t m_FileSize {};
        std::vector<const char*> m_LineList{};
    };

    Result GameCardStatusCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);
        if(!nn::fs::IsGameCardInserted())
        {
            DEVMENUCOMMAND_LOG("Not ");
        }
        DEVMENUCOMMAND_LOG("Inserted\n");
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GameCardEraseCommand(bool* outValue, const Option& option)
    {
        size_t timeout = 0;
        if(option.HasKey("--timeout"))
        {
            sscanf(option.GetValue("--timeout"), "%zd", &timeout);
            DEVMENUCOMMAND_LOG("timeout : %zds\n", timeout);
        }

        *outValue = EraseGameCard(timeout).IsSuccess();
        NN_RESULT_SUCCESS;
    }

    bool CheckAndGetValue(const std::vector<InputCheckItem>& items) NN_NOEXCEPT
    {
        bool isValid = true;

        for (const auto& item : items)
        {
            if (InputCheckItem::Type::FilePath == item.type)
            {
                auto outPath = reinterpret_cast<char*>(item.pOutValue);
                const auto srcSize = nn::util::Strlcpy(outPath, item.inputString, static_cast<int>(item.outSize));

                NN_ASSERT_GREATER(srcSize, 0);

                if (static_cast<size_t>(srcSize) >= item.outSize)
                {
                    DEVMENUCOMMAND_LOG("error : %s file path length must be less than %tu (current: %zd).\n", item.name, item.outSize, srcSize);
                    isValid = false;
                    continue;
                }

                if (!devmenuUtil::IsAbsolutePath(outPath))
                {
                    DEVMENUCOMMAND_LOG("error : %s file path must be absolute path.\n", item.name);
                    isValid = false;
                    continue;
                }

                if (!devmenuUtil::IsNspFile(outPath))
                {
                    DEVMENUCOMMAND_LOG("error : %s file name is not correct. It doesn't include \"nsp\" extension.\n", item.name);
                    isValid = false;
                    continue;
                }

                if (item.contentMetaType == nn::ncm::ContentMetaType::Application ||
                    item.contentMetaType == nn::ncm::ContentMetaType::Patch ||
                    item.contentMetaType == nn::ncm::ContentMetaType::AddOnContent)
                {
                    bool isValidContent;
                    auto result = CheckContentMeta(&isValidContent, outPath, item.contentMetaType);
                    if (result.IsFailure() || !isValidContent)
                    {
                        DEVMENUCOMMAND_LOG( "error : %s file is invalid.\n", item.name );
                        isValid = false;
                        continue;
                    }
                }

                DEVMENUCOMMAND_LOG("%s : %s\n", item.name, outPath);
            }
            else if (InputCheckItem::Type::GameCardSize == item.type)
            {
                auto pSize = reinterpret_cast<nn::fs::GameCardSize*>(item.pOutValue);
                int i = 0;
                sscanf(item.inputString, "%d", &i);
                *pSize = static_cast<nn::fs::GameCardSize>(i);

                if (!IsGameCardSizeRight(*pSize))
                {
                    DEVMENUCOMMAND_LOG("error : --size <1,2,4,8,16,32> or Do not specify --size.\n");
                    isValid = false;
                    continue;
                }
                DEVMENUCOMMAND_LOG("%s : %d\n", item.name, static_cast<uint32_t>(*pSize));
            }
            else if (InputCheckItem::Type::TimeoutSecond == item.type)
            {
                auto pSecond = reinterpret_cast<size_t*>(item.pOutValue);
                sscanf(item.inputString, "%tu", pSecond);
                if (*pSecond == 0)
                {
                    DEVMENUCOMMAND_LOG("error : timeout second is zero or not input. It must be greater than or equal to 1.\n");
                    isValid = false;
                    continue;
                }
                DEVMENUCOMMAND_LOG("%s : %tus\n", item.name, *pSecond);
            }
            else if (InputCheckItem::Type::GameCardClockRate == item.type)
            {
                auto pClockRate = reinterpret_cast<nn::fs::GameCardClockRate*>(item.pOutValue);
                int i = 0;
                sscanf(item.inputString, "%d", &i);
                *pClockRate = static_cast<nn::fs::GameCardClockRate>(i);

                if (!IsGameCardClockRateRight(*pClockRate))
                {
                    DEVMENUCOMMAND_LOG("error : --clock-rate <25, 50> or Do not specify --clock-rate.\n");
                    isValid = false;
                    continue;
                }
                DEVMENUCOMMAND_LOG("%s : %d\n", item.name, static_cast<uint32_t>(*pClockRate));
            }
            else
            {
                NN_ABORT("Must not come here\n");
            }
        }
        return isValid;
    }

    Result GameCardWriteCommand(bool* outValue, const Option& option)
    {
        nn::fs::SetVerifyWriteEnalbleFlag(false);

        char cupPath[MaxFilePathLength] = {}; // TORIAEZU
        nn::fs::GameCardSize cardSize = {};
        nn::fs::GameCardClockRate cardClockRate = {};

        bool isAutoBoot     = false;
        bool isHistoryErase = false;
        bool isRepairTool   = false;
        bool isFastParam = false;
        size_t timeout = 0;

        int nspPathCount = 0;
        NspListFile nspListFile;

        if (option.HasKey("--nsp-list-file"))
        {
            if (!option.HasKey("--multi-application-card"))
            {
                DEVMENUCOMMAND_LOG("Multi-Application card requires --multi-application-card option.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            if (option.GetTargetCount() > 0 || option.HasKey("--on-card-patch") || option.HasKey("--on-card-aoc"))
            {
                DEVMENUCOMMAND_LOG("Please specify nsp paths with nsp list file when you specify --nsp-list-file option.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_DO(nspListFile.Initialize(option.GetValue("--nsp-list-file")));
            nspPathCount = nspListFile.Count();
            if (nspPathCount <= 1)
            {
                DEVMENUCOMMAND_LOG("Nsp list file needs two or more nsp paths.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            if (option.GetTargetCount() > 1 && !option.HasKey("--multi-application-card"))
            {
                DEVMENUCOMMAND_LOG("Multi-Application card requires --multi-application-card option.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            nspPathCount = option.GetTargetCount();

            if (option.HasKey("--on-card-patch"))
            {
                if (option.GetTargetCount() == 0)
                {
                    DEVMENUCOMMAND_LOG("error : application nsp must be also specified.\n");
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
                nspPathCount++;
            }
            if (option.HasKey("--on-card-aoc"))
            {
                if (option.GetTargetCount() == 0)
                {
                    DEVMENUCOMMAND_LOG("error : application nsp must be also specified.\n");
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
                nspPathCount++;
            }
        }
        std::unique_ptr<char[][MaxFilePathLength]> nspPath(new char[nspPathCount][MaxFilePathLength]);
        char macNspPath[MaxFilePathLength];

        std::vector<InputCheckItem> checkItems;

        DEVMENUCOMMAND_LOG("---- given Parameter ----\n");

        int nspIndex = 0;
        // オプションへの入力値のチェック
        if (option.HasKey("--nsp-list-file"))
        {
            for (int i = 0; i < nspPathCount; i++)
            {
                checkItems.push_back(
                    {InputCheckItem::Type::FilePath, nspPath[i], MaxFilePathLength, nspListFile.Get(i), "application nsp", nn::ncm::ContentMetaType::Unknown});
                nspIndex++;
            }
        }
        else
        {
            {
                // アプリケーション本体
                for (int i = 0; option.HasTarget(i); i++)
                {
                    checkItems.push_back(
                        {InputCheckItem::Type::FilePath, nspPath[i], MaxFilePathLength, option.GetTarget(i), "application nsp", nn::ncm::ContentMetaType::Application});
                    nspIndex++;
                }
            }

            if (option.HasKey("--on-card-aoc"))
            {
                checkItems.push_back(
                    {InputCheckItem::Type::FilePath, nspPath[nspIndex++], MaxFilePathLength, option.GetValue("--on-card-aoc"), "aoc nsp", nn::ncm::ContentMetaType::AddOnContent});
            }

            if (option.HasKey("--on-card-patch"))
            {
                checkItems.push_back(
                    {InputCheckItem::Type::FilePath, nspPath[nspIndex++], MaxFilePathLength, option.GetValue("--on-card-patch"), "patch nsp", nn::ncm::ContentMetaType::Patch});
            }
        }

        if (option.HasKey("--update-partition"))
        {
            checkItems.push_back(
                {InputCheckItem::Type::FilePath, cupPath, MaxFilePathLength, option.GetValue("--update-partition"), "upp nsp", nn::ncm::ContentMetaType::Unknown});
        }

        if (option.HasKey("--size"))
        {
            checkItems.push_back(
                {InputCheckItem::Type::GameCardSize, &cardSize, sizeof(cardSize), option.GetValue("--size"), "size", nn::ncm::ContentMetaType::Unknown});
        }

        if (option.HasKey("--clock-rate"))
        {
            checkItems.push_back(
                {InputCheckItem::Type::GameCardClockRate, &cardClockRate, sizeof(cardClockRate), option.GetValue("--clock-rate"), "clock rate", nn::ncm::ContentMetaType::Unknown});
        }

        if (option.HasKey("--timeout"))
        {
            checkItems.push_back(
                {InputCheckItem::Type::TimeoutSecond, &timeout, sizeof(timeout), option.GetValue("--timeout"), "timeout", nn::ncm::ContentMetaType::Unknown});
        }

        if (option.HasKey("--multi-application-card"))
        {
            checkItems.push_back(
                {InputCheckItem::Type::FilePath, macNspPath, MaxFilePathLength, option.GetValue("--multi-application-card"), "multi application card nsp", nn::ncm::ContentMetaType::Unknown});
        }

        if (!CheckAndGetValue(checkItems))
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // Multi-Application Card のチェック
        if (option.HasKey("--multi-application-card"))
        {
            MultiApplicationCardIndicator mac(macNspPath);
            NN_RESULT_DO(mac.Initialize());
            if (!mac.Verify(nspPath.get(), nspIndex))
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }

            if (option.HasKey("--size"))
            {
                DEVMENUCOMMAND_LOG("Warning : --size option overrides card size of multi application card nsp.\n");
            }
            else
            {
                cardSize = mac.GetCardSize();
            }

            if (option.HasKey("--clock-rate"))
            {
                DEVMENUCOMMAND_LOG("Warning : --clock-rate option overrides clock rate of multi application card nsp.\n");
            }
            else
            {
                cardClockRate = static_cast<uint8_t>(cardClockRate) != 0 ? cardClockRate : mac.GetCardClockRate();
            }
        }

        // 有効なオプションのチェック
        if (option.HasKey("--auto"))
        {
            isAutoBoot = true;
            DEVMENUCOMMAND_LOG("autoboot enable\n");
        }

        if (option.HasKey("--nohistory"))
        {
            isHistoryErase = true;
            DEVMENUCOMMAND_LOG("nohistory enable\n");
        }

        if (option.HasKey("--repairtool"))
        {
            isRepairTool = true;
            DEVMENUCOMMAND_LOG("repairtool enable\n");
        }

        if (option.HasKey("--verify"))
        {
            nn::fs::SetVerifyWriteEnalbleFlag(true);
            DEVMENUCOMMAND_LOG("verify enable\n");
        }

        if (option.HasKey("--fast"))
        {
            isFastParam = true;
            DEVMENUCOMMAND_LOG("fastParam enable\n");
        }

#ifdef NN_BUILD_CONFIG_OS_WIN
        char dstPath[MaxFilePathLength] = {}; // TORIAEZU
        if (option.HasKey("--dst"))
        {
            nn::util::Strlcpy(dstPath, option.GetValue("--dst"), MaxFilePathLength);
            DEVMENUCOMMAND_LOG("dst : %s\n", dstPath);
        }
        auto result = (WriteFromNsp(nspPath.get(), nspPathCount, cardSize, cardClockRate, cupPath, isAutoBoot, isHistoryErase, isRepairTool, timeout, isFastParam, dstPath));
#else
        auto result = (WriteFromNsp(nspPath.get(), nspPathCount, cardSize, cardClockRate, cupPath, isAutoBoot, isHistoryErase, isRepairTool, timeout, isFastParam));
#endif
        if (result.IsFailure())
        {
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                DEVMENUCOMMAND_LOG("Input nsp file is not found.\n");
            }
            else if(devmenuUtil::ResultInvalidExternalKey::Includes(result))
            {
                DEVMENUCOMMAND_LOG("External key nsp can not be written to the game card.\n");
            }
            else
            {
                DEVMENUCOMMAND_LOG("Failed to write nsp to game card 0x%08x\n", result.GetInnerValueForDebug());
            }
        }
        *outValue = result.IsSuccess();
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result GameCardMountStatusCommand(bool* outValue, const Option&)
    {
        ns::GameCardAttachmentInfo info;
        ns::GetGameCardAttachmentInfo(&info);

        auto result = ns::GetLastGameCardMountFailureResult();

        rapidjson::Document document;
        document.SetObject();

        document.AddMember("sequence", rapidjson::Value(info.sequence), document.GetAllocator());
        document.AddMember("isAttached", rapidjson::Value(info.isAttached), document.GetAllocator());
        document.AddMember("lastMountFailureResult", rapidjson::Value(devmenuUtil::GetBit32String(result.GetInnerValueForDebug()).data, document.GetAllocator()), document.GetAllocator());

        nne::rapidjson::StringBuffer buffer;
        nne::rapidjson::PrettyWriter<nne::rapidjson::StringBuffer> writer(buffer);
        document.Accept(writer);

        DEVMENUCOMMAND_LOG("%s\n", buffer.GetString());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GameCardEnsureAccessCommand(bool* outValue, const Option&)
    {
        NN_RESULT_DO(ns::EnsureGameCardAccess());
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GameCardListCommand(bool* outValue, const Option&)
    {
        *outValue = false;

        ncm::ApplicationId idList[ns::MaxApplicationCountOnGameCard];
        int count;
        NN_RESULT_DO(ns::ListApplicationIdOnGameCard(&count, idList, static_cast<int>(NN_ARRAY_SIZE(idList))));

        DEVMENUCOMMAND_LOG("Application List (%d applications):\n", count);
        for (int i = 0; i < count; i++)
        {
            DEVMENUCOMMAND_LOG("0x%016llx\n", idList[i]);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result FindGameCardApplication(util::optional<ncm::ApplicationId>* outValue)
    {
        std::vector<ns::ApplicationRecord> recordList(2048);
        auto count = ns::ListApplicationRecord(recordList.data(), static_cast<int>(recordList.size()), 0);
        recordList.resize(static_cast<size_t>(count));
        for (auto& record : recordList)
        {
            ns::ApplicationView view;
            NN_RESULT_DO(ns::GetApplicationView(&view, &record.id, 1));

            if (view.IsGameCard())
            {
                *outValue = record.id;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = util::nullopt;
        NN_RESULT_SUCCESS;
    }

    Result RegisterCommand(bool* outValue, const Option& option)
    {
        auto userIndex = static_cast<size_t>(std::strtoul(option.GetTarget(), nullptr, 10));

        util::optional<ncm::ApplicationId> appId;
        NN_RESULT_DO(FindGameCardApplication(&appId));
        if (!appId)
        {
            DEVMENUCOMMAND_LOG("Not found any gamecard application\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Found application ID 0x%016llx on gamecard\n", appId->value);

        account::InitializeForAdministrator();

        std::vector<nn::account::Uid> uidList(account::UserCountMax);
        int count;
        NN_RESULT_DO(account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
        if (count == 0)
        {
            DEVMENUCOMMAND_LOG("Not found any user account\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        uidList.resize(static_cast<size_t>(count));

        if (userIndex >= uidList.size())
        {
            DEVMENUCOMMAND_LOG("Not found user index %zu\n, userIndex");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto uid = uidList[userIndex];

        auto pointString = option.GetValue("--point");

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int goldPoint = 0;
        if(!pointString)
        {
            ns::AsyncGameCardGoldPoint async;
            NN_RESULT_DO(ns::RequestGameCardRegistrationGoldPoint(&async, uid, *appId));

            {
                Stopwatch stopWatch;
                DEVMENUCOMMAND_LOG("Wating request Application ID 0x%016llx registration status for uid %016llx-%016llx\n", appId->value, uid._data[0], uid._data[1]);
                async.Wait();
            }

            NN_RESULT_DO(async.Get(&goldPoint));
        }
        else
        {
            goldPoint = std::strtol(pointString, nullptr, 10);
        }

        {
            ns::AsyncResult async;
            NN_RESULT_DO(ns::RequestRegisterGameCard(&async, uid, *appId, goldPoint));

            {
                Stopwatch stopWatch;
                DEVMENUCOMMAND_LOG("Wating request register gamecard application ID 0x%016llx for uid %016llx-%016llx\n", appId->value, uid._data[0], uid._data[1]);
                async.Wait();
            }

            NN_RESULT_DO(async.Get());
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckRegistrationCommand(bool* outValue, const Option&)
    {
        util::optional<ncm::ApplicationId> appId;
        NN_RESULT_DO(FindGameCardApplication(&appId));
        if (!appId)
        {
            DEVMENUCOMMAND_LOG("Not found any gamecard application\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Found application ID 0x%016llx on gamecard\n", appId->value);

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        {
            ns::AsyncResult async;
            NN_RESULT_DO(ns::RequestCheckGameCardRegistration(&async, *appId));

            {
                Stopwatch stopWatch;
                DEVMENUCOMMAND_LOG("Wating request Application ID 0x%016llx registration status\n", appId->value);
                async.Wait();
            }

            auto result = async.Get();
            bool registrable = true;
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::ns::ResultGameCardAlreadyRegistered)
                {
                    registrable = false;
                }
            NN_RESULT_END_TRY;
            DEVMENUCOMMAND_LOG("Registration: %s\n", registrable ? "True" : "False");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnregisterCommand(bool* outValue, const Option& option)
    {
        account::InitializeForAdministrator();

        std::vector<nn::account::Uid> uidList(account::UserCountMax);
        int count;
        NN_RESULT_DO(account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
        if (count == 0)
        {
            DEVMENUCOMMAND_LOG("Not found any user account\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        uidList.resize(static_cast<size_t>(count));
        auto userIndex = static_cast<size_t>(std::strtoul(option.GetTarget(), nullptr, 10));

        if (userIndex >= uidList.size())
        {
            DEVMENUCOMMAND_LOG("Not found user index %zu\n, userIndex");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto uid = uidList[userIndex];

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::Bit64 naId;
        NN_RESULT_DO(GetNaId(&naId, uid));

        const size_t BufferSize = 4096;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);
        NN_RESULT_DO(UnregisterGameCardByNaId(naId, buffer.get(), BufferSize));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }


    Result RegistrationGoldPointCommand(bool* outValue, const Option& option)
    {
        auto userIndex = static_cast<size_t>(std::strtoul(option.GetTarget(), nullptr, 10));

        util::optional<ncm::ApplicationId> appId;
        NN_RESULT_DO(FindGameCardApplication(&appId));
        if (!appId)
        {
            DEVMENUCOMMAND_LOG("Not found any gamecard application\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("Found application ID 0x%016llx on gamecard\n", appId->value);

        account::InitializeForAdministrator();

        std::vector<nn::account::Uid> uidList(account::UserCountMax);
        int count;
        NN_RESULT_DO(account::ListAllUsers(&count, uidList.data(), static_cast<int>(uidList.size())));
        if (count == 0)
        {
            DEVMENUCOMMAND_LOG("Not found any user account\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        uidList.resize(static_cast<size_t>(count));

        if (userIndex >= uidList.size())
        {
            DEVMENUCOMMAND_LOG("Not found user index %zu\n, userIndex");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        auto uid = uidList[userIndex];

        nifm::SubmitNetworkRequestAndWait();
        if (!nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available\n");

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        {
            ns::AsyncGameCardGoldPoint async;
            NN_RESULT_DO(ns::RequestGameCardRegistrationGoldPoint(&async, uid, *appId));

            {
                Stopwatch stopWatch;
                DEVMENUCOMMAND_LOG("Wating request register gamecard application ID 0x%016llx for uid %016llx-%016llx\n", appId->value, uid._data[0], uid._data[1]);
                async.Wait();
            }

            int goldPoint;
            NN_RESULT_DO(async.Get(&goldPoint));
            DEVMENUCOMMAND_LOG("Registration gold point: %d\n", goldPoint);
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#endif

    const char HelpMessage[] =
            "usage: " DEVMENUCOMMAND_NAME " gamecard status\n"
            "       " DEVMENUCOMMAND_NAME " gamecard write <absolute_nsp_path> [--timeout <seconds>] [--verify] [--on-card-patch <absolute_nsp_path>] [--on-card-aoc <absolute_nsp_path>] [--multi-application-card <absolute_nsp_path>] [--nsp-list-file <absolute_text_file_path>]\n"
            "       " DEVMENUCOMMAND_NAME " gamecard erase [--timeout <seconds>]\n"
            "       " DEVMENUCOMMAND_NAME " gamecard mount-status\n"
            "       " DEVMENUCOMMAND_NAME " gamecard ensure-access\n"
            "       " DEVMENUCOMMAND_NAME " gamecard list\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
            "       " DEVMENUCOMMAND_NAME " gamecard register <user_account_index>\n"
            "       " DEVMENUCOMMAND_NAME " gamecard unregister\n"
            "       " DEVMENUCOMMAND_NAME " gamecard check-registration\n"
            "       " DEVMENUCOMMAND_NAME " gamecard registration-gold-point <user_account_index>\n"
#endif
        ;


    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
            { "status",         GameCardStatusCommand },
            { "write",          GameCardWriteCommand },
            { "erase",          GameCardEraseCommand },
            { "mount-status",   GameCardMountStatusCommand },
            { "ensure-access",  GameCardEnsureAccessCommand },
            { "list",           GameCardListCommand },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
            { "register",                   RegisterCommand },
            { "check-registration",         CheckRegistrationCommand },
            { "registration-gold-point",    RegistrationGoldPointCommand },
            { "unregister",                 UnregisterCommand },
#endif
    };
}

Result GameCardCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu gamecard command. See '" DEVMENUCOMMAND_NAME " gamecard --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
