﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>
#include <string>
#include <sstream>
#include <nn/nn_Log.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/es/es_ELicenseApiForSystem.h>
#include <nn/es/debug/es_ELicenseArchiveJsonUtil.h>
#include <nn/nim/nim_DynamicRightsApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Json.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_ELicenseUtil.h"
#include "DevMenuCommand_ELicenseCommand.h"

using namespace nn;
using namespace nne;
using namespace devmenuUtil;

namespace
{
    constexpr int JsonBufferSize = 64 * 1024;

    const char g_HelpMessage[] =
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "usage: " DEVMENUCOMMAND_NAME " elicense list\n"
        "       " DEVMENUCOMMAND_NAME " elicense import [--account <user_account_index>] <input_elicense_archive.json>\n"
        "       " DEVMENUCOMMAND_NAME " elicense export [--account <user_account_index>] [<output_elicense_archive.json>] [-f|--force]\n"
        "       " DEVMENUCOMMAND_NAME " elicense create [--account <user_account_index>] --rights-type <permanent|temporary> [--expire-span <TIMESPAN_SPEC>] <rights_id...>\n"
        "       " DEVMENUCOMMAND_NAME " elicense delete-all-archives\n"
        "\n"
        "       TIMESPAN_SPEC  [0-9]+d|[0-9]+h|[0-9]+m|[0-9]+s|[0-9]+ms|[0-9]+us|[0-9]+ns\n"
#endif
        "";

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    Result List(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        es::ELicenseId elicenseIds[ELicenseIdsMax];
        int elicenseIdCount = es::ListELicenseIds(elicenseIds, ELicenseIdsMax);
        DEVMENUCOMMAND_LOG("[ELicense] count = %d\n", elicenseIdCount);
        if (elicenseIdCount > 0)
        {
            DEVMENUCOMMAND_LOG("--------------------------------------------------------------------\n");
            es::ELicenseInfoForSystemWrapper elicenses[ELicenseIdsMax];
            int elicenseCount = es::ListELicenseInfoForDebug(elicenses, ELicenseIdsMax, elicenseIds, elicenseIdCount);
            NN_ABORT_UNLESS(elicenseCount == elicenseIdCount);
            for (int i = 0; i < elicenseCount; i++)
            {
                PrintELicenseInfoForSystemWrapper(elicenses[i]);
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Import(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        bool isELicenseChallengeVerificationForDebugEnabled = false;
        settings::fwdbg::GetSettingsItemValue(&isELicenseChallengeVerificationForDebugEnabled, sizeof(isELicenseChallengeVerificationForDebugEnabled), "es.debug", "ignore_elicense_archive_challenge_verification_for_debug");
        if (!isELicenseChallengeVerificationForDebugEnabled)
        {
            DEVMENUCOMMAND_LOG("Importing elicense requires permission by es.debug ignore_elicense_archive_challenge_verification_for_debug");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::Uid uid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId = account::InvalidNintendoAccountId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        if (option.GetTargetCount() != 1)
        {
            DEVMENUCOMMAND_LOG("import source file must be specified\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        const char* sourcePath = option.GetTarget(0);
        size_t jsonSize = 0;
        std::unique_ptr<char[]> jsonBuffer(new char[JsonBufferSize]);
        NN_RESULT_DO(devmenuUtil::ReadFile(&jsonSize, sourcePath, jsonBuffer.get(), JsonBufferSize));

        es::ELicenseImportContext context;
        NN_RESULT_DO(es::BeginImportELicenseArchive(&context, naId));

        NN_RESULT_DO(es::ImportELicenseArchive(context, jsonBuffer.get(), jsonSize));

        es::ELicenseArchiveId archiveId;
        NN_RESULT_DO(es::EndImportELicenseArchiveForDebug(&archiveId, context));

        char archiveIdString[33];
        DEVMENUCOMMAND_LOG("eLicense archive imported: ArchiveID: %s %s\n", archiveId.ToString(archiveIdString, sizeof(archiveIdString)), sourcePath);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Export(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId = account::InvalidNintendoAccountId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        size_t jsonSize;
        NN_RESULT_TRY(es::GetELicenseArchiveSizeForDebug(&jsonSize, naId))
            NN_RESULT_CATCH(es::ResultELicenseArchiveNotFound)
            {
                DEVMENUCOMMAND_LOG("eLicense archive not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        DEVMENUCOMMAND_LOG("eLicense json size: %u\n", jsonSize);

        std::unique_ptr<char[]> jsonBuffer(new char[jsonSize]);
        size_t tmpJsonSize;
        NN_RESULT_DO(es::GetELicenseArchiveDataForDebug(&tmpJsonSize, jsonBuffer.get(), jsonSize, naId));
        NN_SDK_ASSERT_EQUAL(jsonSize, tmpJsonSize);

        if (option.GetTargetCount() > 0)
        {
            bool isForced = option.HasKey("--force") || option.HasKey("-f");
            NN_RESULT_DO(devmenuUtil::WriteFile(option.GetTarget(0), jsonBuffer.get(), jsonSize, isForced));
        }
        else
        {
            DEVMENUCOMMAND_LOG("eLicense json:\n");
            DEVMENUCOMMAND_LOG("%.*s\n", static_cast<int>(jsonSize), jsonBuffer.get());
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Create(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        account::Uid uid = account::InvalidUid;
        NN_RESULT_DO(GetUidFromOption(&uid, option));
        if (uid == account::InvalidUid)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));

        util::optional<nim::ELicenseType> elicenseType;
        NN_RESULT_DO(GetELicenseTypeFromOption(&elicenseType, option));
        if (!elicenseType)
        {
            DEVMENUCOMMAND_LOG("--type option must be specified at create subcommand\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        else if (*elicenseType == nim::ELicenseType::EndOfContents)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        util::optional<TimeSpan> expireSpan;
        NN_RESULT_DO(GetTimeSpanFromOption(&expireSpan, option, "--expire-span"));

        int rightsIdCount = -1;
        es::RightsId rightsIds[RightsIdsMax] = {};
        NN_RESULT_DO(GetRightsIdsFromTarget(&rightsIdCount, rightsIds, RightsIdsMax, option));
        if (rightsIdCount <= 0)
        {
            DEVMENUCOMMAND_LOG("no rights id is specified\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        es::ELicenseImportContext context;
        NN_RESULT_DO(es::BeginImportELicenseArchive(&context, naId));

        nn::es::debug::ELicenseArchiveInfoForDebug archiveInfo = nn::es::debug::MakeELicenseArchiveInfo(context.GetChallenge().value, naId);
        nn::es::debug::ELicenseInfoForDebug eLicenseInfos[RightsIdsMax];
        for (int i = 0; i < rightsIdCount; i++)
        {
            bool isCreateSucceeded = false;
            switch (*elicenseType)
            {
            case nim::ELicenseType::Permanent:
                isCreateSucceeded = es::debug::MakeDeviceLinkedPermanentELicenseFromTicketDBInfo(&eLicenseInfos[i], rightsIds[i], naId);
                break;
            case nim::ELicenseType::Temporary:
                if (expireSpan == util::nullopt)
                {
                    isCreateSucceeded = es::debug::MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(&eLicenseInfos[i], rightsIds[i], naId);
                }
                else
                {
                    isCreateSucceeded = es::debug::MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(&eLicenseInfos[i], rightsIds[i], naId, *expireSpan);
                }
                break;
            case nim::ELicenseType::Unknown:
            case nim::ELicenseType::DeviceLinkedPermanent:
            case nim::ELicenseType::AccountRestrictivePermanent:
            default:
                DEVMENUCOMMAND_LOG("elicense type is not supported: %s.\n", option.GetValue("--type"));
                *outValue = false;
                NN_RESULT_SUCCESS;
                break;
            }

            if (!isCreateSucceeded)
            {
                DEVMENUCOMMAND_LOG("failed to make elicense because ticket don't exist.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        }

        std::unique_ptr<char[]> jsonBuffer(new char[JsonBufferSize]);
        size_t jsonSize = es::debug::BuildELicenseArchiveJsonString(jsonBuffer.get(), JsonBufferSize, archiveInfo, eLicenseInfos, rightsIdCount);
        NN_ABORT_UNLESS(jsonSize > 0);

        DEVMENUCOMMAND_LOG("eLicense json: %u\n", jsonSize);
        DEVMENUCOMMAND_LOG("%s\n", jsonBuffer.get());

        NN_RESULT_DO(es::ImportELicenseArchive(context, jsonBuffer.get(), jsonSize));

        es::ELicenseArchiveId archiveId;
        NN_RESULT_DO(es::EndImportELicenseArchiveForDebug(&archiveId, context));

        char archiveIdString[33];
        DEVMENUCOMMAND_LOG("eLicense created: ArchiveID: %s\n", archiveId.ToString(archiveIdString, sizeof(archiveIdString)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }
}

Result DeleteAllArchives(bool* outValue, const Option& option) NN_NOEXCEPT
{
    NN_UNUSED(option);

    // 有効な eLicense の情報と保存されている eLicense アーカイブを削除して、eLicense 配下でない状態にする
    es::DeleteAllELicenseArchiveForDebug();

    // 削除自体は再起動なしで行われるが、想定外の状態にならないように再起動を促しておく
    DEVMENUCOMMAND_LOG("Please reboot system.\n");

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result ElicenseCommand(bool* outValue, const Option& option)
{
#if defined(NN_DEVMENU) || defined(NN_DEVMENUSYSTEM)
    NN_UNUSED(g_HelpMessage);
#endif

    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(g_HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help" || option.HasKey("--help"))
    {
        DEVMENUCOMMAND_LOG(g_HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    account::InitializeForAdministrator();
    NN_RESULT_DO(time::InitializeForMenu());

    const SubCommand SubCommands[] =
    {
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "list", List },
        { "import", Import },
        { "export", Export },
        { "create", Create },
        { "delete-all-archives", DeleteAllArchives },
#endif
    };

    for (const auto& subCommand : SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a " DEVMENUCOMMAND_NAME " elicense command. See '" DEVMENUCOMMAND_NAME " elicense --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}

