﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/time/time_CalendarAdditionalInfo.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_PosixTime.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ncm/ncm_ApplyDeltaTaskBase.h>
#include <nn/ncm/ncm_StorageId.h>
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Option.h"

static const char* SdCardMountName = "sdcard";

namespace devmenu
{
    const size_t LaunchParameterSizeMax = 16 * 1024;
}

namespace devmenuUtil
{

bool IsAbsolutePath(const char* path) NN_NOEXCEPT;
bool IsInsertedSdCardPath(const char* path) NN_NOEXCEPT;
bool IsNspFile(const char* path) NN_NOEXCEPT;

bool IsProgramInstalled(const nn::ncm::ProgramId& programId, nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInSystem) NN_NOEXCEPT;

struct ErrorFileInfo
{
    char path[nn::fs::EntryNameLengthMax + 1];
    nn::Result result;
};

struct Bit64String
{
    char data[20];
};

struct Bit32String
{
    char data[12];
};

struct CalendarTimeString
{
    char data[32];
};

nn::util::optional<nn::ncm::ContentMetaType> ParseDownloadableContentMetaType(const char* type) NN_NOEXCEPT;
Bit64String GetBit64String(nn::Bit64 bit) NN_NOEXCEPT;
Bit32String GetBit32String(nn::Bit32 bit) NN_NOEXCEPT;
CalendarTimeString GetCalendarTimeString(const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT;
CalendarTimeString GetCalendarTimeString(const nn::time::PosixTime& time, nn::time::CalendarAdditionalInfo* info = nullptr) NN_NOEXCEPT;

const char* GetDateTimeString(char* out, int maxOutLen, const nn::time::PosixTime& posixTime) NN_NOEXCEPT;

const char* GetContentMetaTypeString(nn::ncm::ContentMetaType type) NN_NOEXCEPT;
const char* GetContentTypeString(nn::ncm::ContentType type) NN_NOEXCEPT;
const char* GetStorageIdString(nn::ncm::StorageId storage) NN_NOEXCEPT;

nn::Result WaitInstallTaskExecute(nn::ncm::InstallTaskBase* task, nn::TimeSpan detachSdCardAfter = 0) NN_NOEXCEPT;
nn::Result WaitInstallTaskExecute(nn::ncm::ApplyDeltaTaskBase* task) NN_NOEXCEPT;

nn::ncm::ApplicationId GetApplicationIdTarget(const Option& option) NN_NOEXCEPT;
nn::es::RightsIdIncludingKeyId GetRightsIdTarget(const Option& option) NN_NOEXCEPT;
nn::util::optional<bool> GetBooleanOptionValue(const Option& option, const char* key) NN_NOEXCEPT;

nn::Result ReadFile(size_t* outSize, const char* path, void* outBuffer, size_t bufferSize) NN_NOEXCEPT;
nn::Result WriteFile(const char* path, void* buffer, size_t bufferSize, bool force) NN_NOEXCEPT;

nn::Result WalkDirectory(const char* path, std::function<nn::Result(const char*, const nn::fs::DirectoryEntry&)> callback) NN_NOEXCEPT;
nn::Result Copy(std::vector<std::unique_ptr<ErrorFileInfo>>* pOutErrorFileInfoList, const char* destination, const char* source, bool force, bool skipsErrorFile) NN_NOEXCEPT;

nn::Result CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT;

nn::ApplicationId GetSelfApplicationId() NN_NOEXCEPT;

}
