﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/bcat.h>
#include <nn/bcat/bcat_ApiAdmin.h>
#include <nn/bcat/bcat_ApiDebug.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/nifm.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/msgpack.h>

#include <nn/idle/idle_SystemApi.h>


// #include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
// #include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_BcatCommand.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    bool ConvertHexId(uint64_t* outId, const char* idString) NN_NOEXCEPT
    {
        char* end = nullptr;
        *outId = static_cast<uint64_t>(strtoull(idString, &end, 16));

        if (*end)
        {
            return false;
        }

        return true;
    }

    void FormatNetworkTime(char* buffer, size_t size, nn::time::PosixTime networkTime) NN_NOEXCEPT
    {
        int length = 0;

        if (networkTime.value > 0)
        {
            nn::time::CalendarTime calendar = {};
            nn::time::CalendarAdditionalInfo additionalInfo = {};

            if (nn::time::ToCalendarTime(&calendar, &additionalInfo, networkTime).IsSuccess())
            {
                length = nn::util::SNPrintf(buffer, size, "%04d-%02d-%02d %02d:%02d:%02d (%s)",
                    calendar.year, calendar.month, calendar.day, calendar.hour, calendar.minute, calendar.second,
                    additionalInfo.timeZone.standardTimeName);
            }
            else
            {
                length = nn::util::SNPrintf(buffer, size, "---------- --:--:-- (---)");
            }
        }
        else
        {
            length = nn::util::SNPrintf(buffer, size, "---------- --:--:-- (---)");
        }

        NN_ABORT_UNLESS_LESS(static_cast<size_t>(length), size);
    }

    const char* GetTaskStatus(nn::bcat::TaskStatus status) NN_NOEXCEPT
    {
        switch (status)
        {
        case nn::bcat::TaskStatus_Runnable:
            return "Runnable";
        case nn::bcat::TaskStatus_Running:
            return "Running";
        case nn::bcat::TaskStatus_Wait:
            return "Wait";
        case nn::bcat::TaskStatus_Done:
            return "Done";
        case nn::bcat::TaskStatus_Error:
            return "Error";
        default:
            return "-";
        }
    }

    const char* GetResultDescription(nn::Result result) NN_NOEXCEPT
    {
        if (nn::bcat::ResultCanceled::Includes(result))
        {
            return " (Canceled)";
        }
        if (nn::bcat::ResultServiceUnavailable::Includes(result))
        {
            if (nn::bcat::ResultNintendoAccountNotLinked::Includes(result))
            {
                return " (ServiceUnavailable:NintendoAccountNotLinked)";
            }
            if (nn::bcat::ResultApplicationUpdateRequired::Includes(result))
            {
                return " (ServiceUnavailable:ApplicationUpdateRequired)";
            }
            return " (ServiceUnavailable)";
        }
        if (nn::bcat::ResultServiceExpired::Includes(result))
        {
            return " (ServiceExpired)";
        }
        if (nn::bcat::ResultHttpError::Includes(result))
        {
            if (nn::bcat::ResultHttpErrorCouldntResolveProxy::Includes(result))
            {
                return " (HttpError:CouldntResolveProxy)";
            }
            if (nn::bcat::ResultHttpErrorCouldntResolveHost::Includes(result))
            {
                return " (HttpError:CouldntResolveHost)";
            }
            if (nn::bcat::ResultHttpErrorCouldntConnect::Includes(result))
            {
                return " (HttpError:CouldntConnect)";
            }
            if (nn::bcat::ResultHttpErrorOperationTimedout::Includes(result))
            {
                return " (HttpError:OperationTimedout)";
            }
            if (nn::bcat::ResultHttpErrorAbortedByCallback::Includes(result))
            {
                return " (HttpError:AbortedByCallback)";
            }
            return " (HttpError)";
        }
        if (nn::bcat::ResultServerError::Includes(result))
        {
            if (nn::bcat::ResultServerError404::Includes(result))
            {
                return " (ServerError:404)";
            }
            if (nn::bcat::ResultServerError503::Includes(result))
            {
                return " (ServerError:503)";
            }
            if (nn::bcat::ResultServerError504::Includes(result))
            {
                return " (ServerError:504)";
            }
            return " (ServerError)";
        }
        if (nn::bcat::ResultInternetRequestNotAccepted::Includes(result))
        {
            return " (InternetRequestNotAccepted)";
        }
        if (nn::bcat::ResultStorageNotFound::Includes(result))
        {
            return " (StorageNotFound)";
        }
        if (nn::bcat::ResultUsableSpaceNotEnough::Includes(result))
        {
            return " (UsableSpaceNotEnough)";
        }

        return "";
    }

    void DumpProgress(const nn::bcat::DeliveryCacheProgress& progress) NN_NOEXCEPT
    {
        switch (progress.GetStatus())
        {
        case nn::bcat::DeliveryCacheProgressStatus_Queued:
            NN_LOG("Progress: status = queued\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Connect:
            NN_LOG("Progress: status = connect\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
            NN_LOG("Progress: status = process list\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Download:
            NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
                progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Commit:
            NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value,
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Done:
            if (progress.GetResult().IsSuccess())
            {
                NN_LOG("Progress: status = done, whole = (%lld/%lld)\n",
                    progress.GetWholeDownloaded(), progress.GetWholeTotal());
            }
            else
            {
                NN_LOG("Progress: status = done, whole = (%lld/%lld), code = 2%03d-%04d%s\n",
                    progress.GetWholeDownloaded(), progress.GetWholeTotal(),
                    progress.GetResult().GetModule(), progress.GetResult().GetDescription(),
                    GetResultDescription(progress.GetResult()));
            }
            break;
        default:
            break;
        }
    }

    void DumpTaskInfo(const nn::bcat::TaskInfo& info) NN_NOEXCEPT
    {
        NN_LOG("----------------------------------------------------------------------------------------------------\n");
        NN_LOG("TaskInfo\n");
        NN_LOG("----------------------------------------------------------------------------------------------------\n");
        NN_LOG("ApplicationID:\n");
        NN_LOG("    0x%016llx\n", info.appId.value);
        NN_LOG("ApplicationVersion:\n");
        NN_LOG("    %u\n", info.appVersion);
        NN_LOG("Status:\n");
        NN_LOG("    %s\n", GetTaskStatus(info.status));

        char lastRunTime[32] = {};
        FormatNetworkTime(lastRunTime, sizeof (lastRunTime), info.lastRunTime);

        NN_LOG("LastRunTime:\n");
        NN_LOG("    %s\n", lastRunTime);

        if (info.lastError != 0)
        {
            nn::Result result;
            std::memcpy(&result, &info.lastError, sizeof (result));

            NN_LOG("LastError:\n");
            NN_LOG("    2%03d-%04d%s\n", result.GetModule(), result.GetDescription(), GetResultDescription(result));
        }

        if (info.status == nn::bcat::TaskStatus_Wait)
        {
            char nextRunnableTime[32] = {};
            FormatNetworkTime(nextRunnableTime, sizeof (nextRunnableTime), info.nextRunnableTime);

            NN_LOG("NextRunnableTime:\n");
            NN_LOG("    %s\n", nextRunnableTime);
        }

        char registrationTime[32] = {};
        FormatNetworkTime(registrationTime, sizeof (registrationTime), info.registrationTime);

        NN_LOG("RegistrationTime:\n");
        NN_LOG("    %s\n", registrationTime);

        NN_LOG("Diagnostics:\n");

        if (info.subscription == nn::bcat::SubscriptionStatus_Subscribed)
        {
            NN_LOG("    The background sync is available.\n");
        }
        else
        {
            NN_LOG("    The background sync is unavailable, because push notification is not yet subscribed.\n");

            if (info.subscription == nn::bcat::SubscriptionStatus_Empty)
            {
                NN_LOG("    Please wait with the Internet connection established...\n");
            }
            else
            {
                NN_LOG("    Please reboot system.\n");
            }
        }

        NN_LOG("----------------------------------------------------------------------------------------------------\n");
    }

#if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)

    void DumpMsgpack(const void* buffer, size_t size) NN_NOEXCEPT
    {
        nne::nlib::MemoryInputStream stream;

        stream.Init(buffer, size);

        NN_UTIL_SCOPE_EXIT
        {
            stream.Close();
        };

        nne::nlib::msgpack::JsonStreamParserSettings parserSettings;

        parserSettings.format = nne::nlib::msgpack::JsonStreamParserSettings::kFormatMsgpack;
        parserSettings.max_depth = 12;
        parserSettings.token_buffer_size = 2048;

        nne::nlib::msgpack::JsonStreamParser parser;

        parser.Init(parserSettings);
        parser.Open(&stream);

        NN_UTIL_SCOPE_EXIT
        {
            parser.Close();
        };

        NN_LOG("----------------------------------------------------------------------------------------------------\n");

        struct Node
        {
            nne::nlib::msgpack::JsonStreamParser::Event event;
            int elementCount;
        };

        static Node s_Nodes[100] = {}; // parserSettings.max_depth の倍以上推奨
        int depth = 0;

        while (parser.HasNext())
        {
            nne::nlib::msgpack::JsonStreamParser::Event event = parser.Next();

            bool push = false;
            bool popIfKey = false;

            switch (event)
            {
            case nne::nlib::msgpack::JsonStreamParser::kEventNull:
                {
                    NN_LOG("null");
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventTrue:
                {
                    NN_LOG("true");
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventFalse:
                {
                    NN_LOG("false");
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventString:
                {
                    NN_LOG("\"%s\"", parser.GetToken().buf);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberInt32:
                {
                    int32_t value;
                    nne::nlib::msgpack::JsonStreamParser::ToInt32(parser.GetToken(), &value);

                    NN_LOG("%d", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberInt64:
                {
                    int64_t value;
                    nne::nlib::msgpack::JsonStreamParser::ToInt64(parser.GetToken(), &value);

                    NN_LOG("%lld", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberUint32:
                {
                    uint32_t value;
                    nne::nlib::msgpack::JsonStreamParser::ToUint32(parser.GetToken(), &value);

                    NN_LOG("%u", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberUint64:
                {
                    uint64_t value;
                    nne::nlib::msgpack::JsonStreamParser::ToUint64(parser.GetToken(), &value);

                    NN_LOG("%llu", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberFloat:
                {
                    float value;
                    nne::nlib::msgpack::JsonStreamParser::ToFloat(parser.GetToken(), &value);

                    NN_LOG("%f", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventNumberDouble:
                {
                    double value;
                    nne::nlib::msgpack::JsonStreamParser::ToDouble(parser.GetToken(), &value);

                    NN_LOG("%f", value);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventStartArray:
                {
                    NN_LOG("%s[", s_Nodes[depth].elementCount > 0 ? "," : "");
                    push = true;

                    s_Nodes[depth].elementCount++;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventEndArray:
                {
                    NN_LOG("]");
                    popIfKey = true;

                    depth--;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventStartMap:
                {
                    NN_LOG("%s{", s_Nodes[depth].elementCount > 0 ? "," : "");
                    push = true;

                    s_Nodes[depth].elementCount++;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventEndMap:
                {
                    NN_LOG("}");
                    popIfKey = true;

                    depth--;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventKeyName:
                {
                    NN_LOG("%s\"%s\":", s_Nodes[depth].elementCount > 0 ? "," : "", parser.GetToken().buf);
                    push = true;

                    s_Nodes[depth].elementCount++;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventBinary:
                {
                    NN_LOG("\"<BINARY(size=%zu)>\"", parser.GetToken().number.size);
                    popIfKey = true;
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventExt:
                {
                    NN_LOG("\"<EXTENSION(type=%d,size=%zu)>\"", parser.GetToken().ext, parser.GetToken().number.size);
                    popIfKey = true;
                }
                break;
            default:
                break;
            }

            if (push)
            {
                depth++;
                s_Nodes[depth].elementCount = 0;
                s_Nodes[depth].event = event;
            }
            else if (popIfKey && s_Nodes[depth].event == nne::nlib::msgpack::JsonStreamParser::kEventKeyName)
            {
                depth--;
            }
        }

        NN_LOG("\n----------------------------------------------------------------------------------------------------\n");
    } // NOLINT(impl/function_size)

#endif

    Result SyncDeliveryCache(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::nifm::Initialize();

        NN_LOG("SubmitNetworkRequestAndWait ...\n");

        nn::nifm::SubmitNetworkRequestAndWait();

        NN_LOG("SubmitNetworkRequestAndWait done!\n");

        if (!nn::nifm::IsNetworkAvailable())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        uint32_t appVersion = 0;

        // バックグラウンドタスクが存在する場合、タスク情報からアプリケーションバージョンを取得する
        {
            static nn::bcat::TaskInfo s_Infos[nn::bcat::TaskCountMax] = {};
            int count = 0;

            NN_RESULT_DO(nn::bcat::EnumerateBackgroundDeliveryTask(&count, s_Infos, NN_ARRAY_SIZE(s_Infos)));

            for (int i = 0; i < count; i++)
            {
                if (appId == s_Infos[i].appId)
                {
                    appVersion = s_Infos[i].appVersion;
                    break;
                }
            }
        }

        nn::bcat::DeliveryCacheProgress progress;

        NN_RESULT_TRY(nn::bcat::RequestSyncDeliveryCache(&progress, appId, appVersion))
            NN_RESULT_CATCH(nn::bcat::ResultStorageNotFound)
            {
                NN_LOG("The delivery cache storage is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();
            DumpProgress(progress);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
            nn::idle::ReportUserIsActive();
#endif
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)

    Result DumpDeliveryList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        static nn::Bit8 s_Buffer[2 * 1024 * 1024];
        size_t size = 0;

        NN_RESULT_TRY(nn::bcat::GetDeliveryList(&size, s_Buffer, sizeof (s_Buffer), appId))
            NN_RESULT_CATCH(nn::bcat::ResultNotFound)
            {
                NN_LOG("The delivery list is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::bcat::ResultStorageNotFound)
            {
                NN_LOG("The delivery cache storage is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::bcat::ResultLocked)
            {
                NN_LOG("The delivery cache storage is locked.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_LOG("Delivery list size = %zu\n", size);

        DumpMsgpack(s_Buffer, size);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#endif

    Result DumpDeliveryCacheStorage(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_TRY(nn::bcat::MountDeliveryCacheStorage(appId))
            NN_RESULT_CATCH(nn::bcat::ResultStorageNotFound)
            {
                NN_LOG("The delivery cache storage is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::bcat::ResultLocked)
            {
                NN_LOG("The delivery cache storage is locked.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::bcat::ResultNintendoAccountNotLinked)
            {
                NN_LOG("The delivery service is unavailable. (Nintendo Account is not linked)\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::bcat::UnmountDeliveryCacheStorage();
        };

        static nn::bcat::DirectoryName dirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
        int dirCount = 0;

        NN_RESULT_DO(nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, dirNames, NN_ARRAY_SIZE(dirNames)));

        NN_LOG("----------------------------------------------------------------------------------------------------\n");

        for (int d = 0; d < dirCount; d++)
        {
            nn::bcat::DeliveryCacheDirectory directory;

            NN_RESULT_DO(directory.Open(dirNames[d]));

            NN_LOG("- %s\n", dirNames[d]);

            static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
            int entryCount = 0;

            NN_RESULT_DO(directory.Read(&entryCount, s_Entries, NN_ARRAY_SIZE(s_Entries)));

            for (int e = 0; e < entryCount; e++)
            {
                NN_LOG("    - %32s, %016llx%016llx, %lld\n",
                    s_Entries[e].name.value, s_Entries[e].digest.value[0], s_Entries[e].digest.value[1], s_Entries[e].size);
            }
        }

        NN_LOG("----------------------------------------------------------------------------------------------------\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearDeliveryCacheStorage(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_TRY(nn::bcat::ClearDeliveryCacheStorage(appId))
            NN_RESULT_CATCH(nn::bcat::ResultStorageNotFound)
            {
                NN_LOG("The delivery cache storage is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::bcat::ResultLocked)
            {
                NN_LOG("The delivery cache storage is locked.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DeleteDeliveryCacheStorage(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_TRY(nn::bcat::DeleteDeliveryCacheStorage(appId))
            NN_RESULT_CATCH(nn::bcat::ResultStorageNotFound)
            {
                NN_LOG("The delivery cache storage is not found.\n");
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetInfo(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(nn::time::InitializeForMenu());

        static nn::bcat::TaskInfo s_Infos[nn::bcat::TaskCountMax] = {};
        int count = 0;

        NN_RESULT_DO(nn::bcat::EnumerateBackgroundDeliveryTask(&count, s_Infos, NN_ARRAY_SIZE(s_Infos)));

        int index = -1;

        for (int i = 0; i < count; i++)
        {
            if (s_Infos[i].appId == appId)
            {
                index = i;
                break;
            }
        }

        if (index >= 0)
        {
            DumpTaskInfo(s_Infos[index]);
        }
        else
        {
            NN_LOG("The background delivery task is not registered.\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result BlockDeliveryTask(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(nn::bcat::BlockDeliveryTask(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result UnblockDeliveryTask(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::ApplicationId appId = {};

        if (!ConvertHexId(&appId.value, option.GetTarget()))
        {
            NN_LOG("The application ID format is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        if (appId == nn::ApplicationId::GetInvalidId())
        {
            NN_LOG("The application ID is invalid.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(nn::bcat::UnblockDeliveryTask(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DumpPushNotificationLog(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        static nn::bcat::PushNotificationLog s_Logs[nn::bcat::PushNotificationLogCountMax] = {};
        int count = 0;

        NN_RESULT_DO(nn::bcat::GetPushNotificationLog(&count, s_Logs, NN_ARRAY_SIZE(s_Logs)));

        NN_RESULT_DO(nn::time::InitializeForMenu());

        NN_LOG("----------------------------------------------------------------------------------------------------\n");

        for (int i = 0; i < count; i++)
        {
            char receivedTime[32] = {};
            FormatNetworkTime(receivedTime, sizeof (receivedTime), s_Logs[i].receivedTime);

            NN_LOG("%s 0x%016llx\n", receivedTime, s_Logs[i].appId.value);
        }

        NN_LOG("----------------------------------------------------------------------------------------------------\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage:\n"
        "  * delivery-cache\n"
        "       " DEVMENUCOMMAND_NAME " bcat sync-delivery-cache <application_id>\n"
#if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "       " DEVMENUCOMMAND_NAME " bcat dump-delivery-list <application_id>\n"
#endif
        "       " DEVMENUCOMMAND_NAME " bcat dump-delivery-cache-storage <application_id>\n"
        // private command
        // "       " DEVMENUCOMMAND_NAME " bcat clear-delivery-cache-storage <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " bcat delete-delivery-cache-storage <application_id>\n"
        "  * application\n"
        "       " DEVMENUCOMMAND_NAME " bcat get-info <application_id>\n"
        // private command
        // "       " DEVMENUCOMMAND_NAME " bcat block-delivery-task <application_id>\n"
        // "       " DEVMENUCOMMAND_NAME " bcat unblock-delivery-task <application_id>\n"
        "  * log\n"
        "       " DEVMENUCOMMAND_NAME " bcat dump-push-notification-log\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"sync-delivery-cache",              SyncDeliveryCache},
#if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)
        {"dump-delivery-list",               DumpDeliveryList},
#endif
        {"dump-delivery-cache-storage",      DumpDeliveryCacheStorage},
        {"clear-delivery-cache-storage",     ClearDeliveryCacheStorage},
        {"delete-delivery-cache-storage",    DeleteDeliveryCacheStorage},
        {"get-info",                         GetInfo},
        {"block-delivery-task",              BlockDeliveryTask},
        {"unblock-delivery-task",            UnblockDeliveryTask},
        {"dump-push-notification-log",       DumpPushNotificationLog},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result BcatCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu bcat command. See '" DEVMENUCOMMAND_NAME " bcat --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
