﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <string>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_ResultPrivate.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_BatchCommand.h"
#include "DevMenuCommand_MakeArgv.h"


using namespace nn;

bool RunCommand(int argc, char** argv) NN_NOEXCEPT;


namespace {

    const size_t ArgvIndexMax = 128;
    char*        g_Argv[ArgvIndexMax];

    const size_t CommandLineBufferSize = 8192;
    char         g_CommandLineBuffer[CommandLineBufferSize];

    const size_t CommandListFileBufferSize = 2 * 1024 * 1024;
    char         g_CommandListFileBuffer[CommandListFileBufferSize];
    size_t       g_CommandListFileSize = 0;

    char* SearchLineEnd(const char* pBegin, const char* pEnd) NN_NOEXCEPT
    {
        const char* p = pBegin;
        for( ; p < pEnd; p++)
        {
            char c = *p;
            if (c == '\r' || c == '\n')
            {
                break;
            }
        }
        return const_cast<char*>(p);
    }

    char* SearchNextLineTop(const char* pBegin, const char* pEnd) NN_NOEXCEPT
    {
        const char* p = pBegin;
        for( ; p < pEnd; p++)
        {
            char c = *p;
            if (c != '\r' && c != '\n')
            {
                break;
            }
        }
        return const_cast<char*>(p);
    }

    char* SkipSpace(const char* pBegin, const char* pEnd) NN_NOEXCEPT
    {
        const char* p = pBegin;
        for( ; (p < pEnd) && std::isspace(*p); p++ ) {}
        return const_cast<char*>(p);
    }

    bool IsValidLine(const char* src) NN_NOEXCEPT
    {
        char c = *src;
        if (c == '#' || c == '/')
        {
            return false;
        }
        return true;
    }

    bool ExecuteCommandLines(int* pOutLine, bool isForce)
    {
        int lineNum = 1;

        char* src    = g_CommandListFileBuffer;
        char* srcEnd = g_CommandListFileBuffer + g_CommandListFileSize;

        bool isAllSuccess = true;

        while (src < srcEnd)
        {
            auto pSrcLineBegin = SkipSpace(src, srcEnd);
            auto pSrcLineEnd   = SearchLineEnd(pSrcLineBegin, srcEnd);
            *pSrcLineEnd = '\0';

            if (IsValidLine(pSrcLineBegin))
            {
                int argc = MakeArgv(g_Argv + 1, g_CommandLineBuffer, pSrcLineBegin, pSrcLineEnd - pSrcLineBegin, ArgvIndexMax - 2);
                if (argc > 0)
                {
                    g_Argv[0] = os::GetHostArgv()[0];
                    ++argc;
                }

#if 0
                // デバッグ用
                NN_LOG("Command[%d]='%s'\n", lineNum, pSrcLineBegin);
                NN_LOG("  argc = %d\n", argc);
                for (int i = 0; i<argc; ++i)
                {
                    NN_LOG("  argv[%d] = '%s'\n", i, g_Argv[i]);
                }
#else
                auto isSuccess = RunCommand(argc, g_Argv);
                if (!isSuccess)
                {
                    if ( isForce )
                    {
                        NN_LOG("Error at line %d.\n", lineNum);
                        isAllSuccess = false;
                    }
                    else
                    {
                        *pOutLine = lineNum;
                        return false;
                    }

                }
#endif
            }

            src = const_cast<char*>( SearchNextLineTop(pSrcLineEnd + 1, srcEnd) );
            ++lineNum;
        }

        *pOutLine = lineNum;
        return isAllSuccess;
    }

    Result ReadCommandListFile(const char* filePath)
    {
        if (!devmenuUtil::IsAbsolutePath(filePath))
        {
            NN_LOG("'%s' is not an absolute path.\n", filePath);
            NN_RESULT_THROW(nn::fs::ResultPathNotFound());
        }

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize64;
        NN_RESULT_DO(fs::GetFileSize(&fileSize64, file));

        auto fileSize = static_cast<size_t>(fileSize64);
        if (fileSize >= CommandListFileBufferSize)
        {
            NN_LOG("The file size exceeds the buffer size. (MAX : %d bytes)\n", CommandListFileBufferSize - 1);
            NN_RESULT_THROW(nn::fs::ResultInvalidSize());
        }
        NN_RESULT_DO(fs::ReadFile(file, 0, g_CommandListFileBuffer, fileSize));
        g_CommandListFileBuffer[fileSize] = '\0';

        g_CommandListFileSize = fileSize;
        NN_RESULT_SUCCESS;
    }


    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " batch <command-file-list> [--igonore-error]\n"
        "";

}   // namespace


Result BatchCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto secondArg = option.GetSubCommand();
    if (std::string(secondArg) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    if (!ReadCommandListFile(secondArg).IsSuccess())
    {
        NN_LOG("Cannot read '%s'.\n", secondArg);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    auto isForce = option.HasKey("--ignore-error");
    int line = 0;
    if (!ExecuteCommandLines(&line, isForce))
    {
        if ( !isForce )
        {
            NN_LOG("Error: '%s' at line %d.\n", secondArg, line);
        }
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}
