﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenuCommand_DebugCommandData.h"

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <vector>
#include <memory>
#include "command/DevMenuCommand_Abbreviations.h"

#include <array>
#include <string>
#include <nn/util/util_Span.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/ae/ae_SystemAppletControlForDebug.h>

namespace nn { namespace devmenucommand { namespace application {

ProcessResult LaunchByName(ncm::ApplicationId applicationId, int userIndex) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    nn::account::InitializeForAdministrator();
    std::array<account::Uid, account::UserCountMax> uidBuffer;
    int n;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&n, uidBuffer.data(), static_cast<int>(uidBuffer.size())));
    auto allUids = util::MakeSpan(uidBuffer.data(), n);
    if (!(userIndex < allUids.size()))
    {
        auto message = "userIndex = " + std::to_string(userIndex) + " is specified, but there are " + std::to_string(allUids.size()) + " accounts in this system";
        return MakeProcessResultFailure(message);
    }
    auto uids = allUids.subspan(userIndex, 1);

    //NN_LOG("ae::RequestLaunchApplicationToSystemAppletForDebug(0x%016llX, [%d]{uid=%016llX,%016llX})\n", applicationId.value, userIndex, static_cast<unsigned long long>(uids[0]._data[0]), static_cast<unsigned long long>(uids[0]._data[1]));
    ae::RequestLaunchApplicationToSystemAppletForDebug(applicationId, uids);
    return ProcessResultSuccess;
#else
    NN_UNUSED(applicationId);
    NN_UNUSED(userIndex);
    return MakeProcessResultFailure("not supported");
#endif
}

const std::vector<SubCommand>& GetApplicationSubCommands() NN_NOEXCEPT
{
    // 記述方法は README.md を参照
    using namespace abbreviation;
    static auto g_SubCommands = AggregateSubCommands(std::vector<std::shared_ptr<ICommandBuilder>>
    {
        Command("launch-by-menu",
            Params(
                NcmApplicationId(CmdArg(0)),
                Int(CmdOpt("--user,-u"), Min(0), Max(account::UserCountMax - 1), ArgumentString("<user_index>"))
            ),
            LaunchByName,
            Private
        ),
    });
    return g_SubCommands;
}

}}}
