﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <cctype>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_AddOnContentCommand.h"
#include "DevMenuCommand_InstallUtil.h"

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StorageId.h"
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_ApplicationEntityApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Stopwatch.h"

// Install Command
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ns/ns_Result.h>

// Uninstall Command

// List Command
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/aoc/aoc_SystemApi.h>

#include <nn/idle/idle_SystemApi.h>

#include <nn/fs/fs_GameCard.h>
#include <nn/ns/ns_GameCardApi.h>

using namespace nn;

namespace
{
    const int MaxAddOnContentCount = 2048;

    uint32_t GetAddOnContentIndex(ncm::ApplicationId appId, ncm::AddOnContentId aocId)
    {
        auto baseId = aoc::GetAddOnContentBaseId({ appId.value });
        return static_cast<uint32_t>(aocId.value - baseId);
    }

    ncm::AddOnContentId GetAddOnContentId(ncm::ApplicationId appId, uint32_t index)
    {
        auto baseId = aoc::GetAddOnContentBaseId({ appId.value });
        return{ baseId + index };
    }

    Result Install(bool* outValue, const Option& option)
    {
        std::unique_ptr<ncm::StorageContentMetaKey[]> keys(new ncm::StorageContentMetaKey[MaxAddOnContentCount]);
        int outCount;
        NN_RESULT_DO(devmenuUtil::InstallCommandCommon(keys.get(), &outCount, MaxAddOnContentCount, outValue, option, ncm::ContentMetaType::AddOnContent));

        if (option.HasKey("--dynamic"))
        {
            NN_RESULT_TRY(ns::TriggerDynamicCommitEvent())
                NN_RESULT_CATCH(ns::ResultRuntimeInstallNotAllowed)
                {
                    DEVMENUCOMMAND_LOG("The application is not configured to support runtime-install add-on content.\n");
                    DEVMENUCOMMAND_LOG("To enable runtime-install, please set RuntimeAddOnContentInstall as AllowAppend in nmeta.\n");
                    *outValue = false;
                }
            NN_RESULT_END_TRY;
        }
        NN_RESULT_SUCCESS;
    }

    nn::util::optional<uint32_t> ToIndex(const char* indexStr)
    {
        return  std::strtoul(indexStr, nullptr, 10);
    }

    Result Uninstall(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        auto index = option.HasKey("-i") ? ToIndex(option.GetValue("-i")) : util::nullopt;

        ns::DeleteRedundantApplicationEntity();

        auto idString = option.GetTarget();
        ncm::ApplicationId appId = { STR_TO_ULL(idString, NULL, 16) };


        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, *storage));

        if (index)
        {
            auto aocId = GetAddOnContentId(appId, *index);
            NN_RESULT_DO(devmenuUtil::DeleteBasedOnDatabase(ncm::ContentMetaType::AddOnContent, aocId.value, *storage));
        }
        else
        {
            NN_RESULT_DO(devmenuUtil::DeleteBasedOnDatabase(ncm::ContentMetaType::AddOnContent, appId, *storage));
        }

        NN_RESULT_DO(devmenuUtil::PushApplicationRecordBasedOnDatabase(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    //! Use in List(bool* outValue, const Option& option)
    struct ListAddonContents
    {
        Bit64               ownerId;            //! owner application identifier.
        uint16_t            index;              //! index.
        uint16_t            version;            //! release-version.
        uint16_t            requiredAppVersion; //! required-app-release-version.
        ncm::StorageId      installedStorageId; //! installed-storage identifer.

        class Collection : public std::vector<ListAddonContents>
        {
        public:
            explicit Collection(size_type initialCapacity = 8)
                : std::vector<ListAddonContents>()
            {
                std::vector<ListAddonContents>::reserve(initialCapacity);
            }

            Result Search(const ncm::StorageId storageId, const int bufferSize = 16 * 1024, const int maxKeys = 2048)
            {
                if (ncm::StorageId::Card == storageId)
                {
                    NN_RESULT_DO(ns::EnsureGameCardAccess());
                }
                else if (ncm::StorageId::SdCard == storageId)
                {
                    NN_RESULT_DO(ns::CheckSdCardMountStatus());
                }
                std::unique_ptr<ncm::ContentMetaKey[]> keys(new ncm::ContentMetaKey[maxKeys]);
                std::unique_ptr<Bit8[]> data(new Bit8[bufferSize]);

                ncm::ContentMetaDatabase db;
                NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, storageId));
                auto listcount = db.ListContentMeta(keys.get(), maxKeys,
                    nn::ncm::ContentMetaType::AddOnContent, ncm::ContentMetaDatabase::AnyApplicationId
                );

                for (int i = 0; i < listcount.listed; ++i)
                {
                    // ApplicationId の取得
                    size_t size;
                    NN_RESULT_DO(db.Get(&size, data.get(), bufferSize, keys[i]));
                    ncm::ContentMetaReader reader(data.get(), size);
                    auto appId = reader.GetApplicationId(keys[i]);
                    auto aocIndex = GetAddOnContentIndex(*appId, {keys[i].id});
                    auto aocHeader = reader.GetExtendedHeader<ncm::AddOnContentMetaExtendedHeader>();

                    // 16ビットシフトは release version への変換
                    push_back(ListAddonContents
                    {
                        appId->value,
                        static_cast<uint16_t>(aocIndex),
                        static_cast<uint16_t>(keys[i].version >> 16),
                        static_cast<uint16_t>(aocHeader->requiredApplicationVersion >> 16),
                        storageId
                    });
                }
                NN_RESULT_SUCCESS;
            }
        };
    };

    Result List(bool* outValue, const Option& option)
    {
        ListAddonContents::Collection collection;
        const auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : util::nullopt;
        if (storage)
        {
            // 従来
            NN_RESULT_DO(collection.Search(*storage));
        }
        else
        {
            // ビルトインは常時検索。
            NN_RESULT_DO(collection.Search(ncm::StorageId::BuiltInUser));

            // SdCard は未挿入状態、データベース無効時はスキップする。
            if (ns::CheckSdCardMountStatus().IsSuccess())
            {
                auto result = collection.Search(ncm::StorageId::SdCard);
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(ncm::ResultContentMetaDatabaseNotActive)
                    NN_RESULT_CATCH_ALL
                    {
                        // 想定外はログ出力して通知。但しコマンド自体は FAILURE にはしません。
                        DEVMENUCOMMAND_LOG("SdCard acccess failed, result => 0x%08x.\n", result.GetInnerValueForDebug());
                    }
                NN_RESULT_END_TRY
            }

            // GameCard は未挿入状態、データベース無効時はスキップする。
            if (fs::IsGameCardInserted())
            {
                auto result = collection.Search(ncm::StorageId::Card);
                NN_RESULT_TRY(result)
                    NN_RESULT_CATCH(ncm::ResultContentMetaDatabaseNotActive)
                    NN_RESULT_CATCH_ALL
                    {
                        // 想定外はログ出力して通知。但しコマンド自体は FAILURE にはしません。
                        DEVMENUCOMMAND_LOG("GameCard acccess failed, result => 0x%08x.\n", result.GetInnerValueForDebug());
                    }
                NN_RESULT_END_TRY
            }
        }

        const auto listedCount = static_cast<int>(collection.size());
        DEVMENUCOMMAND_LOG("%d add-on-content(s) found.\n", listedCount);
        DEVMENUCOMMAND_LOG("id                  index  release-version  required-app-release-version  installed-storage\n");
        for (int i = 0; i < listedCount; ++i)
        {
            const auto& data = collection[i];
            DEVMENUCOMMAND_LOG("0x%016llx   %4d              %3u                         %5u       %12s\n",
                data.ownerId,
                data.index,
                data.version,
                data.requiredAppVersion,
                ToStringFromStorageId(data.installedStorageId)
            );
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result Verify(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        ns::AsyncVerifyAddOnContentsResult asyncResult;
        NN_RESULT_DO(ns::RequestVerifyAddOnContentsRights(&asyncResult, appId));
        for(;;)
        {
            ns::AsyncVerifyAddOnContentsProgress progress;
            asyncResult.GetProgress(&progress);
            DEVMENUCOMMAND_LOG("Current Progress: %d / %d\n", progress.current, progress.total);
            if (asyncResult.TryWait())
            {
                auto result = asyncResult.Get();
                if (result.IsSuccess())
                {
                    DEVMENUCOMMAND_LOG("All addoncontents has rights.\n");
                    break;
                }
                else if (ns::ResultVerifyAddOnContentsRightsUnexpected::Includes(result))
                {
                    DEVMENUCOMMAND_LOG("Verification is failed.\n");
                    auto detailResult = asyncResult.GetDetailResult();
                    DEVMENUCOMMAND_LOG("DetailResult: module = %d, description = %d\n", detailResult.GetModule(), detailResult.GetDescription());
                    NN_RESULT_THROW(result);
                }
                else
                {
                    DEVMENUCOMMAND_LOG("There are addoncontents with no rights. module = %d, description = %d\n", result.GetModule(), result.GetDescription());
                    NN_RESULT_THROW(result);
                }
            }

#ifdef NN_BUILD_CONFIG_OS_HORIZON
            nn::idle::ReportUserIsActive();
#endif
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Cleanup(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        NN_RESULT_DO(ns::CleanupAddOnContentsWithNoRights(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result WithdrawCleanupRecommendation(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        NN_RESULT_DO(ns::WithdrawCleanupAddOnContentsWithNoRightsRecommendation(appId));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result CheckCount(bool* outValue, const Option& option)
    {
        auto storage = option.HasKey("-s") ? ToStorageId(option.GetValue("-s")) : ToStorageId("builtin");
        NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        if (!option.HasKey("-c"))
        {
            DEVMENUCOMMAND_LOG("-c <expectCount> should be specfied.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        auto expectCount = std::strtoul(option.GetValue("-c"), nullptr, 10);

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, *storage));

        auto listcount = db.ListContentMeta(nullptr, 0, nn::ncm::ContentMetaType::AddOnContent, ncm::ContentMetaDatabase::AnyApplicationId);
        if (static_cast<uint32_t>(listcount.total) != expectCount)
        {
            *outValue = false;
            DEVMENUCOMMAND_LOG("Expected count = %d, Actual count = %d\n", expectCount, listcount.total);
            NN_RESULT_SUCCESS;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Corrupt(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        auto appId = devmenuUtil::GetApplicationIdTarget(option);
        auto storage = option.HasKey("-s") ? ToInstallStorage(option.GetValue("-s")) : ToInstallStorage("auto");
        NN_RESULT_THROW_UNLESS(storage, ncm::ResultUnknownStorage());
        if (*storage == ncm::StorageId::SdCard)
        {
            NN_RESULT_DO(nn::ns::CheckSdCardMountStatus());
        }

        auto index = option.HasKey("-i") ? ToIndex(option.GetValue("-i")) : ToIndex("1");
        NN_ABORT_UNLESS(index);

        auto aocId = GetAddOnContentId(appId, *index);
        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            const int ListCount = 16;
            ns::ApplicationContentMetaStatus statusList[ListCount];
            int count;
            NN_RESULT_DO(ns::ListApplicationContentMetaStatus(&count, statusList, ListCount, appId, offset));
            for (int i = 0; i < count; i++)
            {
                if (statusList[i].id == aocId.value)
                {
                    ncm::ContentMetaKey key = ncm::ContentMetaKey::Make(statusList[i].id, statusList[i].version, ncm::ContentMetaType::AddOnContent);
                    NN_RESULT_DO(ns::CorruptContentForDebug(key, *storage));
                }
            }

            if (count < ListCount)
            {
                break;
            }

            offset += count;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

#endif // NN_TOOL_DEVMENUCOMMANDSYSTEM

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " addoncontent install <absolute_nsp_path> [--force] [--dynamic]\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent list\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent uninstall <application_id> [-i <index>]\n"
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        "       " DEVMENUCOMMAND_NAME " addoncontent check-count -c <expectCount> [-s <sdcard|builtin|card>\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent verify <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent cleanup <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent withdraw-cleanup-recommendation <application_id>\n"
        "       " DEVMENUCOMMAND_NAME " addoncontent corrupt <application_id> [-i <index>] [-s <sdcard|builtin>]\n"
#endif // NN_TOOL_DEVMENUCOMMANDSYSTEM
        ;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

    const SubCommand g_SubCommands[] =
    {
        { "install", Install },
        { "list", List },
        { "uninstall", Uninstall },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "check-count", CheckCount },
        { "verify", Verify },
        { "cleanup", Cleanup },
        { "withdraw-cleanup-recommendation", WithdrawCleanupRecommendation },
        { "corrupt", Corrupt },
#endif // NN_TOOL_DEVMENUCOMMANDSYSTEM
    };
}

Result AddOnContentCommand(bool* outValue, const Option& option)
{

    if (!option.HasSubCommand())
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        DEVMENUCOMMAND_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    aoc::Initialize();
    NN_UTIL_SCOPE_EXIT{ aoc::Finalize(); };

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    DEVMENUCOMMAND_LOG("'%s' is not a DevMenu addoncontent command. See '" DEVMENUCOMMAND_NAME " addoncontent --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
