﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <limits>
#include <memory>
#include <string>
#include <type_traits>
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/cal/cal.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_FormatString.h>

#if defined NN_CALWRITER_USE_MANU_INTERFACE
#include <nn/manu/manu_Api.h>
#endif

#include <cal_Settings.h>

#include "CalWriter_Crc16.h"
#include "CalWriter_FileSystem.h"
#include "CalWriter_KeyValidator.h"
#include "CalWriter_IFile.h"
#include "CalWriter_Log.h"

//!< 書式を指定して成功ログを出力します。
#define SUCCESS_F(x, ...) CALWRITER_LOG("[SUCCESS] " x "\n", __VA_ARGS__)

//!< 成功ログを出力します。
#define SUCCESS(x) SUCCESS_F("%s", x)

//!< 書式を指定してエラーログを出力します。
#define ERROR_F(x, ...) CALWRITER_LOG("[ERROR] " x "\n", __VA_ARGS__)

//!< エラーログを出力します。
#define ERROR(x) ERROR_F("%s", x)

//!< 書式を指定して検査ログを出力します。
#define VERIFY_F(x, y, ...) CALWRITER_LOG("[%s] " y "\n", (x) ? "OK" : "NG",\
                                   __VA_ARGS__)

//!< 検査ログを出力します。
#define VERIFY(x, y) VERIFY_F(x, "%s", y)

//!< コマンドを実行します。
#define COMMAND_DO(command)\
    do\
    {\
        const auto& errorCode(command);\
        if (errorCode) { return errorCode; }\
    } while (NN_STATIC_CONDITION(0))

//!< Result を返すコマンドを実行します。
#define RESULT_COMMAND_DO(command)\
    do\
    {\
        const auto& result = command;\
        if (result.IsFailure()) { return ErrorCode_Failure; }\
    } while (NN_STATIC_CONDITION(0))

namespace {

//!< メモリヒープのサイズ
const size_t MemoryHeapSize = 16 * 1024 * 1024;

//!< エラーコードを表す列挙体です。
enum ErrorCode : int
{
    ErrorCode_Success,  //!< 成功を示すエラーコード
    ErrorCode_Failure,  //!< 失敗を示すエラーコード
};

//!< リージョンコードの最大数
const int MaxRegionCodeNum = 6;

//!< リージョンコードの配列表現
const ::nn::settings::system::RegionCode RegionCodeArray[MaxRegionCodeNum] =
{
    ::nn::settings::system::RegionCode_Japan,
    ::nn::settings::system::RegionCode_China,
    ::nn::settings::system::RegionCode_Europe,
    ::nn::settings::system::RegionCode_Usa,
    ::nn::settings::system::RegionCode_Korea,
    ::nn::settings::system::RegionCode_Taiwan,
};

//!< リージョンコードの文字列表現
const char* RegionCodeString[MaxRegionCodeNum] =
{
    "Japan",
    "China",
    "Europe",
    "Usa",
    "Korea",
    "Taiwan",
};

//!< 本体モデルタイプの最大数
const int ProductModelNum = 5;

//!< 本体モデルタイプの文字列表現
const char* ProductModelString[ProductModelNum] =
{
    "Invalid",
    "Nx",
    "Copper",
    "Iowa",
    "Hoag",
};


//!< 本体モデルタイプの文字列表現


//!< コマンド定義を表す構造体です。
struct Command final
{
    //!< コマンドの名前
    const char* name;

    //!< コマンドの実体
    ErrorCode (*function)(const ::std::vector<::std::string>&);
};

//!< プログラム引数のベクタを返します。
::std::vector<::std::string> GetArgs(int argc, char** argv) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(argc, 0, ::std::numeric_limits<int>::max());
    NN_SDK_REQUIRES_NOT_NULL(argv);
    ::std::vector<::std::string> args;
    for (int i = 0; i < argc; ++i)
    {
        args.push_back(argv[i]);
    }
    return args;
}

//!< 16 進数を表す文字か否かを表す値を返します。
bool IsHex(char value) NN_NOEXCEPT
{
    return (('0' <= value && value <= '9') ||
            ('A' <= value && value <= 'F') ||
            ('a' <= value && value <= 'f'));
}

//!< 文字列が 10 進数を表す数値を返します。
int GetIntegerOfNumber(const ::std::string& value) NN_NOEXCEPT
{
    size_t i = 0;

    bool isPositive = true;
    if (value.size() > 1 && value[0] == '-')
    {
        isPositive = false;
        ++i;
    }

    int total = 0;
    while (i < value.size())
    {
        const char& c = value[i++];
        if (c < '0' && '9' < c)
        {
            break;
        }
        else
        {
            total = total * 10 + c - '0';
        }
    }

    return (isPositive ? total : -total);
}

//!< 文字列が 16 進数を表す数値を返します。
size_t GetIntegerOfHex(const ::std::string& value) NN_NOEXCEPT
{
    size_t total = 0;
    for (const char& c : value)
    {
        size_t sub = (('0' <= c && c <= '9') ? (c - '0')
                   : (('A' <= c && c <= 'F') ? (c - 'A' + 10)
                   : (('a' <= c && c <= 'f') ? (c - 'a' + 10) : 0xE05)));
        if (sub == 0xE05)
        {
            break;
        }
        total = total * 16 + sub;
    }
    return total;
}

//!< 文字列の浮動小数点を返します
float GetFloatOfNumber(const ::std::string& value) NN_NOEXCEPT
{
    float res = static_cast<float>(atof(value.c_str()));

    return res;
}

//!< ログ出力用に整形した文字列を返します。
::std::string FormatString(const ::std::string& value) NN_NOEXCEPT
{
    const char* const padding = "[--] ";
    ::std::string string(padding);
    if (value.size() == 0)
    {
        string += '\n';
    }
    else
    {
        size_t head = 0;
        while (head < value.size())
        {
            size_t tail = value.find_first_of('\n', head);
            if (tail == ::std::string::npos)
            {
                string += value.substr(head);
                string += '\n';
                break;
            }
            else
            {
                string += value.substr(head, tail - head + 1);
                string += padding;
                head = tail + 1;
            }
        }
    }
    return string;
}

//!< 印字可能な文字列を返します。
::std::string GetPrintableStringOf(const char* values, size_t count) NN_NOEXCEPT
{
    ::std::string string;
    for (size_t i = 0; i < count; ++i)
    {
        char c = values[i];
        if (c == '\0')
        {
            break;
        }
        else
        {
            if (' ' <= c && c <= '~' && c != '\\')
            {
                string += c;
            }
            else
            {
                char hex[3];
                ::nn::util::SNPrintf(hex, sizeof(hex), "%02X", c);
                string += "\\x";
                string += hex;
            }
        }
    }
    return string;
}

//!< 整数の文字列表現を返します。
::std::string GetStringOf(uint64_t value) NN_NOEXCEPT
{
    char string[32];
    ::nn::util::SNPrintf(string, sizeof(string), "%llu", value);
    return string;
}

//!< 16進数の文字列表現を返します
::std::string GetHexStringOf(uint64_t value) NN_NOEXCEPT
{
    char string[32];
    ::nn::util::SNPrintf(string, sizeof(string), "%x", value);
    return string;
}

//!< 整数タプルの文字列表現を返します。
::std::string GetStringOf(int16_t x, int16_t y, int16_t z) NN_NOEXCEPT
{
    char tuple[32];
    ::nn::util::SNPrintf(tuple, sizeof(tuple), "(%6d, %6d, %6d)", x, y, z);
    return tuple;
}

//!< 浮動小数点タプルの文字列表現を返します。
::std::string GetStringOf(float x, float y, float z) NN_NOEXCEPT
{
    char tuple[32];
    ::nn::util::SNPrintf(tuple, sizeof(tuple), "(%3.3f, %3.3f, %3.3f)", x, y, z);
    return tuple;
}

//!< オクテット列の文字列表現を返します。
::std::string GetStringOf(const ::nn::Bit8* values,
                          size_t count,
                          const ::std::string& separator) NN_NOEXCEPT
{
    ::std::string string;
    for (size_t i = 0; i < count; ++i)
    {
        char hex[3];
        ::nn::util::SNPrintf(hex, sizeof(hex), "%02X", values[i]);
        string += hex;
        if (i != count - 1)
        {
            string += separator;
            if (i % 16 == 15)
            {
                string += "\n";
            }
        }
    }
    return string;
}

//!< SHA-256 ハッシュの文字列表現を返します。
::std::string GetStringOf(const nn::cal::Sha256Hash& value) NN_NOEXCEPT
{
    return GetStringOf(value.data, sizeof(value.data), "");
}

//!< 構成識別子 1 の文字列表現を返します。
::std::string GetStringOf(const ::nn::settings::factory::ConfigurationId1& value
                          ) NN_NOEXCEPT
{
    return GetPrintableStringOf(value.string, sizeof(value.string));
}

//!< 無線 LAN に対して設定されたカントリーコードの文字列表現を返します。
::std::string GetStringOf(const ::nn::settings::factory::CountryCode* values,
                          uint32_t count) NN_NOEXCEPT
{
    ::std::string string;
    for (size_t i = 0; i < count; ++i)
    {
        string += GetPrintableStringOf(values[i].string,
                                       sizeof(values[i].string));
        if (i != count - 1)
        {
            if (i % 16 == 15)
            {
                string += ",\n";
            }
            else
            {
                string += ", ";
            }
        }
    }
    return string;
}

//!< 無線 LAN に対して設定された MAC アドレスの文字列表現を返します。
::std::string GetStringOf(const ::nn::settings::factory::MacAddress& value
                         ) NN_NOEXCEPT
{
    return GetStringOf(value.octets,
                       sizeof(value.octets) / sizeof(value.octets[0]),
                       "::");
}

//!< Bluetooth に対して設定された BD アドレスの文字列表現を返します。
::std::string GetStringOf(const ::nn::settings::factory::BdAddress& value
                         ) NN_NOEXCEPT
{
    return GetStringOf(value.octets,
                       sizeof(value.octets) / sizeof(value.octets[0]),
                       "::");
}

//!< 製品シリアル番号の文字列表現を返します。
::std::string GetStringOf(const ::nn::settings::factory::SerialNumber& value) NN_NOEXCEPT
{
    return GetPrintableStringOf(value.string, sizeof(value.string));
}

//!< ECC-P256 版のデバイス秘密鍵の文字列表現を返します。
::std::string GetStringOf(const nn::cal::EccP256DeviceKey& value) NN_NOEXCEPT
{
    return GetStringOf(value.data, sizeof(value.data), "");
}

//!< ECC-B233 版のデバイス秘密鍵の文字列表現を返します。
::std::string GetStringOf(const nn::cal::EccB233DeviceKey& value) NN_NOEXCEPT
{
    return GetStringOf(value.data, sizeof(value.data), "");
}

//!< リージョンコードの文字列表現を返します
::std::string GetStringOf(const ::nn::settings::system::RegionCode& value) NN_NOEXCEPT
{
    for (int i = 0; i < MaxRegionCodeNum; i++)
    {
        if (RegionCodeArray[i] == value)
        {
            return RegionCodeString[i];
        }
    }

    NN_ABORT("Unexpected region code %d", value);
    return RegionCodeString[0];
}

//!< 文字列のリージョンコード表現を返します
::nn::settings::system::RegionCode GetRegionCodeOf(const ::std::string& value) NN_NOEXCEPT
{
    for (int i = 0; i < MaxRegionCodeNum; i++)
    {
        if (RegionCodeString[i] == value)
        {
            return RegionCodeArray[i];
        }
    }

    NN_ABORT("Unexpected region code %s", value.c_str());
    return RegionCodeArray[0];
}

//!< 文字列の本体モデルタイプ表現を返します
::nn::settings::system::ProductModel GetProductModelOf(const ::std::string& value) NN_NOEXCEPT
{
    for (int i = 0; i < ProductModelNum; i++)
    {
        if (ProductModelString[i] == value)
        {
            return static_cast<::nn::settings::system::ProductModel>(i);
        }
    }

    NN_ABORT("Unexpected Product Model %s", value.c_str());
}

//!< 未初期化のヘッダか否かを返します。
bool IsUninitializedHeader(const nn::cal::CalibrationInfoHeader& header) NN_NOEXCEPT
{
    return header.updateCount == 0 ? true : false;
}

//!< ベリファイ可能なフォーマットか否かを返します。
bool IsVerifiableFormat(
        const nn::cal::CalibrationInfoHeader& header, uint32_t supportedVersion) NN_NOEXCEPT
{
    if (IsUninitializedHeader(header))
    {
        return true;
    }

    return header.version >= supportedVersion ? true : false;
}

//!< 生産時構成情報取得コマンドを実行します。
ErrorCode RunGetCalibrationInfoCommand(nn::cal::CalibrationInfo* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    const ::nn::Result& result = GetCalibrationInfo(outValue);
    if (result.IsSuccess())
    {
        return ErrorCode_Success;
    }
    else
    {
        ERROR_F("Failed to read the calibration file. (%03d-%04d)",
                result.GetModule(),
                result.GetDescription());
        return ErrorCode_Failure;
    }
}

//!< ファイルオープンコマンドを実行します。
template<typename T>
ErrorCode RunOpenFileCommand(::nn::fs::FileHandle* outValue,
    const ::std::string& path,
    int mode,
    T& file) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    ::nn::Result result = file.Open(outValue, path.c_str(), mode);
    if (result.IsFailure())
    {
        ERROR_F("Failed to open the file: %s (%03d-%04d)",
            path.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< ファイルオープンコマンドを実行します。
ErrorCode RunOpenFileCommand(std::shared_ptr<IFile>* outValue,
    const ::std::string& path,
    int mode,
    HostFsFile& file) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    ::nn::Result result = file.Open(outValue, path.c_str(), mode);
    if (result.IsFailure())
    {
        ERROR_F("Failed to open the file: %s (%03d-%04d)",
            path.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< ファイルサイズ取得コマンドを実行します。
ErrorCode RunGetFileSizeCommand(int64_t* outValue,
                                ::nn::fs::FileHandle handle,
                                const ::std::string& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    ::nn::Result result = ::nn::fs::GetFileSize(outValue, handle);
    if (result.IsFailure())
    {
        ERROR_F("Failed to get the size of the file: %s (%03d-%04d)",
                path.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< ファイルサイズ取得コマンドを実行します。
ErrorCode RunGetFileSizeCommand(int64_t* outValue,
    IFile *pFile,
    const ::std::string& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    ::nn::Result result = pFile->GetSize(outValue);
    if (result.IsFailure())
    {
        ERROR_F("Failed to get the size of the file: %s (%03d-%04d)",
            path.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< ファイル読み込みコマンドを実行します。
ErrorCode RunReadFileCommand(size_t* outSize,
    char* buffer,
    size_t bufferSize,
    IFile *pFile,
    const ::std::string& path,
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    auto fileSize = int64_t();
    COMMAND_DO(RunGetFileSizeCommand(&fileSize, pFile, path));
    if (fileSize == 0)
    {
        *outSize = 0;
    }
    else
    {
        ::nn::Result result = pFile->Read(outSize, offset, buffer, bufferSize);
        if (result.IsFailure())
        {
            ERROR_F("Failed to read the file: %s (%03d-%04d)",
                path.c_str(), result.GetModule(), result.GetDescription());
            return ErrorCode_Failure;
        }
    }
    return ErrorCode_Success;
}

//!< ファイル書き込みコマンドを実行します。
ErrorCode RunWriteFileCommand(  char* buffer,
                                size_t bufferSize,
                                IFile *pFile,
                                const ::std::string& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    {
        ::nn::Result result = pFile->Write(0, buffer, bufferSize, true);
        if (result.IsFailure())
        {
            ERROR_F("Failed to write the file: %s (%03d-%04d)",
                    path.c_str(), result.GetModule(), result.GetDescription());
            return ErrorCode_Failure;
        }
    }

    return ErrorCode_Success;
}

//!< ファイルサイズチェックコマンドを実行します。
ErrorCode RunCheckFileSizeCommand(IFile *pFile,
    const ::std::string& path,
    size_t expectedSize) NN_NOEXCEPT
{
    auto fileSize = int64_t();
    COMMAND_DO(RunGetFileSizeCommand(&fileSize, pFile, path));
    if (expectedSize != static_cast<size_t>(fileSize))
    {
        ERROR_F("Expected size is %u byte(s) but %lld byte(s) is the actual "
            "size of the file: %s", expectedSize, fileSize, path.c_str());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< 生産時較正情報パーティション (FAT) のマウントコマンドを実行します。
ErrorCode RunMountFatPartitionCommand(FatPartition& value) NN_NOEXCEPT
{
    ::nn::Result result = value.Mount();
    if (result.IsFailure())
    {
        ERROR_F("Failed to mount the fat partition. (%03d-%04d)",
                result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    return ErrorCode_Success;
}

//!< プログラム引数の個数検査コマンドを実行します。
ErrorCode RunCheckArgumentsCountCommand(
    const ::std::vector<::std::string>& args, size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(args.size(),
                          2u,
                          ::std::numeric_limits<size_t>::max());
    if (args.size() >= count)
    {
        return ErrorCode_Success;
    }
    else
    {
        ERROR_F("Invalid arguments count: %d", args.size() - 2);
        return ErrorCode_Failure;
    }
}

//!< 数値検査コマンドを実行します。
ErrorCode RunCheckNumberCommand(const ::std::string& value) NN_NOEXCEPT
{
    size_t i = 0;

    while (NN_STATIC_CONDITION(true))
    {
        if (value.size() > 1 && value[0] == '-')
        {
            ++i;
        }

        while (i < value.size() && '0' <= value[i] && value[i] <= '9')
        {
            ++i;
        }

        if (i < value.size())
        {
            break;
        }

        return ErrorCode_Success;
    }

    ERROR_F("Invalid number: %s", value.c_str());
    return ErrorCode_Failure;
}

//!< 16 進数の数値検査コマンドを実行します。
ErrorCode RunCheckHexCommand(const ::std::string& value) NN_NOEXCEPT
{
    size_t i = 0;

    while (NN_STATIC_CONDITION(true))
    {
        while (i < value.size() && IsHex(value[i]))
        {
            ++i;
        }

        if (i < value.size())
        {
            break;
        }

        return ErrorCode_Success;
    }

    ERROR_F("Invalid hexadecimal number: %s", value.c_str());
    return ErrorCode_Failure;
}

//!< オクテット検査コマンドを実行します。
ErrorCode RunCheckOctetCommand(const ::std::string& value) NN_NOEXCEPT
{
    if (value.size() == 2 && IsHex(value[0]) && IsHex(value[1]))
    {
        return ErrorCode_Success;
    }
    else
    {
        ERROR_F("Invalid octet: %s", value.c_str());
        return ErrorCode_Failure;
    }
}

//!< リージョンコード検査コマンドを実行します
ErrorCode RunCheckRegionCodeCommand(const ::std::string& value) NN_NOEXCEPT
{
    for (int i = 0; i < MaxRegionCodeNum; i++)
    {
        if (RegionCodeString[i] == value)
        {
            return ErrorCode_Success;
        }
    }

    ERROR_F("Invalid region code: %s", value.c_str());
    return ErrorCode_Failure;
}

//!< 本体モデルタイプ検査コマンドを実行します
ErrorCode RunCheckProductModelCommand(const ::std::string& value) NN_NOEXCEPT
{
    for (int i = 0; i < ProductModelNum; i++)
    {
        if (ProductModelString[i] == value)
        {
            return ErrorCode_Success;
        }
    }

    ERROR_F("Invalid product model: %s", value.c_str());
    return ErrorCode_Failure;
}

//!< ファイル走査コマンドを実行します。
ErrorCode RunReadFileCommand(const ::std::string& path) NN_NOEXCEPT
{
    FatPartitionFile file;
    auto handle = ::nn::fs::FileHandle();
    COMMAND_DO(
        RunOpenFileCommand(&handle, path, ::nn::fs::OpenMode_Read, file));
    auto fileSize = int64_t();
    COMMAND_DO(RunGetFileSizeCommand(&fileSize, handle, path));
    const size_t bufferSize = 16 * 1024;
    ::std::unique_ptr<char[]> buffer(new char[bufferSize]);
    char* const ptr = buffer.get();
    int64_t bias = 0;
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    while (0 < fileSize)
    {
        size_t size = ::std::min(bufferSize, static_cast<size_t>(fileSize));
        ::nn::Result result =
              ::nn::fs::ReadFile(&size, handle, bias, ptr, size);
        if (result.IsFailure())
        {
            ERROR_F("Failed to read the file: %s (%03d-%04d)",
                    path.c_str(), result.GetModule(), result.GetDescription());
            return ErrorCode_Failure;
        }
        fileSize -= static_cast<int64_t>(size);
        bias += static_cast<int64_t>(size);
        generator.Update(ptr, size);
    }
    ::std::unique_ptr<nn::cal::Sha256Hash> hash(new nn::cal::Sha256Hash());
    generator.GetHash(hash.get(), sizeof(nn::cal::Sha256Hash));
    CALWRITER_LOG("%s %s\n", GetStringOf(*hash).substr(0, 11).c_str(), path.c_str());
    return ErrorCode_Success;
}

//!< ディレクトリ走査コマンドを実行します。
ErrorCode RunReadDirectoryCommand(const ::std::string& path) NN_NOEXCEPT
{
    FatPartitionDirectory dir;
    auto handle = ::nn::fs::DirectoryHandle();
    ::nn::Result result =
          dir.Open(&handle, path.c_str(), ::nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        ERROR_F("Failed to open the directory: %s (%03d-%04d)",
                path.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }
    else
    {
        CALWRITER_LOG("%s %s\n", ::std::string(11, '*').c_str(), path.c_str());

        while (NN_STATIC_CONDITION(true))
        {
            auto n = int64_t();

            ::std::unique_ptr<::nn::fs::DirectoryEntry
                              > entry(new ::nn::fs::DirectoryEntry());

            result = ::nn::fs::ReadDirectory(&n, entry.get(), handle, 1);

            if (result.IsFailure())
            {
                ERROR_F("Failed to read the directory: %s (%03d-%04d)",
                        path.c_str(),
                        result.GetModule(), result.GetDescription());
                return ErrorCode_Failure;
            }

            if (n == 0)
            {
                break;
            }

            ::std::string entryPath = path + entry->name;

            auto entryType = static_cast<
                ::nn::fs::DirectoryEntryType>(entry->directoryEntryType);

            entry.release();

            switch (entryType)
            {
            case ::nn::fs::DirectoryEntryType_Directory:
                entryPath += "/";
                COMMAND_DO(RunReadDirectoryCommand(entryPath));
                break;

            case ::nn::fs::DirectoryEntryType_File:
                COMMAND_DO(RunReadFileCommand(entryPath));
                break;

            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        return ErrorCode_Success;
    }
}

//!< ZeroInitialize コマンドを実行します。
ErrorCode RunZeroInitializeCommand(const ::std::vector<::std::string>& args
                                   ) NN_NOEXCEPT
{
    NN_UNUSED(args);
    RESULT_COMMAND_DO(nn::cal::ZeroInitialize());
    return ErrorCode_Success;
}

//!< FinishUp コマンドを実行します。
ErrorCode RunFinishUpCommand(const ::std::vector<::std::string>& args
                             ) NN_NOEXCEPT
{
    NN_UNUSED(args);
    RESULT_COMMAND_DO(nn::cal::FinishUp());
    return ErrorCode_Success;
}

//!< 生産時較正情報のヘッダーをログ出力します。
bool PrintCalibrationInfoHeader(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& header = value.header;
    sub = (header.magicNumber == nn::cal::CalibrationInfoMagicNumber);
    all &= sub;
    VERIFY_F(sub, "MagicNumber : %08X", header.magicNumber);
    sub = (header.version != 0);
    all &= sub;
    VERIFY_F(sub, "Version     : %d", header.version);
    sub = (header.bodySize == sizeof(value.body));
    all &= sub;
    VERIFY_F(sub, "BodySize    : %d", header.bodySize);
    sub = (header.model == nn::cal::CalibrationInfoModel);
    all &= sub;
    VERIFY_F(sub, "Model       : %d", header.model);
    sub = true;
    all &= sub;
    VERIFY_F(sub, "UpdateCount : %d", header.updateCount);
    sub = (GetCrc16(
               &header,
               sizeof(header) - sizeof(header.crc16) - sizeof(header.bodyHash)
               ) == header.crc16);
    all &= sub;
    VERIFY_F(sub, "CRC-16      : %04X", header.crc16);
    ::std::unique_ptr<char[]> bodyHash(new char[sizeof(header.bodyHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(&value.body, sizeof(value.body));
    generator.GetHash(bodyHash.get(), sizeof(header.bodyHash));
    sub = (::std::memcmp(&header.bodyHash,
                         bodyHash.get(),
                         sizeof(header.bodyHash)) == 0);
    all &= sub;
    VERIFY(sub, "BodyHash    :");
    CALWRITER_LOG(FormatString(GetStringOf(header.bodyHash)).c_str());
    return all;
}

//!< 無線 LAN に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoWirelessLan(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.wirelessLanCountryCodesBlock,
                    sizeof(body.wirelessLanCountryCodesBlock) -
                    sizeof(body.wirelessLanCountryCodesBlock.crc16)) ==
           body.wirelessLanCountryCodesBlock.crc16);
    all &= sub;
    VERIFY(sub, "WirelessLanCountryCodes :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.wirelessLanCountryCodesBlock.countryCodes,
                        body.wirelessLanCountryCodesBlock.count)).c_str());
    sub = (GetCrc16(&body.wirelessLanMacAddressBlock.macAddress,
                    sizeof(body.wirelessLanMacAddressBlock.macAddress)) ==
           body.wirelessLanMacAddressBlock.crc16);
    all &= sub;
    VERIFY(sub, "WirelessLanMacAddress :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.wirelessLanMacAddressBlock.macAddress)).c_str());
    return all;
}

//!< 加速度センサに関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoAccelerometer(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.accelerometerOffsetBlock.accelerometerOffset,
                    sizeof(body.accelerometerOffsetBlock.accelerometerOffset)
                    ) == body.accelerometerOffsetBlock.crc16);
    all &= sub;
    VERIFY(sub, "AccelerometerOffset :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.accelerometerOffsetBlock.accelerometerOffset.x,
                body.accelerometerOffsetBlock.accelerometerOffset.y,
                body.accelerometerOffsetBlock.accelerometerOffset.z)).c_str());
    sub = (GetCrc16(&body.accelerometerScaleBlock.accelerometerScale,
                    sizeof(body.accelerometerScaleBlock.accelerometerScale)
                    ) == body.accelerometerScaleBlock.crc16);
    all &= sub;
    VERIFY(sub, "AccelerometerScale :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.accelerometerScaleBlock.accelerometerScale.x,
                body.accelerometerScaleBlock.accelerometerScale.y,
                body.accelerometerScaleBlock.accelerometerScale.z)).c_str());
    return all;
}

//!< ジャイロスコープに関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoGyroscope(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.gyroscopeOffsetBlock.gyroscopeOffset,
                    sizeof(body.gyroscopeOffsetBlock.gyroscopeOffset)
                    ) == body.gyroscopeOffsetBlock.crc16);
    all &= sub;
    VERIFY(sub, "GyroscopeOffset :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.gyroscopeOffsetBlock.gyroscopeOffset.x,
                body.gyroscopeOffsetBlock.gyroscopeOffset.y,
                body.gyroscopeOffsetBlock.gyroscopeOffset.z)).c_str());
    sub = (GetCrc16(&body.gyroscopeScaleBlock.gyroscopeScale,
                    sizeof(body.gyroscopeScaleBlock.gyroscopeScale)
                    ) == body.gyroscopeScaleBlock.crc16);
    all &= sub;
    VERIFY(sub, "GyroscopeScale :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.gyroscopeScaleBlock.gyroscopeScale.x,
                body.gyroscopeScaleBlock.gyroscopeScale.y,
                body.gyroscopeScaleBlock.gyroscopeScale.z)).c_str());
    return all;
}

//!< デバイス登録に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoDeviceRegister(const nn::cal::CalibrationInfo& value
                                        ) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.eccP256DeviceKeyBlock,
                    sizeof(body.eccP256DeviceKeyBlock) -
                    sizeof(body.eccP256DeviceKeyBlock.crc16)) ==
           body.eccP256DeviceKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccP256DeviceKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.eccP256DeviceKeyBlock.deviceKey)).c_str());
    sub = (GetCrc16(&body.eccP256DeviceCertificateBlock,
                    sizeof(body.eccP256DeviceCertificateBlock) -
                    sizeof(body.eccP256DeviceCertificateBlock.crc16)) ==
           body.eccP256DeviceCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccP256DeviceCertificate :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.eccP256DeviceCertificateBlock.deviceCertificate.data,
                sizeof(
                    body.eccP256DeviceCertificateBlock.deviceCertificate.data),
                "")).c_str());
    sub = (GetCrc16(&body.eccB233DeviceKeyBlock,
                    sizeof(body.eccB233DeviceKeyBlock) -
                    sizeof(body.eccB233DeviceKeyBlock.crc16)) ==
           body.eccB233DeviceKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccB233DeviceKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.eccB233DeviceKeyBlock.deviceKey)).c_str());
    sub = (GetCrc16(&body.eccB233DeviceCertificateBlock,
                    sizeof(body.eccB233DeviceCertificateBlock) -
                    sizeof(body.eccB233DeviceCertificateBlock.crc16)) ==
           body.eccB233DeviceCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccB233DeviceCertificate :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.eccB233DeviceCertificateBlock.deviceCertificate.data,
                sizeof(
                    body.eccB233DeviceCertificateBlock.deviceCertificate.data),
                "")).c_str());
    if (IsVerifiableFormat(value.header, 8))
    {
        sub = (GetCrc16(&body.rsa2048DeviceCertificateBlock,
                        sizeof(body.rsa2048DeviceCertificateBlock) -
                        sizeof(body.rsa2048DeviceCertificateBlock.crc16)) ==
               body.rsa2048DeviceCertificateBlock.crc16);
        all &= sub;
        VERIFY(sub, "Rsa2048DeviceCertificate :");
        CALWRITER_LOG(
            FormatString(
                GetStringOf(
                    body.rsa2048DeviceCertificateBlock.rsa2048DeviceCertificate.data,
                    sizeof(
                        body.rsa2048DeviceCertificateBlock.rsa2048DeviceCertificate),
                    "")).c_str());
    }
    return all;
}

//!< eTicket 発行に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoETicket(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.eccP256ETicketKeyBlock,
                    sizeof(body.eccP256ETicketKeyBlock) -
                    sizeof(body.eccP256ETicketKeyBlock.crc16)) ==
           body.eccP256ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccP256ETicketKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.eccP256ETicketKeyBlock.deviceKey)).c_str());
    sub = (GetCrc16(&body.eccP256ETicketCertificateBlock,
                    sizeof(body.eccP256ETicketCertificateBlock) -
                    sizeof(body.eccP256ETicketCertificateBlock.crc16)) ==
           body.eccP256ETicketCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccP256ETicketCertificate :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.eccP256ETicketCertificateBlock.deviceCertificate.data,
                sizeof(
                    body.eccP256ETicketCertificateBlock.deviceCertificate.data),
                "")).c_str());
    sub = (GetCrc16(&body.eccB233ETicketKeyBlock,
                    sizeof(body.eccB233ETicketKeyBlock) -
                    sizeof(body.eccB233ETicketKeyBlock.crc16)) ==
           body.eccB233ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccB233ETicketKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.eccB233ETicketKeyBlock.deviceKey)).c_str());
    sub = (GetCrc16(&body.eccB233ETicketCertificateBlock,
                    sizeof(body.eccB233ETicketCertificateBlock) -
                    sizeof(body.eccB233ETicketCertificateBlock.crc16)) ==
           body.eccB233ETicketCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "EccB233ETicketCertificate :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.eccB233ETicketCertificateBlock.deviceCertificate.data,
                sizeof(
                    body.eccB233ETicketCertificateBlock.deviceCertificate.data),
                "")).c_str());

    sub = (GetCrc16(&body.rsa2048ETicketKeyBlock,
        sizeof(body.rsa2048ETicketKeyBlock) -
        sizeof(body.rsa2048ETicketKeyBlock.crc16)) ==
        body.rsa2048ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "Rsa2048ETicketKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.rsa2048ETicketKeyBlock.rsa2048ETicketKey.data,
                sizeof(
                    body.rsa2048ETicketKeyBlock.rsa2048ETicketKey),
                "")).c_str());


    sub = (GetCrc16(&body.rsa2048ETicketCertificateBlock,
        sizeof(body.rsa2048ETicketCertificateBlock) -
        sizeof(body.rsa2048ETicketCertificateBlock.crc16)) ==
        body.rsa2048ETicketCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "Rsa2048ETicketCertificate :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.rsa2048ETicketCertificateBlock.rsa2048ETicketCertificate.data,
                sizeof(
                    body.rsa2048ETicketCertificateBlock.rsa2048ETicketCertificate),
                "")).c_str());

    return all;
}

//!< SSL クライアント認証に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoSsl(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.sslKeyBlock,
                    sizeof(body.sslKeyBlock) - sizeof(body.sslKeyBlock.crc16)
                    ) == body.sslKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "SslKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.sslKeyBlock.sslKey.data,
                                    sizeof(body.sslKeyBlock.sslKey.data),
                                    "")).c_str());
    sub = (GetCrc16(&body.sslCertificateSizeBlock,
                    sizeof(body.sslCertificateSizeBlock) -
                    sizeof(body.sslCertificateSizeBlock.crc16)) ==
           body.sslCertificateSizeBlock.crc16);
    all &= sub;
    VERIFY(sub, "SslCertificateSize :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.sslCertificateSizeBlock.sslCertificateSize)).c_str());
    auto size = static_cast<size_t>(
        body.sslCertificateSizeBlock.sslCertificateSize);
    const auto maxSize = sizeof(body.sslCertificateBlock.sslCertificate);
    size = size < maxSize ? size : maxSize;
    ::std::unique_ptr<char[]> sslCertificateHash(
        new char[sizeof(body.sslCertificateBlock.sslCertificateHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(body.sslCertificateBlock.sslCertificate.data, size);
    generator.GetHash(sslCertificateHash.get(),
                      sizeof(body.sslCertificateBlock.sslCertificateHash));
    sub = (::std::memcmp(
               &body.sslCertificateBlock.sslCertificateHash,
               sslCertificateHash.get(),
               sizeof(body.sslCertificateBlock.sslCertificateHash)) == 0);
    all &= sub;
    VERIFY(sub, "SslCertificateHash :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.sslCertificateBlock.sslCertificateHash)).c_str());
    return all;
}

//!< 乱数に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoRandomNumber(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    ::std::unique_ptr<char[]> randomNumberHash(
        new char[sizeof(body.randomNumberBlock.randomNumberHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(body.randomNumberBlock.randomNumber.data,
                     sizeof(body.randomNumberBlock.randomNumber));
    generator.GetHash(randomNumberHash.get(),
                      sizeof(body.randomNumberBlock.randomNumberHash));
    sub = (::std::memcmp(
               &body.randomNumberBlock.randomNumberHash,
               randomNumberHash.get(),
               sizeof(body.randomNumberBlock.randomNumberHash)) == 0);
    all &= sub;
    VERIFY(sub, "RandomNumberHash :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.randomNumberBlock.randomNumberHash)).c_str());
    return all;
}

//!< ゲームカード相互認証に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoGameCard(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.gameCardKeyBlock,
                    sizeof(body.gameCardKeyBlock) -
                    sizeof(body.gameCardKeyBlock.crc16)
                    ) == body.gameCardKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "GameCardKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.gameCardKeyBlock.gameCardKey.data,
                                    sizeof(body.gameCardKeyBlock.gameCardKey),
                                    "")).c_str());
    ::std::unique_ptr<char[]> gameCardCertificateHash(
        new char[
            sizeof(body.gameCardCertificateBlock.gameCardCertificateHash)]);
    ::nn::crypto::Sha256Generator generator;
    generator.Initialize();
    generator.Update(body.gameCardCertificateBlock.gameCardCertificate.data,
                     sizeof(body.gameCardCertificateBlock.gameCardCertificate));
    generator.GetHash(gameCardCertificateHash.get(),
                      sizeof(body.gameCardCertificateBlock
                                 .gameCardCertificateHash));
    sub = (::std::memcmp(
               &body.gameCardCertificateBlock.gameCardCertificateHash,
               gameCardCertificateHash.get(),
               sizeof(body.gameCardCertificateBlock.gameCardCertificateHash)
               ) == 0);
    all &= sub;
    VERIFY(sub, "GameCardCertificateHash :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.gameCardCertificateBlock
                            .gameCardCertificateHash)).c_str());
    return all;
}

//!< 電池 LOT に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoBatteryLot(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.batteryLotBlock,
        sizeof(body.batteryLotBlock) -
        sizeof(body.batteryLotBlock.crc16)
        ) == body.batteryLotBlock.crc16);
    all &= sub;
    VERIFY(sub, "BatteryLot :");
    CALWRITER_LOG(FormatString(GetStringOf(
        reinterpret_cast<const ::nn::Bit8*>(body.batteryLotBlock.batteryLot.string),
        sizeof(body.batteryLotBlock.batteryLot),
        "")).c_str());

    return all;
}

//!< スピーカー補正に関する生産時較正情報をログ出力します。
bool PrintCalibrationInfoSpeaker(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.speakerCalibrationValueBlock,
        sizeof(body.speakerCalibrationValueBlock) -
        sizeof(body.speakerCalibrationValueBlock.crc16)
        ) == body.speakerCalibrationValueBlock.crc16);
    all &= sub;
    VERIFY(sub, "SpeakerCalibrationValue :");
    CALWRITER_LOG(FormatString(GetStringOf(body.speakerCalibrationValueBlock.speakerCalibrationValue.data,
        sizeof(body.speakerCalibrationValueBlock.speakerCalibrationValue),
        "")).c_str());

    return all;
}

//!< リージョンコードに関する生産時較正情報をログ出力します
bool PrintCalibrationInfoRegionCode(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.regionCodeBlock,
        sizeof(body.regionCodeBlock) -
        sizeof(body.regionCodeBlock.crc16)
        ) == body.regionCodeBlock.crc16);
    all &= sub;
    VERIFY(sub, "RegionCode :");
    CALWRITER_LOG(FormatString(GetStringOf(
        static_cast<::nn::settings::system::RegionCode>(body.regionCodeBlock.regionCode))).c_str());

    return all;
}

//!< amiibo に関する生産時較正情報をログ出力します
bool PrintCalibrationInfoAmiibo(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.amiiboKeyBlock,
        sizeof(body.amiiboKeyBlock) - sizeof(body.amiiboKeyBlock.crc16)
        ) == body.amiiboKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboKeyBlock.amiiboKey.data,
        sizeof(body.amiiboKeyBlock.amiiboKey),
        "")).c_str());

    sub = (GetCrc16(&body.amiiboEcqvCertificateBlock,
        sizeof(body.amiiboEcqvCertificateBlock) - sizeof(body.amiiboEcqvCertificateBlock.crc16)
        ) == body.amiiboEcqvCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboEcqvCertificate :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboEcqvCertificateBlock.amiiboEcqvCertificate.data,
        sizeof(body.amiiboEcqvCertificateBlock.amiiboEcqvCertificate),
        "")).c_str());

    sub = (GetCrc16(&body.amiiboEcdsaCertificateBlock,
        sizeof(body.amiiboEcdsaCertificateBlock) - sizeof(body.amiiboEcdsaCertificateBlock.crc16)
        ) == body.amiiboEcdsaCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboEcdsaCertificate :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboEcdsaCertificateBlock.amiiboEcdsaCertificate.data,
        sizeof(body.amiiboEcdsaCertificateBlock.amiiboEcdsaCertificate),
        "")).c_str());

    sub = (GetCrc16(&body.amiiboEcqvBlsKeyBlock,
        sizeof(body.amiiboEcqvBlsKeyBlock) - sizeof(body.amiiboEcqvBlsKeyBlock.crc16)
        ) == body.amiiboEcqvBlsKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboEcqvBlsKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboEcqvBlsKeyBlock.amiiboEcqvBlsKey.data,
        sizeof(body.amiiboEcqvBlsKeyBlock.amiiboEcqvBlsKey),
        "")).c_str());

    sub = (GetCrc16(&body.amiiboEcqvBlsCertificateBlock,
        sizeof(body.amiiboEcqvBlsCertificateBlock) - sizeof(body.amiiboEcqvBlsCertificateBlock.crc16)
        ) == body.amiiboEcqvBlsCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboEcqvBlsCertificate :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboEcqvBlsCertificateBlock.amiiboEcqvBlsCertificate.data,
        sizeof(body.amiiboEcqvBlsCertificateBlock.amiiboEcqvBlsCertificate),
        "")).c_str());

    sub = (GetCrc16(&body.amiiboEcqvBlsRootCertificateBlock,
        sizeof(body.amiiboEcqvBlsRootCertificateBlock) - sizeof(body.amiiboEcqvBlsRootCertificateBlock.crc16)
        ) == body.amiiboEcqvBlsRootCertificateBlock.crc16);
    all &= sub;
    VERIFY(sub, "AmiiboEcqvBlsRootCertificate :");
    CALWRITER_LOG(FormatString(GetStringOf(body.amiiboEcqvBlsRootCertificateBlock.amiiboEcqvBlsRootCertificate.data,
        sizeof(body.amiiboEcqvBlsRootCertificateBlock.amiiboEcqvBlsRootCertificate),
        "")).c_str());

    return all;
}

//!< 本体モデルタイプに関する生産時較正情報をログ出力します
bool PrintCalibrationInfoProductModel(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.productModelBlock,
        sizeof(body.productModelBlock) - sizeof(body.productModelBlock.crc16)
        ) == body.productModelBlock.crc16);
    all &= sub;
    VERIFY(sub, "ProductModel :");
    int32_t productModel = body.productModelBlock.productModel.value;
    if (productModel < 0 || productModel >= ProductModelNum)
    {
        productModel = ::nn::settings::system::ProductModel::ProductModel_Invalid;
    }

    CALWRITER_LOG(FormatString(ProductModelString[productModel]).c_str());

    return all;
}

//!< 本体カラーバリエーションに関する生産時較正情報をログ出力します
bool PrintCalibrationInfoColorVariation(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.colorVariationBlock,
        sizeof(body.colorVariationBlock) - sizeof(body.colorVariationBlock.crc16)
        ) == body.colorVariationBlock.crc16);
    all &= sub;
    VERIFY(sub, "ColorVariation :");
    CALWRITER_LOG(FormatString(GetStringOf(body.colorVariationBlock.colorVariation.data,
        sizeof(body.colorVariationBlock.colorVariation),
        "")).c_str());

    return all;
}

//!< LCDバックライト出力の写像の係数に関する生産時較正情報をログ出力します
bool PrintCalibrationInfoLcdBacklightBrightnessMapping(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.lcdBacklightBrightnessMappingBlock,
        sizeof(body.lcdBacklightBrightnessMappingBlock) - sizeof(body.lcdBacklightBrightnessMappingBlock.crc16)
        ) == body.lcdBacklightBrightnessMappingBlock.crc16);
    all &= sub;
    VERIFY(sub, "LcdBacklightBrightnessMapping :");
    CALWRITER_LOG(FormatString(
        GetStringOf(body.lcdBacklightBrightnessMappingBlock.lcdBacklightBrightnessMapping.coefficient[0],
                    body.lcdBacklightBrightnessMappingBlock.lcdBacklightBrightnessMapping.coefficient[1],
                    body.lcdBacklightBrightnessMappingBlock.lcdBacklightBrightnessMapping.coefficient[2]
        )).c_str());

    return all;
}

//!< 拡張された秘密鍵に関する生産時較正情報をログ出力します
bool PrintCalibrationInfoExtendedKey(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;

    sub = (GetCrc16(&body.extendedEccB233DeviceKeyBlock,
        sizeof(body.extendedEccB233DeviceKeyBlock) -
        sizeof(body.extendedEccB233DeviceKeyBlock.crc16)) ==
        body.extendedEccB233DeviceKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedEccB233DeviceKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.extendedEccB233DeviceKeyBlock.extendedDeviceKey.data,
        sizeof(body.extendedEccB233DeviceKeyBlock.extendedDeviceKey),
        "")).c_str());

    sub = (GetCrc16(&body.extendedEccP256ETicketKeyBlock,
        sizeof(body.extendedEccP256ETicketKeyBlock) -
        sizeof(body.extendedEccP256ETicketKeyBlock.crc16)) ==
        body.extendedEccP256ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedEccP256ETicketKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.extendedEccP256ETicketKeyBlock.extendedDeviceKey.data,
        sizeof(body.extendedEccP256ETicketKeyBlock.extendedDeviceKey),
        "")).c_str());
    sub = (GetCrc16(&body.extendedEccB233ETicketKeyBlock,
        sizeof(body.extendedEccB233ETicketKeyBlock) -
        sizeof(body.extendedEccB233ETicketKeyBlock.crc16)) ==
        body.extendedEccB233ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedEccB233ETicketKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.extendedEccB233ETicketKeyBlock.extendedDeviceKey.data,
        sizeof(body.extendedEccB233ETicketKeyBlock.extendedDeviceKey),
        "")).c_str());
    sub = (GetCrc16(&body.extendedRsa2048ETicketKeyBlock,
        sizeof(body.extendedRsa2048ETicketKeyBlock) -
        sizeof(body.extendedRsa2048ETicketKeyBlock.crc16)) ==
        body.extendedRsa2048ETicketKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedRsa2048ETicketKey :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(
                body.extendedRsa2048ETicketKeyBlock.extendedRsa2048ETicketKey.data,
                sizeof(
                    body.extendedRsa2048ETicketKeyBlock.extendedRsa2048ETicketKey),
                "")).c_str());
    sub = (GetCrc16(&body.extendedSslKeyBlock,
        sizeof(body.extendedSslKeyBlock) - sizeof(body.extendedSslKeyBlock.crc16)
        ) == body.extendedSslKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedSslKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.extendedSslKeyBlock.extendedSslKey.data,
        sizeof(body.extendedSslKeyBlock.extendedSslKey),
        "")).c_str());
    sub = (GetCrc16(&body.extendedGameCardKeyBlock,
        sizeof(body.extendedGameCardKeyBlock) -
        sizeof(body.extendedGameCardKeyBlock.crc16)
        ) == body.extendedGameCardKeyBlock.crc16);
    all &= sub;
    VERIFY(sub, "ExtendedGameCardKey :");
    CALWRITER_LOG(FormatString(GetStringOf(body.extendedGameCardKeyBlock.extendedGameCardKey.data,
        sizeof(body.extendedGameCardKeyBlock.extendedGameCardKey),
        "")).c_str());

    if (IsVerifiableFormat(value.header, 8))
    {
        sub = (GetCrc16(&body.extendedRsa2048DeviceKeyBlock,
            sizeof(body.extendedRsa2048DeviceKeyBlock) -
            sizeof(body.extendedRsa2048DeviceKeyBlock.crc16)) ==
            body.extendedRsa2048DeviceKeyBlock.crc16);
        all &= sub;
        VERIFY(sub, "ExtendedRsa2048DeviceKey :");
        CALWRITER_LOG(
            FormatString(
                GetStringOf(
                    body.extendedRsa2048DeviceKeyBlock.extendedDeviceKey.data,
                    sizeof(
                        body.extendedRsa2048DeviceKeyBlock.extendedDeviceKey),
                    "")).c_str());
    }
    return all;
}

//!< LCD のベンダ ID に関する生産時較正情報をログ出力します
bool PrintCalibrationInfoLcdVendorId(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    bool sub = true;
    const auto& body = value.body;
    sub = (GetCrc16(&body.lcdVendorIdBlock,
        sizeof(body.lcdVendorIdBlock) - sizeof(body.lcdVendorIdBlock.crc16)
        ) == body.lcdVendorIdBlock.crc16);
    all &= sub;
    VERIFY(sub, "LcdVendorId :");
    CALWRITER_LOG(FormatString(GetHexStringOf(value.body.lcdVendorIdBlock.lcdVendorId.value)).c_str());

    return all;
}

//!< USB Type-C Power Source 回路バージョンに関する生産時較正情報のベリファイを行います。
bool PrintCalibrationInfoUsbTypeCPowerSourceCircuit(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;
    if (IsVerifiableFormat(value.header, 10))
    {
        bool sub = true;
        const auto& body = value.body;
        sub = (GetCrc16(&body.usbTypeCPowerSourceCircuitBlock,
            sizeof(body.usbTypeCPowerSourceCircuitBlock) - sizeof(body.usbTypeCPowerSourceCircuitBlock.crc16)
            ) == body.usbTypeCPowerSourceCircuitBlock.crc16);
        all &= sub;
        VERIFY(sub, "UsbTypeCPowerSourceCircuit :");
        CALWRITER_LOG(
                FormatString(
                    GetHexStringOf(
                        value.body.usbTypeCPowerSourceCircuitBlock.usbTypeCPowerSourceCircuit.version)).c_str());
    }
    return all;
}

//!< 本体カラーに関する生産時較正情報のベリファイを行います。
bool PrintCalibrationInfoHousingColor(const nn::cal::CalibrationInfo& value) NN_NOEXCEPT
{
    bool all = true;

    if (IsVerifiableFormat(value.header, 13))
    {
        bool sub = true;
        const auto& body = value.body;

        sub = (GetCrc16(&body.housingSubColorBlock,
            sizeof(body.housingSubColorBlock) -
            sizeof(body.housingSubColorBlock.crc16)) ==
            body.housingSubColorBlock.crc16);
        all &= sub;
        VERIFY(sub, "HousingSubColor :");
        CALWRITER_LOG(FormatString(GetStringOf(body.housingSubColorBlock.housingColor.data,
            sizeof(body.housingSubColorBlock.housingColor),
            "")).c_str());
        sub = (GetCrc16(&body.housingBezelColorBlock,
            sizeof(body.housingBezelColorBlock) -
            sizeof(body.housingBezelColorBlock.crc16)) ==
            body.housingBezelColorBlock.crc16);
        all &= sub;
        VERIFY(sub, "HousingBezelColor :");
        CALWRITER_LOG(FormatString(GetStringOf(body.housingBezelColorBlock.housingColor.data,
            sizeof(body.housingBezelColorBlock.housingColor),
            "")).c_str());
        sub = (GetCrc16(&body.housingMainColor1Block,
            sizeof(body.housingMainColor1Block) -
            sizeof(body.housingMainColor1Block.crc16)) ==
            body.housingMainColor1Block.crc16);
        all &= sub;
        VERIFY(sub, "HousingMainColor1 :");
        CALWRITER_LOG(FormatString(GetStringOf(body.housingMainColor1Block.housingColor.data,
            sizeof(body.housingMainColor1Block.housingColor),
            "")).c_str());
        sub = (GetCrc16(&body.housingMainColor2Block,
            sizeof(body.housingMainColor2Block) -
            sizeof(body.housingMainColor2Block.crc16)) ==
            body.housingMainColor2Block.crc16);
        all &= sub;
        VERIFY(sub, "HousingMainColor2 :");
        CALWRITER_LOG(FormatString(GetStringOf(body.housingMainColor2Block.housingColor.data,
            sizeof(body.housingMainColor2Block.housingColor),
            "")).c_str());
        sub = (GetCrc16(&body.housingMainColor3Block,
            sizeof(body.housingMainColor3Block) -
            sizeof(body.housingMainColor3Block.crc16)) ==
            body.housingMainColor3Block.crc16);
        all &= sub;
        VERIFY(sub, "HousingMainColor3 :");
        CALWRITER_LOG(FormatString(GetStringOf(body.housingMainColor3Block.housingColor.data,
            sizeof(body.housingMainColor3Block.housingColor),
            "")).c_str());
    }
    return all;
}

//!< Verify コマンドを実行します。
ErrorCode RunVerifyCommand(const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    NN_UNUSED(args);
    ::std::unique_ptr<nn::cal::CalibrationInfo> calibrationInfo(new nn::cal::CalibrationInfo());
    COMMAND_DO(RunGetCalibrationInfoCommand(calibrationInfo.get()));
    bool all = true;
    bool sub = true;
    CALWRITER_LOG("<<Header>>\n");
    all = PrintCalibrationInfoHeader(*calibrationInfo);
    const auto& body = calibrationInfo->body;
    CALWRITER_LOG("<<Body>>\n");
    sub = (GetCrc16(&body.configurationId1Block.configurationId1,
                    sizeof(body.configurationId1Block.configurationId1)) ==
           body.configurationId1Block.crc16);
    all &= sub;
    VERIFY(sub, "ConfigurationId1 :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.configurationId1Block.configurationId1)).c_str());
    all &= PrintCalibrationInfoWirelessLan(*calibrationInfo);
    sub = (GetCrc16(&body.bluetoothBdAddressBlock.bdAddress,
                    sizeof(body.bluetoothBdAddressBlock.bdAddress)) ==
           body.bluetoothBdAddressBlock.crc16);
    all &= sub;
    VERIFY(sub, "BluetoothBdAddress :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.bluetoothBdAddressBlock.bdAddress)).c_str());
    all &= PrintCalibrationInfoAccelerometer(*calibrationInfo);
    all &= PrintCalibrationInfoGyroscope(*calibrationInfo);
    sub = (GetCrc16(&body.serialNumberBlock,
                    sizeof(body.serialNumberBlock) -
                    sizeof(body.serialNumberBlock.crc16)) ==
           body.serialNumberBlock.crc16);
    all &= sub;
    VERIFY(sub, "SerialNumber :");
    CALWRITER_LOG(
        FormatString(
            GetStringOf(body.serialNumberBlock.serialNumber)).c_str());
    all &= PrintCalibrationInfoDeviceRegister(*calibrationInfo);
    all &= PrintCalibrationInfoETicket(*calibrationInfo);
    all &= PrintCalibrationInfoSsl(*calibrationInfo);
    all &= PrintCalibrationInfoRandomNumber(*calibrationInfo);
    all &= PrintCalibrationInfoGameCard(*calibrationInfo);
    all &= PrintCalibrationInfoBatteryLot(*calibrationInfo);
    all &= PrintCalibrationInfoSpeaker(*calibrationInfo);
    all &= PrintCalibrationInfoRegionCode(*calibrationInfo);
    all &= PrintCalibrationInfoAmiibo(*calibrationInfo);
    all &= PrintCalibrationInfoProductModel(*calibrationInfo);
    all &= PrintCalibrationInfoColorVariation(*calibrationInfo);
    all &= PrintCalibrationInfoLcdBacklightBrightnessMapping(*calibrationInfo);
    all &= PrintCalibrationInfoExtendedKey(*calibrationInfo);
    all &= PrintCalibrationInfoLcdVendorId(*calibrationInfo);
    all &= PrintCalibrationInfoUsbTypeCPowerSourceCircuit(*calibrationInfo);
    all &= PrintCalibrationInfoHousingColor(*calibrationInfo);

    RESULT_COMMAND_DO(nn::cal::Verify(all));

    if (all)
    {
        return ErrorCode_Success;
    }
    else
    {
        ERROR("Verification failed.");
        return ErrorCode_Failure;
    }
}

#if defined NN_CALWRITER_USE_MANU_INTERFACE
//!< VerifyKey コマンドを実行します。
ErrorCode RunVerifyKeyCommand(const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    NN_UNUSED(args);
    if (ValidatePrivateKey())
    {
        return ErrorCode_Success;
    }
    else
    {
        ERROR("Key Verification failed.");
        return ErrorCode_Failure;
    }
}
#endif

//!< WriteConfigurationId1 コマンドを実行します。
ErrorCode RunWriteConfigurationId1Command(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    ::nn::settings::factory::ConfigurationId1 configurationId1;
    ::std::memset(&configurationId1, 0, sizeof(configurationId1));
    if (args.size() > 2)
    {
        ::std::strncpy(reinterpret_cast<char*>(&configurationId1),
                       args[2].c_str(),
                       sizeof(configurationId1) - 1);
    }
    RESULT_COMMAND_DO(nn::cal::WriteConfigurationId1(configurationId1));
    return ErrorCode_Success;
}

//!< WriteWlanCountryCodes コマンドを実行します。
ErrorCode RunWriteWlanCountryCodesCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    size_t count = args.size() - 2;
    ::std::unique_ptr<::nn::settings::factory::CountryCode[]> countryCodes(new ::nn::settings::factory::CountryCode[count]);
    if (count > 0)
    {
        for (size_t i = 0; i < count; ++i)
        {
            const ::std::string& string = args[2 + i];
            for (size_t j = 0; j < 2 && j < string.size(); ++j)
            {
                countryCodes[i].string[j] = string[j];
            }
        }
    }
    RESULT_COMMAND_DO(nn::cal::WriteWlanCountryCodes(countryCodes.get(), count));
    return ErrorCode_Success;
}

//!< WriteWlanMacAddress コマンドを実行します。
ErrorCode RunWriteWlanMacAddressCommand(const ::std::vector<::std::string>& args
                                        ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 8));
    for (size_t i = 0; i < 6; ++i)
    {
        COMMAND_DO(RunCheckOctetCommand(args[i + 2]));
    }
    auto macAddress = ::nn::settings::factory::MacAddress();
    for (size_t i = 0; i < 6; ++i)
    {
        macAddress.octets[i] =
            static_cast<::nn::Bit8>(GetIntegerOfHex(args[i + 2]));
    }
    RESULT_COMMAND_DO(nn::cal::WriteWlanMacAddress(macAddress));
    return ErrorCode_Success;
}

//!< WriteBdAddress コマンドを実行します。
ErrorCode RunWriteBdAddressCommand(const ::std::vector<::std::string>& args
                                   ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 8));
    for (size_t i = 0; i < 6; ++i)
    {
        COMMAND_DO(RunCheckOctetCommand(args[i + 2]));
    }
    auto bdAddress = ::nn::settings::factory::BdAddress();
    for (size_t i = 0; i < 6; ++i)
    {
        bdAddress.octets[i] =
            static_cast<::nn::Bit8>(GetIntegerOfHex(args[i + 2]));
    }
    RESULT_COMMAND_DO(nn::cal::WriteBdAddress(bdAddress));
    return ErrorCode_Success;
}

//!< WriteAccelerometerOffset コマンドを実行します。
ErrorCode RunWriteAccelerometerOffsetCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 5));
    for (size_t i = 0; i < 3; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto x = static_cast<int16_t>(GetIntegerOfNumber(args[2]));
    auto y = static_cast<int16_t>(GetIntegerOfNumber(args[3]));
    auto z = static_cast<int16_t>(GetIntegerOfNumber(args[4]));
    RESULT_COMMAND_DO(nn::cal::WriteAccelerometerOffset(x,y,z));
    return ErrorCode_Success;
}

//!< WriteAccelerometerScale コマンドを実行します。
ErrorCode RunWriteAccelerometerScaleCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 5));
    for (size_t i = 0; i < 3; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto x = static_cast<int16_t>(GetIntegerOfNumber(args[2]));
    auto y = static_cast<int16_t>(GetIntegerOfNumber(args[3]));
    auto z = static_cast<int16_t>(GetIntegerOfNumber(args[4]));
    RESULT_COMMAND_DO(nn::cal::WriteAccelerometerScale(x,y,z));
    return ErrorCode_Success;
}

//!< WriteGyroscopeOffset コマンドを実行します。
ErrorCode RunWriteGyroscopeOffsetCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 5));
    for (size_t i = 0; i < 3; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto x = static_cast<int16_t>(GetIntegerOfNumber(args[2]));
    auto y = static_cast<int16_t>(GetIntegerOfNumber(args[3]));
    auto z = static_cast<int16_t>(GetIntegerOfNumber(args[4]));
    RESULT_COMMAND_DO(nn::cal::WriteGyroscopeOffset(x,y,z));
    return ErrorCode_Success;
}

//!< WriteGyroscopeScale コマンドを実行します。
ErrorCode RunWriteGyroscopeScaleCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 5));
    for (size_t i = 0; i < 3; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto x = static_cast<int16_t>(GetIntegerOfNumber(args[2]));
    auto y = static_cast<int16_t>(GetIntegerOfNumber(args[3]));
    auto z = static_cast<int16_t>(GetIntegerOfNumber(args[4]));
    RESULT_COMMAND_DO(nn::cal::WriteGyroscopeScale(x,y,z));
    return ErrorCode_Success;
}

//!< WriteSerialNumber コマンドを実行します。
ErrorCode RunWriteSerialNumberCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    ::nn::settings::factory::SerialNumber serialNumber;
    ::std::memset(&serialNumber, 0, sizeof(serialNumber));
    if (args.size() > 2)
    {
        ::std::strncpy(reinterpret_cast<char*>(&serialNumber),
                       args[2].c_str(),
                       sizeof(serialNumber) - 1);
    }
    RESULT_COMMAND_DO(nn::cal::WriteSerialNumber(serialNumber));
    return ErrorCode_Success;
}

//!< WriteEccP256DeviceKey コマンドを実行します。
ErrorCode RunWriteEccP256DeviceKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::EccP256DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
                           reinterpret_cast<char*>(&deviceKey),
                           sizeof(deviceKey),
                           pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccP256DeviceKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteEccP256DeviceCertificate コマンドを実行します。
ErrorCode RunWriteEccP256DeviceCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceCertificate = nn::cal::DeviceCertificate();
    COMMAND_DO(
        RunCheckFileSizeCommand(
            pFile.get(), args[2], sizeof(deviceCertificate.data)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
                           reinterpret_cast<char*>(deviceCertificate.data),
                           sizeof(deviceCertificate.data),
                           pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccP256DeviceCertificate(deviceCertificate));
    return ErrorCode_Success;
}

//!< WriteEccB233DeviceKey コマンドを実行します。
ErrorCode RunWriteEccB233DeviceKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::EccB233DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
                           reinterpret_cast<char*>(&deviceKey),
                           sizeof(deviceKey),
                            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccB233DeviceKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteEccB233DeviceCertificate コマンドを実行します。
ErrorCode RunWriteEccB233DeviceCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceCertificate = nn::cal::DeviceCertificate();
    COMMAND_DO(
        RunCheckFileSizeCommand(
            pFile.get(), args[2], sizeof(deviceCertificate.data)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(
            &size,
            reinterpret_cast<char*>(deviceCertificate.data),
            sizeof(deviceCertificate.data),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccB233DeviceCertificate(deviceCertificate));
    return ErrorCode_Success;
}

//!< WriteEccP256ETicketKey コマンドを実行します。
ErrorCode RunWriteEccP256ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::EccP256DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
                           reinterpret_cast<char*>(&deviceKey),
                           sizeof(deviceKey),
                           pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccP256ETicketKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteEccP256ETicketCertificate コマンドを実行します。
ErrorCode RunWriteEccP256ETicketCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceCertificate = nn::cal::ETicketCertificate();
    COMMAND_DO(
        RunCheckFileSizeCommand(
            pFile.get(), args[2], sizeof(deviceCertificate.data)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(
            &size,
            reinterpret_cast<char*>(deviceCertificate.data),
            sizeof(deviceCertificate.data),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccP256ETicketCertificate(deviceCertificate));
    return ErrorCode_Success;
}

//!< WriteEccB233ETicketKey コマンドを実行します。
ErrorCode RunWriteEccB233ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::EccB233DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(&deviceKey),
            sizeof(deviceKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccB233ETicketKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteEccB233ETicketCertificate コマンドを実行します。
ErrorCode RunWriteEccB233ETicketCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceCertificate = nn::cal::ETicketCertificate();
    COMMAND_DO(
        RunCheckFileSizeCommand(
            pFile.get(), args[2], sizeof(deviceCertificate.data)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(
            &size,
            reinterpret_cast<char*>(deviceCertificate.data),
            sizeof(deviceCertificate.data),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteEccB233ETicketCertificate(deviceCertificate));
    return ErrorCode_Success;
}

//!< WriteSslKey コマンドを実行します。
ErrorCode RunWriteSslKeyCommand(const ::std::vector<::std::string>& args
                                ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto sslKey = nn::cal::SslKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(sslKey)));
    auto size = size_t();
    COMMAND_DO(RunReadFileCommand(&size,
        reinterpret_cast<char*>(&sslKey),
        sizeof(sslKey),
        pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteSslKey(sslKey));

    return ErrorCode_Success;
}

//!< WriteSslCertificate コマンドを実行します。
ErrorCode RunWriteSslCertificateCommand(const ::std::vector<::std::string>& args
                                        ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::SslCertificate> sslCertificate(new nn::cal::SslCertificate());
    ::std::memset(sslCertificate.get(), 0, sizeof(nn::cal::SslCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto fileSize = int64_t();
    COMMAND_DO(RunGetFileSizeCommand(&fileSize, pFile.get(), args[2]));
    if (static_cast<size_t>(fileSize) > sizeof(nn::cal::SslCertificate))
    {
        ERROR_F(
            "File size (%lld byte(s)) exceeded the maximum size (%u bytes).",
            fileSize, sizeof(nn::cal::SslCertificate));
        return ErrorCode_Failure;
    }
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(sslCertificate.get()),
            static_cast<size_t>(fileSize),
            pFile.get(), args[2], 0));
    RESULT_COMMAND_DO(nn::cal::WriteSslCertificate(sslCertificate.get(), size));
    return ErrorCode_Success;
}

//!< WriteRandomNumber コマンドを実行します。
ErrorCode RunWriteRandomNumberCommand(const ::std::vector<::std::string>& args
                                      ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::RandomNumber> randomNumber(new nn::cal::RandomNumber());
    ::std::memset(randomNumber.get(), 0, sizeof(nn::cal::RandomNumber));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::RandomNumber)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(randomNumber.get()),
            sizeof(nn::cal::RandomNumber),
            pFile.get(), args[2], 0));
    RESULT_COMMAND_DO(nn::cal::WriteRandomNumber(*randomNumber));
    return ErrorCode_Success;
}

//!< WriteGameCardKey コマンドを実行します。
ErrorCode RunWriteGameCardKeyCommand(const ::std::vector<::std::string>& args
                                     ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::GameCardKey> gameCardKey(new nn::cal::GameCardKey());
    ::std::memset(gameCardKey.get(), 0, sizeof(nn::cal::GameCardKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::GameCardKey)));
    auto size = size_t();
    COMMAND_DO(RunReadFileCommand(&size,
        reinterpret_cast<char*>(gameCardKey.get()),
        sizeof(nn::cal::GameCardKey),
        pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteGameCardKey(*gameCardKey));
    return ErrorCode_Success;
}

//!< WriteGameCardCertificate コマンドを実行します。
ErrorCode RunWriteGameCardCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::GameCardCertificate> gameCardCertificate(new nn::cal::GameCardCertificate());
    ::std::memset(gameCardCertificate.get(), 0, sizeof(nn::cal::GameCardCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(),
                                args[2],
                                sizeof(nn::cal::GameCardCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(gameCardCertificate.get()),
            sizeof(nn::cal::GameCardCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteGameCardCertificate(*gameCardCertificate));
    return ErrorCode_Success;
}

//!< WriteRsa2048ETicketKey コマンドを実行します。
ErrorCode RunWriteRsa2048ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::Rsa2048ETicketKey> rsa2048ETicketKey(new nn::cal::Rsa2048ETicketKey());
    ::std::memset(rsa2048ETicketKey.get(), 0, sizeof(nn::cal::Rsa2048ETicketKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::Rsa2048ETicketKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(rsa2048ETicketKey.get()),
            sizeof(nn::cal::Rsa2048ETicketKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteRsa2048ETicketKey(*rsa2048ETicketKey));
    return ErrorCode_Success;
}

//!< WriteRsa2048ETicketCertificate コマンドを実行します。
ErrorCode RunWriteRsa2048ETicketCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::Rsa2048ETicketCertificate> rsa2048ETicketCertificate(new nn::cal::Rsa2048ETicketCertificate());
    ::std::memset(rsa2048ETicketCertificate.get(), 0, sizeof(nn::cal::Rsa2048ETicketCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::Rsa2048ETicketCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(rsa2048ETicketCertificate.get()),
            sizeof(nn::cal::Rsa2048ETicketCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteRsa2048ETicketCertificate(*rsa2048ETicketCertificate));
    return ErrorCode_Success;
}

//!< WriteBatteryLot コマンドを実行します。
ErrorCode RunWriteBatteryLotCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    ::nn::settings::factory::BatteryLot batteryLot;
    ::std::memset(&batteryLot, 0, sizeof(batteryLot));
    if (args.size() > 2)
    {
        ::std::strncpy(reinterpret_cast<char*>(&batteryLot),
            args[2].c_str(),
            sizeof(batteryLot) - 1);
    }
    RESULT_COMMAND_DO(nn::cal::WriteBatteryLot(batteryLot));
    return ErrorCode_Success;
}

//!< WriteSpeakerCalibrationValue コマンドを実行します。
ErrorCode RunWriteSpeakerCalibrationValueCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::SpeakerCalibrationValue> speakerCalibrationValue(new nn::cal::SpeakerCalibrationValue());
    ::std::memset(speakerCalibrationValue.get(), 0, sizeof(nn::cal::SpeakerCalibrationValue));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::SpeakerCalibrationValue)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(speakerCalibrationValue.get()),
            sizeof(nn::cal::SpeakerCalibrationValue),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteSpeakerCalibrationValue(*speakerCalibrationValue));
    return ErrorCode_Success;
}

//!< WriteRegionCode コマンドを実行します。
ErrorCode RunWriteRegionCodeCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    COMMAND_DO(RunCheckRegionCodeCommand(args[2]));
    auto regionCode = (GetRegionCodeOf(args[2]));
    RESULT_COMMAND_DO(nn::cal::WriteRegionCode(regionCode));
    return ErrorCode_Success;
}

//!< WriteAmiiboKey コマンドを実行します。
ErrorCode RunWriteAmiiboKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboKey> amiiboKey(new nn::cal::AmiiboKey());
    ::std::memset(amiiboKey.get(), 0, sizeof(nn::cal::AmiiboKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboKey.get()),
            sizeof(nn::cal::AmiiboKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboKey(*amiiboKey));
    return ErrorCode_Success;
}

//!< WriteAmiiboEcqvCertificate コマンドを実行します。
ErrorCode RunWriteAmiiboEcqvCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboEcqvCertificate> amiiboEcqvCertificate(new nn::cal::AmiiboEcqvCertificate());
    ::std::memset(amiiboEcqvCertificate.get(), 0, sizeof(nn::cal::AmiiboEcqvCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboEcqvCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboEcqvCertificate.get()),
            sizeof(nn::cal::AmiiboEcqvCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboEcqvCertificate(*amiiboEcqvCertificate));
    return ErrorCode_Success;
}

//!< WriteAmiiboEcdsaCertificate コマンドを実行します。
ErrorCode RunWriteAmiiboEcdsaCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboEcdsaCertificate> amiiboEcdsaCertificate(new nn::cal::AmiiboEcdsaCertificate());
    ::std::memset(amiiboEcdsaCertificate.get(), 0, sizeof(nn::cal::AmiiboEcdsaCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboEcdsaCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboEcdsaCertificate.get()),
            sizeof(nn::cal::AmiiboEcdsaCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboEcdsaCertificate(*amiiboEcdsaCertificate));
    return ErrorCode_Success;
}

//!< WriteAmiiboEcqvBlsKey コマンドを実行します。
ErrorCode RunWriteAmiiboEcqvBlsKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboEcqvBlsKey> amiiboEcqvBlsKey(new nn::cal::AmiiboEcqvBlsKey());
    ::std::memset(amiiboEcqvBlsKey.get(), 0, sizeof(nn::cal::AmiiboEcqvBlsKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboEcqvBlsKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboEcqvBlsKey.get()),
            sizeof(nn::cal::AmiiboEcqvBlsKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboEcqvBlsKey(*amiiboEcqvBlsKey));
    return ErrorCode_Success;
}

//!< WriteAmiiboEcqvBlsCertificate コマンドを実行します。
ErrorCode RunWriteAmiiboEcqvBlsCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboEcqvBlsCertificate > amiiboEcqvBlsCertificate (new nn::cal::AmiiboEcqvBlsCertificate ());
    ::std::memset(amiiboEcqvBlsCertificate .get(), 0, sizeof(nn::cal::AmiiboEcqvBlsCertificate ));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboEcqvBlsCertificate )));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboEcqvBlsCertificate .get()),
            sizeof(nn::cal::AmiiboEcqvBlsCertificate ),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboEcqvBlsCertificate (*amiiboEcqvBlsCertificate ));
    return ErrorCode_Success;
}

//!< WriteAmiiboEcqvBlsRootCertificate コマンドを実行します。
ErrorCode RunWriteAmiiboEcqvBlsRootCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::AmiiboEcqvBlsRootCertificate> amiiboEcqvBlsRootCertificate(new nn::cal::AmiiboEcqvBlsRootCertificate());
    ::std::memset(amiiboEcqvBlsRootCertificate.get(), 0, sizeof(nn::cal::AmiiboEcqvBlsRootCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::AmiiboEcqvBlsRootCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(amiiboEcqvBlsRootCertificate.get()),
            sizeof(nn::cal::AmiiboEcqvBlsRootCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteAmiiboEcqvBlsRootCertificate(*amiiboEcqvBlsRootCertificate));
    return ErrorCode_Success;
}

//!< WriteProductModel コマンドを実行します。
ErrorCode RunWriteProductModelCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    auto productModel = nn::cal::ProductModel();
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    COMMAND_DO(RunCheckProductModelCommand(args[2]));
    productModel.value = (GetProductModelOf(args[2]));

    RESULT_COMMAND_DO(nn::cal::WriteProductModel(productModel));
    return ErrorCode_Success;
}

//!< WriteColorVariation コマンドを実行します。
ErrorCode RunWriteColorVariationCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::ColorVariation> colorVariation(new nn::cal::ColorVariation());
    ::std::memset(colorVariation.get(), 0, sizeof(nn::cal::ColorVariation));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::ColorVariation)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(colorVariation.get()),
            sizeof(nn::cal::ColorVariation),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteColorVariation(*colorVariation));
    return ErrorCode_Success;
}

//!< WriteLcdBacklightBrightnessMappingCommand コマンドを実行します。
ErrorCode RunWriteLcdBacklightBrightnessMappingCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 5));
    ::std::unique_ptr<nn::cal::LcdBacklightBrightnessMapping> lcdBacklightBrightnessMapping(new nn::cal::LcdBacklightBrightnessMapping());
    ::std::memset(lcdBacklightBrightnessMapping.get(), 0, sizeof(nn::cal::LcdBacklightBrightnessMapping));

    lcdBacklightBrightnessMapping.get()->coefficient[0] = GetFloatOfNumber(args[2]);
    lcdBacklightBrightnessMapping.get()->coefficient[1] = GetFloatOfNumber(args[3]);
    lcdBacklightBrightnessMapping.get()->coefficient[2] = GetFloatOfNumber(args[4]);

    RESULT_COMMAND_DO(nn::cal::WriteLcdBacklightBrightnessMapping(*lcdBacklightBrightnessMapping));
    return ErrorCode_Success;
}

//!< WriteEccB233DeviceKey コマンドを実行します。
ErrorCode RunWriteExtendedEccB233DeviceKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::ExtendedEccB233DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(&deviceKey),
            sizeof(deviceKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedEccB233DeviceKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteExtendedEccP256ETicketKey コマンドを実行します。
ErrorCode RunWriteExtendedEccP256ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::ExtendedEccP256DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(&deviceKey),
            sizeof(deviceKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedEccP256ETicketKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteExtendedEccB233ETicketKey コマンドを実行します。
ErrorCode RunWriteExtendedEccB233ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto deviceKey = nn::cal::ExtendedEccB233DeviceKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(deviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(&deviceKey),
            sizeof(deviceKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedEccB233ETicketKey(deviceKey));
    return ErrorCode_Success;
}

//!< WriteExtendedRsa2048ETicketKey コマンドを実行します。
ErrorCode RunWriteExtendedRsa2048ETicketKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::ExtendedRsa2048ETicketKey> extendedRsa2048ETicketKey(new nn::cal::ExtendedRsa2048ETicketKey());
    ::std::memset(extendedRsa2048ETicketKey.get(), 0, sizeof(nn::cal::ExtendedRsa2048ETicketKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::ExtendedRsa2048ETicketKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(extendedRsa2048ETicketKey.get()),
            sizeof(nn::cal::ExtendedRsa2048ETicketKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedRsa2048ETicketKey(*extendedRsa2048ETicketKey));
    return ErrorCode_Success;
}

//!< WriteExtendedSslKey コマンドを実行します。
ErrorCode RunWriteExtendedSslKeyCommand(const ::std::vector<::std::string>& args
    ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    auto ExtendedSslKey = nn::cal::ExtendedSslKey();
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(ExtendedSslKey)));
    auto size = size_t();
    COMMAND_DO(RunReadFileCommand(&size,
        reinterpret_cast<char*>(&ExtendedSslKey),
        sizeof(ExtendedSslKey),
        pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedSslKey(ExtendedSslKey));

    return ErrorCode_Success;
}

//!< WriteExtendedGameCardKey コマンドを実行します。
ErrorCode RunWriteExtendedGameCardKeyCommand(const ::std::vector<::std::string>& args
    ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::ExtendedGameCardKey> extendedGameCardKey(new nn::cal::ExtendedGameCardKey());
    ::std::memset(extendedGameCardKey.get(), 0, sizeof(nn::cal::ExtendedGameCardKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::ExtendedGameCardKey)));
    auto size = size_t();
    COMMAND_DO(RunReadFileCommand(&size,
        reinterpret_cast<char*>(extendedGameCardKey.get()),
        sizeof(nn::cal::ExtendedGameCardKey),
        pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedGameCardKey(*extendedGameCardKey));
    return ErrorCode_Success;
}

//!< WriteLcdVendorId コマンドを実行します。
ErrorCode RunWriteLcdVendorIdCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    auto LcdVendorId = nn::cal::LcdVendorId();
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    COMMAND_DO(RunCheckHexCommand(args[2]));
    LcdVendorId.value = static_cast<uint32_t>(GetIntegerOfHex(args[2]));

    RESULT_COMMAND_DO(nn::cal::WriteLcdVendorId(LcdVendorId));
    return ErrorCode_Success;
}

//!< WriteExtendedRsa2048DeviceKey コマンドを実行します。
ErrorCode RunWriteExtendedRsa2048DeviceKeyCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::ExtendedRsa2048DeviceKey> extendedRsa2048DeviceKey(new nn::cal::ExtendedRsa2048DeviceKey());
    ::std::memset(extendedRsa2048DeviceKey.get(), 0, sizeof(nn::cal::ExtendedRsa2048DeviceKey));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::ExtendedRsa2048DeviceKey)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(extendedRsa2048DeviceKey.get()),
            sizeof(nn::cal::ExtendedRsa2048DeviceKey),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteExtendedRsa2048DeviceKey(*extendedRsa2048DeviceKey));
    return ErrorCode_Success;
}

//!< WriteRsa2048DeviceCertificate コマンドを実行します。
ErrorCode RunWriteRsa2048DeviceCertificateCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::Rsa2048DeviceCertificate> rsa2048DeviceCertificate(new nn::cal::Rsa2048DeviceCertificate());
    ::std::memset(rsa2048DeviceCertificate.get(), 0, sizeof(nn::cal::Rsa2048DeviceCertificate));
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::Rsa2048DeviceCertificate)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(rsa2048DeviceCertificate.get()),
            sizeof(nn::cal::Rsa2048DeviceCertificate),
            pFile.get(), args[2], 0));
    NN_UNUSED(size);
    RESULT_COMMAND_DO(nn::cal::WriteRsa2048DeviceCertificate(*rsa2048DeviceCertificate));
    return ErrorCode_Success;
}

//!< WriteUsbTypeCPowerSourceCircuitVersion コマンドを実行します。
ErrorCode RunWriteUsbTypeCPowerSourceCircuitVersionCommand(
    const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    auto usbTypeCPowerSourceCircuit = nn::cal::UsbTypeCPowerSourceCircuit();
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    COMMAND_DO(RunCheckHexCommand(args[2]));
    usbTypeCPowerSourceCircuit.version = static_cast<::nn::Bit8>(GetIntegerOfHex(args[2]));
    RESULT_COMMAND_DO(
            nn::cal::WriteUsbTypeCPowerSourceCircuitVersion(usbTypeCPowerSourceCircuit));
    return ErrorCode_Success;
}

//!< WriteHousingSubColor コマンドを実行します。
ErrorCode RunWriteHousingSubColorCommand(
        const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 6));
    for (size_t i = 0; i < 4; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto housingColor = nn::cal::HousingColor();
    housingColor.data[0] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[2]));
    housingColor.data[1] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[3]));
    housingColor.data[2] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[4]));
    housingColor.data[3] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[5]));
    RESULT_COMMAND_DO(nn::cal::WriteHousingSubColor(housingColor));
    return ErrorCode_Success;
}

//!< WriteHousingBezelColor コマンドを実行します。
ErrorCode RunWriteHousingBezelColorCommand(
        const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 6));
    for (size_t i = 0; i < 4; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto housingColor = nn::cal::HousingColor();
    housingColor.data[0] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[2]));
    housingColor.data[1] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[3]));
    housingColor.data[2] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[4]));
    housingColor.data[3] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[5]));
    RESULT_COMMAND_DO(nn::cal::WriteHousingBezelColor(housingColor));
    return ErrorCode_Success;
}

//!< WriteHousingMainColor1 コマンドを実行します。
ErrorCode RunWriteHousingMainColor1Command(
        const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 6));
    for (size_t i = 0; i < 4; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto housingColor = nn::cal::HousingColor();
    housingColor.data[0] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[2]));
    housingColor.data[1] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[3]));
    housingColor.data[2] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[4]));
    housingColor.data[3] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[5]));
    RESULT_COMMAND_DO(nn::cal::WriteHousingMainColor1(housingColor));
    return ErrorCode_Success;
}

//!< WriteHousingMainColor2 コマンドを実行します。
ErrorCode RunWriteHousingMainColor2Command(
        const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 6));
    for (size_t i = 0; i < 4; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto housingColor = nn::cal::HousingColor();
    housingColor.data[0] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[2]));
    housingColor.data[1] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[3]));
    housingColor.data[2] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[4]));
    housingColor.data[3] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[5]));
    RESULT_COMMAND_DO(nn::cal::WriteHousingMainColor2(housingColor));
    return ErrorCode_Success;
}

//!< WriteHousingMainColor3 コマンドを実行します。
ErrorCode RunWriteHousingMainColor3Command(
        const ::std::vector<::std::string>& args) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 6));
    for (size_t i = 0; i < 4; ++i)
    {
        COMMAND_DO(RunCheckNumberCommand(args[i + 2]));
    }
    auto housingColor = nn::cal::HousingColor();
    housingColor.data[0] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[2]));
    housingColor.data[1] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[3]));
    housingColor.data[2] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[4]));
    housingColor.data[3] = static_cast<::nn::Bit8>(GetIntegerOfNumber(args[5]));
    RESULT_COMMAND_DO(nn::cal::WriteHousingMainColor3(housingColor));
    return ErrorCode_Success;
}

//!< PutCalibrationFile コマンドを実行します。
ErrorCode RunPutCalibrationFileCommand(const ::std::vector<::std::string>& args
                                       ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));
    ::std::unique_ptr<nn::cal::RawCalibrationInfo> calibrationInfo(new nn::cal::RawCalibrationInfo());
    HostFsFile file;
    std::shared_ptr<IFile> pFile;
    COMMAND_DO(
        RunOpenFileCommand(&pFile, args[2], ::nn::fs::OpenMode_Read, file));
    COMMAND_DO(
        RunCheckFileSizeCommand(pFile.get(), args[2], sizeof(nn::cal::RawCalibrationInfo)));
    auto size = size_t();
    COMMAND_DO(
        RunReadFileCommand(&size,
            reinterpret_cast<char*>(calibrationInfo.get()),
            sizeof(nn::cal::RawCalibrationInfo),
            pFile.get(), args[2], 0));
    RESULT_COMMAND_DO(nn::cal::PutCalibrationData(*calibrationInfo));
    ::std::vector<::std::string> verifyArgs(2);
    verifyArgs[0] = args[0];
    verifyArgs[1] = "Verify";
    COMMAND_DO(RunVerifyCommand(verifyArgs));
    return ErrorCode_Success;
}

//!< PutFile コマンドを実行します。
ErrorCode RunPutFileCommand(const ::std::vector<::std::string>& args
                            ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 4));

    HostFsFile srcFile;
    std::shared_ptr<IFile> pFile;
    auto& srcPath  = args[2];
    COMMAND_DO(RunOpenFileCommand(
                &pFile, srcPath, ::nn::fs::OpenMode_Read, srcFile));
    int64_t srcFileSize = 0;
    COMMAND_DO(RunGetFileSizeCommand(&srcFileSize, pFile.get(), srcPath));
    ::std::unique_ptr<char[]> buffer(new char[static_cast<uint32_t>(srcFileSize)]);
    size_t readSize = 0;
    COMMAND_DO(RunReadFileCommand(&readSize,
        reinterpret_cast<char*>(buffer.get()),
        static_cast<size_t>(srcFileSize),
        pFile.get(), srcPath, 0));
    NN_ABORT_UNLESS(static_cast<int64_t>(readSize) <= srcFileSize);

    auto& dstPath  = args[3];
    RESULT_COMMAND_DO(nn::cal::WriteFile(buffer.get(), static_cast<size_t>(srcFileSize), dstPath));

    return ErrorCode_Success;
}

//!< GetFile コマンドを実行します。
ErrorCode RunGetFileCommand(const ::std::vector<::std::string>& args
                            ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 4));

    bool isFileExist = true;
    auto& srcPath  = args[2];
    RESULT_COMMAND_DO(nn::cal::IsFileExist(isFileExist, srcPath));
    if(!isFileExist)
    {
        ERROR_F("File not found in FAT : %s", srcPath.c_str());
        return ErrorCode_Failure;
    }

    int64_t srcFileSize = 0;
    RESULT_COMMAND_DO(nn::cal::GetFileSize(&srcFileSize, srcPath));
    ::std::unique_ptr<char[]> buffer(new char[static_cast<uint32_t>(srcFileSize)]);
    RESULT_COMMAND_DO(nn::cal::ReadFile(buffer.get(), static_cast<size_t>(srcFileSize), srcPath));

    HostFsFile dstFileSystem;
    auto& dstPath  = args[3];
    std::shared_ptr<IFile> dstFile;
    COMMAND_DO(
        RunOpenFileCommand(
            &dstFile, dstPath, ::nn::fs::OpenMode_Write, dstFileSystem));

    COMMAND_DO(RunWriteFileCommand(buffer.get(), static_cast<size_t>(srcFileSize), dstFile.get(), dstPath));

    return ErrorCode_Success;
}

//!< DeleteFile コマンドを実行します。
ErrorCode RunDeleteFileCommand(const ::std::vector<::std::string>& args
                               ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    bool isFileExist = true;
    auto& deleteFilePath  = args[2];
    RESULT_COMMAND_DO(nn::cal::IsFileExist(isFileExist, deleteFilePath));
    if(!isFileExist)
    {
        ERROR_F("File not found in FAT : %s", deleteFilePath.c_str());
        return ErrorCode_Failure;
    }

    ::nn::Result result = nn::cal::DeleteFile(deleteFilePath);
    if (result.IsFailure())
    {
        ERROR_F("Failed to delete the file: %s (%03d-%04d)",
                deleteFilePath.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }

    return ErrorCode_Success;
}

//!< MakeDirectory コマンドを実行します。
ErrorCode RunMakeDirectoryCommand(const ::std::vector<::std::string>& args
                                  ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    bool isDirectoryExist = true;
    auto& createDirectoryPath  = args[2];
    RESULT_COMMAND_DO(nn::cal::IsDirectoryExist(isDirectoryExist, createDirectoryPath));
    if(isDirectoryExist)
    {
        ERROR_F("Directory already exist in FAT : %s", createDirectoryPath.c_str());
        return ErrorCode_Failure;
    }
    ::nn::Result result = nn::cal::CreateDirectory(createDirectoryPath);
    if (result.IsFailure())
    {
        ERROR_F("Failed to create the directory: %s (%03d-%04d)",
                createDirectoryPath.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }

    return ErrorCode_Success;
}

//!< DeleteDirectory コマンドを実行します。
ErrorCode RunDeleteDirectoryCommand(const ::std::vector<::std::string>& args
                                    ) NN_NOEXCEPT
{
    COMMAND_DO(RunCheckArgumentsCountCommand(args, 3));

    bool isDirectoryExist = true;
    auto& deleteDirectoryPath  = args[2];
    RESULT_COMMAND_DO(nn::cal::IsDirectoryExist(isDirectoryExist, deleteDirectoryPath));
    if(!isDirectoryExist)
    {
        ERROR_F("Directory not found in FAT : %s", deleteDirectoryPath.c_str());
        return ErrorCode_Failure;
    }
    ::nn::Result result = nn::cal::DeleteDirectoryRecursively(deleteDirectoryPath);
    if (result.IsFailure())
    {
        ERROR_F("Failed to delete the directory: %s (%03d-%04d)",
                deleteDirectoryPath.c_str(), result.GetModule(), result.GetDescription());
        return ErrorCode_Failure;
    }

    return ErrorCode_Success;
}

//!< ListFiles コマンドを実行します。
ErrorCode RunListFilesCommand(const ::std::vector<::std::string>& args
                              ) NN_NOEXCEPT
{
    NN_UNUSED(args);
    FatPartition fatPartition;
    COMMAND_DO(RunMountFatPartitionCommand(fatPartition));
    COMMAND_DO(RunReadDirectoryCommand(GetFatDummyRootPath()));
    return ErrorCode_Success;
}

//!< コマンドの配列
const Command g_Commands[] =
{
    { "ZeroInitialize", RunZeroInitializeCommand },
    { "FinishUp", RunFinishUpCommand },
    { "Verify", RunVerifyCommand },
#if defined NN_CALWRITER_USE_MANU_INTERFACE
    { "VerifyKey", RunVerifyKeyCommand },
#endif
    { "WriteConfigurationId1", RunWriteConfigurationId1Command },
    { "WriteWlanCountryCodes", RunWriteWlanCountryCodesCommand },
    { "WriteWlanMacAddress", RunWriteWlanMacAddressCommand },
    { "WriteBdAddress", RunWriteBdAddressCommand },
    { "WriteAccelerometerOffset", RunWriteAccelerometerOffsetCommand },
    { "WriteAccelerometerScale", RunWriteAccelerometerScaleCommand },
    { "WriteGyroscopeOffset", RunWriteGyroscopeOffsetCommand },
    { "WriteGyroscopeScale", RunWriteGyroscopeScaleCommand },
    { "WriteSerialNumber", RunWriteSerialNumberCommand },
    { "WriteEccP256DeviceKey", RunWriteEccP256DeviceKeyCommand },
    {
        "WriteEccP256DeviceCertificate",
        RunWriteEccP256DeviceCertificateCommand
    },
    { "WriteEccB233DeviceKey", RunWriteEccB233DeviceKeyCommand },
    {
        "WriteEccB233DeviceCertificate",
        RunWriteEccB233DeviceCertificateCommand
    },
    { "WriteEccP256ETicketKey", RunWriteEccP256ETicketKeyCommand },
    {
        "WriteEccP256ETicketCertificate",
        RunWriteEccP256ETicketCertificateCommand
    },
    { "WriteEccB233ETicketKey", RunWriteEccB233ETicketKeyCommand },
    {
        "WriteEccB233ETicketCertificate",
        RunWriteEccB233ETicketCertificateCommand
    },
    { "WriteSslKey", RunWriteSslKeyCommand },
    { "WriteSslCertificate", RunWriteSslCertificateCommand },
    { "WriteRandomNumber", RunWriteRandomNumberCommand },
    { "WriteGameCardKey", RunWriteGameCardKeyCommand },
    { "WriteGameCardCertificate", RunWriteGameCardCertificateCommand },
    { "WriteRsa2048ETicketKey", RunWriteRsa2048ETicketKeyCommand },
    { "WriteRsa2048ETicketCertificate", RunWriteRsa2048ETicketCertificateCommand },
    { "WriteBatteryLot", RunWriteBatteryLotCommand },
    { "WriteSpeakerCalibrationValue", RunWriteSpeakerCalibrationValueCommand },
    { "WriteRegionCode", RunWriteRegionCodeCommand },
    { "WriteAmiiboKey", RunWriteAmiiboKeyCommand },
    { "WriteAmiiboEcqvCertificate", RunWriteAmiiboEcqvCertificateCommand },
    { "WriteAmiiboEcdsaCertificate", RunWriteAmiiboEcdsaCertificateCommand },
    { "WriteAmiiboEcqvBlsKey", RunWriteAmiiboEcqvBlsKeyCommand },
    { "WriteAmiiboEcqvBlsCertificate", RunWriteAmiiboEcqvBlsCertificateCommand },
    { "WriteAmiiboEcqvBlsRootCertificate", RunWriteAmiiboEcqvBlsRootCertificateCommand },
    { "WriteProductModel", RunWriteProductModelCommand },
    { "WriteColorVariation", RunWriteColorVariationCommand },
    { "WriteLcdBacklightBrightnessMapping", RunWriteLcdBacklightBrightnessMappingCommand },
    { "WriteExtendedEccB233DeviceKey", RunWriteExtendedEccB233DeviceKeyCommand },
    { "WriteExtendedEccP256ETicketKey", RunWriteExtendedEccP256ETicketKeyCommand },
    { "WriteExtendedEccB233ETicketKey", RunWriteExtendedEccB233ETicketKeyCommand },
    { "WriteExtendedRsa2048ETicketKey", RunWriteExtendedRsa2048ETicketKeyCommand },
    { "WriteExtendedSslKey", RunWriteExtendedSslKeyCommand },
    { "WriteExtendedGameCardKey", RunWriteExtendedGameCardKeyCommand },
    { "WriteLcdVendorId", RunWriteLcdVendorIdCommand },
    { "WriteExtendedRsa2048DeviceKey", RunWriteExtendedRsa2048DeviceKeyCommand},
    { "WriteRsa2048DeviceCertificate", RunWriteRsa2048DeviceCertificateCommand},
    { "WriteUsbTypeCPowerSourceCircuitVersion", RunWriteUsbTypeCPowerSourceCircuitVersionCommand },
    { "WriteHousingSubColor", RunWriteHousingSubColorCommand },
    { "WriteHousingBezelColor", RunWriteHousingBezelColorCommand },
    { "WriteHousingMainColor1", RunWriteHousingMainColor1Command },
    { "WriteHousingMainColor2", RunWriteHousingMainColor2Command },
    { "WriteHousingMainColor3", RunWriteHousingMainColor3Command },
    { "PutCalibrationFile", RunPutCalibrationFileCommand },
    { "PutFile", RunPutFileCommand },
    { "GetFile", RunGetFileCommand },
    { "DeleteFile", RunDeleteFileCommand },
    { "MakeDirectory", RunMakeDirectoryCommand },
    { "DeleteDirectory", RunDeleteDirectoryCommand },
    { "ListFiles", RunListFilesCommand },
};

} // namespace

extern "C" void nninitStartup()
{
    // メモリヒープのサイズを設定
    ::nn::os::SetMemoryHeapSize(MemoryHeapSize);

    // メモリヒープを確保
    auto address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MemoryHeapSize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    MemoryHeapSize);
}

#if defined NN_CALWRITER_USE_MANU_INTERFACE
extern "C" void nndiagStartup()
{
}
#endif

extern "C" void nnMain()
{
#if defined NN_CALWRITER_USE_MANU_INTERFACE
    ::nn::manu::InitializeUfio();
#endif

    // プログラム引数を取得
    const ::std::vector<::std::string> args = GetArgs(::nn::os::GetHostArgc(),
                                                      ::nn::os::GetHostArgv());
    if (args.size() < 2)
    {
        // コマンドが指定されなかった場合は失敗
        ERROR("No command specified.");
        return;
    }

    // ファイルシステムを初期化
    InitializeFileSystem();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    EnableDebugMode();
#endif

    const auto& name = [&]() -> ::std::string
    {
        if (args[1].find("@") != 0)
        {
            return args[1];
        }
        else
        {
            EnableDebugMode();
            return args[1].substr(1);
        }
    }();

    for (const Command& command : g_Commands)
    {
        // 配列を走査して検出されたコマンドを実行
        if (name == command.name)
        {
            if (command.function(args) == ErrorCode_Success)
            {
                SUCCESS("Done.");
            }

            return;
        }
    }

    // コマンドが見つからなかった場合は失敗
    ERROR_F("Command '%s' not found.", name.c_str());

#if defined NN_CALWRITER_USE_MANU_INTERFACE
    ::nn::manu::FinalizeUfio();
#endif

    return;
}
