﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*! @file
    @brief    SubStateの定義
*/

#pragma once

namespace nn {
namespace wlan {

/*!
 * @brief WLANプロセスの内部ステートを示す列挙体です。
 */
enum WdrvMainState
{
    /*!
        @brief 無線デバイスドライバー未初期状態です。
     */
    WdrvMainState_Stop,

    /*!
        @brief 無線デバイスドライバーの初期化が行われた状態です。
               各モードをCloseModeすると、このステートに遷移します。
               このステートから、各モードにOpenModeすることが可能です。
      */
    WdrvMainState_Ready,

    /*!
        @brief インフラストラクチャモードのアイドル状態です。
               無線アクセスポイントへの接続とスキャンが可能です。
     */
    WdrvMainState_InfraIdle,

    /*!
        @brief インフラストラクチャモードの接続が確立された状態です。
               データフレームの送受信、無線アクセスポイントからの切断、スキャンが可能です。
     */
    WdrvMainState_InfraSta,

    /*!
        @brief ローカル通信モードの親機のアイドル状態です。
               親機の開始、スキャンが可能です。
     */
    WdrvMainState_LocalMasterIdle,

    /*!
        @brief ローカル通信モードの親機でBSSを展開している状態です。
               データフレームの送受信、子機の切断が可能です。
     */
    WdrvMainState_LocalMasterBss = 5,

    /*!
        @brief ローカル通信モードの子機のアイドル状態です。
                 親機への接続、スキャンが可能です。
     */
    WdrvMainState_LocalClientIdle,

    /*!
        @brief ローカル通信モードの観戦子機のアイドル状態です。
                 親機への同期、スキャンが可能です。
     */
    WdrvMainState_LocalSpectatorIdle,

    /*!
        @brief ローカル通信モードで親機との接続が確立された子機状態です。
               データフレームの送受信、Masterからの切断、スキャンが可能です。
     */
    WdrvMainState_LocalClient,

    /*!
        @brief ローカル通信モードで親機への同期が確立された観戦子機状態です。
               Broadcastデータフレームの受信、同期解除、スキャンが可能です。
     */
    WdrvMainState_LocalSpectator,


    /*!
        @brief スリープモード。無線チップは省電力稼働状態または完全に停止状態です。
     */
    WdrvMainState_Sleep,


    //! EXITステート。プロセス終了時に遷移するステートです。
    WdrvMainState_Exit,

    /*!
        @brief wlanプロセスで無線デバイスへの処理が中止されている状態です。
               工程検査などwlanで無線処理が動いては不味い場合などに使用されます。
     */
    WdrvMainState_OutOfService,

    /*!
        @brief ローカル通信LCSモードの親機のアイドル状態です。
               親機の開始、スキャンが可能です。
     */
    WdrvMainState_LocalLcsMasterIdle,

    /*!
        @brief ローカル通信LCSモードの親機でBSSを展開している状態です。
               データフレームの送受信、子機の切断が可能です。
     */
    WdrvMainState_LocalLcsMasterBss,

    /*!
        @brief ローカル通信LCSモードの子機のアイドル状態です。
                 親機への接続、スキャンが可能です。
     */
    WdrvMainState_LocalLcsClientIdle,

    /*!
        @brief ローカル通信LCSモードで親機との接続が確立された子機状態です。
               データフレームの送受信、Masterからの切断、スキャンが可能です。
     */
    WdrvMainState_LocalLcsClient,

    /*!
        @brief すれちがい通信を行うモードです。非通信状態。
     */
    WdrvMainState_DetectIdle,

    /*!
        @brief すれちがい通信を行うモードです。通信状態。
     */
    WdrvMainState_Detect,

    WdrvMainState_NUM
};

enum WdrvSubState
{
    // I/Fがdownしている状態
    WdrvSubState_None,

    // I/Fがupしている状態
    WdrvSubState_Idle,

    // I/Fがupして、かつ接続が維持されている状態。
    // LocalApの場合はBSSを展開している状態。
    // LocalSpectatorの場合は親機と同期している状態。
    WdrvSubState_Connected,

    // I/Fがupして、かつScan中の状態
    WdrvSubState_Scanning,

    // I/Fがupして、かつ接続試行中の状態
    WdrvSubState_Connecting,

    // I/Fがupして、Scan中断中の状態
    WdrvSubState_CancelingScan,

    // I/Fがupして、かつ切断試行中の状態
    WdrvSubState_Disconnecting,

    LocalSubState_Num
}; // end of SubState

static const char* MainStateStr[] = {
        "Stop",
        "Ready",
        "InfraIdle",
        "InfraSta",
        "LocalMasterIdle",
        "LocalMasterBss",
        "LocalClientIdle",
        "LocalSpectatorIdle",
        "LocalClient",
        "LocalSpectator",
        "Exit",
        "NUM",
};

static const char* SubStateStr[] = {
        "None",
        "Idle",
        "Connected",
        "Scanning",
        "Connecting",
        "CancelingScan",
        "Disconnecting",
        "NUM",
};

} // end of namespace wlan
} // end of namespace nn

