﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/wlan/detail/wlan_InternalTypes.h>

#include "wlan_DebugLog.h"
#include "wlan_State.h"
#include "wlan_BaseFunctions.h"
#include "wlan_MsgBuffer.h"
#include "wlan_CommonType.h"
#include "wlan_RxEntries.h"
#include "wlan_WpaSupplicant.h"

namespace nn { namespace wlan {

class WlanBaseFunctions;
class WpaSupplicant;

class StateMachine
{
public:
    static const size_t EventMessageCountMax = 4 * 64;
    static const size_t EventSleepMessageCountMax = 16;  // スリープステート中に発生したイベントを溜め込んでおける最大数
    static const size_t CommandMessageCountMax = 32;
    static const int    InvalidRssiValue = -999;

    StateMachine() NN_NOEXCEPT;
    ~StateMachine() NN_NOEXCEPT;

    void ExitStateMachineRequest() NN_NOEXCEPT;

    WlanCommand* WlanGetCommandBuff(size_t arg_size) NN_NOEXCEPT;
    void WlanReleaseCommandBuff(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    WlanEvent* WlanGetEventBuff(size_t arg_size) NN_NOEXCEPT;
    void WlanReleaseEventBuff(WlanEvent* pevnbuff) NN_NOEXCEPT;

    void Initialize(int num_evn, int num_cmd,
                    nn::wlan::WlanBaseFunctions* pfuncs,
                    nn::wlan::WpaSupplicant* pSuppFuncs) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    // StateMachineの初期化が完了するまで待ちます
    void WaitInitialization() NN_NOEXCEPT;

    /* データパスの確認用 */
    inline bool IsAuthorized() NN_NOEXCEPT
    {
        return m_Authorized;
    }

    inline bool IsConnectAndWaitAuthorized() NN_NOEXCEPT
    {
        bool ret = false;
        if( m_WdrvState.GetSub() == WdrvSubState_Connected && IsAuthorized() == false )
        {
            ret = true;
        }
        return ret;
    }

    nn::Result PutFrame(nn::mbuf::Mbuf** pMbuf) NN_NOEXCEPT
    {
        // TODO Spectatorモード時の扱い
        // m_Authorizedではなく、別の状態を表現出来る変数を用意する
        nn::Result result;
        if( m_WdrvState.GetMain() == WdrvMainState_LocalMasterBss )
        {
            if( m_pWlanFuncs->GetConnectedClientCount() == 0 )
            {
                WLAN_LOG_DEBUG("Drop the tx packet due to no connected clients\n");
                nn::mbuf::MbufFreem(*pMbuf);
                *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
                return ResultNotAuthorized();
            }
        }

        nn::os::LockMutex(&m_TxOperateMutex);
        if( m_IsSleepState == true )
        {
            nn::mbuf::MbufFreem(*pMbuf);
            *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
            nn::os::UnlockMutex(&m_TxOperateMutex);
            return ResultInvalidState();
        }

        if( IsAuthorized() == true )
        {
            result = m_pWlanFuncs->PutFrame(*pMbuf);
            if( ResultCommandFailure().Includes(result) == true )
            {
                nn::mbuf::MbufFreem(*pMbuf);
                *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
            }
        }
        else
        {
            WLAN_LOG_DEBUG("Failed to putframe due to not authorized\n");
            nn::mbuf::MbufFreem(*pMbuf);
            *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
            result = ResultNotAuthorized();
        }
        nn::os::UnlockMutex(&m_TxOperateMutex);
        return result;
    }

    // 強制的にPutFrameを呼ぶ
    nn::Result PutFrameForce(nn::mbuf::Mbuf** pMbuf) NN_NOEXCEPT
    {
        nn::Result result;
        nn::os::LockMutex(&m_TxOperateMutex);
        if( m_IsSleepState == true )
        {
            nn::mbuf::MbufFreem(*pMbuf);
            *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
            nn::os::UnlockMutex(&m_TxOperateMutex);
            return ResultInvalidState();
        }

        result = m_pWlanFuncs->PutFrame(*pMbuf);
        if( ResultCommandFailure().Includes(result) == true )
        {
            nn::mbuf::MbufFreem(*pMbuf);
            *pMbuf = NULL;  // MbufFreemではpMbufはNULLにはならないので、ここでしておく
        }
        nn::os::UnlockMutex(&m_TxOperateMutex);
        return result;
    }

    /*
     * 特殊用途用 基本はステートマシーンの流れに沿ってステート取得するのが
     * きれいなので、この関数では無く、他のスキャン等のコマンドを使って
     * 状態取得するのがおすすめ
     */
    void GetStateMachineState(WdrvMainState *mstate, WdrvSubState *sstate) NN_NOEXCEPT
    {
        *mstate = m_WdrvState.GetMain();
        *sstate = m_WdrvState.GetSub();
    }

    void ProcessStateMachine() NN_NOEXCEPT;
    bool PostEventMessage(WlanEvent* pevebuff) NN_NOEXCEPT;
    bool PostCommandMessage(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool PostEventSleepMessage(WlanEvent* pevebuff) NN_NOEXCEPT;
    void SignalEventMessage() NN_NOEXCEPT;
    void RegisterEventCallback( void (*m_StateMachineEventCallback)(WlanEventId id, void* ptr) ) NN_NOEXCEPT;

    // SystemEventのハンドルを取得するための関数
    nn::os::NativeHandle GetSystemEventReadableHandle(detail::SystemEventType type) NN_NOEXCEPT;

    void GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT;
    void GetClientStatus(WlanClientStatusInfo* pOutInfo) NN_NOEXCEPT;

    // 受信データ管理用関数群 (start)------------------------------------------------
    inline Result AddRxEntry(uint32_t* pOutRxId, uint32_t capacity, const uint16_t ethertypes[], uint32_t count, bool bOverWrite) NN_NOEXCEPT
    {
        return m_RxEntries.Add(pOutRxId, capacity, ethertypes, count, bOverWrite);
    }

    inline Result RemoveRxEntry(uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntries.Remove(rxId);
    }

    inline Result AddEthertypeToRxEntry(uint32_t rxId, uint16_t ethertype) NN_NOEXCEPT
    {
        return m_RxEntries.AddType(rxId, ethertype);
    }

    inline Result RemoveEthertypeFromRxEntry(uint32_t* pOutRxId, uint16_t ethertype) NN_NOEXCEPT
    {
        return m_RxEntries.RemoveType(pOutRxId, ethertype);
    }

    inline bool IsMatchDataInfoRegistered(uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntries.IsMatchDataInfoRegistered(rxId);
    }

    inline Result PushRxBuffer(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
    {
        return m_RxEntries.PushBuffer(pMbuf);
    }

    inline Result PushRxBuffer(uint32_t rxId, nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
    {
        return m_RxEntries.PushBuffer(rxId, pMbuf);
    }

    inline Result PullRxBuffer(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntries.PullBuffer(pOutMbuf, rxId);
    }

    inline Result PullRxBufferEx(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId, bool block) NN_NOEXCEPT
    {
        return m_RxEntries.PullBufferEx(pOutMbuf, rxId, block);
    }

    inline Result CancelRx(uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntries.Cancel(rxId);
    }

    inline void ActivateRxBuffer() NN_NOEXCEPT
    {
        return m_RxEntries.SetMute(false);
    }

    inline void InactivateRxBuffer() NN_NOEXCEPT
    {
        return m_RxEntries.SetMute(true);
    }
    // 受信データ管理用関数群 (end)------------------------------------------------

    // 受信ActionFrame管理用関数群 (start) --------------------------------------
    inline Result AddRxEntryForActionFrame(uint32_t* pOutRxId, uint32_t capacity, const uint16_t subtypes[], uint32_t count, bool bOverWrite) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.Add(pOutRxId, capacity, subtypes, count, bOverWrite);
    }

    inline Result RemoveRxEntryForActionFrame(uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.Remove(rxId);
    }

    inline Result AddSubtypeToRxEntryForActionFrame(uint32_t rxId, uint16_t subtype) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.AddType(rxId, subtype);
    }

    inline Result RemoveSubtypeFromRxEntryForActionFrame(uint32_t* pOutRxId, uint16_t subtype) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.RemoveType(pOutRxId, subtype);
    }

    inline Result PushRxActionFrame(nn::mbuf::Mbuf* pMbuf)
    {
        return m_RxEntriesForAf.PushBuffer(pMbuf);
    }

    inline Result PushRxActionFrameEx(nn::mbuf::Mbuf* pMbuf)
    {
        return m_RxEntriesForAf.PushBufferEx(pMbuf);
    }

    inline Result PullRxActionFrame(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.PullBuffer(pOutMbuf, rxId);
    }

    inline Result CancelRxActionFrame(uint32_t rxId) NN_NOEXCEPT
    {
        return m_RxEntriesForAf.Cancel(rxId);
    }

    inline void ActivateRxActionFrame() NN_NOEXCEPT
    {
        return m_RxEntriesForAf.SetMute(false);
    }

    inline void InactivateRxActionFrame() NN_NOEXCEPT
    {
        return m_RxEntriesForAf.SetMute(true);
    }
    // 受信ActionFrame管理用関数群 (end)-----------------------------------------


    // StaticAESの鍵を覚えさせる
    void SetStaticAesKey(const Bit8* key)
    {
        NN_ABORT_UNLESS_NOT_NULL(key);
        for( uint8_t i = 0; i < StaticAesKeyLength; i++ )
        {
            m_StaticAesKey[i] = key[i];
        }
        return;
    }

    void GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT;

    Result RegisterTcpSessionInfo(WlanWowlSetupParams params) NN_NOEXCEPT;
    Result RemoveTcpSessionInfo() NN_NOEXCEPT;

    WowlWakeReason ConvertWakeupReason(uint32_t reason) NN_NOEXCEPT;
    void GetAllowedChannels(int16_t* pBuf, size_t length, uint32_t* pCount) NN_NOEXCEPT;
    // 入力されたチャンネルが使用可能かどうかチェックします
    Result CheckChannels(const int16_t* pBuf, size_t length) NN_NOEXCEPT;

    DetectSleepCondition GetDetectSleepCondition() NN_NOEXCEPT
    {
        DetectSleepCondition ret;
        if( m_IsReservedDetectSleep == true )
        {
            if( m_DetectSetupParams.length < sizeof(NintendoActionFrameHeader) + sizeof(DetectHeader) )
            {
                ret = DetectSleepCondition_NoData;
            }
            else
            {
                ret = DetectSleepCondition_Ok;
            }
        }
        else
        {
            ret = DetectSleepCondition_NotReserved;
        }
        return ret;
    }

    void ReserveDetectSleep(bool isReserved) NN_NOEXCEPT
    {
        m_IsReservedDetectSleep = isReserved;
    }

    void SetActionFrameForDetectSleep(uint8_t subtype, DetectHash hash, const char* pData, size_t size) NN_NOEXCEPT;
    nn::Result SetHashListForDetectSleep(const char* pHashList, size_t size) NN_NOEXCEPT;
    void SetPeriodicActionFrameCycle(const DetectPeriodicAfCycle& param, DetectPeriodicAfCycleTarget target) NN_NOEXCEPT;

    void ClearAfCache() NN_NOEXCEPT
    {
        m_pWlanFuncs->ClearAfCache();
    }

    void SetDetectHomeChannel(uint32_t channel) NN_NOEXCEPT
    {
        m_pWlanFuncs->SetDetectHomeChannel(channel);
    }

    uint32_t GetDetectHomeChannel() NN_NOEXCEPT
    {
        return m_pWlanFuncs->GetDetectHomeChannel();
    }

    uint64_t GetDetectSaTotalRecvCnt() NN_NOEXCEPT
    {
        return m_pWlanFuncs->GetDetectSaTotalRecvCnt();
    }

public:
    nn::wlan::WlanBaseFunctions* m_pWlanFuncs;
    nn::wlan::WpaSupplicant* m_pSuppFuncs;

private:
    void DisplayIlligalMessage(uint32_t id, const char* pfunc_name) NN_NOEXCEPT
    {
//        NN_SDK_LOG("WDM: StateErr(Curst[%s, %s], RcvId[%d], in %s\n",
//               MainStateStr[m_WdrvState.GetMain()], SubStateStr[m_WdrvState.GetSub()],
//               id, pfunc_name);
    };

    void AppendWaitEventList(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    void PostWaitingEventList(uint32_t id, bool result) NN_NOEXCEPT;
    int WaitMessage() NN_NOEXCEPT;
    void CallEventCallbackFunction(uint32_t id, void* ptr) NN_NOEXCEPT;
    bool EventProcess(int signal) NN_NOEXCEPT;

    // Functions for handling event
    void ScanComplete(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void ScanIndication(WlanEvent *pevnbuff) NN_NOEXCEPT;
    void ConnectionIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void DisassocIndicateEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void DeauthIndicateEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void DisassocCompleteEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void DeauthCompleteEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void BmissIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void LinkChangeIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void IfChangeIndicate(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void WpaConnectionSuccessEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void WpaConnectionFailureEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void WpsConnectionFailureEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void GtkSetEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void GtkClearEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void StartToConnectEv(WlanEvent* pevnbuff) NN_NOEXCEPT;
    void StartToCancelConnectEv(WlanEvent* pevnbuff) NN_NOEXCEPT;

    // Functions for handling command
    bool InitalizeWlanDriver(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    // skip_boot=trueの時だけ利用可能な無線ドライバー初期化関数(SIGLO-67650)
    bool InitalizeWlanDriverOnSkipBoot(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool FinalizeWlanDriver(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    void SetEachStateToIdle() NN_NOEXCEPT;
    bool SetWlanMode(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetWlanInterfaceUpDown(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetWlanInterfaceUpDown(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetWlanEventMask(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetWlanEventMask(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetApConfigration(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ScanRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool CancelScan(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetMacAddress(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetMacAddress(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ExecuteAcsd(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool CreateAp(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool DestroyAp(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool JoinNetworkSta(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool JoinNetworkStaWithWps(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool JoinNetworkSpectator(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool CancelJoinNetwork(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool DisconnectRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool DeauthRequest(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool AddIe(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool DeleteIe(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool AddMatchingDataInfoToRxEntry(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool RemoveMatchDataInfoToRxEntry(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetStaticAesMode(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetClientTimeout(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetBeaconLostTimeout(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetRxChain(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetMaxAssociationNumber(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool SetActionFrameWithBeacon(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool CancelActionFrameWithBeacon(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool GetRssi(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool GetState(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetStateBeforeSleep(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool SetTsfTimerEventmask(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetFwVersion(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool RequestSleep(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool RequestWakeUp(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetWakeupReasonRaw(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetWakeupReasonRaw(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool EnableWowlFeatures(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetWowlWakeCount(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetWowlSleepStats(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ClearWowlWakeCount(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ClearWowlSleepStats(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetGtkInfo(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ClearGtkInfo(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool EmulateDriverInitFail(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool GetChannelStats(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool PutOneShotActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool StartPeriodicActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool CancelPeriodicActionFrame(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool RequestSleepForDetect(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool RequestWakeUpForDetect(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    void ClearDetectData() NN_NOEXCEPT;
    void InitializeDetectData() NN_NOEXCEPT;
    bool SetDetectRecvConfig(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool ClearDetectSaTotalRecvCnt(WlanCommand* pcmdbuff) NN_NOEXCEPT;
    bool SetMulticastList(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    bool CommandProcess() NN_NOEXCEPT;
    bool InitMessageSystem() NN_NOEXCEPT;
    bool FinalizeMessageSystem() NN_NOEXCEPT;
    void InitializeSystemEvent() NN_NOEXCEPT;
    void FinalizeSystemEvent() NN_NOEXCEPT;
    bool PostMassage(nn::os::MessageQueueType* msgq, uintptr_t msg) NN_NOEXCEPT;

    void BssInfoListInitialize() NN_NOEXCEPT;
    void BssInfoListFinalize() NN_NOEXCEPT;
    void BssInfoListClear() NN_NOEXCEPT;
    void BssInfoListAdd(WlanBssInfo* pbssinfo) NN_NOEXCEPT;
    bool BssInfoListGet(WlanCommand* pcmdbuff) NN_NOEXCEPT;

    // ビーコンロストなどで切断が起きた際に自発的にDisassocを打っておくための関数を仮置き
    // FWのステートをきっちり切断状態に持っていくのが目的
    void PostDisassocCommand() NN_NOEXCEPT;

    void UpdateConnectionStatus(const ConnectionStatus& status) NN_NOEXCEPT;

    // Clear system events and initialize connection status
    void ResetSystemEventsAndConnectionStatus() NN_NOEXCEPT;

//    void AppendWaitEventList(WlanCommand* pcmdbuff);
//    void PostWaitingEventList(uint32_t id);

private:
    size_t m_EventMessageMax;
    size_t m_CommandMessageMax;
    size_t m_MessageSemaphoreMax;

    void (*m_StateMachineEventCallback)(WlanEventId id, void* ptr);

    bool m_IsInit;
    bool m_Authorized;
    bool m_IsMatchingDataRegistered;  // ローカル通信のみで使用。マッチング用のデータが既に登録されているか否か。
    static bool m_IsSleepState; // スリープ状態またはスリープ準備中にTxさせないための判断フラグ
    static nn::os::MutexType m_TxOperateMutex; // 送信処理を確実に逐次実行させるためのmutex(SIGLO-68030)

    nn::os::EventType m_StateMachineInitEvent;   // StateMachineの初期化完了待ち用イベント

    /* Event処理よりも優先して抜けたい場合があるかもしれないので用意してみた */
    bool m_exitStateMachine;

    // 自分自身のMACアドレス
    MacAddress               m_ownMacAddr;

    // イベント/コマンドメッセージキューに関係するもの(定数値を除く)
    nn::os::SemaphoreType    m_MessageSemaphore;        // 操作対象のセマフォ
    nn::os::SemaphoreType    m_MessageSleepSemaphore;   // Sleepステート時にポストされたイベントメッセージの処理を開始するためのセマフォ
    nn::os::MessageQueueType m_CommandQueue;            // メッセージキュー
    nn::os::MessageQueueType m_EventQueue;              // イベントキュー
    nn::os::MessageQueueType m_EventSleepQueue;         // イベントキュー(sleepステート時)
    uintptr_t                *m_EventBuffer;
    uintptr_t                *m_CommandBuffer;
    uintptr_t                *m_EventSleepBuffer;

    nn::wlan::State          m_WdrvState;
    nn::wlan::State          m_WdrvStateBeforeSleep;    // スリープ突入前のステート保存用
    WlanBssInfo*             m_BssInfoList;
    int                      m_BssInfoListCnt;
    nn::os::MutexType        m_BssInfoListMutex;

    ConnectionStatus         m_ConnectionStatus;

    // ローカル通信とインフラ通信でSystemEventを分けておいた方が良い？
    nn::os::SystemEventType  m_ConnectionStatusEvent;  // 接続状態変化通知用イベント
    nn::os::SystemEventType  m_ScanCompletionEvent;    // スキャン完了通知用イベント
    nn::os::SystemEventType  m_ConnectCompletionEvent; // 接続完了通知用イベント

    nn::os::MutexType        m_WaitingEventMutex;
    nn::os::MutexType        m_ConnectionStatusMutex;

    // 受信データ管理用
    RxEntries                m_RxEntries;
    // ActionFrame受信管理用
    RxEntriesForActionFrame  m_RxEntriesForAf;

    Bit8                     m_StaticAesKey[16]; // ローカル通信のStaticAES鍵を保存しておくための変数
    bool                     m_StaticAesMode;    // StaticAESモードかどうか

    // Beacon付随ActionFrame関連
    bool                     m_IsBeaconActionFrameSet;  // Beacon用ActionFrameがセットされているかどうか
    bool                     m_IsBeaconActionFrameTransferring; // Beacon用ActionFrameが送信中かどうか

    struct BeaconActionFrame {
        char data[BeaconActionFrameSizeMax];
        uint16_t size;
        uint8_t bssid[MacAddress::MacAddressSize];
    };
    BeaconActionFrame        m_BeaconActionFrame;
    WlanAllowedChannels      m_AllowedChannel;

    // CoexMC2モードを有効化中かどうか
    bool                     m_IsEnabledMc2;

    // Scan結果管理用構造体
    struct ScanResultInfo {
        int minRssiValue;  // 格納しているスキャン結果の中で最も低いRSSI値
        WlanBssInfo* pMinRssiInfo;  // 格納しているスキャン結果の中で最も低いRSSI値を持つものへのポインタ
    };
    ScanResultInfo           m_ScanResultInfo;
    static bool              m_IsDrvInitFail;
    bool                     m_IsSetTcpInfoWoWL;  // WoWLの起床用パケットに関する情報がセットされているかどうか
    WlanWowlSetupParams      m_WowlParams;
    uint32_t                 m_WowlWakeReason;
    WowlWakeCount            m_WowlWakeCount;
    WowlSleepStats           m_WowlSleepStats;

    bool                     m_IsGtkSet;  // GTKがFWに登録されているか否か

    bool                     m_IsLcsMode;  // ローカル通信LCSモード中か否か

    bool                     m_IsLinkDownBeforeWpaFailureEv;  //(SIGLO-64396)の問題に対応するためのフラグ

    bool                     m_IsAntennaChangedByUser;  // アンテナ設定がユーザーに変更されているかどうか
    int                      m_EventMessageSleepCnt;   // sleepステート時に処理をスルーしたイベントメッセージの数


    // すれちがい通信用
    WlanDetectSetupParams        m_DetectSetupParams;  // すれちがいスリープ用パラメータ
    DetectPeriodicAfCycle          m_DetectHdComuPat;    // HDモードActionFrame通信パターン
    uint32_t                 m_DetectWakeReason;
    DetectWakeCount  m_DetectWakeCount;
    bool                     m_IsReservedDetectSleep;  // すれちがいスリープが予約されているかどうか

    SleepMode                m_sleepMode;  // 何のモードのスリープに入ったか
};


void StateMachineThreadFunc(void *pArg) NN_NOEXCEPT;
}}

