﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/wlan/wlan_Result.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/wlan/wlan_Types.h>

#include "wlan_DebugLog.h"
#include "wlan_RingBuffer.h"

namespace nn { namespace wlan {

class RxEntries
{
public:
    // 登録できるRxEntryの最大数
    static const uint32_t RxEntriesCountMax = 8;
    static const uint32_t RxCapacityMax = RingBuffer::CapacityMax;

    RxEntries() NN_NOEXCEPT;
    ~RxEntries() NN_NOEXCEPT;

    virtual Result Add(uint32_t* pOutRxId, uint32_t capacity, const uint16_t ethertypes[], uint32_t count, bool bOverWrite) NN_NOEXCEPT;
    Result Remove(uint32_t rxId) NN_NOEXCEPT;

    virtual Result RemoveType(uint32_t* pOutRxId, uint16_t type) NN_NOEXCEPT;
    virtual Result AddType(uint32_t rxId, uint16_t type) NN_NOEXCEPT;
    Result AddMatchingInfo(uint32_t rxId, const ReceivedDataMatchInfo& pMatchDataInfo) NN_NOEXCEPT;
    Result RemoveMatchInfo(uint32_t rxId, const ReceivedDataMatchInfo& pMatchDataInfo) NN_NOEXCEPT;
    bool IsMatchDataInfoRegistered(uint32_t rxId) NN_NOEXCEPT;

    void RemoveAll() NN_NOEXCEPT;

    virtual Result PushBuffer(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT;
    virtual Result PushBuffer(uint32_t rxId, nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT;
    virtual Result PullBuffer(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId) NN_NOEXCEPT;
    virtual Result PullBufferEx(nn::mbuf::Mbuf** pOutMbuf, uint32_t rxId, bool block) NN_NOEXCEPT;

    void SetMute(bool bMute) NN_NOEXCEPT;
    Result Cancel(uint32_t rxId) NN_NOEXCEPT;

protected:
    void Cancel() NN_NOEXCEPT;
    void Cleanup() NN_NOEXCEPT;

    RingBuffer* FindRingBufferByRxId(uint32_t rxId) NN_NOEXCEPT;
    RingBuffer* FindRingBufferByType(uint16_t type) NN_NOEXCEPT;
    Result FindRxIdFromRingBuffer(uint32_t* pOutRxId, const RingBuffer* pRingBuffer) const NN_NOEXCEPT;

    typedef std::map<uint16_t, RingBuffer*>  RingBufferMap;
    typedef std::map<uint32_t, RingBuffer*>  EntryMap;
    typedef std::pair<uint16_t, RingBuffer*> RingBufferPair;
    typedef std::pair<uint32_t, RingBuffer*> EntryPair;
    nn::os::MutexType m_Mutex;
    RingBufferMap     m_RingBuffers;
    EntryMap          m_Entries;
    uint32_t          m_LastRxId;
    bool              m_bMute;
};

class RxEntriesForActionFrame : public RxEntries
{
public:
    RxEntriesForActionFrame() NN_NOEXCEPT {};
    virtual ~RxEntriesForActionFrame() NN_NOEXCEPT {};

    virtual Result Add(uint32_t* pOutRxId, uint32_t capacity, const uint16_t types[], uint32_t count, bool bOverWrite) NN_NOEXCEPT;

    virtual Result RemoveType(uint32_t* pOutRxId, uint16_t subtype) NN_NOEXCEPT;
    virtual Result AddType(uint32_t rxId, uint16_t subtype) NN_NOEXCEPT;

    virtual Result PushBuffer(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT;
    virtual Result PushBufferEx(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT;
private:
    Result PushBufferCore(nn::mbuf::Mbuf* pMbuf, bool isEx) NN_NOEXCEPT;
};

}}

