﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/wlan/wlan_Types.h>

#include "wlan_DebugLog.h"

namespace nn { namespace wlan {

class RingBuffer
{
public:
    static const uint32_t CapacityMax = 96;
    // 受信データのマッチング用データの最大登録数
    static const uint32_t ReceivedDataMatchInfoCountMax = 10;  // TORIAEZU

    explicit RingBuffer(uint32_t capacity, bool bOverWrite) NN_NOEXCEPT
     : m_WaitingDequeueCount(0),
       m_bOverWrite(bOverWrite),
       m_MessageQueue(m_Buffer, capacity),
       m_MatchInfoCount(0),
       m_bForDetect(false)
    {
        for( uint32_t i = 0; i < ReceivedDataMatchInfoCountMax; i++ )
        {
            std::memset(&m_MatchInfoArray[i], 0, sizeof(ReceivedDataMatchInfo));
        }
    }

    ~RingBuffer() NN_NOEXCEPT;

    /*
     * RingBufferへデータを強制的にPushします。キューが満杯の場合、最も古いデータを消してから再度Pushを行います。
     */
    void ForceEnqueue(nn::mbuf::Mbuf* pData) NN_NOEXCEPT;

    bool TryDequeue(nn::mbuf::Mbuf** pOutData) NN_NOEXCEPT;

    nn::mbuf::Mbuf* Dequeue() NN_NOEXCEPT;

    /*
     * RingBufferに入っている要素を空にします。
     * RingBufferにはmbufへのポインタが入っているので、それらを使って、mbufの解放を行います。
     */
    void Cleanup() NN_NOEXCEPT;

    /*
     * Dequeue待ちをしているスレッドを解放します。
     */
    void Cancel() NN_NOEXCEPT;

    /*
     * 受信データに対するマッチング用データを登録します。
     */
    bool SetMatchInfo(const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT;

    /*
     * マッチング用データを削除します。該当するデータが存在しなかった場合、何もしません。
     */
    void RemoveMatchInfo(const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT;

    bool IsMatchDataInfoRegistered() NN_NOEXCEPT
    {
        return (m_MatchInfoCount > 0) ? true : false;
    }

    void SetParamForDetect(bool var) NN_NOEXCEPT
    {
        m_bForDetect = var;
    }

    bool IsForDetect() NN_NOEXCEPT
    {
        return m_bForDetect;
    }

private:
    void DumpMatchInfoArray() NN_NOEXCEPT; // デバッグ用

private:
    uintptr_t              m_Buffer[CapacityMax];
    uint32_t               m_WaitingDequeueCount;  // Dequeue待ちをしているスレッドの数
    bool                   m_bOverWrite;           // RingBufferがいっぱいの場合に上書きしても良いか
    nn::os::MessageQueue   m_MessageQueue;
    ReceivedDataMatchInfo  m_MatchInfoArray[ReceivedDataMatchInfoCountMax]; // 受信データに対するマッチングを行う場合に使用される
    uint32_t               m_MatchInfoCount;  // マッチング用データの登録数
    bool                   m_bForDetect;           // すれちがいデータ用のRingBufferかどうか
};

}}

