﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include "wlan_RingBuffer.h"
#include "wlan_MemoryInit.h"

namespace nn { namespace wlan {

RingBuffer::~RingBuffer() NN_NOEXCEPT
{
    Cleanup();
}

void RingBuffer::ForceEnqueue(nn::mbuf::Mbuf* pData) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("enter %s\n", __FUNCTION__);
    // マッチング用のデータが登録されていたら、比較する
    if( m_MatchInfoCount > 0 && pData != NULL ) // キャンセル用のデータの場合NULLなので、そうじゃなかった場合のみ比較
    {
        bool contains = false;
        // 比較先は、DIX形式のEtherTypeの次のフィールドから。
        // 受信データは常にDIX形式なので、先頭から15オクテット目からを比較すれば良い
        nn::Bit8 *ptr = reinterpret_cast<nn::Bit8*>(nn::mbuf::MbufTod(pData));

        for( uint32_t i = 0; i < m_MatchInfoCount; i++ )
        {
            if( std::memcmp(&m_MatchInfoArray[i].matchData[0],
                    &ptr[14],
                    m_MatchInfoArray[i].matchLength) == 0 )
            {
                // 該当したので、このデータをキューに入れる
                WLAN_LOG_DEBUG("Match data info!\n");
                contains = true;
                break;
            }
        }

        if( contains == false )
        {
            // 該当しなかったので、データを破棄して抜ける
            WLAN_LOG_INFO("Not matched.\n");
            if( pData->_pGeneral != NULL )
            {
                nnwlanFreeNormal(pData->_pGeneral);
            }
            nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf*>(pData));
            return;
        }
    }

    if( m_MessageQueue.TrySend(reinterpret_cast<uintptr_t>(pData)) != true )
    {
        if( m_bOverWrite == true )
        {
            // キューが一杯なので最後尾のデータを削除してから突っ込む
            uintptr_t data;
            if( m_MessageQueue.TryReceive(&data) != true )
            {
                NN_SDK_ASSERT(false, "Failed to dequeue data from ring buffer despite the existence of data.\n");
            }
            // 最後尾のデータがCancelによるNULLが入っている場合があるのでチェック
            if( data != NULL )
            {
                if( reinterpret_cast<nn::mbuf::Mbuf*>(data)->_pGeneral != NULL )
                {
                    nnwlanFreeNormal(reinterpret_cast<nn::mbuf::Mbuf*>(data)->_pGeneral);
                }
                nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf*>(data));
            }

            if( m_MessageQueue.TrySend(reinterpret_cast<uintptr_t>(pData)) != true )
            {
                NN_SDK_ASSERT( false, "Failed to enqueue mbuf to ring buffer\n");
            }
        }
        else
        {
            // 上書きしないので、データは破棄する
            // CancelによるNULLデータかチェック
            if( pData != NULL )
            {
                if( pData->_pGeneral != NULL )
                {
                    nnwlanFreeNormal(pData->_pGeneral);
                }
                nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf*>(pData));
            }
        }
    }
    else
    {
        WLAN_LOG_DEBUG("Data is enqueued.\n");
    }
}

bool RingBuffer::TryDequeue(nn::mbuf::Mbuf** pOutData) NN_NOEXCEPT
{
    uintptr_t data;
    bool ret = m_MessageQueue.TryReceive(&data);
    if( ret == true )
    {
        *pOutData = reinterpret_cast<nn::mbuf::Mbuf*>(data);
    }

    return ret;
}

nn::mbuf::Mbuf* RingBuffer::Dequeue() NN_NOEXCEPT
{
    uintptr_t data;

    m_WaitingDequeueCount++;

    m_MessageQueue.Receive(&data);

    m_WaitingDequeueCount--;

    return reinterpret_cast<nn::mbuf::Mbuf*>(data);
}

void RingBuffer::Cleanup() NN_NOEXCEPT
{
    nn::mbuf::Mbuf* pMbuf = nullptr;
    uintptr_t data = NULL;

    // リングバッファが空になるまで繰り返し
    while( m_MessageQueue.TryReceive(&data) == true )
    {
        // キャンセル用に NULL が入っている場合があるのでチェック
        if( data != NULL )
        {
            pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(data);
            if( pMbuf->_pGeneral != NULL )
            {
                nnwlanFreeNormal(pMbuf->_pGeneral);
            }
            nn::mbuf::MbufFreem(pMbuf);
            data = NULL;
        }
    }

    // マッチングデータもクリアしておく
    for( uint32_t i = 0; i < ReceivedDataMatchInfoCountMax; i++ )
    {
        std::memset(&m_MatchInfoArray[i], 0, sizeof(ReceivedDataMatchInfo));
    }
    m_MatchInfoCount = 0;
}

void RingBuffer::Cancel() NN_NOEXCEPT
{

    do
    {
        // ブロック待ちしているスレッドがあればリングバッファに強制的に NULL を格納する
        // あふれたバッファは内部で自動的に破棄される
        ForceEnqueue(NULL);  // nullptrではなくNULL（＝0）をいれておく

        // 受信側が NULL を受け取ったら特別ならエラーを返す
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
    while( m_WaitingDequeueCount > 0 );

}

bool RingBuffer::SetMatchInfo(const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
{
    if( m_MatchInfoCount >= ReceivedDataMatchInfoCountMax )
    {
        WLAN_LOG_ERROR("[WLAN]Failed to set new received data match info because received data match info count is max.\n");
        return false;
    }

    NN_SDK_ASSERT(pMatchInfo.matchLength <= ReceivedDataMatchLengthMax);

    // 重複登録チェック
    for( uint32_t i = 0; i < m_MatchInfoCount; i++ )
    {
        if( std::memcmp(&m_MatchInfoArray[i].matchData[0],
                &pMatchInfo.matchData[0],
                pMatchInfo.matchLength) == 0 )
        {
            WLAN_LOG_ERROR("[WLAN]Failed to set new received data match info because the same one is already registered\n");
            return false;
        }
    }

    std::memcpy(&m_MatchInfoArray[m_MatchInfoCount].matchData[0],
            &pMatchInfo.matchData[0],
            pMatchInfo.matchLength);
    m_MatchInfoArray[m_MatchInfoCount].matchLength = pMatchInfo.matchLength;
    WLAN_LOG_DEBUG("Set new received data match info\n");

    m_MatchInfoCount++;

    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Debug )
    {
        DumpMatchInfoArray();
    }

    return true;
}

void RingBuffer::RemoveMatchInfo(const ReceivedDataMatchInfo& pMatchInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pMatchInfo.matchLength <= ReceivedDataMatchLengthMax);

    for( uint32_t i = 0; i < m_MatchInfoCount; i++ )
    {
        if( std::memcmp(&m_MatchInfoArray[i].matchData[0],
                &pMatchInfo.matchData[0],
                pMatchInfo.matchLength) == 0 )
        {
            // 該当したので、後ろの要素を前に詰めることで削除する。
            for( uint32_t j = i; j < ReceivedDataMatchInfoCountMax - 1; j++ )
            {
                m_MatchInfoArray[j] = m_MatchInfoArray[j + 1];
            }
            // 最後尾を0埋め。
            std::memset(&m_MatchInfoArray[ReceivedDataMatchInfoCountMax - 1], 0, sizeof(ReceivedDataMatchInfo));

            WLAN_LOG_DEBUG("Removed received data match info\n");
            m_MatchInfoCount--;
            break;
        }
    }
    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Debug )
    {
        DumpMatchInfoArray();
    }
}

void RingBuffer::DumpMatchInfoArray() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[WLAN] ----- Dump Match Info Array -----\n");
    for( uint32_t i = 0; i < ReceivedDataMatchInfoCountMax; i++ )
    {
        WLAN_LOG_DEBUG("[%d] Length : %d\n", i, m_MatchInfoArray[i].matchLength);
        for( uint32_t j = 0; j < m_MatchInfoArray[i].matchLength; j++ )
        {
            WLAN_LOG_DEBUG("%02X ", m_MatchInfoArray[i].matchData[j]);
        }
        WLAN_LOG_DEBUG("\n");
    }
}

}}

