﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include "wlan_MezcalType.h"
#include "wlan_MsgBuffer.h"
#include "wlan_MemoryInit.h"

namespace nn { namespace wlan {


//WlanMessageBuffer g_WlanMessage[ WlanMessageBufferMax ];
nn::os::MutexType g_WlanMessageBufferMutex;

//#define MALLOC_VERSION    1
//#define TRACE_MSG

/* 暫定 */
static int g_CommandBuffCounterMax;
static int g_CommandBuffCounter;
static int g_EventBuffCounterMax;
static int g_EventBuffCounter;

/* WlanMassageの再初期化 --------------------------------------------------- */
static void InitalWlanMessage(WlanMessageBuffer *pmsgbuff)
{
    pmsgbuff->_magicWord    = nn::wlan::WlanTypeMagic;
    pmsgbuff->_usingFlag    = false;
    pmsgbuff->id            = WLAN_COMMAND_START;
    pmsgbuff->Args          = NULL;
    pmsgbuff->EvnWait       = WAIT_CMD_FINISH;
    pmsgbuff->Result        = false;

    nn::os::InitializeEvent( &pmsgbuff->_finishEvent,
                             false,
                             nn::os::EventClearMode_AutoClear );
}

/* WlanMessageBuffer の確保 ------------------------------------------------ */
static WlanMessageBuffer* AllocateMessageBuffer(size_t arg_size)
{
    WlanMessageBuffer* pmsgbuff = NULL;

    //pmsgbuff = static_cast<WlanMessageBuffer*>(BlackMalloc( sizeof(WlanCommand) ));
    pmsgbuff = new WlanMessageBuffer;
    NN_SDK_ASSERT_NOT_NULL(pmsgbuff);

    InitalWlanMessage( pmsgbuff );

    if(arg_size > 0)
    {
        pmsgbuff->Args = nnwlanMallocNormal( arg_size );
        NN_SDK_ASSERT_NOT_NULL(pmsgbuff->Args);
    }

    return pmsgbuff;
}

/* WlanMessageBuffer の解放処理 -------------------------------------------- */
static void FreeMessaguBuffer(WlanMessageBuffer *pmsgbuff)
{
    if(pmsgbuff->Args != NULL)
    {
        // 型情報を失っているので、もしクラスのオブジェクトが割り当てられていたら、デストラクタは呼ばれないので注意
        // 今のところ構造体しか割り当てられないのでデストラクタのことは気にしなくて良い
        // TODO deleterのようなものを作りちゃんと対処する
        // TORIAEZU deleteは使わずfreeでclangのWarningからは逃げておく
        // delete pmsgbuff->Args;
        nnwlanFreeNormal(pmsgbuff->Args);
    }
    nn::os::FinalizeEvent( &pmsgbuff->_finishEvent );
    delete pmsgbuff;
}

/* WlanMessage の初期化処理 ------------------------------------------------ */
void InitalWlanMessageStruct(int num_cmdbuff, int num_evnbuff)
{
    nn::os::InitializeMutex(&g_WlanMessageBufferMutex, false, 0);

    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    /* limitter */
    g_CommandBuffCounterMax = num_cmdbuff;
    g_EventBuffCounterMax   = num_evnbuff;

    g_CommandBuffCounter = 0;
    g_EventBuffCounter   = 0;

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );
}

/* WlanMessage の終了処理 -------------------------------------------------- */
void FinalizeWlanMessageStruct()
{
#if defined (TRACE_MSG)
    NN_SDK_LOG("WDMM: %s enter\n", __FUNCTION__);
#endif
    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    g_CommandBuffCounterMax = 0;
    g_EventBuffCounterMax   = 0;

    g_CommandBuffCounter = 0;
    g_EventBuffCounter   = 0;

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );

    nn::os::FinalizeMutex( &g_WlanMessageBufferMutex );
}

/* コマンドバッファの取得 -------------------------------------------------- */
WlanCommand* GetCommandBuffer(size_t arg_size)
{
    WlanCommand* pcmdbuff = NULL;

    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    if(g_CommandBuffCounter < g_CommandBuffCounterMax)
    {
        pcmdbuff = static_cast<WlanCommand*>( AllocateMessageBuffer( arg_size ) );
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);
        g_CommandBuffCounter ++;
#if defined (TRACE_MSG)
        NN_SDK_LOG("WDMBC: Cmd GetAddr=%lp\n", pcmdbuff);
#endif
    }
    else
    {
        NN_SDK_ASSERT(false, "!!!!!!!!!!!!!!!!!! WDMB: Command buffer full\n");
    }

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );

    return pcmdbuff;
}

/* コマンドバッファの解放 -------------------------------------------------- */
void ReleaseCommandBuffer(WlanCommand* pcmdbuff)
{
#if defined (TRACE_MSG)
    NN_SDK_LOG("WDMBC: %s[%lp]\n", __FUNCTION__, pcmdbuff);
#endif

    FreeMessaguBuffer( static_cast<WlanMessageBuffer*>(pcmdbuff) );

    /* 守りたいのはカウンタなので ここだけ排他 */
    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    NN_SDK_REQUIRES(g_CommandBuffCounter);
    g_CommandBuffCounter --;

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );
}

/* イベントバッファの取得 -------------------------------------------------- */
WlanEvent* GetEventBuffer(size_t arg_size)
{
    WlanEvent* pevnbuff = NULL;

    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    if(g_EventBuffCounter < g_EventBuffCounterMax)
    {
        pevnbuff = static_cast<WlanEvent*>( AllocateMessageBuffer( arg_size ) );
        NN_SDK_ASSERT_NOT_NULL(pevnbuff);
        g_EventBuffCounter ++;
#if defined (TRACE_MSG)
        NN_SDK_LOG("WDMBC: Evn GetAddr=%lp\n", pevnbuff);
#endif
    }
    else
    {
        NN_SDK_ASSERT(false, "!!!!!!!!!!!!!!!!!! WDMB: Event buffer full\n");
    }

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );

    return pevnbuff;
}

/* イベントバッファの解放 -------------------------------------------------- */
void ReleaseEventBuffer(WlanEvent* pevnbuff)
{
#if defined (TRACE_MSG)
    NN_SDK_LOG("WDMBC: Evn Freebuffer[%lp]\n", pevnbuff);
#endif

    FreeMessaguBuffer( static_cast<WlanMessageBuffer*>(pevnbuff) );

    nn::os::LockMutex( &g_WlanMessageBufferMutex );

    /* 守りたいのはカウンタなので ここだけ排他 */
    NN_SDK_REQUIRES(g_EventBuffCounter);
    g_EventBuffCounter --;

    nn::os::UnlockMutex( &g_WlanMessageBufferMutex );
}

}}
