﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemApi.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Result.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/util/util_StringUtil.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#endif

#include <bwl.h>
#include <dhd_horizon.h>
#include <bcmwifi_channels.h>

#include <acsd_svr.h>

#include "wlan_MezcalType.h"
#include "wlan_StateMachine.h"
#include "wlan_Mezcal.h"
#include "driver/wlan_driver.Mezcal.h"

#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_Definitions.h>

#include "wlan_MemoryInit.h"
#include "wlan_WpaSupplicant.h"
#include "wlan_Util.h"

//#define TICK_EVENTCB_LOCAL
//#define TICK_RXCB_LOCAL
//#define TICK_TXCOMPCB
//#define PRINT_THROUGHPUT

namespace nn { namespace wlan {

#ifdef TICK_TXCOMPCB
static nn::os::Tick g_TxValidateTick;
static int8_t g_TxValidateFlag;
#endif

#ifdef PRINT_THROUGHPUT
static nn::os::Tick g_ThroughTick;
static int32_t throughput_received;
static size_t throughput_rxBytes;
static const int32_t MEASURE_INTERVAL = 10; // sec
#endif

static void DumpLine(void *ptr, uint32_t skip, uint32_t len)
{
    uint8_t *p = static_cast<uint8_t*>(ptr);

    NN_SDK_LOG("%p| ", (static_cast<uint32_t>(reinterpret_cast<uintptr_t>(ptr) & 0xfffffff0))); // 64bit環境では情報落ちする
    for(uint32_t i=0; i<skip; i++)
    {
        NN_SDK_LOG("   ");
    }
    for(uint32_t i=0; i<len; i++)
    {
        NN_SDK_LOG("%02x ", *p);
        p ++;
    }
    for(uint32_t i=(skip + len); i < 0x10; i++)
    {
        NN_SDK_LOG("   ");
    }
    NN_SDK_LOG("|");

    for(uint32_t i=0; i < skip; i++)
    {
        NN_SDK_LOG(" ");
    }
    p = static_cast<uint8_t*>(ptr);
    for(uint32_t i=0; i<len; i++)
    {
        NN_SDK_LOG("%c", (((*p<0x20)||*p>=0x80)? 0x2e: *p));
        p ++;
    }
    NN_SDK_LOG("\n");
}

void DumpFunc(void* buffp, size_t size)
{
    uint8_t* buff = reinterpret_cast<uint8_t*>(buffp);
    uint32_t skip_len;
    uint32_t disp_len;

    NN_SDK_LOG("Dump Buffer(size=%ld)\n", size);
    NN_SDK_LOG("Addr    | +0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +a +b +c +d +e +f |0123456789abcdef\n");

    skip_len = static_cast<uint32_t>(reinterpret_cast<uintptr_t>(buff) & 0xf);
    disp_len = (0x10 - static_cast<uint32_t>(reinterpret_cast<uintptr_t>(buff) & 0xf));
    if( disp_len > size )
    {
        disp_len = size;
    }

    do {
        DumpLine(buff, skip_len, disp_len);
        skip_len = 0;

        buff += disp_len;
        size -= disp_len;
        if(size > 0x10)    disp_len = 0x10;
        else            disp_len = size;
    } while(size > 0);
}

#ifdef ENABLE_WOWL_AGING_DEBUG
void ParseTcpData(void* pBuf, size_t size, uint64_t* pAckNum)
{
    uint8_t* ptr = reinterpret_cast<uint8_t*>(pBuf);

    // IPパケットかチェック
    struct macHeader {
        uint8_t src[6];
        uint8_t dst[6];
        uint16_t type;
    };
    macHeader* pMac = reinterpret_cast<macHeader*>(ptr);
    uint16_t ethtype = ReverseEndian16(pMac->type);
    WLAN_LOG_DEBUG("Ether type:%02X\n", ethtype);
    if( ethtype != 0x0800 )
    {
        WLAN_LOG_DEBUG("Not IP(0x%04X)\n", ethtype);
        return;
    }

    // TCPパケットかチェック
    struct ipHeader {
        uint16_t version;
        uint16_t ipDataSize;
        uint16_t id;
        uint16_t pad;
        uint8_t ffh;
        uint8_t protocol;
        uint16_t checksum;
        uint8_t srcip[4];
        uint8_t dstip[4];
    };
    ptr += sizeof(macHeader);
    ipHeader* pIp = reinterpret_cast<ipHeader*>(ptr);
    if( pIp->protocol != 0x06 )
    {
        WLAN_LOG_DEBUG("Not TCP(0x%02X)\n", pIp->protocol);
        return;
    }
    uint16_t ipSize = ReverseEndian16(pIp->ipDataSize);

    // TCP情報取得
    struct tcpHeader {
        uint16_t srcPort;
        uint16_t dstPort;
        uint32_t seqNum;
        uint32_t ackNum;
        uint8_t headerSize;
        uint8_t flags;
        uint16_t windowSize;
        uint16_t checkSum;
    };
    ptr += sizeof(ipHeader);
    tcpHeader* pTcp = reinterpret_cast<tcpHeader*>(ptr);
    uint32_t seqNum = ReverseEndian32(pTcp->seqNum);
    WLAN_LOG_DEBUG("Seq num : 0x%08X\n", seqNum);
    uint16_t tcpHeaderSize = ((pTcp->headerSize & 0xF0) >> 4) * 4;
    WLAN_LOG_DEBUG("Header size : %d bytes\n", tcpHeaderSize);
    uint32_t tcpDataLen = ipSize - 20 - tcpHeaderSize;
    WLAN_LOG_DEBUG("TCP data length : %d\n", tcpDataLen);
    WLAN_LOG_DEBUG("New ack num : 0x%08X\n", seqNum + tcpDataLen);
    if( tcpDataLen > 0 )
    {
        uint64_t num = seqNum + tcpDataLen;
        NN_SDK_LOG("Update acknum:0x%08X\n", num);
        *pAckNum = num;
        DumpFunc(pBuf, size);
    }
}
#endif

static void DisplayDhdEvent(wl_event_msg_t* pevn)
{
    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Debug )
    {
        NN_SDK_LOG("=== DHD event ===\n");
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Waddress-of-packed-member"
#endif
        NN_SDK_LOG("Ver(2)      : %d[%lp]\n", pevn->version, &pevn->version);
        NN_SDK_LOG("Flag(2)     : %d[%lp]\n", pevn->flags, &pevn->flags);
        NN_SDK_LOG("Type(4)     : %d[%lp]\n", pevn->event_type, &pevn->event_type);
        NN_SDK_LOG("Stat(4)     : %d[%lp]\n", pevn->status, &pevn->event_type);
        NN_SDK_LOG("Reason(4)   : %d[%lp]\n", pevn->reason, &pevn->reason);
        NN_SDK_LOG("AuthT(4)    : %d[%lp]\n", pevn->auth_type, &pevn->auth_type);
        NN_SDK_LOG("Len(4)      : %d[%lp]\n", pevn->datalen, &pevn->datalen);
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic pop
#endif
        NN_SDK_LOG("Ether(6)    : (Skip)[%lp]\n", &pevn->addr);
        NN_SDK_LOG("Name(16)    : %s[%lp]\n", pevn->ifname, &pevn->ifname);
        NN_SDK_LOG("Idx(1)      : %d[%lp]\n", pevn->ifidx, &pevn->ifidx);
        NN_SDK_LOG("BssCfgIdx(1): %d[%lp]\n", pevn->bsscfgidx, &pevn->bsscfgidx);
    }
}

static StateMachine* g_pStateMachine; /* 暫定 */
/* Mezcal クラスにステートマシーンクラスを覚えさせる ----------------------- */
void BindStateMachine(nn::wlan::StateMachine* statm) NN_NOEXCEPT
{
    // 暫定
    g_pStateMachine = statm;
}

ConnectionStatus Mezcal::m_MezcalConStat;
ClientStatusManager Mezcal::m_ClientStatusManager;
nn::os::MutexType Mezcal::m_ConnectionStatusMutex;
bool Mezcal::m_bExecutingAcsd;
struct ether_addr Mezcal::m_OwnMacAddr;
nn::os::MutexType Mezcal::m_GeneralMutex;
nn::os::MutexType Mezcal::m_TsfMutex;
WlanTsfTimerValue Mezcal::m_TsfTimerValue;
int64_t Mezcal::m_DeltaTimeBetweenTsfAndSys;
#ifdef ENABLE_LOCAL_TXFLOW_CTRL
int32_t Mezcal::m_TxCount;
nn::os::MutexType Mezcal::m_TxCountMutex;
#endif
bool Mezcal::m_bWaitIfUpEvent;
uint32_t Mezcal::m_wakeReasonForSet;
uint32_t Mezcal::m_wowlFeatures;
#ifdef ENABLE_WOWL_AGING_DEBUG
uint64_t Mezcal::m_ackNum;
#endif
static int g_aloeAfCount = -1;
uint32_t Mezcal::m_dongleTime;
nn::os::Tick Mezcal::m_sysTick(0);
/* コンストラクタ  --------------------------------------------------------- */
Mezcal::Mezcal() NN_NOEXCEPT
{
    m_InterfaceHandle = NULL;
    nn::os::InitializeMutex(&m_ConnectionStatusMutex, false, 0);
    nn::os::InitializeMutex(&m_GeneralMutex, false, 0);
    nn::os::InitializeMutex(&m_TsfMutex, false, 0);
    m_bExecutingAcsd = false;
    m_bDcsEnabled = false;
    m_bWaitIfUpEvent = false;
    std::memset(&m_OwnMacAddr.octet[0], 0x00, ETHER_ADDR_LEN);
    m_TsfTimerValue.high = 0;
    m_TsfTimerValue.low = 0;
    std::memset(m_DefaultEventmask, 0, WL_EVENTING_MASK_LEN);
#ifdef TICK_TXCOMPCB
    g_TxValidateFlag = 0;
#endif
#ifdef ENABLE_LOCAL_TXFLOW_CTRL
    m_TxCount = 0;
    nn::os::InitializeMutex(&m_TxCountMutex, false, 0);
#endif
#ifdef PRINT_THROUGHPUT
    g_ThroughTick = nn::os::GetSystemTick();
    throughput_received = 0;
    throughput_rxBytes = 0;
#endif
    m_wakeReasonForSet = 0;
    m_wowlFeatures = 0;
#ifdef ENABLE_WOWL_AGING_DEBUG
    m_ackNum = 0;
#endif
    m_IsLcsMode = false;
    nn::psm::Initialize();
    m_DetectHomeCh = DetectHomeChannel;
    m_dongleTime = 0;
    m_DetectSaTotalRecvCnt = 0;
}

Mezcal::~Mezcal() NN_NOEXCEPT
{
    nn::psm::Finalize();
    m_InterfaceHandle = NULL;
    nn::os::FinalizeMutex(&m_ConnectionStatusMutex);
    nn::os::FinalizeMutex(&m_GeneralMutex);
    nn::os::FinalizeMutex(&m_TsfMutex);
    m_TsfTimerValue.high = 0;
    m_TsfTimerValue.low = 0;
#ifdef ENABLE_LOCAL_TXFLOW_CTRL
    nn::os::FinalizeMutex(&m_TxCountMutex);
#endif
}

/*
 * TODO ドライバAPIが失敗した場合はケアしようがないので、すべてFATALエラーとして扱う
 */

void Mezcal::SetRandomMacAddress() NN_NOEXCEPT
{
    struct ether_addr tempMacAddr = {
            {0x40, 0xD2, 0x8A, 0x00, 0x00, 0x00}
    };
    // 下位3オクテットを乱数で埋める
    std::srand(static_cast<uint32_t>(nn::os::GetSystemTick().GetInt64Value()));
    for( uint8_t i = 3; i < ETHER_ADDR_LEN; i++ )
    {
        tempMacAddr.octet[i] = static_cast<uint8_t>(std::rand() % 0xFF);
    }

    int bwl_ret = bwl_set_curetheraddr(m_InterfaceHandle, &tempMacAddr);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    WLAN_LOG_INFO("Set random MAC address [%02X:%02X:%02X:%02X:%02X:%02X]\n\n",
            tempMacAddr.octet[0], tempMacAddr.octet[1], tempMacAddr.octet[2],
            tempMacAddr.octet[3], tempMacAddr.octet[4], tempMacAddr.octet[5]);
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
void Mezcal::SetUniqueMacAddress() NN_NOEXCEPT
{
    struct ether_addr macAddr;
    nn::settings::factory::MacAddress wirelessUniqueMac;
    // Get unique MAC address and country code
    nn::Result result = nn::settings::factory::GetWirelessLanMacAddress(&wirelessUniqueMac);
    if( result.IsSuccess() )
    {
        WLAN_LOG_INFO("Wireless MAC address written in NAND\n  %02X:%02X:%02X:%02X:%02X:%02X\n",
                wirelessUniqueMac.octets[0], wirelessUniqueMac.octets[1], wirelessUniqueMac.octets[2],
                wirelessUniqueMac.octets[3], wirelessUniqueMac.octets[4], wirelessUniqueMac.octets[5]);
        std::memcpy(&macAddr.octet[0], &wirelessUniqueMac.octets[0], MacAddress::MacAddressSize);

        int bwl_ret = bwl_set_curetheraddr(m_InterfaceHandle, &macAddr);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

        WLAN_LOG_INFO("Set MAC address [%02X:%02X:%02X:%02X:%02X:%02X]\n\n",
                macAddr.octet[0], macAddr.octet[1], macAddr.octet[2], macAddr.octet[3], macAddr.octet[4], macAddr.octet[5]);
    }
    else
    {
#if defined(RANDMAC)
        SetRandomMacAddress();
#endif
    }
}

void Mezcal::SetCountryCode() NN_NOEXCEPT
{
    // Get country code written in NAND.
    int ccCount = 0;
    nn::Result result = nn::settings::factory::GetWirelessLanCountryCodeCount(&ccCount);
    if( result.IsSuccess() )
    {
        nn::settings::factory::CountryCode* pCountryCode;
        pCountryCode = new nn::settings::factory::CountryCode[ccCount];
        NN_ABORT_UNLESS_NOT_NULL(pCountryCode);
        int getCount;
        result = nn::settings::factory::GetWirelessLanCountryCodes(&getCount, pCountryCode, ccCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        wl_country_t drvCountry;
        // 全世界仕様であるカントリーコードで初期化しておく
        std::memcpy(&drvCountry, &(driver::NxCountryCodesEdev[0].drvCountryCode), sizeof(wl_country_t));

        // NxCountryCodesの中で一致したカントリーコードを適用
        bool isCountyCodeMatched = false;
        for( int i = 0; i < driver::CountryCodeCounts; i++ )
        {
            if( std::strcmp(pCountryCode[0].string, driver::NxCountryCodes[i].hwCountryCode.string) == 0 )
            {
                WLAN_LOG_INFO("Region Code:%s\n", pCountryCode[0].string);
                std::memcpy(&drvCountry, &(driver::NxCountryCodes[i].drvCountryCode), sizeof(wl_country_t));
                isCountyCodeMatched = true;
                break;
            }
        }

        // NxCountryCodesのどれとも一致しなければ 製品か開発機かチェック
        if( !isCountyCodeMatched )
        {
            nn::settings::factory::ConfigurationId1 boardType;
            nn::settings::factory::GetConfigurationId1(&boardType);
            WLAN_LOG_INFO("%s\n",boardType.string);
            const int COMPARISON_LENGTH = 4;
            char stringSdev[COMPARISON_LENGTH + 1] = "SDEV";
            if( nn::util::Strncmp(boardType.string, stringSdev, COMPARISON_LENGTH) == 0 )
            {
                WLAN_LOG_INFO("This device is SDEV\n");
                int i = 0;
                for( i = 0; i < driver::DeviceCounts; i++ )
                {
                    if( std::strcmp(pCountryCode[0].string, driver::NxCountryCodesSdev[i].hwCountryCode.string) == 0 )
                    {
                        WLAN_LOG_INFO("Region Code:%s\n", pCountryCode[0].string);
                        std::memcpy(&drvCountry, &(driver::NxCountryCodesSdev[i].drvCountryCode), sizeof(wl_country_t));
                        break;
                    }
                }
                if( i >= driver::DeviceCounts )
                {
                    WLAN_LOG_ERROR("Unknown region code:%s\n", pCountryCode[0].string);
                    WLAN_LOG_ERROR("Set SDEV MP country code.\n");
                    std::memcpy(&drvCountry, &(driver::NxCountryCodesSdev[0].drvCountryCode), sizeof(wl_country_t));
                }
            }
            else
            {
                WLAN_LOG_INFO("This device is not SDEV\n");
                int i = 0;
                for( i = 0; i < driver::DeviceCounts; i++ )
                {
                    if( std::strcmp(pCountryCode[0].string, driver::NxCountryCodesEdev[i].hwCountryCode.string) == 0 )
                    {
                        WLAN_LOG_INFO("Region Code:%s\n", pCountryCode[0].string);
                        std::memcpy(&drvCountry, &(driver::NxCountryCodesEdev[i].drvCountryCode), sizeof(wl_country_t));
                        break;
                    }
                }
                if( i >= driver::DeviceCounts )
                {
                    WLAN_LOG_ERROR("Unknown region code:%s\n", pCountryCode[0].string);
                    WLAN_LOG_ERROR("Set EDEV MP country code.\n");
                    std::memcpy(&drvCountry, &(driver::NxCountryCodesEdev[0].drvCountryCode), sizeof(wl_country_t));
                }
            }
        }
        delete[] pCountryCode;

        int bwl_ret = bwl_country(m_InterfaceHandle,&drvCountry, true);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

        if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Info )
        {
            // Country Code
            NN_SDK_LOG("Set the country code: %s/%d\n",
                    drvCountry.country_abbrev, drvCountry.rev);

            // Print out allowed channels.
            wl_uint32_list_t* chList;
            uint32_t chBuf[WL_NUMCHANNELS + 1];
            chList = reinterpret_cast<wl_uint32_list_t*>(chBuf);
            chList->count = WL_NUMCHANNELS;
            bwl_ret = wlu_get(m_InterfaceHandle, WLC_GET_VALID_CHANNELS, chBuf, sizeof(chBuf));
            WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
            NN_SDK_LOG(" --- Allowed channels ---\n");
            for( int i = 0; i < chList->count; i++ )
            {
                NN_SDK_LOG("%d ", chList->element[i]);
            }
            NN_SDK_LOG("\n");
        }
    }
}
#endif

/* 初期化コマンド作成 ------------------------------------------------------ */
bool Mezcal::InitializeDriver() NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s enter\n", __FUNCTION__);

    /* ドライバ初期化とコールバック関数の一括登録 */
    bwl_ret = bwl_dhd_initialize(&m_InterfaceHandle);
    if( bwl_ret != BCME_OK )
    {
        WLAN_LOG_ERROR("Dhd initialization failed! (%d)\n\n", bwl_ret);
        return false;
    }

    dhd_register_event_cb(m_InterfaceHandle,
                                    EventCallBackFromDriverInfraMode);  // just for wpa supplicant initialization
    dhd_register_rxdata_cb(m_InterfaceHandle,
                                     RxCallBackFromDriverInfraMode);  // just for wpa supplicant initialization
    dhd_register_txcompletion_cb(m_InterfaceHandle,
                                 TxCompletionCallBackFromDriver);

    // interface down
    bwl_ret = bwl_down(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set own MAC address
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    SetUniqueMacAddress();
    SetCountryCode();
#elif defined(RANDMAC)
    SetRandomMacAddress();
#endif

    // Get initial mac address
    bwl_ret = bwl_get_curetheraddr(m_InterfaceHandle, &m_OwnMacAddr);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Get default eventmask
    bwl_ret = bwl_get_eventmask(m_InterfaceHandle, m_DefaultEventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set wake reason for wowl (default value)
    m_wakeReasonForSet = (WL_WOWL_DIS | WL_WOWL_BCN | WL_WOWL_GTK_FAILURE
#if defined(USE_WOWL_WAKEPATTERN)
            | WL_WOWL_TCPKEEP_DATA
#endif
            );
    WLAN_LOG_INFO("Default wake reason:0x%08X\n", m_wakeReasonForSet);
    // Set flag for wowl features
    m_wowlFeatures = (WowlFeatures_ArpOffload | WowlFeatures_TcpKeepAlive);

    // FW ver
    if( WlanLogLevel >= LogLevel_Info )
    {
        char str[WLC_IOCTL_SMLEN];
        wlu_iovar_get(m_InterfaceHandle, "ver", str, WLC_IOCTL_SMLEN);
        NN_SDK_LOG("[Wireless firmware ver]\n%s\n", str);
    }

    WLAN_LOG_DEBUG("[MEZ] %s leave\n", __FUNCTION__);

    return true;
}

/* ドライバ終了処理 -------------------------------------------------------- */
void Mezcal::FinalizeDriver() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s enter\n", __FUNCTION__);

    bwl_dhd_del_all_sta_force(m_InterfaceHandle);

    bwl_dhd_deinitialize(m_InterfaceHandle);

    WLAN_LOG_DEBUG("[MEZ] %s leave\n", __FUNCTION__);
}

void Mezcal::InitializeEventmask(uint32_t mode) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s enter\n", __FUNCTION__);

    // Set eventmask
    uint8_t eventmask[WL_EVENTING_MASK_LEN];
    if( mode == WlanModeDetect )
    {
        std::memset(eventmask, 0, WL_EVENTING_MASK_LEN);
        setbit(eventmask, WLC_E_IF);
        setbit(eventmask, WLC_E_ACTION_FRAME_RX);
        setbit(eventmask, WLC_E_ESCAN_RESULT);
    }
    else
    {
        std::memcpy(eventmask, m_DefaultEventmask, WL_EVENTING_MASK_LEN);
        setbit(eventmask, WLC_E_DEAUTH_IND);
        clrbit(eventmask, WLC_E_ROAM);
        clrbit(eventmask, WLC_E_ACTION_FRAME_COMPLETE);
        clrbit(eventmask, WLC_E_ACTION_FRAME);
    }

    bwl_ret = bwl_set_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
}

/* InfraとLocal共通のパラメーターをセット ---------------------------------------------- */
void Mezcal::SetCommonParams() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = bwl_down(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    bwl_ret = bwl_set_mpc(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // modeに関わらずinframodeは常にInfraBSSモード。
    bwl_ret = bwl_set_inframode(m_InterfaceHandle, 1);  //0がIBSS, 1がInfraBSS
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set action frame subtype
    bwl_ret = bwl_clear_afsubtypes(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    for( int i = 0; i < 3; i++ )
    {
        bwl_ret = bwl_add_afsubtype(m_InterfaceHandle, 4);
        // (SIGLO-70145)返り値がBCME_NOMEMの場合、リトライで成功する可能性があるので最大3回リトライする
        if( bwl_ret != BCME_NOMEM )
        {
            break;
        }
    }
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    for( int i = 0; i < 3; i++ )
    {
        bwl_ret = bwl_add_afsubtype(m_InterfaceHandle, 5);
        // (SIGLO-70145)返り値がBCME_NOMEMの場合、リトライで成功する可能性があるので最大3回リトライする
        if( bwl_ret != BCME_NOMEM )
        {
            break;
        }
    }
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set action frame oui filter // TODO 上位層から指定できるようにする
    uint8_t ouibuf[] = {0x00, 0x22, 0xAA};
    bwl_ret = bwl_nsc_subtypeoui(m_InterfaceHandle, ouibuf, sizeof(ouibuf), true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 全通信モードの全データ通信設定においてLDPCを無効化しておく
    {
        bwl_rate_info_t gRateInfo = {0};
        std::memset(&gRateInfo, 0, sizeof(bwl_rate_info_t));
        gRateInfo.auto_set = true;
        gRateInfo.legacy_rate = -1;
        gRateInfo.htmcsindex = -1;
        gRateInfo.vhtmcsindex = -1;
        bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, false, BWL_FRAME_TYPE_DATA, &gRateInfo);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
        bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, true, BWL_FRAME_TYPE_DATA, &gRateInfo);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
        bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_DATA, &gRateInfo);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
        bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, true, BWL_FRAME_TYPE_DATA, &gRateInfo);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

        int var = 0;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ldpc_cap", &var, sizeof(int));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Set band
    bwl_ret = bwl_set_band(m_InterfaceHandle, WLC_BAND_AUTO);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set beacon lost timeout to 10sec
    int bcnTimeout = 10;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "bcn_timeout", &bcnTimeout, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set static aes mode to off
    bwl_ret = bwl_set_wsec(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    int wpaAuth = 0;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpa_auth", &wpaAuth, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // サポートするバンド幅を 2.4G/5G 共に20MHz のみと設定しておく
    struct {
        uint32 band;
        uint32 bw_cap;
    } param = {0, 0};

    param.band = WLC_BAND_2G;
    param.bw_cap = 0x1;

    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "bw_cap", &param, sizeof(param));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    param.band = WLC_BAND_5G;
    param.bw_cap = 0x1;

    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "bw_cap", &param, sizeof(param));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Disable ROAM function because our platform does not use it.
    int roam = 1;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "roam_off", &roam, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
} //NOLINT(impl/function_size)

/* Infraのパラメーターをセット ---------------------------------------------- */
void Mezcal::SetInfraParams() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t bwl_ret;

    // コールバック関数登録
    bwl_ret = dhd_register_event_cb(m_InterfaceHandle,
                                    EventCallBackFromDriverInfraMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    bwl_ret = dhd_register_rxdata_cb(m_InterfaceHandle,
                                     RxCallBackFromDriverInfraMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // APモードのセット
    bwl_ret = bwl_set_apmode(m_InterfaceHandle, 0);  // 0: STAモード
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // DS3モードのセット
    bwl_ret = bwl_set_ds3mode(m_InterfaceHandle, 0); // 0: normalモード
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // rate設定用パラメータ
    bwl_rate_info_t rateinfo = {0};
    // インフラ通信モードではBroadcastの通信レートはautoにしておく(2.4, 5GHzとも)
    std::memset(&rateinfo, 0, sizeof(bwl_rate_info_t));
    rateinfo.auto_set = true;
    rateinfo.legacy_rate = -1;
    rateinfo.htmcsindex = -1;
    rateinfo.vhtmcsindex = -1;

    // 2.4GHz broadcast通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 5GHz broadcast通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 送受信アンテナ設定
    int rxChain = 3;
    int txChain = 3;
    // Set txchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &txChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set rxchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &rxChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set ampdu setting
    // For infra mode, use default settings (enable ampdu for all tid)
    for( int i = 0; i < 8; i++ )
    {
        struct ampdu_tid_control atc;
        atc.tid = i;
        atc.enable = 1;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ampdu_tid", &atc, sizeof(atc));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Legacy power save
    wl_pm2_sleep_ret_ext_t sleep_ret_ext;
    sleep_ret_ext.logic                = WL_DFRTS_LOGIC_AND; /* DFRTS logic: see WL_DFRTS_LOGIC_* below */
    sleep_ret_ext.low_ms               = 2000;               /* Low FRTS timeout */
    sleep_ret_ext.high_ms              = 5000;               /* High FRTS timeout */
    sleep_ret_ext.rx_pkts_threshold    = 10;                 /* switching threshold: # rx pkts */
    sleep_ret_ext.tx_pkts_threshold    = 0;                  /* switching threshold: # tx pkts */
    sleep_ret_ext.txrx_pkts_threshold  = 0;                  /* switching threshold: # (tx+rx) pkts */
    sleep_ret_ext.rx_bytes_threshold   = 0;                  /* switching threshold: # rx bytes */
    sleep_ret_ext.tx_bytes_threshold   = 0;                  /* switching threshold: # tx bytes */
    sleep_ret_ext.txrx_bytes_threshold = 0;                  /* switching threshold: # (tx+rx) bytes */
    bwl_ret = bwl_set_pm2_sleep_ret_ext(m_InterfaceHandle, &sleep_ret_ext);
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 2);

    // Remove wpa ie
    int wpaie = 0;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpaie", &wpaie, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set vht mode
    bwl_ret = bwl_set_vhtmode(m_InterfaceHandle, 1);  // VHT mode enabled
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // set mpc to 1
    bwl_ret = bwl_set_mpc(m_InterfaceHandle, 1);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set bphy and ofdm rx desense level cap
    int cap = 20;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_bphy", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_ofdm", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Enable the dynamic coex desense
    bwl_ret = bwl_radio_tune_onoff(m_InterfaceHandle, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

} //NOLINT(impl/function_size)

/* Localのパラメーターをセット ---------------------------------------------- */
void Mezcal::SetLocalParams(uint32_t mode) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = dhd_register_event_cb(m_InterfaceHandle,
                                    EventCallBackFromDriverLocalMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    bwl_ret = dhd_register_rxdata_cb(m_InterfaceHandle,
                                     RxCallBackFromDriverLocalMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set power management to constant mode
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // rate設定用パラメータ
    bwl_rate_info_t rateinfo = {0};
    std::memset(&rateinfo, 0, sizeof(bwl_rate_info_t));

    // APモード/DS3モード設定用変数
    int apmode = 0;  // AP mode
    int ds3mode = 0; // DS3 mode

    /* mode チェック */
    if(mode & WlanModeLocalAp)
    {
        // APモードのセット。1がAP。
        apmode = 1;
        // DS3モードのセット。2がAPにおいてSTAからのローカルブロードキャストを受け取れるモード。
        ds3mode = 2;
        // ローカル通信モードではBroadcastの通信レートを11n MCS3に固定しておく
        rateinfo.auto_set = false;
        rateinfo.legacy_rate = -1;
        rateinfo.htmcsindex = 3;
        rateinfo.vhtmcsindex = -1;
        rateinfo.sgi = true;
        rateinfo.bw_val = WL_RSPEC_BW_20MHZ;
    }
    else if(mode & WlanModeLocalClient)
    {
        // APモードのセット。0がSTA。
        apmode = 0;
        // DS3モードのセット。1がSTAにおいてローカルブロードキャストの送受信が出来るモード。
        ds3mode = 1;
        // ローカル通信モードではBroadcastの通信レートを11n MCS3に固定しておく
        rateinfo.auto_set = false;
        rateinfo.legacy_rate = -1;
        rateinfo.htmcsindex = 3;
        rateinfo.vhtmcsindex = -1;
        rateinfo.sgi = true;
        rateinfo.bw_val = WL_RSPEC_BW_20MHZ;
    }
    else if(mode & WlanModeLocalSpectator)
    {
        // APモードのセット。0がSTA。
        apmode = 0;
        // DS3モードのセット。4がSpectatorモード。
        ds3mode = 4;
        // SpectatorモードはBroadcast送信しないので、通信レートはautoにしておく
        rateinfo.auto_set = true;
        rateinfo.legacy_rate = -1;
        rateinfo.htmcsindex = -1;
        rateinfo.vhtmcsindex = -1;
    }

    // APモードのセット
    bwl_ret = bwl_set_apmode(m_InterfaceHandle, apmode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // DS3モードのセット
    bwl_ret = bwl_set_ds3mode(m_InterfaceHandle, ds3mode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    if( apmode == 1 )
    {
        // Set the max association number to 7
        int maxAssocCnt = 7;
        wlu_iovar_set(m_InterfaceHandle, "maxassoc", &maxAssocCnt, sizeof(int)); // set
    }
    else
    {
        // Remove wpa ie
        int wpaie = 0;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpaie", &wpaie, sizeof(int));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // 2.4GHz ブロードキャスト通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 5GHz ブロードキャスト通信レートのセット(5GHzでも11n MCS3レートを使用するようにしておく)
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    int rxChain = 2;
    int txChain = 2;
    // Set txchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &txChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    // Set rxchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &rxChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set ampdu setting
    // For local mode, disable ampdu for all tid
    for( int i = 0; i < 8; i++ )
    {
        struct ampdu_tid_control atc;
        atc.tid = i;
        atc.enable = 0;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ampdu_tid", &atc, sizeof(atc));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Set action frame phy rate
    bwl_rate_info_t afRateInfo = {0};
    std::memset(&afRateInfo, 0, sizeof(bwl_rate_info_t));
    afRateInfo.auto_set = false;
    afRateInfo.legacy_rate = -1;
    afRateInfo.htmcsindex = 3;
    afRateInfo.vhtmcsindex = -1;
    afRateInfo.sgi = true;
    afRateInfo.bw_val = WL_RSPEC_BW_20MHZ;
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_AF, &afRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set vht mode
    bwl_ret = bwl_set_vhtmode(m_InterfaceHandle, 0);  // VHT mode disabled
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set bphy and ofdm rx desense level cap
    int cap = 5;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_bphy", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_ofdm", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // set mpc to 0
    bwl_ret = bwl_set_mpc(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Disable the dynamic coex desense
    bwl_ret = bwl_radio_tune_onoff(m_InterfaceHandle, false);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

} //NOLINT(impl/function_size)

/* Localのパラメーターをセット ---------------------------------------------- */
void Mezcal::SetLocalLcsParams(uint32_t mode) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = dhd_register_event_cb(m_InterfaceHandle,
                                    EventCallBackFromDriverLocalMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    bwl_ret = dhd_register_rxdata_cb(m_InterfaceHandle,
                                     RxCallBackFromDriverLocalMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set power management to constant mode
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // rate設定用パラメータ
    bwl_rate_info_t rateinfo = {0};
    std::memset(&rateinfo, 0, sizeof(bwl_rate_info_t));

    // APモード/DS3モード設定用変数
    int apmode = 0;  // AP mode
    int ds3mode = 0; // DS3 mode

    /* mode チェック */
    if(mode & WlanModeLocalLcsAp)
    {
        // APモードのセット。1がAP。
        apmode = 1;
        // DS3モードのセット。2がAPにおいてSTAからのローカルブロードキャストを受け取れるモード。
        ds3mode = 2;
        // ローカル通信モードではBroadcastの通信レートを11n MCS3に固定しておく
        rateinfo.auto_set = false;
        rateinfo.legacy_rate = -1;
        rateinfo.htmcsindex = 3;
        rateinfo.vhtmcsindex = -1;
        rateinfo.sgi = true;
        rateinfo.bw_val = WL_RSPEC_BW_20MHZ;
    }
    else if(mode & WlanModeLocalLcsClient)
    {
        // APモードのセット。0がSTA。
        apmode = 0;
        // DS3モードのセット。1がSTAにおいてローカルブロードキャストの送受信が出来るモード。
        ds3mode = 1;
        // ローカル通信モードではBroadcastの通信レートを11n MCS3に固定しておく
        rateinfo.auto_set = false;
        rateinfo.legacy_rate = -1;
        rateinfo.htmcsindex = 3;
        rateinfo.vhtmcsindex = -1;
        rateinfo.sgi = true;
        rateinfo.bw_val = WL_RSPEC_BW_20MHZ;
    }
    else
    {
        NN_SDK_ASSERT(false, "%s: invalid mode value(0x%08X).\n", __FUNCTION__, mode);
    }

    // APモードのセット
    bwl_ret = bwl_set_apmode(m_InterfaceHandle, apmode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // DS3モードのセット
    bwl_ret = bwl_set_ds3mode(m_InterfaceHandle, ds3mode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    if( apmode == 1 )
    {
        // Set the max association number to 7
        int maxAssocCnt = 7;
        wlu_iovar_set(m_InterfaceHandle, "maxassoc", &maxAssocCnt, sizeof(int)); // set
    }
    else
    {
        // Remove wpa ie
        int wpaie = 0;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpaie", &wpaie, sizeof(int));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // 2.4GHz ブロードキャスト通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 5GHz ブロードキャスト通信レートのセット(5GHzでも11n MCS3レートを使用するようにしておく)
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // LCSモードの場合、アンテナは2本使用する
    int rxChain = 3;
    int txChain = 3;
    // Set txchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &txChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    // Set rxchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &rxChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set ampdu setting
    // For local mode, disable ampdu for all tid
    for( int i = 0; i < 8; i++ )
    {
        struct ampdu_tid_control atc;
        atc.tid = i;
        atc.enable = 1;  // LCSモードではA-MPDU有効
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ampdu_tid", &atc, sizeof(atc));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Set action frame phy rate
    bwl_rate_info_t afRateInfo = {0};
    std::memset(&afRateInfo, 0, sizeof(bwl_rate_info_t));
    afRateInfo.auto_set = false;
    afRateInfo.legacy_rate = -1;
    afRateInfo.htmcsindex = 3;
    afRateInfo.vhtmcsindex = -1;
    afRateInfo.sgi = true;
    afRateInfo.bw_val = WL_RSPEC_BW_20MHZ;
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_AF, &afRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set vht mode
    bwl_ret = bwl_set_vhtmode(m_InterfaceHandle, 0);  // VHT mode disabled
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set bphy and ofdm rx desense level cap
    int cap = 5;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_bphy", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_ofdm", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // set mpc to 0
    bwl_ret = bwl_set_mpc(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Disable the dynamic coex desense
    bwl_ret = bwl_radio_tune_onoff(m_InterfaceHandle, false);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    m_IsLcsMode = true;

} //NOLINT(impl/function_size)

void Mezcal::SetDetectParams() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    // コールバック関数登録
    bwl_ret = dhd_register_event_cb(m_InterfaceHandle,
                                    EventCallBackFromDriverDetectMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    bwl_ret = dhd_register_rxdata_cb(m_InterfaceHandle,
                                     RxCallBackFromDriverDetectMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // APモードのセット
    bwl_ret = bwl_set_apmode(m_InterfaceHandle, 0);  // 0: STAモード
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // DS3モードのセット
    bwl_ret = bwl_set_ds3mode(m_InterfaceHandle, 0); // 0: normalモード
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // rate設定用パラメータ
    bwl_rate_info_t rateinfo = {0};
    std::memset(&rateinfo, 0, sizeof(bwl_rate_info_t));
    rateinfo.auto_set = true;
    rateinfo.legacy_rate = -1;
    rateinfo.htmcsindex = -1;
    rateinfo.vhtmcsindex = -1;

    // 2.4GHz broadcast通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 5GHz broadcast通信レートのセット
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, false, false, BWL_FRAME_TYPE_DATA, &rateinfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // 送受信アンテナ設定
    int rxChain = 1;
    int txChain = 1;
    // Set txchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &txChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    // Set rxchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &rxChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set ampdu setting
    for( int i = 0; i < 8; i++ )
    {
        struct ampdu_tid_control atc;
        atc.tid = i;
        atc.enable = 1;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ampdu_tid", &atc, sizeof(atc));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Remove wpa ie
    int wpaie = 0;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpaie", &wpaie, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set action frame phy rate
    bwl_rate_info_t afRateInfo = {0};
    std::memset(&afRateInfo, 0, sizeof(bwl_rate_info_t));
    afRateInfo.auto_set = false;
    afRateInfo.legacy_rate = -1;
    afRateInfo.htmcsindex = 1;
    afRateInfo.vhtmcsindex = -1;
    afRateInfo.sgi = false;
    afRateInfo.bw_val = WL_RSPEC_BW_20MHZ;
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_AF, &afRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set vht mode
    bwl_ret = bwl_set_vhtmode(m_InterfaceHandle, 0);  // VHT mode disabled
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set bphy and ofdm rx desense level cap
    int cap = 5;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_bphy", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_ofdm", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // set mpc to 0
    bwl_ret = bwl_set_mpc(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Disable the dynamic coex desense
    bwl_ret = bwl_radio_tune_onoff(m_InterfaceHandle, false);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set action frame subtype
    bwl_ret = bwl_clear_afsubtypes(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    for( int i = 0; i < 3; i++ )
    {
        bwl_ret = bwl_add_afsubtype(m_InterfaceHandle, 7);
        // (SIGLO-70145)返り値がBCME_NOMEMの場合、リトライで成功する可能性があるので最大3回リトライする
        if( bwl_ret != BCME_NOMEM )
        {
            break;
        }
    }
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set channel
    bwl_ret = bwl_set_channel(m_InterfaceHandle, m_DetectHomeCh);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
} //NOLINT(impl/function_size)

/* 通信に関わるパラメーターをリセット ---------------------------------------------- */
void Mezcal::ResetParams() NN_NOEXCEPT
{
    int32_t bwl_ret;
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    SetCommonParams();

    // APモードのセット
    bwl_ret = bwl_set_apmode(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // DS3モードのセット
    bwl_ret = bwl_set_ds3mode(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set power management to constant mode
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set vht mode
    bwl_ret = bwl_set_vhtmode(m_InterfaceHandle, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset the max association number
    int maxAssocCnt = 7;
    wlu_iovar_set(m_InterfaceHandle, "maxassoc", &maxAssocCnt, sizeof(int)); // set

    int rxChain = 3;
    int txChain = 3;
    // Reset txchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &txChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    // Reset rxchain
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &rxChain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset action frame phy rate
    bwl_rate_info_t afRateInfo = {0};
    std::memset(&afRateInfo, 0, sizeof(bwl_rate_info_t));
    afRateInfo.auto_set = true;
    afRateInfo.legacy_rate = -1;
    afRateInfo.htmcsindex = -1;
    afRateInfo.vhtmcsindex = -1;
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_AF, &afRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset ampdu setting
    for( int i = 0; i < 8; i++ )
    {
        struct ampdu_tid_control atc;
        atc.tid = i;
        atc.enable = 1;
        bwl_ret = wlu_iovar_set(m_InterfaceHandle, "ampdu_tid", &atc, sizeof(atc));
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Remove wpa ie
    int wpaie = 0;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "wpaie", &wpaie, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset desense level cap
    int cap = 90;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_bphy", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    cap = 48;
    bwl_ret = wlu_iovar_set(m_InterfaceHandle, "aci_max_desense_ofdm", &cap, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Enable the dynamic coex desense
    bwl_ret = bwl_radio_tune_onoff(m_InterfaceHandle, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    m_IsLcsMode = false;

} //NOLINT(impl/function_size)

/* ドライバの動作モードを変更 ---------------------------------------------- */
bool Mezcal::ChangeWirelessMode(uint32_t mode) NN_NOEXCEPT
{
    /*
     * 動作モードの指定には3種のパラメータのセットが必要。
     * IBSSかInfraBSSかを指定するInframode。
     * APかSTAかを指定するApmode。
     * そのモードにおける特定動作を可能にするためのds3mode。
     */

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_DEBUG("[MEZ] Mode = %lx\n", mode);

#ifdef ENABLE_LOCAL_TXFLOW_CTRL
    // Reset tx count
    m_TxCount = 0;
#endif

    // Infra,Local共通のパラメータをセット
    SetCommonParams();

    if( mode == WlanModeInfraSta )
    {
        SetInfraParams();
    }
    else if( mode == WlanModeLocalAp ||
            mode == WlanModeLocalClient )
    {
        SetLocalParams(mode);
    }
    else if( mode == WlanModeLocalLcsAp ||
            mode == WlanModeLocalLcsClient )
    {
        SetLocalLcsParams(mode);
    }
    else if( mode == WlanModeDetect )
    {
        SetDetectParams();
    }
    else
    {
        NN_ABORT("%s: mode is unexpected value\n");
    }

    return true;
} //NOLINT(impl/function_size)

/* Interface のUp/Down ----------------------------------------------------- */
bool Mezcal::SetIfUpDown(uint32_t up_down) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    if(up_down == 0)
    {
#ifdef ENABLE_DCS_OPERATION
        if( m_bDcsEnabled == true )
        {
            acsd_end();
            m_bDcsEnabled = false;
        }
#endif
        m_bWaitIfUpEvent = false;
        bwl_ret = bwl_down(m_InterfaceHandle);
    }
    else
    {
        m_bWaitIfUpEvent = true;
        bwl_ret = bwl_up(m_InterfaceHandle);
    }

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

/* Interface の Up/Down 状態を取得する ------------------------------------- */
bool Mezcal::GetIfUpDown(uint32_t* p_up_down) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    bwl_ret = bwl_isup(m_InterfaceHandle, p_up_down);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* イベントマスクの設定 ---------------------------------------------------- */
bool Mezcal::SetEventMask(uint8_t mask) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s[%x]\n", __FUNCTION__, mask);

    bwl_ret = bwl_set_eventmask(m_InterfaceHandle,  &mask);
    WLAN_LOG_DEBUG("[MEZ] retval(bwl_set_eventmask) = %d\n", bwl_ret);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* イベントマスクのクリア ---------------------------------------------------- */
bool Mezcal::ClearEventMask() NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    uint8_t eventmask[WL_EVENTING_MASK_LEN];
    std::memset(eventmask, 0, sizeof(eventmask));
    bwl_ret = bwl_set_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* イベントマスクの取得 ---------------------------------------------------- */
bool Mezcal::GetEventMask(uint8_t* pmask) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    bwl_ret = bwl_get_eventmask(m_InterfaceHandle, pmask);
    WLAN_LOG_DEBUG("[MEZ] retval(bwl_get_eventmask) = %d\n", bwl_ret);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

bool Mezcal::SetMacAddress(const WlanMacAddressData* mac) NN_NOEXCEPT
{
    NN_SDK_LOG("WDMMZ: %s\n", __FUNCTION__);

    int32_t bwl_ret;

    std::memcpy(&m_OwnMacAddr.octet[0], &mac->addr[0], ETHER_ADDR_LEN);

    bwl_ret = bwl_set_curetheraddr(m_InterfaceHandle, &m_OwnMacAddr);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

/* APの起動設定 ------------------------------------------------------------ */
bool Mezcal::SetApConfigration(WlanLocalBssConfiguration* pbssinfo) NN_NOEXCEPT
{
    int32_t bwl_ret;
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // WLC_E_DEAUTHを取得出来るようにしておく (SIGLO-41209の問題のため)
    // CLIENTにDeauthが届かなかった場合、WLC_E_DEAUTHで知らせてくれる。
    // DHDではこれを受けてSTA情報の削除を行う。
    // Deauthが届いていた場合はWLC_E_DISASSOC_INDが発生し、それを受けてSTA情報の削除が行われる。
    uint8_t eventmask[WL_EVENTING_MASK_LEN];
    bwl_ret = bwl_get_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    setbit(eventmask, WLC_E_DEAUTH);
    bwl_ret = bwl_set_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // bandの設定およびチャンネルの設定
    if( pbssinfo->channel <= WirelessChannel_13ch )
    {
        bwl_set_band(m_InterfaceHandle, WLC_BAND_2G);
        if( pbssinfo->channel != -1 )
        {
            bwl_set_channel(m_InterfaceHandle, pbssinfo->channel);
        }
    }
    else
    {
        bwl_set_band(m_InterfaceHandle, WLC_BAND_5G);
        bwl_set_channel(m_InterfaceHandle, pbssinfo->channel);
    }

    // ステルスSSIDの設定
    WLAN_LOG_DEBUG("[MEZ]set hidden/unhidden SSID\n");
    int32_t closed;
    if( pbssinfo->hiddenSsid == true )
    {
        closed = 1;
    }
    else
    {
        closed = 0;
    }
    bwl_ret = wlu_set(m_InterfaceHandle, WLC_SET_CLOSED, &closed, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::ExecuteAcsd() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int ret;

    ret = acsd_main();
    if( ret != ACSD_OK )
    {
        WLAN_LOG_ERROR("acsd_main() failed (%d)\n", ret);
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAcsdError());
    }
#ifdef ENABLE_DCS_OPERATION
    m_bDcsEnabled = true;
#endif

    return true;
}

bool Mezcal::CreateAp(WlanSsidInfo* pSsidInfo) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t bwl_ret;

    // change beacon phy rate
    bwl_rate_info_t bcnRateInfo = {0};
    bcnRateInfo.auto_set = false;
    bcnRateInfo.legacy_rate = 48;  // 24Mbps in 500kbps units
#ifndef CHANGE_BASIC_RATE_SET_OFDM
    uint32_t curCh;
    bwl_get_channel(m_InterfaceHandle, &curCh);
    if( curCh <= WirelessChannel_13ch )
    {
        bcnRateInfo.legacy_rate = 22;  // 11Mbps in 500kbps units
    }
#endif
    bcnRateInfo.htmcsindex = -1;
    bcnRateInfo.vhtmcsindex = -1;
    bwl_ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_BCN, &bcnRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Initialize ClientStatus before starting Ap
    m_ClientStatusManager.Initialize();

    // Coex MC2有効化（LCSモードでは無い場合）
    if( m_IsLcsMode == false )
    {
        bwl_ret = bwl_enable_mc2(m_InterfaceHandle);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }
    else
    {
        // CWの変更 LCSモード用
        SetCwParam(AccessCategoryIndex_BE, LCS_AC_BE_CWMIN, LCS_AC_BE_CWMAX);
        WLAN_LOG_INFO("%s: Change CW value\n", __FUNCTION__);
        // ACI mitigation値をLCS用のものにセット
        SetAciMitigation(AciMitigation_LocalLcs);
    }

    // Coex desense levelの固定化
    SetBtcDesenseLevel(0);

    // CreateBss
    bwl_ret = bwl_create(m_InterfaceHandle,
                         reinterpret_cast<uint8_t*>(pSsidInfo->ssid),
                         pSsidInfo->length);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    if( GetAcsdFlag() == true && WlanLogLevel >= LogLevel_Debug )
    {
        uint8_t* pChannelStatsBuf = new uint8_t[1024 * 16];
        uint8_t* pChannelScoresBuf = new uint8_t[256];
        NN_SDK_REQUIRES_NOT_NULL(pChannelStatsBuf);
        NN_SDK_REQUIRES_NOT_NULL(pChannelScoresBuf);

        bwl_ret = acsd_get_chanim(&m_OwnMacAddr.octet[0],
                reinterpret_cast<void*>(pChannelStatsBuf), reinterpret_cast<void*>(pChannelScoresBuf));
        if( bwl_ret != ACSD_OK )
        {
            WLAN_LOG_ERROR("acsd_get_chanim() failed (%d)\n", bwl_ret);
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAcsdError());
        }

        //DumpAcsdResults(reinterpret_cast<wl_chanim_stats_t*>(pChannelStatsBuf), reinterpret_cast<chan_score_t*>(pChannelScoresBuf));
        delete[] pChannelScoresBuf;
        delete[] pChannelStatsBuf;
    }

    if( GetAcsdFlag() == true )
    {
        WLAN_LOG_DEBUG("Finalize acsd function.\n");
#ifndef ENABLE_DCS_OPERATION
        acsd_end();
#endif
    }

    return true;
}

bool Mezcal::DestroyAp() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int ret;

    // MC2機能の無効化
    if( m_IsLcsMode == false )
    {
        ret = bwl_disable_mc2(m_InterfaceHandle);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);
    }
    else
    {
        // CWを戻す
        SetCwParam(AccessCategoryIndex_BE, DEFAULT_AC_BE_CWMIN, DEFAULT_AC_BE_CWMAX);
        WLAN_LOG_INFO("%s: Revert CW value\n", __FUNCTION__);
        // ACI mitigation値を戻す
        SetAciMitigation(AciMitigation_Default);
    }

    // Coex desense levelの初期化
    SetBtcDesenseLevel(0);

    // 一旦SSIDを設定しないとrate setがBAND帯に則したものに再設定されないので
    // わざと空のSSIDをセットしておく
    wlc_ssid_t ssid = {
            0,
            ""
    };
    ret = wlu_set(m_InterfaceHandle, WLC_SET_SSID, &ssid, sizeof(wlc_ssid_t));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    ret = bwl_down(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    ret = bwl_set_band(m_InterfaceHandle, WLC_BAND_2G);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    ret = bwl_set_channel(m_InterfaceHandle, 1);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    // Set beacon phy rate to default
    bwl_rate_info_t bcnRateInfo = {0};
    bcnRateInfo.auto_set = true;
    bcnRateInfo.legacy_rate = -1;
    bcnRateInfo.htmcsindex = -1;
    bcnRateInfo.vhtmcsindex = -1;
    ret = bwl_set_fixedrate(m_InterfaceHandle, true, false, BWL_FRAME_TYPE_BCN, &bcnRateInfo);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    // event maskの初期化
    InitializeEventmask(WlanBaseFunctions::WlanModeLocalAp);

    // DHDがClient情報を全て削除したかどうかチェック。
    // 正常であれば全て削除しきっているはず。そうでなかったら問題なのでASSERTを仕掛けておく。
    // 漏れがあったとしても少量のメモリリークであり、通信に影響はしばらく出ないのでABORTにはしない。
    // また、本体スリープ時に漏れの分も完全に削除される。
    if( dhd_total_sta(m_InterfaceHandle) != 0 )
    {
        NN_SDK_ASSERT(false, "Dhd fails to remove sta infomation list completely.\n");
    }

    return true;
}

bool Mezcal::SetStaticAesMode(bool mode) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // StaticAESモードの設定
    bwl_ret = bwl_static_aesmode(m_InterfaceHandle, mode);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

bool Mezcal::SetStaticAesKey(WlanStaticAesInfo* pKeyInfo) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    NN_SDK_ASSERT_NOT_NULL(pKeyInfo);

    ether_addr addr;

    std::memcpy(&addr.octet[0], &pKeyInfo->peerMacAddr[0], ETHER_ADDR_LEN);
    bwl_ret = bwl_setaeskey(m_InterfaceHandle, pKeyInfo->keyMode, pKeyInfo->key, &addr);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

bool Mezcal::SetClientTimeoutPeriod(uint32_t time) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    // Clientに対する自動切断時間の設定(APモード専用)

    NN_SDK_REQUIRES_LESS_EQUAL(time, 180);
    wl_scb_probe_t scbProbe;
    scbProbe.scb_timeout = (time == 0) ? 100000 : time;  // timeに0が指定されていた場合は、大きな値を入れることで実質的に本機能の無効化を行う。
    scbProbe.scb_activity_time = 3;  // ドライバのdefault値は5秒
    scbProbe.scb_max_probe = 6;     // ドライバのdefault値は10発
    int bwl_ret = bwl_set_activity_time(m_InterfaceHandle, &scbProbe);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::SetBeaconLostTimeout(int time) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    NN_SDK_REQUIRES_RANGE(time, 1, 31);

    int ret = wlu_iovar_set(m_InterfaceHandle, "bcn_timeout", &time, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    return true;
}

bool Mezcal::SetTxChain(int chain) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_RANGE(chain, RxAntennaPattern_0, RxAntennaPattern_Both + 1);

    int ret = wlu_iovar_set(m_InterfaceHandle, "txchain", &chain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    return true;
}

bool Mezcal::SetRxChain(int chain) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_RANGE(chain, RxAntennaPattern_0, RxAntennaPattern_Both + 1);

    int ret = wlu_iovar_set(m_InterfaceHandle, "rxchain", &chain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    return true;
}

bool Mezcal::SetMaxAssociationNumber(int num) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    if( num < 1 || num > ConnectableClientsCountMax )
    {
        NN_SDK_REQUIRES(false, "Max association number must be in 1 ~ %d", ConnectableClientsCountMax);
        return false;
    }

    int ret = wlu_iovar_set(m_InterfaceHandle, "maxassoc", &num, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    return true;
}

/* Scan 開始 --------------------------------------------------------------- */
bool Mezcal::ScanRequest(WlanScanParameters* scan_params) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    uint32_t scanType;
    int32_t activeTime;
    int32_t passiveTime;

    if(scan_params->scanType == nn::wlan::ScanType_Active)
    {
        scanType = 0;
        activeTime = scan_params->channelScanTime;
        passiveTime = 0;
    }
    else
    {
        scanType = 1;
        activeTime = 0;
        passiveTime = scan_params->channelScanTime;
    }

    struct ether_addr* pBssid;
    struct ether_addr bssid;
    Bit8 broadcastMac[] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

    if( std::memcmp(&scan_params->bssid[0], &broadcastMac[0], nn::wlan::MacAddress::MacAddressSize) == 0 )
    {
        pBssid = NULL;
    }
    else
    {
        std::memcpy(&bssid.octet[0], &scan_params->bssid[0], nn::wlan::MacAddress::MacAddressSize);
        pBssid = &bssid;
    }

    struct ssid_list* pSsidList;
    if( scan_params->ssidCount == 0 )
    {
        pSsidList = NULL;
    }
    else
    {
        // Fill the rest space of ssid list with 0
        for( uint8_t i = scan_params->ssidCount; i < nn::wlan::ScanningSsidCountMax; i++ )
        {
            scan_params->ssidList[i].length = 0;
            std::memset(&scan_params->ssidList[i].ssid[0], 0x00, nn::wlan::Ssid::SsidLengthMax);
        }
        pSsidList = reinterpret_cast<struct ssid_list*>(scan_params->ssidList);
    }

    bwl_ret = bwl_advscan(m_InterfaceHandle,                       // I/F
            pSsidList,                                             // ssid list
            static_cast<int32_t>(scan_params->ssidCount),          // ssid count
            pBssid,                                                // bssid
            scanType,                                              // scan type
            -1,                                                    // nprobes (固定運用)
            activeTime,                                            // active time
            passiveTime,                                           // passive time
            scan_params->homeChannelTime,                          // home channel time
            static_cast<int32_t>(scan_params->channelCount),       // channel count
            scan_params->channelList );                            // channel list

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* Scan 中断 --------------------------------------------------------------- */
bool Mezcal::CancelScan() NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    bwl_ret = bwl_advscan_abort( m_InterfaceHandle );

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* 無線ネットワークへの接続 (Infra/LocalClient用) ---------------------------------- */
bool Mezcal::JoinNetworkSta(WlanConnectinoParameters* pConnectParam) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    // 正しいSSIDを指定したうえでのActiveScanをする
    wl_join_params_t join_params;
    uint32_t join_params_size = WL_JOIN_PARAMS_FIXED_SIZE + sizeof(chanspec_t);

    std::memset(&join_params, 0, join_params_size);
    std::memcpy(&join_params.ssid.SSID[0], pConnectParam->ssid.GetSsidData(), pConnectParam->ssid.GetLength());
    join_params.ssid.SSID_len = pConnectParam->ssid.GetLength();
    std::memcpy(&join_params.params.bssid.octet[0], pConnectParam->bssid.GetMacAddressData(), ETHER_ADDR_LEN);
    join_params.params.bssid_cnt = 0;
    // チャンネル指定
    if( pConnectParam->channel != -1 )
    {
        NN_SDK_REQUIRES_RANGE(pConnectParam->channel, WirelessChannel_1ch, WirelessChannel_165ch + 1);
        chanspec_t channel = CH20MHZ_CHSPEC(pConnectParam->channel);
        join_params.params.chanspec_list[0] = channel;
        join_params.params.chanspec_num = 1;
    }
    else
    {
        join_params.params.chanspec_num = 0;
    }

    // Record the ssid on the connection status beforehand
    nn::os::LockMutex(&m_ConnectionStatusMutex);
    m_MezcalConStat.ssid.Set(pConnectParam->ssid.GetSsidData(), pConnectParam->ssid.GetLength());
    nn::os::UnlockMutex(&m_ConnectionStatusMutex);

    bwl_ret = bwl_join(m_InterfaceHandle, &join_params, join_params_size, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

/* 無線ネットワークへの同期 (LocalSpectator用) --------------------------------------- */
bool Mezcal::JoinNetworkSpectator(WlanConnectinoParameters* pConnectParam) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    wl_extjoin_params_t* joinParam = reinterpret_cast<wl_extjoin_params_t*>(nnwlanMallocNormal(sizeof(wl_extjoin_params_t) + sizeof(chanspec_t)));
    uint32_t joinParamSize = sizeof(wl_extjoin_params_t) + sizeof(chanspec_t);
    int passive = 1; // Spectatorは同期を行うだけなのでactiveスキャンは不要

    // 0で初期化
    memset(joinParam, 0, joinParamSize);

    // wlc_ssid_t埋め
    std::memcpy(&joinParam->ssid.SSID[0], pConnectParam->ssid.GetSsidData(), pConnectParam->ssid.GetLength());
    joinParam->ssid.SSID_len = pConnectParam->ssid.GetLength();

    // wl_assoc_param_t埋め
    std::memcpy(&joinParam->assoc.bssid.octet[0], pConnectParam->bssid.GetMacAddressData(), ETHER_ADDR_LEN);
    WLAN_LOG_DEBUG("BSSID %02X:%02X:%02X:%02X:%02X:%02X\n", pConnectParam->bssid.GetMacAddressData()[0],
            pConnectParam->bssid.GetMacAddressData()[1],
            pConnectParam->bssid.GetMacAddressData()[2],
            pConnectParam->bssid.GetMacAddressData()[3],
            pConnectParam->bssid.GetMacAddressData()[4],
            pConnectParam->bssid.GetMacAddressData()[5]);
    joinParam->assoc.bssid_cnt = 1;
    // チャンネル指定
    if( pConnectParam->channel != -1 )
    {
        NN_SDK_REQUIRES_RANGE(pConnectParam->channel, WirelessChannel_1ch, WirelessChannel_165ch + 1);
        chanspec_t channel = CH20MHZ_CHSPEC(pConnectParam->channel);
        joinParam->assoc.chanspec_list[0] = channel;
        joinParam->assoc.chanspec_num = 0;
    }
    else
    {
        joinParam->assoc.chanspec_num = 0;
    }

    // wl_join_scan_param_t埋め
    joinParam->scan.scan_type = WL_SCANFLAGS_PASSIVE;
    joinParam->scan.nprobes = -1;
    joinParam->scan.active_time = -1;
    joinParam->scan.passive_time = 150;  // 150ms固定にしておく (上位層から指定してもらった方が良い？)
    joinParam->scan.home_time = -1;

    bwl_ret = bwl_join(m_InterfaceHandle, joinParam, joinParamSize, passive);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    nnwlanFreeNormal(joinParam);

    return true;
}

/* 無線ネットワークからの離脱 ---------------------------------------------- */
bool Mezcal::Disassociate() NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // 接続中かどうかチェック
    char ssidStr[32];
    int16_t rssi = 0;
    bwl_ret = bwl_get_assocstatus( m_InterfaceHandle, ssidStr, sizeof(ssidStr), &rssi );
    if( bwl_ret == BCME_NOTASSOCIATED )
    {
        // 接続中でない
        WLAN_LOG_DEBUG("[MEZ]Not associated.\n");
        return false;
    }
    else
    {
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    WLAN_LOG_DEBUG("[MEZ]Associated state.\n");

    bwl_ret = bwl_disassoc( m_InterfaceHandle );

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

nn::Result Mezcal::Deauthenticate(WlanDisconnectInfo* deauthParam) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // STAが一台も接続していないとDeauthイベントが発生しないので、まずはSTAが接続しているかチェック
    struct ether_addr cliList[ConnectableClientsCountMax];
    for( uint8_t i = 0; i < ConnectableClientsCountMax; i++ )
    {
        std::memset(&cliList[i], 0, sizeof(ether_addr));
    }
    uint32_t count = ConnectableClientsCountMax;
    bwl_ret = bwl_get_assoclist( m_InterfaceHandle, cliList, &count);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    if( count == 0 )
    {
        WLAN_LOG_DEBUG("No STA exists\n");
        return ResultNoClients();
    }

    // 1台以上のSTAが接続しているのでDeauthを投げる。ただし、実際に接続している相手かどうか判断する。
    // 宛先がBroadcastの場合は判断不要。
    // (暫定)Broadcastアドレス指定のbwl_deauthの動作が不安定なので、個別指定にしておく
    MacAddress mac(deauthParam->PeerAddr);
    if( mac != MacAddress::CreateBroadcastMacAddress() )
    {
        bool isFound = false;
        for( uint8_t i = 0; i < count; i++ )
        {
            if( std::memcmp(&cliList[i].octet[0], &deauthParam->PeerAddr[0], ETHER_ADDR_LEN) == 0 )
            {
                WLAN_LOG_DEBUG("Now sends deauth frame to the connected STA\n");
                isFound = true;
                bwl_ret = bwl_deauth( m_InterfaceHandle, &cliList[i], deauthParam->reason );
                WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
                break;
            }
        }
        if( isFound == false )
        {
            WLAN_LOG_DEBUG("Given mac address is not in the list\n");
            return ResultClientNotFound();
        }
    }
    else
    {
        for( uint8_t i = 0; i < count; i++ )
        {
            bwl_ret = bwl_deauth( m_InterfaceHandle, &cliList[i], deauthParam->reason );
            WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));  // TORIAEZU 随時イベントが上がってくるので少し待つ TODO StateMachineから一個ずつ呼ぶべき
        }
        // TODO bwl_get_assoclistで獲得したClientリストとm_ClientStatusManagerが保持しているClientリストが完全一致するかチェック
    }

    // ClientStatusの更新
    ClientStatus clientStatus;
    clientStatus.state = ConnectionState_Disconnected;
    clientStatus.cause = CauseOfInfo_RecieveDisconnect;  // CLIENT視点ではMASTERから切断されたことになるので、この理由となる
    clientStatus.clientMacAddress.Set(&deauthParam->PeerAddr[0]);
    clientStatus.statusReasonCode = deauthParam->reason;
    clientStatus.capabilityInfo = 0;
    clientStatus.rssi = -128;

    m_ClientStatusManager.UpdateClientStatus(clientStatus);

    return ResultSuccess();
}

/* MACアドレスの取得 --------------------------------------------------------------- */
bool Mezcal::GetMacAddress(uint8_t macAddress[MacAddress::MacAddressSize]) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    bwl_ret = bwl_get_curetheraddr(m_InterfaceHandle, &m_OwnMacAddr);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    std::memcpy(macAddress, &m_OwnMacAddr.octet[0], ETHER_ADDR_LEN);

    return true;
}

/* VendorSpecificIEの追加 --------------------------------------------------------------- */
bool Mezcal::AddIe(uint32_t* pOutIndex, WlanIeContainer* pInIe) NN_NOEXCEPT
{
    int32_t ret;
    int32_t bwl_ret;

    // IE管理クラスにIEを覚えさせておく
    // pInIeのNULL評価等はCreateIeの中でやっているので、ここでは不要
    ret = m_vieManager.CreateIe(pOutIndex, pInIe);
    if(ret == false)
    {
        return false;
    }

    bwl_ret = bwl_add_ie(m_InterfaceHandle, pInIe->flag,
            reinterpret_cast<uint8_t*>(pInIe->oui), reinterpret_cast<uint8_t*>(pInIe->body), pInIe->length - 3);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

/* VendorSpecificIEの削除 --------------------------------------------------------------- */
bool Mezcal::DeleteIe(uint32_t ieIndex) NN_NOEXCEPT
{
    int32_t bwl_ret;

    WlanIeContainer* pContainer = NULL;

    // IE管理クラスから、ieIndexに該当するIE情報を取得する
    m_vieManager.GetWlanIeContainer(&pContainer, ieIndex);
    if( pContainer == NULL )
    {
        WLAN_LOG_DEBUG("DeleteIe index[%d] is not found.\n", ieIndex);
        return false;
    }

    bwl_ret = bwl_del_ie(m_InterfaceHandle, pContainer->flag,
            reinterpret_cast<uint8_t*>(pContainer->oui), reinterpret_cast<uint8_t*>(pContainer->body), pContainer->length - 3);

    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // IE管理クラスからIE情報を削除しておく
    m_vieManager.RemoveIe(ieIndex);

    return true;
}

bool Mezcal::GetChannel(int16_t* pOutChannel) NN_NOEXCEPT
{
    int32_t bwl_ret;
    uint32_t channel;

    NN_SDK_ASSERT_NOT_NULL(pOutChannel);

    bwl_ret = bwl_get_channel(m_InterfaceHandle, &channel);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    *pOutChannel = static_cast<int16_t>(channel);

    return true;
}

bool Mezcal::GetAllowedChannels(WlanAllowedChannels* allowedChannel) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_ABORT_UNLESS_NOT_NULL(allowedChannel);

    wl_uint32_list_t* chList;
    uint32_t chBuf[WL_NUMCHANNELS + 1];
    chList = reinterpret_cast<wl_uint32_list_t*>(chBuf);
    chList->count = WL_NUMCHANNELS;
    int32_t bwl_ret = wlu_get(m_InterfaceHandle, WLC_GET_VALID_CHANNELS, chBuf, sizeof(chBuf));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    memset(allowedChannel, 0, sizeof(WlanAllowedChannels));
    if( chList->count < driver::ChannelCountMax )
    {
        allowedChannel->count = chList->count;
    }
    else
    {
        allowedChannel->count = driver::ChannelCountMax;
    }
    for( int i = 0; i < allowedChannel->count; i++ )
    {
        allowedChannel->channel[i] = static_cast<int16_t>(chList->element[i]);
    }
    std::sort(allowedChannel->channel, allowedChannel->channel + allowedChannel->count);

    return true;
}

bool Mezcal::EnableCoexMc2() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = bwl_enable_mc2(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // ついでにCoex desense levelの固定化も行う
    SetBtcDesenseLevel(0);

    return true;
}

bool Mezcal::DisableCoexMc2() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = bwl_disable_mc2(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // ついでにCoex desense levelの初期化も行う
    SetBtcDesenseLevel(0);
    return true;
}

bool Mezcal::UpdateClientStatus(const ClientStatus& status) NN_NOEXCEPT
{
    return m_ClientStatusManager.UpdateClientStatus(status);
}

void Mezcal::GetClientStatusList(ClientStatus* pOutList, uint8_t count, Bit32* pOutBitmap, bool IsClear) NN_NOEXCEPT
{
    m_ClientStatusManager.GetClientStatusList(pOutList, count, pOutBitmap, IsClear);

    // Get rssi of each connected client
    for( uint8_t i = 0; i < count; i++ )
    {
        if( pOutList[i].state == ConnectionState_Connected )
        {
            GetRssiForAp(&pOutList[i].rssi, pOutList[i].clientMacAddress);
        }
    }
}

WowlWakeReason Mezcal::ConvertWakeupReason(uint32_t reason) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    WowlWakeReason ret;

    if( reason == 0 )
    {
        ret = WowlWakeReason_Nothing;
    }
    else if( reason & WL_WOWL_NET )
    {
        ret = WowlWakeReason_PatternData;
    }
    else if( reason & WL_WOWL_TCPKEEP_DATA )
    {
        ret = WowlWakeReason_TcpSessionData;
    }
    else if( reason & WL_WOWL_MAGIC )
    {
        ret = WowlWakeReason_Magicpacket;
    }
    else
    {
        ret = WowlWakeReason_Linkdown;
    }

    return ret;
}

bool Mezcal::GetWakeupReasonRaw(uint32_t* pOutReason, WowlWakeCount* pCounts) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    NN_SDK_ASSERT_NOT_NULL(pOutReason);
    NN_SDK_ASSERT_NOT_NULL(pCounts);

    uint32_t pcieInd = 0;  // pci wakeind can be ignored.
    uint32_t wakeInd = 0;
    bwl_ret = bwl_wowl_wakeind(m_InterfaceHandle, false, &pcieInd, &wakeInd);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    if( wakeInd == 0 )
    {
        // システム起因の起床の場合はこの関数自体呼び出されないので、ここには来ることはない
        WLAN_LOG_ERROR("%s is called in spite of system trigger wakeup.\n", __FUNCTION__);
        pCounts->system += 1;
    }
    else if( wakeInd & (WL_WOWL_NET | WL_WOWL_TCPKEEP_DATA | WL_WOWL_MAGIC |
            WL_WOWL_DIS | WL_WOWL_BCN | WL_WOWL_GTK_FAILURE) )
    {
        WLAN_LOG_INFO("Wakeind %08X\n", wakeInd);
        if( wakeInd & WL_WOWL_MAGIC )
        {
            pCounts->magic += 1;
        }
        else if( wakeInd & WL_WOWL_NET )
        {
            pCounts->netPattern += 1;
        }
        else if( wakeInd & WL_WOWL_DIS )
        {
            pCounts->disconnect += 1;
        }
        else if( wakeInd & WL_WOWL_BCN )
        {
            pCounts->beaconLost += 1;
        }
        else if( wakeInd & WL_WOWL_GTK_FAILURE )
        {
            pCounts->gtkFail += 1;
        }
        else if( wakeInd & WL_WOWL_TCPKEEP_DATA )
        {
            pCounts->tcpData += 1;
        }
    }
    else
    {
        pCounts->other += 1;
        WLAN_LOG_ERROR("Unknown wakeind %08X\n", wakeInd);
    }

    *pOutReason = wakeInd;

    return true;
}

bool Mezcal::ClearWakeupReason() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    uint32_t pcieInd = 0;
    uint32_t ucodeInd = 0;
    bwl_ret = bwl_wowl_wakeind(m_InterfaceHandle, true, &pcieInd, &ucodeInd);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::SetupWowlParams(WlanWowlSetupParams* pParams) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    NN_SDK_ASSERT_NOT_NULL(pParams);

#if defined(USE_WOWL_WAKEPATTERN)
    ipv4_addr srcIp = {
            { pParams->srcIp.addr[0], pParams->srcIp.addr[1], pParams->srcIp.addr[2], pParams->srcIp.addr[3] }
    };
    ipv4_addr dstIp = {
            { pParams->dstIp.addr[0], pParams->dstIp.addr[1], pParams->dstIp.addr[2], pParams->dstIp.addr[3] }
    };
    WLAN_LOG_DEBUG("Set tcp pattern.\n");
    WLAN_LOG_DEBUG("    src ip: %d.%d.%d.%d\n", srcIp.addr[0], srcIp.addr[1], srcIp.addr[2], srcIp.addr[3]);
    WLAN_LOG_DEBUG("    dst ip: %d.%d.%d.%d\n", dstIp.addr[0], dstIp.addr[1], dstIp.addr[2], dstIp.addr[3]);
    WLAN_LOG_DEBUG("    src port:%d\n", pParams->srcPort);
    WLAN_LOG_DEBUG("    dst port:%d\n", pParams->dstPort);
    WLAN_LOG_DEBUG("    ack num :%08x\n", pParams->ackNum);
    WLAN_LOG_DEBUG("    windowSize num :%d\n", pParams->windowSize);

    // Enable arp offload
    if( m_wowlFeatures & WowlFeatures_ArpOffload )
    {
        WLAN_LOG_INFO("Enable ARP offload feature\n");
        bwl_ret = bwl_arp_offload_enable(m_InterfaceHandle, &srcIp);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }
    // Set tcp ka info
    if( m_wowlFeatures & WowlFeatures_TcpKeepAlive )
    {
        WLAN_LOG_INFO("Enable TCP KA feature\n");
        bwl_tcpka_session_t tcpkaSession;
        memcpy(tcpkaSession.dst_mac.octet, pParams->dstMac.GetMacAddressData(), MacAddress::MacAddressSize);
        memcpy(&tcpkaSession.src_ip, &srcIp, sizeof(ipv4_addr));
        memcpy(&tcpkaSession.dst_ip, &dstIp, sizeof(ipv4_addr));
        tcpkaSession.ipid = 2000;
        tcpkaSession.src_port = pParams->srcPort;
        tcpkaSession.dst_port = pParams->dstPort;
#ifdef ENABLE_WOWL_AGING_DEBUG
        tcpkaSession.ack = m_ackNum;
#else
        tcpkaSession.ack = pParams->ackNum;
#endif
        tcpkaSession.tcpwin = pParams->windowSize;
        bwl_ret = bwl_tcpka_response(m_InterfaceHandle, &tcpkaSession);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
        bwl_ret = bwl_tcpka_enable(m_InterfaceHandle, true, NULL);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }
#endif

    // Set gpio wake pin level
    int gpiopol = 0; // 0:Set active high when wake up, 1:Set active low when wake up
    int ret = wlu_iovar_set(m_InterfaceHandle, "wowl_gpiopol", &gpiopol, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    // Clear wake up reason
    ClearWakeupReason();

    // Set wake up reason
    WLAN_LOG_INFO("Set wake up reason: 0x%08X\n", m_wakeReasonForSet);
    bwl_ret = bwl_wowl(m_InterfaceHandle, &m_wakeReasonForSet, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Clear wowl stats
    wl_sleep_stats_t stats;
    bwl_ret = bwl_get_sleep_stats(m_InterfaceHandle, &stats, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set beacon listen interval
    uint32_t interval;
#if WOWL_BEACON_LISTEN_INTERVAL == 0
    // If a charger is connected, beacon listen interval is set to 1.
    // If not, beacon listen interval is set to 2.
    nn::psm::ChargerType chargerType = nn::psm::GetChargerType();
    if (chargerType == nn::psm::ChargerType_EnoughPower
        || chargerType == nn::psm::ChargerType_LowPower)
    {
        WLAN_LOG_DEBUG("Charger is connected.\n");
        interval = 1;
    }
    else
    {
        WLAN_LOG_DEBUG("Charger is not connected.\n");
        interval = 2;
    }
#elif WOWL_BEACON_LISTEN_INTERVAL == 1
    interval = 1;
    WLAN_LOG_DEBUG("BEACON_LISTEN_INTERVAL 1\n");
#else
    interval = 2;
    WLAN_LOG_DEBUG("BEACON_LISTEN_INTERVAL 2\n");
#endif
    bwl_ret = bwl_bcn_li(m_InterfaceHandle, &interval, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Change legacy power save parameter for WoWL (SIGLO-72198)
    wl_pm2_sleep_ret_ext_t sleep_ret_ext;
    sleep_ret_ext.logic                = WL_DFRTS_LOGIC_AND; /* DFRTS logic: see WL_DFRTS_LOGIC_* below */
    sleep_ret_ext.low_ms               = 10;                 /* Low FRTS timeout */
    sleep_ret_ext.high_ms              = 30;                 /* High FRTS timeout */
    sleep_ret_ext.rx_pkts_threshold    = 10;                 /* switching threshold: # rx pkts */
    sleep_ret_ext.tx_pkts_threshold    = 0;                  /* switching threshold: # tx pkts */
    sleep_ret_ext.txrx_pkts_threshold  = 0;                  /* switching threshold: # (tx+rx) pkts */
    sleep_ret_ext.rx_bytes_threshold   = 0;                  /* switching threshold: # rx bytes */
    sleep_ret_ext.tx_bytes_threshold   = 0;                  /* switching threshold: # tx bytes */
    sleep_ret_ext.txrx_bytes_threshold = 0;                  /* switching threshold: # (tx+rx) bytes */
    bwl_ret = bwl_set_pm2_sleep_ret_ext(m_InterfaceHandle, &sleep_ret_ext);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 2);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::ClearWowlParams() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

#if defined(USE_WOWL_WAKEPATTERN)
    // Clear all the TCP session information
    char op[] = "clr";
    bwl_ret = bwl_wowl_pattern(m_InterfaceHandle, op, 0, NULL, NULL, NULL, 0);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    // Disable arp offload
    if( m_wowlFeatures & WowlFeatures_ArpOffload )
    {
        bwl_ret = bwl_arp_offload_disable(m_InterfaceHandle);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }
    // Disable tcp keep alive
    if( m_wowlFeatures & WowlFeatures_TcpKeepAlive )
    {
        uint16_t ipId = 0;
        bwl_ret = bwl_tcpka_enable(m_InterfaceHandle, false, &ipId);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
        WLAN_LOG_INFO("tcpka disable ipid : %d\n", ipId);
    }
#endif

    // Clear wake up reason
    ClearWakeupReason();

    // Reset beacon listen interval
    uint32_t interval = 0;
    bwl_ret = bwl_bcn_li(m_InterfaceHandle, &interval, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset legacy power save parameter
    wl_pm2_sleep_ret_ext_t sleep_ret_ext;
    sleep_ret_ext.logic                = WL_DFRTS_LOGIC_AND; /* DFRTS logic: see WL_DFRTS_LOGIC_* below */
    sleep_ret_ext.low_ms               = 2000;               /* Low FRTS timeout */
    sleep_ret_ext.high_ms              = 5000;               /* High FRTS timeout */
    sleep_ret_ext.rx_pkts_threshold    = 10;                 /* switching threshold: # rx pkts */
    sleep_ret_ext.tx_pkts_threshold    = 0;                  /* switching threshold: # tx pkts */
    sleep_ret_ext.txrx_pkts_threshold  = 0;                  /* switching threshold: # (tx+rx) pkts */
    sleep_ret_ext.rx_bytes_threshold   = 0;                  /* switching threshold: # rx bytes */
    sleep_ret_ext.tx_bytes_threshold   = 0;                  /* switching threshold: # tx bytes */
    sleep_ret_ext.txrx_bytes_threshold = 0;                  /* switching threshold: # (tx+rx) bytes */
    bwl_ret = bwl_set_pm2_sleep_ret_ext(m_InterfaceHandle, &sleep_ret_ext);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = bwl_set_pm(m_InterfaceHandle, 2);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Re-initialize event mask
    InitializeEventmask(WlanBaseFunctions::WlanModeInfraSta);

    return true;
}

bool Mezcal::ActivateWowl() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    uint32_t result;
    int32_t bwl_ret = bwl_wowl_activate(m_InterfaceHandle, &result);
    if( bwl_ret != BWL_ERR_SUCCESS )
    {
        WLAN_LOG_ERROR("%s failed due to %d\n", bwl_ret);
        return false;
    }

    if( result == 1 )
    {
        return true;
    }
    else
    {
        WLAN_LOG_ERROR("Failed to activate WoWL mode\n");
        return false;
    }
}

bool Mezcal::DisableWowl() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    bwl_ret = bwl_wowl_clear(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::SetDevicePowerState(bool isFullAwake) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    if( isFullAwake == true )
    {
        bwl_ret = bwl_set_powerstate(m_InterfaceHandle, DHD_PCIE_D0);
    }
    else
    {
        bwl_ret = bwl_set_powerstate(m_InterfaceHandle, DHD_PCIE_D3hot);
    }
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

void Mezcal::SetWakeupReasonRaw(uint32_t reason) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_INFO("%s: %08X will be set for wowl wake reason.\n", __FUNCTION__, reason);
    m_wakeReasonForSet = reason;
}

void Mezcal::EnableWowlFeatures(uint32_t features) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_INFO("%s: %08X is given.\n", __FUNCTION__, features);
    m_wowlFeatures = features;
}

void Mezcal::GetWowlSleepStats(WowlSleepStats* pStats) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_ASSERT_NOT_NULL(pStats);

    int32_t bwl_ret;

    wl_sleep_stats_t stats;
    bwl_ret = bwl_get_sleep_stats(m_InterfaceHandle, &stats, false);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    pStats->cpuWakes += stats.cpu_wakes;
    pStats->cpuActiveTime += stats.cpu_up_us;
    pStats->totalMeasuredTime += stats.total_time_ms;
    pStats->dtim += stats.dtim;
    pStats->txPackets += stats.tx_packets;
    pStats->txBytes += stats.tx_bytes;
    pStats->txActiveTime += stats.tx_us;
    pStats->rxPackets += stats.rx_packets;
    pStats->rxBytes += stats.rx_bytes;
    pStats->rxActiveTime += stats.rx_us;
    pStats->idleTime += stats.idle_us;
    pStats->arpReply += stats.arp_resp_cnt;
    pStats->tcpKaAck += stats.tcpka_ack_cnt;
    pStats->gtkRenewal += stats.grp_key_renew_cnt;

    if( WlanLogLevel >= LogLevel_Info )
    {
        NN_SDK_LOG("[WLAN] Wowl sleep stats\n");
        NN_SDK_LOG("cpu wakes:%d\n", pStats->cpuWakes);
        NN_SDK_LOG("cpu active:%d[us]\n", pStats->cpuActiveTime);
        NN_SDK_LOG("total measured:%d[ms]\n", pStats->totalMeasuredTime);
        NN_SDK_LOG("dtim:%d\n", pStats->dtim);
        NN_SDK_LOG("tx packets:%d\n", pStats->txPackets);
        NN_SDK_LOG("tx bytes:%d[byte]\n", pStats->txBytes);
        NN_SDK_LOG("tx active:%d[us]\n", pStats->txActiveTime);
        NN_SDK_LOG("rx packets:%d\n", pStats->rxPackets);
        NN_SDK_LOG("rx bytes:%d[byte]\n", pStats->rxBytes);
        NN_SDK_LOG("rx active:%d[us]\n", pStats->rxActiveTime);
        NN_SDK_LOG("idle:%d[us]\n", pStats->idleTime);
        NN_SDK_LOG("arp:%d\n", pStats->arpReply);
        NN_SDK_LOG("tcp ka:%d\n", pStats->tcpKaAck);
        NN_SDK_LOG("gtk renewal:%d\n", pStats->gtkRenewal);
    }

}

bool Mezcal::SetCwParam(AccessCategoryIndex accessCategory, int cwmin, int cwmax) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    int ac;
    switch( accessCategory )
    {
    case AccessCategoryIndex_BE:
        ac = AC_BE;
        break;
    case AccessCategoryIndex_BK:
        ac = AC_BK;
        break;
    case AccessCategoryIndex_VI:
        ac = AC_VI;
        break;
    case AccessCategoryIndex_VO:
        ac = AC_VO;
        break;
    default:
        ac = AC_BE;
        break;
    }

    bwl_ret = bwl_acparams(m_InterfaceHandle, ac, cwmin, cwmax, true);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    bwl_ret = bwl_acparams(m_InterfaceHandle, ac, cwmin, cwmax, false);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

void Mezcal::SetAciMitigation(AciMitigation id) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t val;
    int32_t ret = BCME_OK;

    switch( id )
    {
    case AciMitigation_Default:
        val = 47;
        break;
    case AciMitigation_LocalNormal:
        val = 9;
        break;
    case AciMitigation_LocalLcs:
        val = 9;
        break;
    case AciMitigation_Detect:
        val = 9;
        break;
    default:
        val = 47;
        break;
    }

    ret = wlu_set(m_InterfaceHandle, WLC_SET_INTERFERENCE_OVERRIDE_MODE, &val, sizeof(val));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);
    WLAN_LOG_INFO("%s: Set %d\n", __FUNCTION__, val);
}

bool Mezcal::SetStaKeepAlive(int intervalms) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t bwl_ret;
    bool ret = true;

    bwl_ret = bwl_set_keepalive(m_InterfaceHandle, static_cast<uint32_t>(intervalms));
    if( bwl_ret != BCME_OK )
    {
        WLAN_LOG_ERROR("Failed to set keepalive (%d)\n", bwl_ret);
        ret = false;
    }
    WLAN_LOG_DEBUG("Set keep alive with %dms interval\n", intervalms);

    return ret;
}

void Mezcal::SetMulticastList(WlanAllowedMulticastList* pList) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;
    struct bwl_multi_filter_list filterList;
    maclist_t *pMacList;

    NN_SDK_ASSERT_NOT_NULL(pList);
    filterList.count = static_cast<uint>(pList->count);
    for( int i = 0; i < filterList.count; i++ )
    {
        std::memcpy(&filterList.ea[i].octet[0], &pList->addr[i].addr[0], MacAddress::MacAddressSize);
    }
    pMacList = reinterpret_cast<maclist_t*>(&filterList);

    bwl_ret = bwl_set_mcast_list(m_InterfaceHandle, pMacList);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return;
}

bool Mezcal::PutActionFramePeriodicallyForDetect(const char* pData, uint16_t size, DetectPeriodicAfCycle pattern) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_NOT_NULL(pData);

    // Set periodic action frame
    bwl_aloe_periodic_cfg_t afCfg = {
            pattern.txInterval, // tx interval
            pattern.txCount,  // tx cnt
            pattern.idleCount, // idle cnt
            pattern.rxStartCount, // rx start
            pattern.rxCount, // rxcnt
    };
    wl_aloe_af_t* pAfData = reinterpret_cast<wl_aloe_af_t*>(const_cast<char*>(pData));

    int32_t bwl_ret = bwl_aloe_send_periodic_af(m_InterfaceHandle, &afCfg, size, pAfData);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::SetupDetectSleepParams(WlanDetectSetupParams* pParams) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    NN_SDK_ASSERT_NOT_NULL(pParams);

    // Get dongle time and the system time
    bwl_ret = bwl_aloe_get_time(m_InterfaceHandle, &m_dongleTime);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    m_sysTick = nn::os::GetSystemTick();

    // Clear the hash table
    bwl_ret = bwl_aloe_clear_hash(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set the wake trigger for the AF cache
    bwl_ret = bwl_aloe_set_wake_afcnt(m_InterfaceHandle, pParams->cacheCntMax);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Set the hash table
    // Check hash pointer
    if( pParams->hashList.pHashList != NULL )
    {
        bwl_ret = bwl_aloe_set_hash(m_InterfaceHandle, pParams->hashList.pHashList, pParams->hashList.numOfHash);
        WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    }

    // Set periodic action frame
    bwl_aloe_periodic_cfg_t afCfg = {
            pParams->sleepParam.txInterval,
            pParams->sleepParam.txCount,
            pParams->sleepParam.idleCount,
            pParams->sleepParam.rxStartCount,
            pParams->sleepParam.rxCount,
    };
    WLAN_LOG_DEBUG("Set sleep pattern\ntxint=%d\ntxcnt=%d\nidle=%d\nrxstart=%d\nrxcnt=%d\n",
            afCfg.tx_int, afCfg.tx_cnt, afCfg.idle_cnt, afCfg.rx_start, afCfg.rx_cnt);
    wl_aloe_af_t* pAfData = reinterpret_cast<wl_aloe_af_t*>(&pParams->actionFrame[0]);

    bwl_ret = bwl_aloe_send_periodic_af(m_InterfaceHandle, &afCfg, pParams->length, pAfData);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Shut out event (SIGLO-75396)
    uint8_t eventmask[WL_EVENTING_MASK_LEN];
    std::memset(eventmask, 0, sizeof(eventmask));
    bwl_ret = bwl_set_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::ClearDetectSleepParams() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    // Stop periodic af
    bwl_ret = bwl_cancel_periodic_af(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Re-initialize event mask
    InitializeEventmask(WlanBaseFunctions::WlanModeDetect);

    return true;
}

bool Mezcal::ActivateDetectSleep() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    // Clear the Action Frame cache
    bwl_ret = bwl_aloe_clear_buffer_af(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Reset wake-up trigger
    bwl_ret = bwl_aloe_clear_wake(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    g_aloeAfCount = -1;

    return true;
}

bool Mezcal::DisableDetectSleep() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    // Reset wake-up trigger
    bwl_ret = bwl_aloe_clear_wake(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

void Mezcal::PullActionFrameFromDongle() NN_NOEXCEPT
{
    WLAN_LOG_INFO("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    uint32_t got;
    uint32_t remain;
    // After calling the following API, actoin frame data starts to be sent from FW as WLC_E_ACTION_FRAME_RX
    bwl_ret = bwl_aloe_get_buffer_af(m_InterfaceHandle, true, &got, &remain);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    g_aloeAfCount = 0;
    WLAN_LOG_INFO("action frame in the cache: got(%d) remain(%d)\n", got, remain);
    m_DetectSaTotalRecvCnt += static_cast<uint64_t>(got);
}

bool Mezcal::GetWakeupReasonRawDetectSleep(uint32_t* pOutReason, DetectWakeCount* pCounts) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    NN_SDK_ASSERT_NOT_NULL(pOutReason);
    NN_SDK_ASSERT_NOT_NULL(pCounts);

    uint32_t reason = 0;
    bwl_ret = bwl_aloe_get_wake_reason(m_InterfaceHandle, &reason);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    if( reason == 0 )
    {
        pCounts->system += 1;
    }
    else if( reason & (1 << 1) )  // TODO
    {
        WLAN_LOG_INFO("reason %08X\n", reason);
        if( reason & (1 << 1) )
        {
            pCounts->cacheFull += 1;
        }
    }
    else
    {
        pCounts->other += 1;
        WLAN_LOG_ERROR("Unknown wakeind %08X\n", reason);
    }

    *pOutReason = reason;

    return true;
}

DetectWakeReason Mezcal::ConvertWakeupReasonDetectSleep(uint32_t reason) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    DetectWakeReason ret;

    if( reason == 0 )
    {
        ret = DetectWakeReason_Nothing;
    }
    else if( reason & ( 1 << 1 ) )  // TODO
    {
        ret = DetectWakeReason_CacheFull;
    }
    else
    {
        ret = DetectWakeReason_CacheFull;
    }

    return ret;
}

void Mezcal::SetActionFrameRecvMode(ActionFrameRecvMode mode) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t bwl_ret;
    Aloe_Mode_t aloeMode;
    const bwl_aloe_cfg_t* pCfg;

    if( mode == ActionFrameRecvMode_DetectWake )
    {
        aloeMode = ALOE_MODE_HDA;
        pCfg = &m_detectHdMode;
    }
    else if( mode == ActionFrameRecvMode_DetectSleep )
    {
        aloeMode = ALOE_MODE_SAA;
        pCfg = &m_detectSaMode;
    }
    else
    {
        aloeMode = ALOE_MODE_DISABLE;
        pCfg = &m_detectDisableMode;
    }

    // Check current setting.
    Aloe_Mode_t outAloeMode;
    bwl_ret = bwl_aloe_get_mode(m_InterfaceHandle, &outAloeMode);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    if( outAloeMode == aloeMode )
    {
        WLAN_LOG_DEBUG("Aloe mode is already the requested mode.\n");
        return;
    }

    bwl_ret = bwl_aloe_set_mode(m_InterfaceHandle, aloeMode, const_cast<bwl_aloe_cfg_t*>(pCfg));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
}

void Mezcal::ClearAfCache() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    int32_t bwl_ret;

    // Clear the Action Frame cache
    bwl_ret = bwl_aloe_clear_buffer_af(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    // Clear the hash table
    bwl_ret = bwl_aloe_clear_hash(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
}

/* Set Ssid Event ---------------------------------------------------------- */
WlanEvent* SetSsidEventProcess(wl_event_msg_t* msg)
{
    WlanEvent* pevnbuff = NULL;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    ptrdiff_t u8p; /* 型変換の一時データ退避用 */
    uintptr_t *p;  /* 型変換の一時データ退避用 */
    u8p = reinterpret_cast<ptrdiff_t>(msg);
    p = reinterpret_cast<uintptr_t*>(u8p + sizeof(wl_event_msg_t));

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanConnectIndicate));
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);
    WlanConnectIndicate* pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);
    pevnbuff->id = WLAN_EVENT_CONNECTED;

    switch( msg->status )
    {
    case WLC_E_STATUS_SUCCESS:
        WLAN_LOG_INFO("[MEZ] Connection success!\n");
        pind->Result = true;
        std::memcpy(&pind->PeerAddr[0], &msg->addr.octet[0], MacAddress::MacAddressSize);

        nn::os::LockMutex(&Mezcal::m_ConnectionStatusMutex);

        // WLC_E_SET_SSIDイベントは、wl_event_msg_tの後にSSIDがそのまま詰まっている
        // SSIDの保存
        if( msg->datalen == 0 || msg->datalen > 32 )
        {
            Mezcal::m_MezcalConStat.ssid = Ssid();
        }
        else
        {
            Mezcal::m_MezcalConStat.ssid.Set(reinterpret_cast<Bit8*>(*p), static_cast<size_t>(msg->datalen));
        }
        Mezcal::m_MezcalConStat.bssid.Set(&msg->addr.octet[0]);
        Mezcal::m_MezcalConStat.state = ConnectionState_Connected;
        Mezcal::m_MezcalConStat.cause = CauseOfInfo_ConnectRequest;
        Mezcal::m_MezcalConStat.statusReasonCode = Dot11StatusCode_Success;

        nn::os::UnlockMutex(&Mezcal::m_ConnectionStatusMutex);

        break;
    default:
        WLAN_LOG_ERROR("[MEZ] SetSsid failed(%d)\n", msg->status);
        pind->Result = false;
        std::memcpy(&pind->PeerAddr[0], ZeroMacAddr, MacAddress::MacAddressSize);
        break;
    }

    return pevnbuff;
}

/* Scan 処理中に通知された 無線ネットワーク情報の解析 ---------------------- */
static WlanEvent* ParseWlanNetworkInfo( wl_event_msg_t* pdrvmsg )
{
    wl_scan_results_t *scanresult;
    wl_bss_info_t* pdrvbssinfo;
    ptrdiff_t u8p; /* 型変換の一時データ退避用 */
    uintptr_t *p;  /* 型変換の一時データ退避用 */
    WlanEvent* pevnbuff = NULL;

    /* 型変換
     * DHDから届く、wl_event_msg_t 構造体の実態が、wl_event_msg_t の
     * 直後4Byteに、wl_scan_results_t への構造体のアドレスが入っている
     * そのため、pdrvmsg に wl_scan_results_t のサイズ分を足して、その直後の
     * ポインタアドレスを抜き取る仕組みにしている。
     */
    u8p = reinterpret_cast<ptrdiff_t>(pdrvmsg);
    p = reinterpret_cast<uintptr_t*>(u8p + sizeof(wl_event_msg_t));
    scanresult = reinterpret_cast<wl_scan_results_t*>(*p);
    pdrvbssinfo = scanresult->bss_info;
#if 0
    NN_SDK_LOG("wl_event_msg_t size = %xh\n", sizeof(wl_event_msg_t));
    NN_SDK_LOG("wl_scan_results_t start = %p, size=%xh\n", p, sizeof(wl_scan_results_t));
    NN_SDK_LOG("ie offset = %p, size=%xh\n", pdrvbssinfo->ie_offset, pdrvbssinfo->ie_length);
    NN_SDK_LOG("pdrvmsg->datalen=%d\n", pdrvmsg->datalen);
    DumpFunc(pdrvmsg, (pdrvmsg->datalen&0xff));
    DumpFunc(pdrvbssinfo, sizeof(wl_bss_info_t));
    DumpFunc((uint8_t*)pdrvbssinfo + pdrvbssinfo->ie_offset, pdrvbssinfo->ie_length);
#endif

    if((scanresult->version == WL_BSS_INFO_VERSION) &&
       (pdrvbssinfo->version == WL_BSS_INFO_VERSION))
    {

        /* IE データを入れるので、その分を追加で確保する
         *
         * +--------------------+ <- pevnbuff         -
         * |    WlanBssInfo     |                     |
         * |                    |                     | size = sizeof(WlanBssInfo)
         * |                    |                     |
         * +--------------------+ <- pbssinfo->IeData -
         * |                    |                     |
         * |      IE Info       |                     | size = pbssinfo->IeLength
         * |                    |                     |
         * +--------------------+                     -
         *
         * pevnbuffが解放される際に、IEのバッファも解放されるようにしている
         */
        pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanBssInfo) + pdrvbssinfo->ie_length);
        if(pevnbuff != NULL)
        {
            WlanBssInfo* pbssinfo = static_cast<WlanBssInfo*>(pevnbuff->Args);

//          NN_SDK_LOG("Mezcal: [scanresults]buflen=%d, ver=%ld, count=%d\n",
//                 scanresult->buflen, scanresult->version, scanresult->count);

            /* リスト構造体用ポインタ */
            pbssinfo->NextItem = NULL;
            pbssinfo->PrevItem = NULL;

            /* BssInfo本体のコピー */

            /* BSSID */
            std::memcpy(pbssinfo->Bssid, pdrvbssinfo->BSSID.octet, ETHER_ADDR_LEN);

            /* ESSID */
            std::memset(&pbssinfo->SsidInfo.ssid, 0x00, sizeof(pbssinfo->SsidInfo.ssid));
            pbssinfo->SsidInfo.length = pdrvbssinfo->SSID_len;
            if(pbssinfo->SsidInfo.length > 32)
            {
                pbssinfo->SsidInfo.length = 32;
            }
            std::memcpy(pbssinfo->SsidInfo.ssid, pdrvbssinfo->SSID,
                        pbssinfo->SsidInfo.length);

            /* その他 */
            // Endian変換は不要？
            pbssinfo->BeaconPeriod   = pdrvbssinfo->beacon_period;
            pbssinfo->Capability     = static_cast<Bit16>(pdrvbssinfo->capability);
            pbssinfo->NCapability    = pdrvbssinfo->n_cap;
            pbssinfo->Rssi           = pdrvbssinfo->RSSI;
            /* 暫定の対処 */
            pbssinfo->Channel        = wf_chspec_ctlchan(pdrvbssinfo->chanspec);
            pbssinfo->ControlChannel = pdrvbssinfo->ctl_ch;
            pbssinfo->RateCount      = pdrvbssinfo->rateset.count;

            /* IE のコピー */
            {
                uint8_t* iesrc;
                uint8_t* iedst;

                iesrc = reinterpret_cast<uint8_t*>(pdrvbssinfo) + pdrvbssinfo->ie_offset;
                iedst = reinterpret_cast<uint8_t*>(pevnbuff->Args) + sizeof(WlanBssInfo);
                std::memcpy(iedst, iesrc, pdrvbssinfo->ie_length);

                pbssinfo->IeLength = pdrvbssinfo->ie_length;
                pbssinfo->IeData   = iedst;
            }

            if( pbssinfo->Rssi == 0 )
            {
                // RSSI値が0になる場合があるという不具合があったが、ReleaeJ4で修正されている。
                WLAN_LOG_ERROR("RSSI 0 info comes.\n");
            }
        }
        else
        {
            WLAN_LOG_ERROR("Failed to allocate Eventbuffer @ %s\n", __FUNCTION__);
        }
    }
    else
    {
        /* 想定外のScan results */
        WLAN_LOG_DEBUG("[scanresults]version error[ver=%ld]\n",
               scanresult->version);

        WLAN_LOG_DEBUG("DumpScanResult\n");
        if( WlanLogLevel >= LogLevel_Debug )
        {
            DumpFunc(scanresult, sizeof(wl_scan_results_t));
        }
    }

    return pevnbuff;
}

/* Scan Complete イベントの処理 -------------------------------------------- */
WlanEvent* ScanResultEventProcess(wl_event_msg_t* msg)
{
    WlanEvent* pevnbuff = NULL;

    if(msg->status == WLC_E_STATUS_PARTIAL)
    {
        /* Scan indication */
        pevnbuff = ParseWlanNetworkInfo( msg );
        if(pevnbuff != NULL)
        {
            pevnbuff->id = WLAN_EVENT_SCAN_INDICATION;
        }
    }
    else if(msg->status == WLC_E_STATUS_SUCCESS)
    {
        /* Scan complete */
        pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(bool));
        if(pevnbuff != NULL)
        {
            bool* result = static_cast<bool*>(pevnbuff->Args);
            pevnbuff->id = WLAN_EVENT_SCAN_COMPLETE;
            *result = true;
        }
    }
    else if(msg->status == WLC_E_STATUS_ABORT)
    {
        /* Scan abort */
        WLAN_LOG_DEBUG("[MEZ] Scan Abort fail\n");
        pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(bool));
        if(pevnbuff != NULL)
        {
            bool* result = static_cast<bool*>(pevnbuff->Args);
            pevnbuff->id = WLAN_EVENT_SCAN_COMPLETE;
            *result = false;
        }
    }
    else
    {
        WLAN_LOG_DEBUG("Unknown Scan result[stat=%d]\n", msg->status);
    }

    return pevnbuff;
}

/* STAモードで、Accociationが完了した ---------------------------------------- */
void AssociationEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // 接続成功自体はWLC_E_SET_SSIDで判断するので、ここでは情報の取得だけしておく
    dot11_assoc_resp* assocResp;
    ptrdiff_t u8p; /* 型変換の一時データ退避用 */
    uintptr_t *p;  /* 型変換の一時データ退避用 */

    u8p = reinterpret_cast<ptrdiff_t>(msg);
    p = reinterpret_cast<uintptr_t*>(u8p + sizeof(wl_event_msg_t));
    assocResp = reinterpret_cast<dot11_assoc_resp*>(*p);
    struct ether_addr ea;
    memcpy(&ea, &msg->addr, ETHER_ADDR_LEN);

    // WLAN_LOG_DEBUG("[MEZ]AssocEvent : length = %d\n", msg->datalen);
    // DumpFunc(assocResp, msg->datalen);

    // 接続情報を保存する。
    nn::os::LockMutex(&Mezcal::m_ConnectionStatusMutex);
    Mezcal::m_MezcalConStat.bssid.Set(ea.octet);                     // 接続相手のBSSID
    Mezcal::m_MezcalConStat.aid = assocResp->aid & 0x03FF;           // Endian変換の必要なし aidは最高位の2ビットは使用しないので間引いておく
    Mezcal::m_MezcalConStat.capabilityInfo = assocResp->capability;  // Endian変換の必要なし
    Mezcal::m_MezcalConStat.statusReasonCode = assocResp->status;    // Endian変換の必要なし
    WLAN_LOG_DEBUG("[MEZ]aid : %d (0x%04X)\n", Mezcal::m_MezcalConStat.aid, assocResp->aid & 0x03FF);
    nn::os::UnlockMutex(&Mezcal::m_ConnectionStatusMutex);

    return;
}

/* APモードで、他の端末が接続された ---------------------------------------- */
WlanEvent* AssociactionIndEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WlanEvent* pevnbuff;

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanConnectIndicate));
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);
    WlanConnectIndicate* pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);
    pevnbuff->id = WLAN_EVENT_CONNECTED;

    if(msg->status == WLC_E_STATUS_SUCCESS)
    {
        // Client情報の保存
        dot11_assoc_req* assocReq;
        ptrdiff_t u8p; /* 型変換の一時データ退避用 */
        uintptr_t *p;  /* 型変換の一時データ退避用 */

        u8p = reinterpret_cast<ptrdiff_t>(msg);
        p = reinterpret_cast<uintptr_t*>(u8p + sizeof(wl_event_msg_t));
        assocReq = reinterpret_cast<dot11_assoc_req*>(*p);

        // ClientStatusManagerに値は保存しておくのでStateMachineでは↓は不要
        // 一応いれておく
        pind->capability = assocReq->capability;
        std::memcpy(pind->PeerAddr, msg->addr.octet, MacAddress::MacAddressSize);

        pind->Result = true;

        // ClientStatusManagerにStatus情報を送る
        ClientStatus status;
        status.state = ConnectionState_Connected;
        status.clientMacAddress.Set(msg->addr.octet);
        status.statusReasonCode = 0;
        status.cause = CauseOfInfo_ConnectRequest;
        status.capabilityInfo = assocReq->capability;
        Mezcal::m_ClientStatusManager.UpdateClientStatus(status);

    }
    else
    {
        // 端末接続失敗の場合、特にClientStatusの更新は行わない
        pind->Result = false;
    }

    return pevnbuff;
}

/* Disassociate complete Event ------------------------------------------------- */
WlanEvent* DisassociateEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    /* 802.11 Reason code 6 や 7 による理由で自動送信されていた場合はWDMでは無視する。
     * FYI:
     * Reason code 6 : Class 2 frame received from nonauthenticated station
     * Reason code 7 : Class 3 frame received from nonassociated station
     */
    if( msg->reason == 6 || msg-> reason == 7 )
    {
        return NULL;
    }

    WlanEvent* pevnbuff;
    pevnbuff = g_pStateMachine->WlanGetEventBuff(0);
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);

    pevnbuff->id = WLAN_EVENT_DISASSOCIATE;

    if(msg->status == WLC_E_STATUS_SUCCESS)
    {
        // ConnectionStatusの更新はStateMachineで行う。
        pevnbuff->Result = true;
        nn::os::LockMutex(&Mezcal::m_ConnectionStatusMutex);
        Mezcal::m_MezcalConStat.statusReasonCode = static_cast<uint16_t>(msg->reason);
        nn::os::UnlockMutex(&Mezcal::m_ConnectionStatusMutex);
    }
    else
    {
        // 常にWLC_E_STATUS_SUCCESSが入っているはずなので、ここに来ていたらABORTさせる
        NN_ABORT("[WLAN][MEZ]Failed to send disassoc due to %d\n", msg->status);
    }

    return pevnbuff;
}

/* Deauthenticate complete Event ------------------------------------------------- */
// TODO 削除予定
WlanEvent* DeauthenticateEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    /* 802.11 Reason code 6 や 7 による理由で自動送信されていた場合はWDMでは無視する。
     * FYI:
     * Reason code 6 : Class 2 frame received from nonauthenticated station
     * Reason code 7 : Class 3 frame received from nonassociated station
     */
    if( msg->reason == 6 || msg-> reason == 7 )
    {
        return NULL;
    }

    WlanEvent* pevnbuff;
    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(uint8_t));
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);
    uint8_t* pFlag = reinterpret_cast<uint8_t*>(pevnbuff->Args);
    NN_SDK_ASSERT_NOT_NULL(pFlag);

    pevnbuff->id = WLAN_EVENT_DEAUTHENTICATE;
    *pFlag = 0;

    if(msg->status == WLC_E_STATUS_SUCCESS)
    {
        // ClientTimeoutによりClientの自動切断を行った場合もこのイベントが発生する。
        // 自動切断でない場合は既にClientStatusは更新されているので、イベントに記録されているClientがまだConnected状態なら自動切断と判断出来る。
        // TODO 自動切断の場合、msg->reason == 4なので、それで判断しても良い。また、WLC_E_DISASSOC_INDが続いて発生するので、そちらで処理しても良い。
        ClientStatus status;
        if( Mezcal::m_ClientStatusManager.GetClientStatusByMac(&status, MacAddress(msg->addr.octet)) == true )
        {
            if( status.state == ConnectionState_Connected )
            {
                // 自動切断によるDeauth送信だったので、接続情報を更新しておく
                NN_SDK_LOG("Client is disconnected due to timeout.\n");
                status.state = ConnectionState_Disconnected;
                status.statusReasonCode = Dot11ReasonCode_DisassocInactivity;
                status.cause = CauseOfInfo_ClientTimeout;
                status.capabilityInfo = 0;
                status.rssi = -128;
                Mezcal::m_ClientStatusManager.UpdateClientStatus(status);
                *pFlag = 1;  // 自動切断によるDeauthイベントであることを示すためのフラグ
            }
        }
        pevnbuff->Result = true;
    }
    else
    {
        pevnbuff->Result = false;
        // Deauthの送信失敗理由調査のため、アサート
        NN_SDK_ASSERT(false, "[WLAN][MEZ]Failed to send deauth due to %d\n", msg->status);
    }

    return pevnbuff;
}

/* Disassociate Indication ------------------------------------------------- */
WlanEvent* DisassocIndEventProcess(wl_event_msg_t* msg)
{
    // Broadcomの動作仕様により、通常の例以外に、切断済みのデバイスにフレームを投げようとしたり、
    // 切断済みのデバイスからフレームを受け取っても本イベントが発生する(?)
    WlanEvent* pevnbuff;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanDisconnectInfo));
    NN_ABORT_UNLESS(pevnbuff != NULL, "Failed to allocate event buffer");

    WlanDisconnectInfo* pind = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
    pevnbuff->id = WLAN_EVENT_DISASSOCIATE_IND;

    // 切断情報のコピー
    pind->reason = static_cast<uint16_t>(msg->reason);
    WLAN_LOG_DEBUG("[MEZ]Disassoc frame is received due to %d(0x%04X)\n", msg->reason, msg->reason);
    std::memcpy(&pind->PeerAddr[0], &msg->addr.octet[0], nn::wlan::MacAddress::MacAddressSize);

    // APが受けたのか、STAが受けたのかここでは判別つかないので、ConnectionStatusを更新すべきかClientStatusを更新すべきか
    // StateMachineのステートで判断するしかない

    return pevnbuff;
}

/* Deauth Indication ------------------------------------------------- */
WlanEvent* DeauthIndEventProcess(wl_event_msg_t* msg)
{
    WlanEvent* pevnbuff;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanDisconnectInfo));
    NN_ABORT_UNLESS(pevnbuff != NULL, "Failed to allocate event buffer");

    WlanDisconnectInfo* pind = static_cast<WlanDisconnectInfo*>(pevnbuff->Args);
    pevnbuff->id = WLAN_EVENT_DEAUTHENTICATE_IND;

    // 切断情報のコピー
    pind->reason = msg->reason;
    std::memcpy(&pind->PeerAddr[0], &msg->addr.octet[0], nn::wlan::MacAddress::MacAddressSize);

    return pevnbuff;
}

/* Link change event ------------------------------------------------------- */
WlanEvent* LinkChangeEventProcess(wl_event_msg_t* msg)
{
    WlanEvent* pevnbuff = NULL;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanLinkInfo));
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);
    pevnbuff->id = WLAN_EVENT_LINKCHANGE;
    WlanLinkInfo* linkInfo = static_cast<WlanLinkInfo*>(pevnbuff->Args);
    std::memcpy(&linkInfo->PeerAddr[0], &msg->addr.octet[0], nn::wlan::MacAddress::MacAddressSize);

    switch( msg->reason )
    {
    case 0:
        linkInfo->reason = LinkChangeReason_Up;
        break;
    case WLC_E_LINK_BCN_LOSS:
        linkInfo->reason = LinkChangeReason_BcnLoss;
        break;
    case WLC_E_LINK_DISASSOC:
        linkInfo->reason = LinkChangeReason_Disassoc;
        break;
    case WLC_E_LINK_ASSOC_REC:
        linkInfo->reason = LinkChangeReason_AssocRecFail;
        break;
    case WLC_E_LINK_BSSCFG_DIS:
        linkInfo->reason = LinkChangeReason_BsscfgDown;
        break;
    default:
        NN_SDK_ASSERT(false, "Unknown reason %d", msg->reason);
    }

    return pevnbuff;
}

/* ビーコンロストイベント ------------------------------------------ */
WlanEvent* BeaconLostEventProcess(wl_event_msg_t* msg)
{
    WlanEvent* pevnbuff = NULL;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_DEBUG("[MEZ]Beacon lost event occurs. STATUS : %d\n", msg->status);

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(WlanConnectIndicate));
    NN_SDK_ASSERT_NOT_NULL(pevnbuff);
    WlanConnectIndicate* pind = static_cast<WlanConnectIndicate*>(pevnbuff->Args);
    pevnbuff->id = WLAN_EVENT_BMISS;
    pind->Result = true;
    std::memcpy(pind->PeerAddr, msg->addr.octet, MacAddress::MacAddressSize);

    return pevnbuff;
}

/* Generic fail event process ---------------------------------------------- */
WlanEvent* GenericFailEventProcess(wl_event_msg_t* msg, WlanEventId id)
{
    WlanEvent* pevnbuff;

    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    pevnbuff = g_pStateMachine->WlanGetEventBuff(sizeof(bool));
    if(pevnbuff != NULL)
    {
        bool* result = static_cast<bool*>(pevnbuff->Args);
        pevnbuff->id = id;
        *result = false;
    }
    else
    {
        WLAN_LOG_DEBUG("[MEZ] Fail get event buffer\n");
    }

    return pevnbuff;
}

void ProcessActionFrameEvent(wl_event_msg_t* msg, nn::mbuf::Mbuf* pMbuf)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_DEBUG("[MEZ]received action frame from %02X:%02X:%02X:%02X:%02X:%02X\n",
            msg->addr.octet[0],msg->addr.octet[1],msg->addr.octet[2],
            msg->addr.octet[3],msg->addr.octet[4],msg->addr.octet[5]);
    //DumpFunc(nn::mbuf::MbufTod(pMbuf), nn::mbuf::MbufLength(pMbuf, NULL));

    auto result = g_pStateMachine->PushRxActionFrame(pMbuf);
    if( result.IsFailure() )
    {
        nn::mbuf::MbufFreem(pMbuf);
    }
}

void ProcessActionFrameEventEx(wl_event_msg_t* msg, nn::mbuf::Mbuf* pMbuf)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    WLAN_LOG_DEBUG("[MEZ]received action frame from %02X:%02X:%02X:%02X:%02X:%02X\n",
            msg->addr.octet[0],msg->addr.octet[1],msg->addr.octet[2],
            msg->addr.octet[3],msg->addr.octet[4],msg->addr.octet[5]);
    //DumpFunc(nn::mbuf::MbufTod(pMbuf), nn::mbuf::MbufLength(pMbuf, NULL));

    // allocate buffer to store channel, rssi and mactime
    if( pMbuf->_pGeneral != NULL )
    {
        nnwlanFreeNormal(pMbuf->_pGeneral);
    }
    WlanActionFrameRecvInfo* pInfo = reinterpret_cast<WlanActionFrameRecvInfo*>(nnwlanMallocNormal(sizeof(WlanActionFrameRecvInfo)));
    if( pInfo != NULL )
    {
        wl_event_rx_frame_data_t* pData;
        char* ptr = reinterpret_cast<char*>(nn::mbuf::MbufTod(pMbuf));
        pData = reinterpret_cast<wl_event_rx_frame_data_t*>(ptr + sizeof(bcm_event_t));
        pInfo->channel = (ReverseEndian16(pData->channel) & WL_CHANSPEC_CHAN_MASK);
        pInfo->rssi = static_cast<int32_t>(ReverseEndian32(static_cast<uint32_t>(pData->rssi)));
        uint32_t mactime = ReverseEndian32(static_cast<uint32_t>(pData->mactime));
        if( mactime > 0 )
        {
            // convert recv time to systick
            WLAN_LOG_DEBUG("action frame received in SAA mode\n");
            nn::os::Tick tick(nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(mactime - Mezcal::m_dongleTime)));
            pInfo->sysTick = (Mezcal::m_sysTick + tick).GetInt64Value();
        }
        else
        {
            // not SAA mode's action frame
            WLAN_LOG_DEBUG("action frame received in not SAA mode\n");
            pInfo->sysTick = nn::os::GetSystemTick().GetInt64Value();
        }
        pMbuf->_pGeneral = pInfo;
    }
    else
    {
        WLAN_LOG_ERROR("Failed to malloc buffer for WlanActionFrameRecvInfo\n");
        pMbuf->_pGeneral = NULL;
    }

    auto result = g_pStateMachine->PushRxActionFrameEx(pMbuf);
    if( result.IsFailure() )
    {
        if( pMbuf->_pGeneral != NULL )
        {
            nnwlanFreeNormal(pMbuf->_pGeneral);
        }
        nn::mbuf::MbufFreem(pMbuf);
    }
}

WlanEvent* InterfaceEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    if( Mezcal::m_bWaitIfUpEvent == true )
    {
        WlanEvent* pevnbuff = NULL;

        if( WlanLogLevel >= LogLevel_Debug )
        {
            wl_event_data_if_t* pIf;
            ptrdiff_t u8p; /* 型変換の一時データ退避用 */
            uintptr_t *p;  /* 型変換の一時データ退避用 */
            u8p = reinterpret_cast<ptrdiff_t>(msg);
            p = reinterpret_cast<uintptr_t*>(u8p + sizeof(wl_event_msg_t));
            pIf = reinterpret_cast<wl_event_data_if_t*>(*p);
            WLAN_LOG_DEBUG("[MEZ]WLC_E_IF event occurs. STATUS : %d\n", msg->status);
            WLAN_LOG_DEBUG("    ifidx:%02X\n", pIf->ifidx);
            WLAN_LOG_DEBUG("    opcode:%02X\n", pIf->opcode);
            WLAN_LOG_DEBUG("    reserved:%02X\n", pIf->reserved);
            WLAN_LOG_DEBUG("    bssidx:%02X\n", pIf->bssidx);
            WLAN_LOG_DEBUG("    role:%02X\n", pIf->role);
        }

        pevnbuff = g_pStateMachine->WlanGetEventBuff(0);
        NN_SDK_ASSERT_NOT_NULL(pevnbuff);
        pevnbuff->id = WLAN_EVENT_IF_UP;
        pevnbuff->Result = true;

        Mezcal::m_bWaitIfUpEvent = false;

        return pevnbuff;
    }

    return NULL;
}

void AuthEventProcess(wl_event_msg_t* msg)
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    switch( msg->status )
    {
    case WLC_E_STATUS_SUCCESS:
        nn::os::LockMutex(&Mezcal::m_ConnectionStatusMutex);

        // 接続先のBSSIDが入っているので保存しておく
        Mezcal::m_MezcalConStat.bssid.Set(&msg->addr.octet[0]);

        nn::os::UnlockMutex(&Mezcal::m_ConnectionStatusMutex);
        break;
    default:
        WLAN_LOG_INFO("[MEZ] Failed to send auth frame %d\n", msg->status);
        break;
    }

    return;
}

/* Callback function called by DHD when receiving event (Local mode)------------ */
int Mezcal::EventCallBackFromDriverLocalMode(void* event, void* buf) NN_NOEXCEPT
{
#ifdef TICK_EVENTCB_LOCAL
    uint32_t eventType;
    nn::os::Tick startTick = nn::os::GetSystemTick();
#endif
    NN_SDK_REQUIRES_NOT_NULL(event);
    NN_SDK_REQUIRES_NOT_NULL(buf);

    WlanEvent* pevnbuff = NULL;
    wl_event_msg_t* msg = static_cast<wl_event_msg_t*>(event);
    bool freeBufFlag = true;  // イベントデータのメモリ解放を行って良いか。イベント毎に判断する。基本は解放する。
#ifdef TICK_EVENTCB_LOCAL
    eventType = msg->event_type;
#endif

    nn::mbuf::Mbuf* pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(buf);  // 2つ目の引数bufはmbuf形式

//    WLAN_LOG_DEBUG("[MEZ] %s enter\n", __FUNCTION__);

//    WLAN_LOG_DEBUG("[MEZ] Event code = %ld\n", msg->event_type);
//    WLAN_LOG_DEBUG("[MEZ] Event stat = %ld\n", msg->status);
//    WLAN_LOG_DEBUG("[MEZ] event len = %d\n", msg->datalen);

    if(msg->version == BCM_EVENT_MSG_VERSION)
    {
        switch( msg->event_type )
        {
        case WLC_E_SET_SSID: // 0 APに接続時に発生するイベント。相手先のSSID情報などが入っている。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_SET_SSID(status=%d, datalen=%d)\n", msg->status, msg->datalen);
            pevnbuff = SetSsidEventProcess( msg );
            break;

        case WLC_E_JOIN:
            WLAN_LOG_DEBUG("[MEZ] WLC_E_JOIN(status=%d)\n", msg->status);
            break;

        case WLC_E_START:
            WLAN_LOG_DEBUG("[MEZ] WLC_E_START(status=%d)\n", msg->status);
            break;

        case WLC_E_AUTH:    // 3 Authenticate フレーム送信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_AUTH(status=%d)\n", msg->status);
            // 接続先のBSSIDが入っている
            AuthEventProcess(msg);
            break;

        case WLC_E_AUTH_IND: // 4 Authenticate フレームを受信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_AUTH_IND(status=%d)\n", msg->status);
            break;

        case WLC_E_DEAUTH:  // 5 Deauth フレーム送信時に発生するイベント。
            // 処理不要
            //pevnbuff = DeauthenticateEventProcess( msg );
            break;
        case WLC_E_DEAUTH_IND: // 6 Deauth フレーム受信時に発生するイベント。
            pevnbuff = DeauthIndEventProcess( msg );
            break;

        case WLC_E_ASSOC: // 7 Associate 要求完了時に発生するイベント。
        case WLC_E_REASSOC:
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC(type=%d, status=%d, reason=%d)\n", msg->event_type, msg->status, msg->reason);
            AssociationEventProcess( msg );
            break;

        case WLC_E_ASSOC_IND: // 8 Associate 要請受領時に発生するイベント。
        case WLC_E_REASSOC_IND:
            pevnbuff = AssociactionIndEventProcess( msg );
            break;

        case WLC_E_DISASSOC: // Disassociate フレーム送信時に発生するイベント。
            pevnbuff = DisassociateEventProcess( msg );
            break;

        case WLC_E_DISASSOC_IND: // 12 Disassociate フレーム受信時に発生するイベント。
            pevnbuff = DisassocIndEventProcess( msg );
            break;

        case WLC_E_LINK: // 16 LINK状態に変化が起きた時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_LINK(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            pevnbuff = LinkChangeEventProcess( msg );
            break;

        case WLC_E_BCNSENT_IND:  // BSS状態でビーコン送信時に発生するイベント
            WLAN_LOG_DEBUG("*");
            break;

        case WLC_E_BCNLOST_MSG: // 31 ビーコンロスト時に発生するイベント。 bwl_set_bcnlossthreshで設定した回数分連続で落とすと発生する。
            pevnbuff = BeaconLostEventProcess( msg );
            break;

        case WLC_E_IF:  // 54 I/Fに変化があったときに発生するイベント。
            pevnbuff = InterfaceEventProcess( msg );
            break;

         case WLC_E_ACTION_FRAME:  // 59
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ACTION_FRAME(status=%d, len=%d)\n", msg->status, msg->datalen);
            ProcessActionFrameEvent( msg, pMbuf );
            freeBufFlag = false; // ActionFrameは貯めておくので解放しない。
            break;

        case WLC_E_ESCAN_RESULT: // 69 ESCANの結果が入っているイベント。BSS情報そのものが入っているときと、スキャン完了/中断を意味する時の2通りがある。
            if( GetAcsdFlag() == true )
            {
                int ret;
                ret = acsd_scan_results(nn::mbuf::MbufTod(pMbuf));
                if( ret != ACSD_OK )
                {
                    WLAN_LOG_ERROR("acsd_scan_results() failed (%d)\n", ret);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAcsdError());
                }
            }
            else
            {
                pevnbuff = ScanResultEventProcess( msg );
            }
            break;

        case WLC_E_ACTION_FRAME_RX:  // 75
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ACTION_FRAME_RX(status=%d, len=%d)\n", msg->status, msg->datalen);
            ProcessActionFrameEventEx( msg, pMbuf );
            freeBufFlag = false; // ActionFrameは貯めておくので解放しない。
            break;

        case WLC_E_ASSOC_REQ_IE: // 87
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC_REQ_IE(status=%d)\n", msg->status);
            break;

        case WLC_E_ASSOC_RESP_IE: // 88
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC_RESP_IE(status=%d)\n", msg->status);
            break;
#ifdef ENABLE_DCS_OPERATION
            // For DCS operation
        case WLC_E_DCS_REQUEST:
        case WLC_E_SCAN_COMPLETE:
        case WLC_E_PKTDELAY_IND:
        case WLC_E_TXFAIL_THRESH:
            {
                WLAN_LOG_DEBUG("[MEZ] %ld event occurs\n", msg->event_type);
                acsd_event_recv(reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pMbuf)),
                        sizeof(bcm_event_t) + msg->datalen + 2);
            }
            break;
#endif
        case WLC_E_HTSFSYNC:
            {
                WLAN_LOG_DEBUG("[MEZ] WLC_E_HTSFSYNC\n");
                void* evData = reinterpret_cast<void*>(
                        reinterpret_cast<char*>(nn::mbuf::MbufTod(pMbuf)) + sizeof(bcm_event_t));
                nn::os::Tick sysTick = nn::os::GetSystemTick();
                int64_t tsfTick64Value = 0;  // unit is micro sec.
                m_TsfTimerValue.low = (reinterpret_cast<uint32_t*>(evData))[0];
                m_TsfTimerValue.high = (reinterpret_cast<uint32_t*>(evData))[1];
                tsfTick64Value = static_cast<int64_t>(m_TsfTimerValue.low);
                tsfTick64Value |= ((static_cast<int64_t>(m_TsfTimerValue.high) << 32) & 0xffffffff00000000);

                nn::os::LockMutex(&m_TsfMutex);
                m_DeltaTimeBetweenTsfAndSys = sysTick.ToTimeSpan().GetMicroSeconds() - tsfTick64Value;
                WLAN_LOG_DEBUG("[TSF]Sys:%lld[us], TSF:%lld[us], delta:%lld[us]\n",
                        sysTick.ToTimeSpan().GetMicroSeconds(), tsfTick64Value, m_DeltaTimeBetweenTsfAndSys);
                nn::os::UnlockMutex(&m_TsfMutex);
            }
            break;

        case WLC_E_DHD_NOTIFY:
            WLAN_LOG_ERROR("Wireless chip fatal error\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultChipFatalError());
            break;

        default:
            WLAN_LOG_DEBUG("[MEZ] unknown event[id=%ld, status=%d]\n",
                   msg->event_type, msg->status);
            DisplayDhdEvent(msg);
        }

        if(pevnbuff != NULL)
        {
            /* Event Bufferは、StateMachine側で解放される */
            g_pStateMachine->PostEventMessage( pevnbuff );
        }
    }
    else
    {
        WLAN_LOG_DEBUG("[MEZ] Unknown event version(%d)\n", msg->version);
    }

//    WLAN_LOG_DEBUG("[MEZ] %s leave\n", __FUNCTION__);

    if( freeBufFlag == true )
    {
        nn::mbuf::MbufFreem(pMbuf);
    }
#ifdef TICK_EVENTCB_LOCAL
    NN_SDK_LOG("%lld [usec] evid=%d @ EventCb\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMicroSeconds(), eventType);
#endif

    return BWL_ERR_SUCCESS;
} //NOLINT(impl/function_size)

/* Callback function called by DHD when receiving data (Local mode)------------ */
int Mezcal::RxCallBackFromDriverLocalMode(void *buf) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
#ifdef TICK_RXCB_LOCAL
    nn::os::Tick startTick = nn::os::GetSystemTick();
#endif

    /* データ受信時の処理 */
    if(buf != NULL)
    {
        nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);

        auto result = g_pStateMachine->PushRxBuffer(pMbuf);
        if( result.IsFailure() )
        {
            WLAN_LOG_DEBUG("[MEZ]Discard rx data\n");
            nn::mbuf::MbufFreem(pMbuf);
        }
#if 0
        if(g_pStateMachine->IsAuthorized() == true)
        {
            WLAN_LOG_DEBUG("To upper layer\n");
            //DumpFunc(reinterpret_cast<void*>(nn::mbuf::MbufTod(pMbuf)), nn::mbuf::MbufLength(pMbuf, NULL));

            // RxEntriesに受信データを送る。
            // 受け取り可能なRxEntryが存在していれば、保管される。
            // 受け取れなかったまたは、そのようなRxEntryが存在していなかった場合、破棄する。
            auto result = g_pStateMachine->PushRxBuffer(pMbuf);
            if( result.IsFailure() )
            {
                WLAN_LOG_DEBUG("[MEZ]Discard rx data\n");
                nn::mbuf::MbufFreem(pMbuf);
            }
        }
        else if(g_pStateMachine->IsConnectAndWaitAuthorized() == true)
        {
            WLAN_LOG_DEBUG("EAPoL?\n");

            // TORIAEZU
            nn::mbuf::MbufFreem(pMbuf);
        }
        else
        {
            WLAN_LOG_DEBUG("Discard rx data due to not authorized\n");
            nn::mbuf::MbufFreem(pMbuf);
        }
#endif
    }
#ifdef TICK_RXCB_LOCAL
    NN_SDK_LOG("%lld [usec] @ RxCb\n", (nn::os::GetSystemTick() - startTick).ToTimeSpan().GetMicroSeconds());
#endif
    return BWL_ERR_SUCCESS;
}

/* Callback function called by DHD when receiving event (Infra mode)------------ */
int Mezcal::EventCallBackFromDriverInfraMode(void* event, void* buf) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(event);
    NN_SDK_REQUIRES_NOT_NULL(buf);

    WlanEvent* pevnbuff = NULL;
    wl_event_msg_t* msg = static_cast<wl_event_msg_t*>(event);
    bool freeBufFlag = true;  // イベントデータのメモリ解放を行って良いか。イベント毎に判断する。基本は解放する。

    nn::mbuf::Mbuf* pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(buf);  // 2つ目の引数bufはmbuf形式

//    WLAN_LOG_DEBUG("[MEZ] %s enter\n", __FUNCTION__);

//    WLAN_LOG_DEBUG("[MEZ] Event code = %ld\n", msg->event_type);
//    WLAN_LOG_DEBUG("[MEZ] Event stat = %ld\n", msg->status);
//    WLAN_LOG_DEBUG("[MEZ] event len = %d\n", msg->datalen);

    if(msg->version == BCM_EVENT_MSG_VERSION)
    {
        switch( msg->event_type )
        {
        case WLC_E_SET_SSID: // 0 APに接続時に発生するイベント。相手先のSSID情報などが入っている。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_SET_SSID(status=%d, datalen=%d)\n", msg->status, msg->datalen);
            pevnbuff = SetSsidEventProcess( msg );  // TODO to change the function so that the event is not sent up to StateMachine.
            break;

        case WLC_E_AUTH:    // 3 Authenticate フレーム送信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_AUTH(status=%d)\n", msg->status);
            break;

        case WLC_E_AUTH_IND: // 4 Authenticate フレームを受信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_AUTH_IND(status=%d)\n", msg->status);
            break;

        case WLC_E_DEAUTH:  // 5 Deauth フレーム送信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_DEAUTH(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            // 処理不要
            //pevnbuff = DeauthenticateEventProcess( msg );
            break;
        case WLC_E_DEAUTH_IND: // 6 Deauth フレーム受信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_DEAUTH_IND(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            pevnbuff = DeauthIndEventProcess( msg );
            break;

        case WLC_E_ASSOC: // 7 Associate 要求完了時に発生するイベント。
        case WLC_E_REASSOC:
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC(type=%d, status=%d, reason=%d)\n", msg->event_type, msg->status, msg->reason);
            AssociationEventProcess( msg );
            break;

        case WLC_E_ASSOC_IND: // 8 Associate 要請受領時に発生するイベント。
            // In infra mode, this event should not come up.
            WLAN_LOG_ERROR("WLC_E_ASSOC(status=%d)\n", msg->status);
            break;

        case WLC_E_REASSOC_IND:
            // In infra mode, this event should not come up.
            WLAN_LOG_ERROR("WLC_E_REASSOC(status=%d)\n", msg->status);
            break;

        case WLC_E_DISASSOC: // Disassociate フレーム送信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_DISASSOC(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            pevnbuff = DisassociateEventProcess( msg );
            break;

        case WLC_E_DISASSOC_IND: // 12 Disassociate フレーム受信時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_DISASSOC_IND(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            pevnbuff = DisassocIndEventProcess( msg );
            break;

        case WLC_E_LINK: // 16 LINK状態に変化が起きた時に発生するイベント。
            WLAN_LOG_DEBUG("[MEZ] WLC_E_LINK(status=%d, reason=%d, auth_type=%d, datalen=%d)\n", msg->status, msg->reason, msg->auth_type, msg->datalen);
            pevnbuff = LinkChangeEventProcess( msg );
            break;

        case WLC_E_BCNSENT_IND:  // BSS状態でビーコン送信時に発生するイベント
            WLAN_LOG_DEBUG("*");
            break;

        case WLC_E_BCNLOST_MSG: // 31 ビーコンロスト時に発生するイベント。 bwl_set_bcnlossthreshで設定した回数分連続で落とすと発生する。
            pevnbuff = BeaconLostEventProcess( msg );
            break;

        case WLC_E_IF:  // 54 I/Fに変化があったときに発生するイベント。
            pevnbuff = InterfaceEventProcess( msg );
            break;

        case WLC_E_ACTION_FRAME:  // 59
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ACTION_FRAME(status=%d, len=%d)\n", msg->status, msg->datalen);
            ProcessActionFrameEvent( msg, pMbuf );
            freeBufFlag = false; // ActionFrameは貯めておくので解放しない。
            break;

        case WLC_E_ESCAN_RESULT: // 69 ESCANの結果が入っているイベント。BSS情報そのものが入っているときと、スキャン完了/中断を意味する時の2通りがある。
#if defined(USE_WPA_SUPPLICANT)
            // if using wpa supplicant, scan complete event will be occured by wpa supplicant.
            if( msg->status != WLC_E_STATUS_SUCCESS )
            {
                pevnbuff = ScanResultEventProcess( msg );
            }
#else
            pevnbuff = ScanResultEventProcess( msg );
#endif
            break;

        case WLC_E_ACTION_FRAME_RX:  // 75
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ACTION_FRAME_RX(status=%d, len=%d)\n", msg->status, msg->datalen);
            ProcessActionFrameEventEx( msg, pMbuf );
            freeBufFlag = false; // ActionFrameは貯めておくので解放しない。
            break;

        case WLC_E_ASSOC_REQ_IE: // 87
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC_REQ_IE(status=%d)\n", msg->status);
            break;

        case WLC_E_ASSOC_RESP_IE: // 88
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ASSOC_RESP_IE(status=%d)\n", msg->status);
            break;

        case WLC_E_HTSFSYNC:
            {
                WLAN_LOG_DEBUG("[MEZ] WLC_E_HTSFSYNC\n");
                void* evData = reinterpret_cast<void*>(
                        reinterpret_cast<char*>(nn::mbuf::MbufTod(pMbuf)) + sizeof(bcm_event_t));
                nn::os::Tick sysTick = nn::os::GetSystemTick();
                int64_t tsfTick64Value = 0;  // unit is micro sec.
                m_TsfTimerValue.low = (reinterpret_cast<uint32_t*>(evData))[0];
                m_TsfTimerValue.high = (reinterpret_cast<uint32_t*>(evData))[1];
                tsfTick64Value = static_cast<int64_t>(m_TsfTimerValue.low);
                tsfTick64Value |= ((static_cast<int64_t>(m_TsfTimerValue.high) << 32) & 0xffffffff00000000);

                nn::os::LockMutex(&m_TsfMutex);
                m_DeltaTimeBetweenTsfAndSys = sysTick.ToTimeSpan().GetMicroSeconds() - tsfTick64Value;
                WLAN_LOG_DEBUG("[TSF]Sys:%lld[us], TSF:%lld[us], delta:%lld[us]\n",
                        sysTick.ToTimeSpan().GetMicroSeconds(), tsfTick64Value, m_DeltaTimeBetweenTsfAndSys);
                nn::os::UnlockMutex(&m_TsfMutex);
            }
            break;

        case WLC_E_DHD_NOTIFY:
            WLAN_LOG_ERROR("Wireless chip fatal error\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultChipFatalError());
            break;

        default:
            WLAN_LOG_DEBUG("[MEZ] unknown event[id=%ld, status=%d]\n",
                   msg->event_type, msg->status);
            DisplayDhdEvent(msg);

        }

        if(pevnbuff != NULL)
        {
            /* Event Bufferは、StateMachine側で解放される */
            g_pStateMachine->PostEventMessage( pevnbuff );
        }
    }
    else
    {
        WLAN_LOG_DEBUG("[MEZ] Unknown event version(%d)\n", msg->version);
    }

//    WLAN_LOG_DEBUG("[MEZ] %s leave\n", __FUNCTION__);
#if defined(USE_WPA_SUPPLICANT)
    // WPA_SUPP: forward events to WPA supplicant, supplicant makes own copy of buffer
    g_pStateMachine->m_pSuppFuncs->ForwardEvent(event, buf);
#endif

    if( freeBufFlag == true )
    {
        nn::mbuf::MbufFreem(pMbuf);
    }

    return BWL_ERR_SUCCESS;
} //NOLINT(impl/function_size)

/* Callback function called by DHD when receiving data (Infra mode)------------ */
int Mezcal::RxCallBackFromDriverInfraMode(void *buf) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    /* データ受信時の処理 */
    if(buf != NULL)
    {
        nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);
#ifdef ENABLE_WOWL_AGING_DEBUG
        ParseTcpData(reinterpret_cast<void*>(nn::mbuf::MbufTod(pMbuf)), nn::mbuf::MbufLength(pMbuf, NULL), &m_ackNum);
#endif
#ifdef PRINT_THROUGHPUT
        throughput_received++;
        throughput_rxBytes += nn::mbuf::MbufLength(pMbuf, NULL);
        if( (nn::os::GetSystemTick() - g_ThroughTick).ToTimeSpan().GetSeconds() > MEASURE_INTERVAL )
        {
            double throughput = ( static_cast<double>(throughput_rxBytes) * 8.0f / static_cast<double>(MEASURE_INTERVAL) ) / ( 1024 * 1024 ); // Mbps
            NN_SDK_LOG("[WLAN]RX : %.2f[Mbps], recv counts:%d\n", throughput, throughput_received);
            g_ThroughTick = nn::os::GetSystemTick();
            throughput_received = 0;
            throughput_rxBytes = 0;
        }
#endif

        if (pMbuf->_protocol == WpaSupplicant::EtherProtocolPae)
        {
            WLAN_LOG_DEBUG("[MEZ]To upper layer\n");
#if defined(USE_WPA_SUPPLICANT)
            // WPA_SUPP: Forward RxEapol to WPA supplicant, supplicant makes own copy of buffer
            WLAN_LOG_DEBUG("WDMMZ: Rx EAPOL received from DHD, will forward to supplicant\n");
            g_pStateMachine->m_pSuppFuncs->ForwardRxEapol(buf);  // not use wpa supp func until wpa supp becomes stable
#endif
            nn::mbuf::MbufFreem(pMbuf);
        }
        else if(g_pStateMachine->IsAuthorized() == true)
        {
            WLAN_LOG_DEBUG("To upper layer\n");
            //DumpFunc(reinterpret_cast<void*>(nn::mbuf::MbufTod(pMbuf)), nn::mbuf::MbufLength(pMbuf, NULL));

            // RxEntriesに受信データを送る。
            // 受け取り可能なRxEntryが存在していれば、保管される。
            // 受け取れなかったまたは、そのようなRxEntryが存在していなかった場合、破棄する。
            auto result = g_pStateMachine->PushRxBuffer(pMbuf);
            if( result.IsFailure() )
            {
                WLAN_LOG_DEBUG("[MEZ]Discard rx data\n");
                nn::mbuf::MbufFreem(pMbuf);
            }
        }
        else if(g_pStateMachine->IsConnectAndWaitAuthorized() == true)
        {
            WLAN_LOG_DEBUG("EAPoL?\n");
            // TODO: NOT SURE IF THIS STATEMENT APPLIES, ALREADY IN WDM CODE
            if (pMbuf->_protocol == WpaSupplicant::EtherProtocolPae)
            {
#if defined(USE_WPA_SUPPLICANT)
                // WPA_SUPP: Forward RxEapol to WPA supplicant, supplicant makes own copy of buffer
                WLAN_LOG_DEBUG("WDMMZ: Rx EAPOL received from DHD, will forward to supplicant\n");
                g_pStateMachine->m_pSuppFuncs->ForwardRxEapol(buf);  // not use wpa supp func until wpa supp becomes stable
#endif
            }

            // TORIAEZU
            nn::mbuf::MbufFreem(pMbuf);
        }
        else
        {
            WLAN_LOG_DEBUG("Discard rx data due to not authorized\n");
            nn::mbuf::MbufFreem(pMbuf);
        }
    }

    return BWL_ERR_SUCCESS;
}

/* Callback function called by DHD when receiving event (Detect mode)------------ */
int Mezcal::EventCallBackFromDriverDetectMode(void* event, void* buf) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(event);
    NN_SDK_REQUIRES_NOT_NULL(buf);

    WlanEvent* pevnbuff = NULL;
    wl_event_msg_t* msg = static_cast<wl_event_msg_t*>(event);
    bool freeBufFlag = true;  // イベントデータのメモリ解放を行って良いか。イベント毎に判断する。基本は解放する。

    nn::mbuf::Mbuf* pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(buf);  // 2つ目の引数bufはmbuf形式

    if(msg->version == BCM_EVENT_MSG_VERSION)
    {
        switch( msg->event_type )
        {
        case WLC_E_IF:  // 54 I/Fに変化があったときに発生するイベント。
            pevnbuff = InterfaceEventProcess( msg );
            break;

        case WLC_E_ESCAN_RESULT: // 69 ESCANの結果が入っているイベント。BSS情報そのものが入っているときと、スキャン完了/中断を意味する時の2通りがある。
            pevnbuff = ScanResultEventProcess( msg );
            break;

        case WLC_E_ACTION_FRAME_RX:
            WLAN_LOG_DEBUG("[MEZ] WLC_E_ACTION_FRAME_RX(status=%d, len=%d)\n", msg->status, msg->datalen);
            if( g_aloeAfCount >= 0 )
            {
                g_aloeAfCount++;
                WLAN_LOG_DEBUG("Cached action frame has come (%d times)\n", g_aloeAfCount);
            }
            ProcessActionFrameEventEx( msg, pMbuf );
            freeBufFlag = false; // ActionFrameは貯めておくので解放しない。
            break;

        case WLC_E_HTSFSYNC:
            {
                WLAN_LOG_DEBUG("[MEZ] WLC_E_HTSFSYNC\n");
                void* evData = reinterpret_cast<void*>(
                        reinterpret_cast<char*>(nn::mbuf::MbufTod(pMbuf)) + sizeof(bcm_event_t));
                nn::os::Tick sysTick = nn::os::GetSystemTick();
                int64_t tsfTick64Value = 0;  // unit is micro sec.
                m_TsfTimerValue.low = (reinterpret_cast<uint32_t*>(evData))[0];
                m_TsfTimerValue.high = (reinterpret_cast<uint32_t*>(evData))[1];
                tsfTick64Value = static_cast<int64_t>(m_TsfTimerValue.low);
                tsfTick64Value |= ((static_cast<int64_t>(m_TsfTimerValue.high) << 32) & 0xffffffff00000000);

                nn::os::LockMutex(&m_TsfMutex);
                m_DeltaTimeBetweenTsfAndSys = sysTick.ToTimeSpan().GetMicroSeconds() - tsfTick64Value;
                WLAN_LOG_DEBUG("[TSF]Sys:%lld[us], TSF:%lld[us], delta:%lld[us]\n",
                        sysTick.ToTimeSpan().GetMicroSeconds(), tsfTick64Value, m_DeltaTimeBetweenTsfAndSys);
                nn::os::UnlockMutex(&m_TsfMutex);
            }
            break;

        case WLC_E_DHD_NOTIFY:
            WLAN_LOG_ERROR("Wireless chip fatal error\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultChipFatalError());
            break;

        default:
            WLAN_LOG_DEBUG("[MEZ] unknown event[id=%ld, status=%d]\n",
                   msg->event_type, msg->status);
            DisplayDhdEvent(msg);
        }

        if(pevnbuff != NULL)
        {
            /* Event Bufferは、StateMachine側で解放される */
            g_pStateMachine->PostEventMessage( pevnbuff );
        }
    }
    else
    {
        WLAN_LOG_DEBUG("[MEZ] Unknown event version(%d)\n", msg->version);
    }

//    WLAN_LOG_DEBUG("[MEZ] %s leave\n", __FUNCTION__);

    if( freeBufFlag == true )
    {
        nn::mbuf::MbufFreem(pMbuf);
    }

    return BWL_ERR_SUCCESS;
} //NOLINT(impl/function_size)

int Mezcal::RxCallBackFromDriverDetectMode(void *buf) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // データは受信することはないはずだが、その場合はそのまま破棄する。
    if(buf != NULL)
    {
        nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);
        nn::mbuf::MbufFreem(pMbuf);
    }

    return BWL_ERR_SUCCESS;
}

/* 無線ドライバからのコールバック関数(データ送信) -------------------------- */
int Mezcal::TxCompletionCallBackFromDriver(void *buf, int result) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    // resultが失敗を意味する値の場合ログを出す。上位層に送信失敗通知を送る？
    WLAN_LOG_DEBUG("Tx result : %d\n", result);

    if(buf != NULL)
    {
        nn::mbuf::Mbuf *pMbuf;
        pMbuf = reinterpret_cast<nn::mbuf::Mbuf*>(buf);
        //DumpFunc(reinterpret_cast<void*>(nn::mbuf::MbufTod(pMbuf)), nn::mbuf::MbufLength(pMbuf, NULL));
        nn::mbuf::MbufFreem(pMbuf);

#ifdef ENABLE_LOCAL_TXFLOW_CTRL
        nn::os::LockMutex(&m_TxCountMutex);
        m_TxCount--;
#ifdef TICK_TXCOMPCB
        if( g_TxValidateFlag == 1 )
        {
            NN_SDK_LOG("tx free %lld [usec]\n",
                    (nn::os::GetSystemTick() - g_TxValidateTick).ToTimeSpan().GetMicroSeconds());
            g_TxValidateFlag = 2;
        }
#endif
        nn::os::UnlockMutex(&m_TxCountMutex);
#endif
    }

    return BWL_ERR_SUCCESS;
}

/* TX Packet --------------------------------------------------------------- */
nn::Result Mezcal::PutFrame(nn::mbuf::Mbuf* pMbuf) NN_NOEXCEPT
{
    nn::Result result;
    int bwl_ret = -1;

    WLAN_LOG_DEBUG("[MEZ]Tx Data Dump\n");
    if( WlanLogLevel >= LogLevel_Debug )
    {
        DumpFunc(nn::mbuf::MbufTod(pMbuf), nn::mbuf::MbufLength(pMbuf, NULL));
    }

#ifdef ENABLE_LOCAL_TXFLOW_CTRL
    nn::os::LockMutex(&m_TxCountMutex);
#ifdef TICK_TXCOMPCB
    g_TxValidateTick = nn::os::GetSystemTick();
    if( g_TxValidateFlag == 0 || g_TxValidateFlag == 2 )
    {
        g_TxValidateTick = nn::os::GetSystemTick();
        g_TxValidateFlag = 1;
    }
    else
    {
        NN_SDK_LOG("Tx Overtaking!\n");
    }
#endif

    if( m_TxCount >= TxCountMax )
    {
        WLAN_LOG_DEBUG("Tx queue is full (%d)\n", m_TxCount);
        nn::os::UnlockMutex(&m_TxCountMutex);
        return ResultTxQueueIsFull();
    }
    m_TxCount++;
    nn::os::UnlockMutex(&m_TxCountMutex);
#endif

    bwl_ret = bwl_txpkt(m_InterfaceHandle, pMbuf);
    if( bwl_ret == BCME_OK )
    {
        result = ResultSuccess();
    }
    else
    {
#ifdef ENABLE_LOCAL_TXFLOW_CTRL
        nn::os::LockMutex(&m_TxCountMutex);
        m_TxCount--;
        nn::os::UnlockMutex(&m_TxCountMutex);
#endif

        if( bwl_ret == BCME_NORESOURCE )
        {
            result = ResultTxQueueIsFull();
        }
        else
        {
            result = ResultCommandFailure();
            WLAN_LOG_ERROR("Failed to putframe due to %d\n", bwl_ret);
        }
    }

    return result;
}

// ActionFrame一発送信
nn::Result Mezcal::PutActionFrameOneShot(const char* pData, uint16_t size, uint8_t dstMac[MacAddress::MacAddressSize],
        uint8_t bssid[MacAddress::MacAddressSize], uint32_t channel, uint32_t dwellTime) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_NOT_NULL(dstMac);

    struct ether_addr eaBssid = {0};
    struct ether_addr eaDstMacAddr;

    // If bssid is zero macaddr, use own mac address
    if( std::memcmp(&eaBssid.octet[0], &bssid[0], ETHER_ADDR_LEN) == 0 )
    {
        std::memcpy(&eaBssid, &m_OwnMacAddr, ETHER_ADDR_LEN);
    }
    else
    {
        std::memcpy(&eaBssid.octet[0], &bssid[0], ETHER_ADDR_LEN);
    }

    // dst mac addr
    std::memcpy(&eaDstMacAddr.octet[0], &dstMac[0], ETHER_ADDR_LEN);

    WLAN_LOG_DEBUG("[MEZ]Action frame data size : %d\n", size);

    nn::Result result;
    int32_t bwl_ret = bwl_send_one_af(m_InterfaceHandle, reinterpret_cast<uint8_t*>(const_cast<char*>(pData)),
            size, &eaBssid, &eaDstMacAddr, channel, dwellTime);
    if( bwl_ret == BCME_OK )
    {
        result = ResultSuccess();
    }
    else
    {
        if( bwl_ret == BCME_NOMEM )
        {
            result = ResultNoMemory();
        }
        else
        {
            result = ResultCommandFailure();
            WLAN_LOG_ERROR("[MEZ]%s is failure. Reason : %d\n", __FUNCTION__, bwl_ret);
        }
    }

    return result;
}

bool Mezcal::PutActionFramePeriodically(const char* pData, uint16_t size, uint8_t bssid[MacAddress::MacAddressSize],
        uint32_t count, uint32_t interval) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_NOT_NULL(bssid);

    struct ether_addr eaDstAddr = {
            {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF} // 送信先はBroadcastアドレス
    };
    struct ether_addr eaBssid;

    std::memcpy(&eaBssid.octet[0], &bssid[0], ETHER_ADDR_LEN);

    WLAN_LOG_DEBUG("[MEZ]Periodic action frame data size : %d\n", size);

    int32_t bwl_ret = bwl_send_periodic_af(m_InterfaceHandle, reinterpret_cast<uint8_t*>(const_cast<char*>(pData)),
            size, &eaBssid, &eaDstAddr, count, interval);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

bool Mezcal::GetBeaconInterval(uint32_t* pOutInterval) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    NN_SDK_REQUIRES_NOT_NULL(pOutInterval);

    int interval = 0;
    int32_t bwl_ret = wlu_get(m_InterfaceHandle, WLC_GET_BCNPRD, &interval, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    WLAN_LOG_DEBUG("[MEZ]Beacon interval = %d[ms]\n", interval);

    *pOutInterval = interval;

    return true;
}

bool Mezcal::CancelPutActionFramePeriodically() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t bwl_ret = bwl_cancel_periodic_af(m_InterfaceHandle);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);

    return true;
}

bool Mezcal::GetRssiForSta(int32_t* rssi) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_NOT_NULL(rssi);

    int32_t val = 0;
    bool ret = false;
    int retVal = wlu_get(m_InterfaceHandle, WLC_GET_RSSI, &val, sizeof(val));
    if( retVal < 0 )
    {
        ret = false;
        *rssi = -128;
        WLAN_LOG_INFO("%s failed due to %d\n", __FUNCTION__, retVal);
    }
    else
    {
        ret = true;
        *rssi = val;
    }

    return ret;
}

bool Mezcal::GetRssiForAp(int32_t* rssi, const MacAddress& mac) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_NOT_NULL(rssi);

    scb_val_t val;
    std::memcpy(&val.ea.octet[0], mac.GetMacAddressData(), ETHER_ADDR_LEN);

    bool ret = false;
    int retVal = wlu_get(m_InterfaceHandle, WLC_GET_RSSI, &val, sizeof(val));
    if( retVal < 0 )
    {
        ret = false;
        *rssi = -128;
        WLAN_LOG_INFO("%s failed due to %d\n", __FUNCTION__, retVal);
    }
    else
    {
        ret = true;
        *rssi = static_cast<int32_t>(val.val);
    }

    return ret;
}

bool Mezcal::SetTsfTimerEventmask(bool enable) NN_NOEXCEPT
{
    uint8_t eventmask[WL_EVENTING_MASK_LEN];
    int32_t bwl_ret = bwl_get_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    if( enable == true )
    {
        setbit(eventmask, WLC_E_HTSFSYNC);
    }
    else
    {
        clrbit(eventmask, WLC_E_HTSFSYNC);
    }
    bwl_ret = bwl_set_eventmask(m_InterfaceHandle, eventmask);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(bwl_ret);
    return true;
}

bool Mezcal::GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    NN_SDK_REQUIRES_NOT_NULL(pOutDeltaTime);

    nn::os::LockMutex(&m_TsfMutex);
    *pOutDeltaTime = m_DeltaTimeBetweenTsfAndSys;
    nn::os::UnlockMutex(&m_TsfMutex);

    return true;
}

bool Mezcal::IsMatchAssocList(uint8_t macAddr[MacAddress::MacAddressSize]) NN_NOEXCEPT
{
    struct ether_addr cliList[ConnectableClientsCountMax];
    for( uint8_t i = 0; i < ConnectableClientsCountMax; i++ )
    {
        std::memset(&cliList[i], 0, sizeof(ether_addr));
    }
    uint32_t count = ConnectableClientsCountMax;
    bwl_get_assoclist( m_InterfaceHandle, cliList, &count);

    for( uint8_t i = 0; i < count; i++ )
    {
        if( std::memcmp(cliList[i].octet, macAddr, ETHER_ADDR_LEN) == 0 )
        {
            return true;
        }
    }
    return false;
}

void Mezcal::DumpAcsdResults(wl_chanim_stats_t* pStats, chan_score_t* pScores) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStats);
    NN_SDK_REQUIRES_NOT_NULL(pScores);

    chanim_stats_t* pSinglesSats;

    /* dump the channel statistics */
    NN_SDK_LOG("chanspec tx   inbss   obss   nocat   nopkt   doze     txop     "
           "goodtx  badtx   glitch   badplcp  knoise chanIdle  timestamp\n");
    for( int i = 0; i < pStats->count; i++ )
    {
        pSinglesSats = &pStats->stats[i];
        NN_SDK_LOG("0x%4x\t", pSinglesSats->chanspec);
        for (int j = 0; j < CCASTATS_MAX; j++ )
        {
            NN_SDK_LOG("%d\t", pSinglesSats->ccastats[j]);
        }
        NN_SDK_LOG("%d\t%d\t%d\t%d\t%d", (pSinglesSats->glitchcnt), (pSinglesSats->badplcp),
            pSinglesSats->bgnoise, (pSinglesSats->chan_idle), (pSinglesSats->timestamp));
        NN_SDK_LOG("\n");
    }

    /* dump the score of channel selection */
    for( int i = 0; i < pScores->count; i++ )
    {
        NN_SDK_LOG("Chan:%#x, bss number:%d, intf score:%d, bg noise:%d\n",
        pScores->chan_score_info[i].chan, pScores->chan_score_info[i].bss_score,
        pScores->chan_score_info[i].intf_score, pScores->chan_score_info[i].bg_noise_avg);
    }
}

// Coex desense levelの設定
void Mezcal::SetBtcDesenseLevel(int level) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int rxgain = level;
    int ret = wlu_iovar_set(m_InterfaceHandle, "phy_btc_restage_rxgain", &rxgain, sizeof(int));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);
}

bool Mezcal::GetFwVersion(WlanIoctlResult* pOutResult)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResult);
    NN_SDK_REQUIRES_NOT_NULL(pOutResult->pStr);
    NN_SDK_REQUIRES_GREATER_EQUAL(pOutResult->size, WLC_IOCTL_SMLEN);
    int32_t ret = wlu_iovar_get(m_InterfaceHandle, "ver", pOutResult->pStr, WLC_IOCTL_SMLEN);
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);
    return true;
}

bool Mezcal::SetSimultaneousTx(SimultaneousTxParam param) NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);

    int32_t ret = BCME_OK;
    char var[256];
    uint32_t flag = 82;
    uint32_t val;
    uint32_t offset = sizeof(uint32_t);

    switch( param )
    {
    case SimultaneousTxParam_Default:
        val = 0x40;
        break;
    case SimultaneousTxParam_1Antenna:
        val = 0x400;
        break;
    case SimultaneousTxParam_2Antenna:
        val = 0x0;
        break;
    default:
        val = 0x40;
        break;
    }

    memset(var, 0, sizeof(var));
    memcpy(var, reinterpret_cast<char*>(&flag), sizeof(uint32_t));
    memcpy(&var[offset], reinterpret_cast<char*>(&val), sizeof(uint32_t));
    ret = wlu_iovar_set(m_InterfaceHandle, "btc_params", var, sizeof(var));
    WLAN_ABORT_UNLESS_DRIVER_RESULT_SUCCESS(ret);

    return true;
}

void Mezcal::DisableFixedDesense() NN_NOEXCEPT
{
    WLAN_LOG_DEBUG("[MEZ] %s\n", __FUNCTION__);
    SetBtcDesenseLevel(0);
}

bool Mezcal::GetChannelStats(ChannelStats* pStats, size_t length, uint32_t* pCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStats);
    NN_SDK_REQUIRES_GREATER_EQUAL(length, BWL_MAX_CHANNEL_NUM);
    NN_SDK_REQUIRES_NOT_NULL(pCount);

    int32_t bwl_ret;
    wl_chanim_stats_t* pWlStats;
    char* pBuf = new char[BWL_GET_CHANIM_STATS_BUF_SZ];
    if( pBuf == NULL )
    {
        WLAN_LOG_ERROR("%s: Failed to allocate buffer\n", __FUNCTION__);
        return false;
    }
    pWlStats = reinterpret_cast<wl_chanim_stats_t*>(pBuf);

    bwl_ret = bwl_get_chanim_stats(m_InterfaceHandle, pWlStats, BWL_GET_CHANIM_STATS_BUF_SZ);
    if( bwl_ret != BCME_OK )
    {
        WLAN_LOG_ERROR("%s: bwl_get_chanim_stats() failed.\n", __FUNCTION__);
        delete[] pBuf;
        return false;
    }

    chanim_stats_t* pSinglesSats;

    /* dump the channel statistics */
    if( nn::wlan::WlanLogLevel >= nn::wlan::LogLevel_Info )
    {
        NN_SDK_LOG("chanspec tx   inbss   obss   nocat   nopkt   doze     txop     "
           "goodtx  badtx   glitch   badplcp  knoise chanIdle  timestamp\n");
        for( int i = 0; i < pWlStats->count; i++ )
        {
            pSinglesSats = &pWlStats->stats[i];
            NN_SDK_LOG("0x%4d\t", pSinglesSats->chanspec & WL_CHANSPEC_CHAN_MASK);
            for (int j = 0; j < CCASTATS_MAX; j++ )
            {
                NN_SDK_LOG("%d\t", pSinglesSats->ccastats[j]);
            }
            NN_SDK_LOG("%d\t%d\t%d\t%d\t%d", (pSinglesSats->glitchcnt), (pSinglesSats->badplcp),
                pSinglesSats->bgnoise, (pSinglesSats->chan_idle), (pSinglesSats->timestamp));
            NN_SDK_LOG("\n");
        }
    }
    for( int i = 0; i < pWlStats->count; i++ )
    {
        pSinglesSats = &pWlStats->stats[i];
        pStats[i].channel = static_cast<uint16_t>(pSinglesSats->chanspec & WL_CHANSPEC_CHAN_MASK);
        pStats[i].noiseLevel = pSinglesSats->bgnoise;
        pStats[i].txop = pSinglesSats->ccastats[6];
    }
    *pCount = pWlStats->count;

    delete[] pBuf;

    return true;
}

}}
