﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/util/util_TypedStorage.h>

#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/wlan/driver/wlan_DriverTypes.h>
#include "wlan_MemoryInit.h"

#include <dhd_horizon.h>
#include <acsd_horizon.h>
#include <horizon_main.h>
#include <wlu_hos_memmgt.h>

#include "wlan_DebugLog.h"

namespace {
    const size_t g_MemorySizeForMbuf           = 1224 * 1024;
    const size_t g_MemorySizeForMallocNormal   = 1400 * 1024;
    const size_t g_MemorySizeForMallocUncached = UNCACHED_MEMORY_SIZE * 1024;
    const size_t g_MemorySizeForMallocScanBuf  = 100 * 1024;

    NN_ALIGNAS(4096) uint8_t g_BufferPoolForMbuf[g_MemorySizeForMbuf];
    NN_ALIGNAS(4096) uint8_t g_BufferPoolForMallocNormal[g_MemorySizeForMallocNormal];
    NN_ALIGNAS(4096) uint8_t g_BufferPoolForMallocUncached[g_MemorySizeForMallocUncached];
    NN_ALIGNAS(4096) uint8_t g_BufferPoolForMallocScanBuf[g_MemorySizeForMallocScanBuf];

    nn::os::SystemEventType  g_SystemEventTypes[nn::wlan::driver::MbufType_Num];

    nn::lmem::HeapHandle g_HeapHandleNormal;
    nn::lmem::HeapHandle g_HeapHandleUncached;
    nn::lmem::HeapHandle g_HeapHandleScanBuf;
}


namespace nn {
namespace wlan {

void InitializeMemory() NN_NOEXCEPT
{
    nn::Result result;

    // congirures 2 types of malloc buffer
    // normal buffer
    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(g_BufferPoolForMallocNormal),
                               g_MemorySizeForMallocNormal,
                               nn::os::MemoryAttribute_Normal);
    g_HeapHandleNormal = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(g_BufferPoolForMallocNormal),
                                                 g_MemorySizeForMallocNormal,
                                                 nn::lmem::CreationOption_ThreadSafe);
    if( g_HeapHandleNormal == NULL )
    {
        NN_ABORT("Can't get heap handle\n");
    }

    // uncached buffer
    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(g_BufferPoolForMallocUncached),
                               g_MemorySizeForMallocUncached,
                               nn::os::MemoryAttribute_Uncached);
    g_HeapHandleUncached = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(g_BufferPoolForMallocUncached),
                                                   g_MemorySizeForMallocUncached,
                                                   nn::lmem::CreationOption_NoOption);
    if( g_HeapHandleUncached == NULL )
    {
        NN_ABORT("Can't get heap handle\n");
    }

    // scan buffer
    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(g_BufferPoolForMallocScanBuf),
                               g_MemorySizeForMallocScanBuf,
                               nn::os::MemoryAttribute_Normal);
    g_HeapHandleScanBuf = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(g_BufferPoolForMallocScanBuf),
                                                 g_MemorySizeForMallocScanBuf,
                                                 nn::lmem::CreationOption_ThreadSafe);
    if( g_HeapHandleScanBuf == NULL )
    {
        NN_ABORT("Can't get heap handle\n");
    }

    // (ja)DHDにアロケーターの関数ポインタを渡す
    // (en)Passes the function pointer of malloc/free to DHD.
    MemoryAllocators wlanAllocators = {
            nnwlanMallocNormal,
            nnwlanFreeNormal,
            nnwlanMallocUncached,
            nnwlanFreeUncached
    };
    bcm_SetAllocators(&wlanAllocators);

    // ACSD向けのアロケーター
    AcsdMemoryAllocators acsdAllocators = {
            nnwlanMallocNormal,
            nnwlanFreeNormal
    };
    acsd_SetAllocators(&acsdAllocators);

    // WPAサプリカント向けアロケーター
    WpaMemoryAllocators wpaAllocators = {
            nnwlanMallocNormal,
            nnwlanFreeNormal,
            nnwlanCallocNormal,
            nnwlanReallocNormal,
            nnwlanMallocNormalAligned,
            nnwlanMallocUsableSize
    };
    wpa_SetAllocators(&wpaAllocators);

    // Wl cmd tool
    WluMemoryAllocators wluAllocators = {
            nnwlanMallocNormal,
            nnwlanFreeNormal,
            nnwlanCallocNormal,
    };
    wlu_SetAllocators(&wluAllocators);

}

void InitializeMbuf() NN_NOEXCEPT
{
    // (ja)Mbuf poolの初期化
    // (en)Initializes Mbuf pool

    // Mbuf pool の Type を複数用意します。
    nn::mbuf::Type types[driver::MbufType_Num];
    const bool isInterProcess = false;
    for (int i = 0; i < driver::MbufType_Num; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(
            &g_SystemEventTypes[i], nn::os::EventClearMode_AutoClear, isInterProcess));
        types[i].pEvent = &g_SystemEventTypes[i];
    }

    // Mbuf poolのunit sizeを決める。unit sizeより大きなmbufを確保することは出来ない。
    // Mbuf poolは複数設けることが出来、それぞれにunit sizeを指定できる。
    // mbufを確保する時、Mbuf poolを順に見て行き、確保したいサイズより大きなunit sizeが設定されているMbuf poolがあった場合、そこから確保する。
    // Configures the unit size of mbuf pool.
    // You can not get mbuf whose size is larger than unit size of mbuf pool.
    // Multiple mbuf pools can be created. And each mbuf pool has each unit size.
    // When mbuf is allocated, mbuf allocator looks for a mbuf pool which has larger unit size than mbuf size.
    // If mbuf allocator cannot find such mbuf pool, NULL will return.
    const nn::mbuf::Config configs[] =
    {
            { 2048 + nn::mbuf::MbufHeaderSize, driver::MbufPoolCountTx,    driver::MbufType_TxData },
            { 2048 + nn::mbuf::MbufHeaderSize, driver::MbufPoolCountRx,    driver::MbufType_RxData },
            { 2048 + nn::mbuf::MbufHeaderSize, driver::MbufPoolCountEvent, driver::MbufType_EventData },
            { 8192 + nn::mbuf::MbufHeaderSize, driver::MbufPoolCountIoctl, driver::MbufType_IoctlCmdRes },
    };
    // MbufHeaderSize is 52 bytes.
    // Mbuf system management header size is 10440 bytes.
    const int poolCount = sizeof(configs) / sizeof(configs[0]);

    // mbuf 用のシステムイベントを初期化します。


    // mbuf ライブラリを初期化します。
    nn::mbuf::Initialize(types,
                         driver::MbufType_Num,
                         configs,
                         poolCount,
                         reinterpret_cast<void*>(g_BufferPoolForMbuf),
                         g_MemorySizeForMbuf);
}


void FinalizeMemory() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(g_HeapHandleNormal);
    nn::lmem::DestroyExpHeap(g_HeapHandleUncached);
    nn::lmem::DestroyExpHeap(g_HeapHandleScanBuf);
}

void FinalizeMbuf() NN_NOEXCEPT
{
    nn::mbuf::Finalize();
    for (int i = 0; i < driver::MbufType_Num; ++i)
    {
        nn::os::DestroySystemEvent(&g_SystemEventTypes[i]);
    }
}

void* MallocScanBuf(size_t size) NN_NOEXCEPT
{
    void* p = nn::lmem::AllocateFromExpHeap(g_HeapHandleScanBuf, size);
    return p;
}

void FreeScanBuf(void* ptr) NN_NOEXCEPT
{
    if( ptr != NULL )
    {
        nn::lmem::FreeToExpHeap(g_HeapHandleScanBuf, ptr);
    }
}

}
}


void* operator new(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandleNormal, size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    if( ptr != NULL )
    {
        nn::lmem::FreeToExpHeap(g_HeapHandleNormal, ptr);
    }
}

void* operator new[](size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandleNormal, size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    if( ptr != NULL )
    {
        nn::lmem::FreeToExpHeap(g_HeapHandleNormal, ptr);
    }
}

extern "C" void* nnwlanMallocNormal(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandleNormal, size);
}

extern "C" void nnwlanFreeNormal(void* ptr)
{
    if( ptr != NULL )
    {
        nn::lmem::FreeToExpHeap(g_HeapHandleNormal, ptr);
    }
}

extern "C" void* nnwlanMallocUncached(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandleUncached, size);
}

extern "C" void nnwlanFreeUncached(void* ptr)
{
    if( ptr != NULL )
    {
        nn::lmem::FreeToExpHeap(g_HeapHandleUncached, ptr);
    }
}

extern "C" void* nnwlanMallocNormalAligned(int alignment, size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandleNormal, size, alignment);
}

extern "C" void* nnwlanCallocNormal(size_t num, size_t size)
{
    size_t sum = num * size;
    void* ptr = nnwlanMallocNormal(sum);
    if( ptr != NULL )
    {
        std::memset(ptr, 0, sum);
    }
    return ptr;
}

extern "C" void* nnwlanReallocNormal(void* addr, size_t newSize)
{
    if( addr == NULL )
    {
        return nnwlanMallocNormal(newSize);
    }
    if( newSize == 0 )
    {
        nnwlanFreeNormal(addr);
        return NULL;
    }

    // Get the size of the old area
    size_t oldSize = nn::lmem::GetExpHeapBlockSize(addr);
    if (oldSize == 0)
    {
        return NULL;
    }

    void* ptr = nnwlanMallocNormal(newSize);
    if( ptr == NULL )
    {
        return NULL;
    }

    // Copy the contents from the old area to the new area
    if (oldSize < newSize)
    {
        std::memcpy(ptr, addr, oldSize);
    }
    else
    {
        std::memcpy(ptr, addr, newSize);
    }
    nnwlanFreeNormal(addr);

    return ptr;
}

extern "C" size_t nnwlanMallocUsableSize(void* ptr)
{
    if( ptr == NULL )
    {
        return 0;
    }
    return nn::lmem::GetExpHeapBlockSize(ptr);
}

extern "C" void nnwlanMemAddr(wlanMem *memBase)
{
    memBase->mbufBase = reinterpret_cast<uintptr_t>(g_BufferPoolForMbuf);
    memBase->mbuf_size = g_MemorySizeForMbuf;
    memBase->memCacheBase = reinterpret_cast<uintptr_t>(g_BufferPoolForMallocNormal);
    memBase->memCache_size = g_MemorySizeForMallocNormal;
    memBase->memUncacheBase = reinterpret_cast<uintptr_t>(g_BufferPoolForMallocUncached);
    memBase->memUncache_size = g_MemorySizeForMallocUncached;
}

extern "C" size_t nnwlanGetTotalFreeSize()
{
    return nn::lmem::GetExpHeapTotalFreeSize(g_HeapHandleNormal);
}
