﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/wlan/wlan_MacAddress.h>
#include <nn/wlan/driver/wlan_DriverTypes.h>

#include "wlan_SocketApiImpl.h"
#include "../wlan_MsgBuffer.h"

namespace nn { namespace wlan { namespace server
{
    const size_t                ThreadStackSize = 8192 * 2;
    NN_ALIGNAS(4096) char       g_pTxThreadStack[ ThreadStackSize ];
    NN_ALIGNAS(4096) char       g_pRxThreadStack[ ThreadStackSize ];

    nn::os::SharedMemoryType    g_ClientSharedMemory;
    nn::os::SystemEvent         g_RxPacketEvent;
    nn::os::SystemEvent         g_TxPacketEvent;
    nn::os::SystemEvent         g_RxCbufEvent;
    nn::os::SystemEvent         g_TxCbufEvent;
    nn::cbuf::Cbuf*             g_pTxCbuf;
    nn::cbuf::Cbuf*             g_pRxCbuf;
    nn::os::ThreadType          g_TxThread;
    nn::os::ThreadType          g_RxThread;
    bool                        g_EnableSharedMemoryThread = false;

    nn::Result SocketApiImpl::PutFrameRaw(const nn::sf::InBuffer& pTxData) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[Socket::%s]\n", __FUNCTION__);

        // pTxDataをmbufに詰め込む
        nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(NULL, pTxData.GetSize(), nn::mbuf::MbufAllocationMode_Wait,
                nn::wlan::driver::MbufType_TxData);
        NN_SDK_ASSERT_NOT_NULL(pMbuf);
        void* pData = nn::mbuf::MbufTod(pMbuf);
        std::memcpy(pData, pTxData.GetPointerUnsafe(), pTxData.GetSize());
        nn::mbuf::MbufExpand(pMbuf, pTxData.GetSize());

        nn::Result result;
        result = m_pStateMachine->PutFrame(&pMbuf);
        if( ResultTxQueueIsFull().Includes(result) == true )
        {
            // 400us待ってリトライして無理なら諦める
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
            result = m_pStateMachine->PutFrame(&pMbuf);
            if( ResultTxQueueIsFull().Includes(result) == true )
            {
                if( pMbuf != NULL )
                {
                    nn::mbuf::MbufFreem(pMbuf);
                }
            }
        }
        // ResultTxQueueIsFull以外の場合は、mbufの解放は内部で適切に行われるので、ここで行う必要はない

        return result;
    }

    nn::Result SocketApiImpl::CancelGetFrame(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        return m_pStateMachine->CancelRx(rxId);
    }

    nn::Result SocketApiImpl::CreateRxEntry(nn::sf::Out<std::uint32_t> pOutRxId, const nn::sf::InArray<std::uint16_t>& pEthertypeArray, std::uint32_t capacity) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);
        // パラメータチェック
        if( pEthertypeArray.GetLength() > EthertypeCountMax )
        {
            NN_SDK_REQUIRES(false, "The number of ethertypes must be less than %d\n", EthertypeCountMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }
        if( capacity > DataFrameRxCapacityMax )
        {
            NN_SDK_REQUIRES(false, "Capacity of RxEntry must be less than %d\n", DataFrameRxCapacityMax + 1);
            return nn::wlan::ResultInvalidArgument();
        }

        uint32_t rxId; // 出力用変数
        auto result = m_pStateMachine->AddRxEntry(&rxId, capacity, pEthertypeArray.GetData(), pEthertypeArray.GetLength(), false);  // RingBufferの上書き禁止にしておく。つまり満杯の場合、新しいものは捨てられる。
        if( result.IsSuccess() )
        {
            pOutRxId.Set(rxId);
        }

        return result;
    }

    nn::Result SocketApiImpl::DeleteRxEntry(std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        auto result = m_pStateMachine->RemoveRxEntry(rxId);

        return result;
    }

    nn::Result SocketApiImpl::AddEthertypeToRxEntry(std::uint32_t rxId, std::uint16_t ethertype) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        auto result = m_pStateMachine->AddEthertypeToRxEntry(rxId, ethertype);

        return result;
    }

    nn::Result SocketApiImpl::DeleteEthertypeFromRxEntry(nn::sf::Out<std::uint32_t> pOutRxId, std::uint16_t ethertype) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        uint32_t rxId; // 出力用変数
        auto result = m_pStateMachine->RemoveEthertypeFromRxEntry(&rxId, ethertype);
        pOutRxId.Set(rxId);

        return result;
    }

    nn::Result SocketApiImpl::SwitchTsfTimerFunction(bool enable) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(bool));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_SET_TSF_TIMER_EVENT;
        bool* pMode = static_cast<bool*>(pcmdbuff->Args);
        *pMode = enable;

        m_pStateMachine->PostCommandMessage( pcmdbuff );
        auto result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );
        return result;
    }

    nn::Result SocketApiImpl::GetDeltaTimeBetweenSystemAndTsf(nn::sf::Out<std::int64_t> pOutDeltaTime) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        int64_t deltaTime; // 出力用変数

        m_pStateMachine->GetDeltaTimeBetweenSystemAndTsf(&deltaTime);
        pOutDeltaTime.Set(deltaTime);

        NN_RESULT_SUCCESS;
    }

    nn::Result SocketGetFrameApiImpl::GetFrameRaw(const nn::sf::OutBuffer& pOutRxData, nn::sf::Out<std::uint32_t> pOutSize, std::uint32_t rxId) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[Socket::%s]\n", __FUNCTION__);

        nn::mbuf::Mbuf* pOutMbuf;
        auto result = m_pStateMachine->PullRxBuffer(&pOutMbuf, rxId);
        size_t dataSize = 0;

        if( result.IsSuccess() )
        {
            NN_SDK_ASSERT_NOT_NULL(pOutMbuf);
            // データコピー
            // 出力先バッファの容量が足りているかチェック
            dataSize = nn::mbuf::MbufLength(pOutMbuf, NULL);
            if( pOutRxData.GetSize() >=  dataSize )
            {
                std::memcpy(pOutRxData.GetPointerUnsafe(), nn::mbuf::MbufTod(pOutMbuf), dataSize);
            }
            else
            {
                WLAN_LOG_DEBUG("Failed to copy rx data due to capacity shortage of out buffer\n");
                result = ResultBufferTooShort();
            }

            // コピー版なので、mbufはここで破棄
            nn::mbuf::MbufFreem(pOutMbuf);
        }

        pOutSize.Set(dataSize);

        return result;
    }

    nn::Result SocketApiImpl::RegisterSharedMemory(
            nn::sf::NativeHandle memHandle,
            uint32_t memSize,
            nn::sf::NativeHandle rxEventHandle,
            nn::sf::NativeHandle txEventHandle,
            nn::sf::NativeHandle rxCbufEventHandle,
            nn::sf::NativeHandle txCbufEventHandle) NN_NOEXCEPT
    {
        if(!g_EnableSharedMemoryThread)
        {
            g_EnableSharedMemoryThread = true;

            nn::os::AttachSharedMemory(&g_ClientSharedMemory, memSize, memHandle.GetOsHandle(), memHandle.IsManaged());

            uint8_t* pMemory = (uint8_t*)nn::os::MapSharedMemory(&g_ClientSharedMemory, nn::os::MemoryPermission_ReadWrite);
            NN_ABORT_UNLESS(pMemory != nullptr);
            memHandle.Detach();

            g_pRxCbuf  = reinterpret_cast<nn::cbuf::Cbuf*>(pMemory);
            g_pTxCbuf  = reinterpret_cast<nn::cbuf::Cbuf*>(pMemory + memSize / 2);

            g_RxPacketEvent.AttachReadableHandle(
                                    rxEventHandle.GetOsHandle(),
                                    rxEventHandle.IsManaged(),
                                    nn::os::EventClearMode_AutoClear);

            g_TxPacketEvent.AttachWritableHandle(
                                    txEventHandle.GetOsHandle(),
                                    txEventHandle.IsManaged(),
                                    nn::os::EventClearMode_AutoClear);

            txEventHandle.Detach();
            rxEventHandle.Detach();

            g_RxCbufEvent.AttachReadableHandle(
                                    rxCbufEventHandle.GetOsHandle(),
                                    rxCbufEventHandle.IsManaged(),
                                    nn::os::EventClearMode_AutoClear);

            g_TxCbufEvent.AttachWritableHandle(
                                    txCbufEventHandle.GetOsHandle(),
                                    txCbufEventHandle.IsManaged(),
                                    nn::os::EventClearMode_AutoClear);

            txCbufEventHandle.Detach();
            rxCbufEventHandle.Detach();

            nn::os::CreateThread(&g_TxThread,
                                 TxThreadEntry,
                                 this,
                                 g_pTxThreadStack,
                                 sizeof(g_pTxThreadStack),
                                 NN_SYSTEM_THREAD_PRIORITY(wlan, SharedMemoryTx));
            nn::os::SetThreadNamePointer(&g_TxThread, NN_SYSTEM_THREAD_NAME(wlan, SharedMemoryTx));

            nn::os::CreateThread(&g_RxThread,
                                 RxThreadEntry,
                                 this,
                                 g_pRxThreadStack,
                                 sizeof(g_pRxThreadStack),
                                 NN_SYSTEM_THREAD_PRIORITY(wlan, SharedMemoryRx));
            nn::os::SetThreadNamePointer(&g_RxThread, NN_SYSTEM_THREAD_NAME(wlan, SharedMemoryRx));
        }

        return ResultSuccess();
    }

    nn::Result SocketApiImpl::UnregisterSharedMemory() NN_NOEXCEPT
    {
        if(g_EnableSharedMemoryThread)
        {
            g_EnableSharedMemoryThread = false;

            g_RxPacketEvent.Signal();

            nn::os::WaitThread(&g_TxThread);
            nn::os::WaitThread(&g_RxThread);
            nn::os::DestroyThread(&g_TxThread);
            nn::os::DestroyThread(&g_RxThread);

            nn::os::UnmapSharedMemory(&g_ClientSharedMemory);
        }

        return ResultSuccess();
    }

    nn::Result SocketApiImpl::EnableSharedMemory() NN_NOEXCEPT
    {
        nn::os::StartThread(&g_TxThread);
        nn::os::StartThread(&g_RxThread);

        return ResultSuccess();
    }

    void SocketApiImpl::ReceiveThread() NN_NOEXCEPT
    {
        nn::cbuf::Frame* pFrame;
        bool isRetry;

        while (g_EnableSharedMemoryThread)
        {
            g_RxPacketEvent.Wait();

            while ((pFrame = g_pRxCbuf->GetReadPointer()) != nullptr)
            {
                nn::mbuf::Mbuf* pMbuf = nn::mbuf::MbufGetm(NULL, pFrame->size, nn::mbuf::MbufAllocationMode_Wait, 0);

                NN_SDK_ASSERT_NOT_NULL(pMbuf);

                std::memcpy(nn::mbuf::MbufTod(pMbuf), pFrame->data, pFrame->size);

                g_pRxCbuf->AcknowledgeRead();
                g_TxCbufEvent.Signal();

                nn::mbuf::MbufExpand(pMbuf, pFrame->size);

                // PutFrame は一度だけ 400usec 待ってリトライをします。リトライでもダメなら捨てます。
                isRetry = false;
                while (1)
                {
                    nn::Result result = m_pStateMachine->PutFrame(&pMbuf);
                    if( ResultTxQueueIsFull().Includes(result) == true && isRetry == false)
                    {
                        //NN_SDK_LOG("out of wlan buffers\n");
                        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(400));
                        isRetry = true;
                    }
                    else if( ResultTxQueueIsFull().Includes(result) == true && isRetry == true )
                    {
                        if( pMbuf != NULL )
                        {
                            nn::mbuf::MbufFreem(pMbuf);
                        }
                        break;
                    }
                    else
                    {
                        break;
                    }
                }
            }
        }
    }

    void SocketApiImpl::SendThread() NN_NOEXCEPT
    {
        nn::cbuf::Frame* pFrame;
        nn::mbuf::Mbuf*  pOutMbuf;
        nn::Result       result;
        size_t           dataSize;
        bool             block = true;

        while (g_EnableSharedMemoryThread)
        {
            nn::cbuf::Frame* pFrame = g_pTxCbuf->GetWritePointer();

            if (pFrame == nullptr)
            {
                g_TxPacketEvent.Signal();
                g_RxCbufEvent.Wait();
                continue;
            }

            result = m_pStateMachine->PullRxBufferEx(&pOutMbuf, g_pTxCbuf->GetRxId(), block);
            if (result.IsSuccess())
            {
                block    = false;
                dataSize = nn::mbuf::MbufLength(pOutMbuf, NULL);
                if (dataSize <= sizeof(pFrame->data))
                {
                    std::memcpy(pFrame->data, nn::mbuf::MbufTod(pOutMbuf), dataSize);
                    nn::mbuf::MbufFreem(pOutMbuf);
                    pFrame->size = dataSize;
                    g_pTxCbuf->CommitWrite();
                }
            }
            else if (ResultNoRxData().Includes(result) == true || ResultGetFrameCancelled().Includes(result) == true ||
                    ResultRxEntryIsMuted().Includes(result) == true)
            {
                // no more packets, signal...
                g_TxPacketEvent.Signal();
                block = true;
            }
            else
            {
                NN_SDK_LOG("wlan SendThread Error %d:%d\n", result.GetModule(), result.GetDescription());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
        }
    }

    nn::Result SocketApiImpl::SetMulticastFilter(const nn::wlan::detail::SfdlMulticastFilterInfo& sfdlInfo) NN_NOEXCEPT
    {
        WLAN_LOG_DEBUG("[%s]\n", __FUNCTION__);

        WlanCommand* pcmdbuff;
        pcmdbuff = m_pStateMachine->WlanGetCommandBuff(sizeof(WlanAllowedMulticastList));
        NN_ABORT_UNLESS_NOT_NULL(pcmdbuff);

        pcmdbuff->id = WLAN_SET_MULTICAST_LIST;
        WlanAllowedMulticastList* pList = reinterpret_cast<WlanAllowedMulticastList*>(pcmdbuff->Args);
        NN_ABORT_UNLESS_NOT_NULL(pList);

        pList->count = sfdlInfo.count;
        for(int i = 0; i < pList->count; i++)
        {
            std::memcpy(&pList->addr[i].addr[0], &sfdlInfo.address[i].data[0], MacAddress::MacAddressSize);
        }

        m_pStateMachine->PostCommandMessage(pcmdbuff);
        nn::Result result = pcmdbuff->nnResult;
        m_pStateMachine->WlanReleaseCommandBuff( pcmdbuff );

        return result;
    }
}}} // end of namespace nn::wlan::server

