﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "shell_Shell.h"
#include "shell_Console.h"
#include "shell_CommandParser.h"
#include "shell_CommandExecutorSet.h"
#include <cstring>

namespace nn { namespace shell {

namespace
{
    const char* const Prompt = "shell# ";
    const char* const ExitCommandName = "exit";
}

Shell::Shell(Console *pConsole, CommandExecutorSet* pExecutorSet) NN_NOEXCEPT
    : m_pConsole(pConsole),
      m_pExecutorSet(pExecutorSet)
{
}

// コマンド入力、実行
void Shell::Run() NN_NOEXCEPT
{
    while (true)
    {
        char commandBuffer[1024];

        m_pConsole->PutString(Prompt);
        m_pConsole->Flush();

        if (!m_pConsole->ReadLine(commandBuffer, sizeof(commandBuffer)))
        {
            break;
        }

        char parserBuffer[sizeof(commandBuffer)];
        CommandParser parser(parserBuffer, commandBuffer);
        while (parser.HasNext())
        {
            InputCommand command;
            auto result = parser.ReadCommand(&command);
            if(result != CommandParser::ReadResult_Ok)
            {
                switch(result)
                {
                case CommandParser::ReadResult_TooManyArguments:
                    {
                        m_pConsole->TPrintf("Too many arguments (max. %d).\n", MaxCommandArgumentCount);
                    }
                    break;
                case CommandParser::ReadResult_UnmatchedQuote:
                    {
                        m_pConsole->PutString("Unmatched \".\n");
                    }
                    break;
                default:
                    {
                        m_pConsole->TPrintf("Unknown CommandParser::ReadResult(=%d)\n", result);
                    }
                    break;
                }

                break;
            }

            // 終了コマンドだけここで処理する
            if (command.argc == 1 && std::strcmp(command.argv[0], ExitCommandName) == 0)
            {
                return;
            }

            int64_t commandResult = Execute(&command);
            m_pConsole->TPrintf("  -> %lld\n\n", commandResult);
        }
    }
}

int64_t Shell::Execute(InputCommand* command) NN_NOEXCEPT
{
    if (command->argc == 0)
    {
        // 空文字列なので何も表示せず戻る
        return 0;
    }

    return m_pExecutorSet->Execute(m_pConsole,
                command->argc, command->argv,
                command->pCommandLine, command->commandLineBytes, command->pArgumentPositions);
}

}}  // namespace nn { namespace shell {
