﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include "shell_Console.h"
#include "shell_CommandExecutorSet.h"
#include "shell_CommandParser.h"
#include "shell_ProcessManagementExecutor.h"
#include "shell_ProcessManagement.h"
#include <cstdlib>
#include <cstring>
#include <alloca.h>
#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/os/os_Thread.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Synchronization.h>

#include <nn/dmnt0/dmnt0_Api.h>

namespace nn { namespace shell {
namespace {
    const int CommandIndexLaunchProgram = 0;
    const int CommandIndexTerminateProcess = 1;
    const int CommandIndexGdbServerProgram = 2;
    const int CommandIndexAttachProcess = 3;
    const int CommandIndexPrintPort = 4;
    const int CommandIndexDumpThread = 5;
    const int CommandIndexDumpCallStack = 6;
    const int CommandIndexDumpKernelObject = 7;
    const int CommandIndexDumpHandle = 8;
    const int CommandIndexDumpMemory = 9;
    const int CommandIndexDumpPageTable = 10;
    const int CommandIndexCpuUtilization = 11;
    const int CommandIndexDumpProcess = 12;
    const int CommandIndexSetTraceTypeFilter = 13;
    const int CommandIndexSetTraceProcessFilter = 14;
    const int CommandIndexStartTrace = 15;
    const int CommandIndexStopTrace = 16;
    const int CommandIndexInfoTrace = 17;
    const int CommandIndexSuspendProcess = 18;
    const int CommandIndexResumeProcess = 19;

    const int CommandIndexAccessBreak = 20;
    const int CommandIndexWriteBreak  = 21;
    const int CommandIndexReadBreak   = 22;
    const int CommandIndexExecBreak   = 23;
    const int CommandIndexShowBreak   = 24;
    const int CommandIndexDeleteBreak = 25;
}

namespace {
    bool g_GdbServerInitialized = false;
    nn::os::Mutex g_GdbServerLock(false);

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A57) || defined(NN_BUILD_CONFIG_CPU_CORTEX_A53)
    const size_t NumHwBreak = 6;
    const size_t NumHwBreakCtx = 2;
    const size_t NumHwWatch = 4;
#else
#error unknown cpu
#endif
    enum HwBreakType
    {
        HwBreakType_InstructionAddressMatch,
        HwBreakType_ContexIdMatch,
    };
    enum HwWatchType
    {
        HwWatchType_Access,
        HwWatchType_Write,
        HwWatchType_Read,
    };
    struct HwBreakControl
    {
        HwBreakType type;
        Bit64 pid;
        Bit64 addr;
        int ref;
    };
    struct HwWatchControl
    {
        HwWatchType type;
        Bit64 pid;
        Bit64 addr;
        int ref;
    };

    HwBreakControl g_HwBreakControl[NumHwBreak];
    HwWatchControl g_HwWatchControl[NumHwWatch];

    void ShowBreakPoint()
    {
        NN_SDK_LOG("Hardware Breakpoints\n");
        for (size_t i = 0; i < NumHwBreak; i++)
        {
            if (g_HwBreakControl[i].ref > 0 &&
                    g_HwBreakControl[i].type == HwBreakType_InstructionAddressMatch)
            {
                NN_SDK_LOG("%ld: pid=%lld addr=%llx Exec\n", i, g_HwBreakControl[i].pid, g_HwBreakControl[i].addr);
            }
        }
        for (size_t i = 0; i < NumHwWatch; i++)
        {
            if (g_HwWatchControl[i].ref > 0)
            {
                const char* s = "Access";
                if (g_HwWatchControl[i].type == HwWatchType_Write) { s = "Write"; }
                if (g_HwWatchControl[i].type == HwWatchType_Read) { s = "Read"; }
                NN_SDK_LOG("%ld: pid=%lld addr=%llx type=%s\n", i + NumHwBreak, g_HwWatchControl[i].pid, g_HwWatchControl[i].addr, s);
            }
        }
    }

    HwBreakControl* AssignContexId(Bit64 pid)
    {
        for (size_t i = NumHwBreak - NumHwBreakCtx; i < NumHwBreak; i++)
        {
            if (g_HwBreakControl[i].ref > 0 &&
                    g_HwBreakControl[i].type == HwBreakType_ContexIdMatch &&
                    g_HwBreakControl[i].pid == pid)
            {
                return &g_HwBreakControl[i];
            }
        }
        for (size_t i = NumHwBreak - NumHwBreakCtx; i < NumHwBreak; i++)
        {
            if (g_HwBreakControl[i].ref == 0)
            {
                nn::svc::Handle debug;
                nn::Result result;
                result = nn::svc::DebugActiveProcess(&debug, pid);
                if (result.IsFailure())
                {
                    NN_SDK_LOG("attach fail %lld\n", pid);
                    return nullptr;
                }
                Bit64 ctrl = (0x3 << 20) | (0<<16) | (0 << 5) | 1;

                int idealCore;
                Bit64 affnity;
                nn::os::GetThreadCoreMask(&idealCore, &affnity, nn::os::GetCurrentThread());
                Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
                while (coreMask != 0)
                {
                    int coreNo = 63 - __builtin_clzll(coreMask);
                    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNo, 1ull << coreNo);
                    NN_ABORT_UNLESS(nn::svc::SetHardwareBreakPoint(
                                static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_I0 + i),
                                ctrl, static_cast<nnHandle>(debug).value).IsSuccess());
                    coreMask &= ~(1ull << coreNo);
                }
                nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), idealCore, affnity);

                nn::svc::CloseHandle(debug);

                g_HwBreakControl[i].type = HwBreakType_ContexIdMatch;
                g_HwBreakControl[i].pid = pid;
                return &g_HwBreakControl[i];
            }
        }
        return nullptr;
    }

    void SetBreakPoint(Bit64 pid, uint64_t addr)
    {
        HwBreakControl* ctx = AssignContexId(pid);
        if (ctx != nullptr)
        {
            HwBreakControl* brk = nullptr;
            for (size_t i = 0; i < NumHwBreak; i++)
            {
                if (g_HwBreakControl[i].ref == 0)
                {
                    brk = &g_HwBreakControl[i];
                }
            }
            if (brk != nullptr)
            {
                brk->pid = pid;
                brk->type = HwBreakType_InstructionAddressMatch;
                brk->addr = addr;

                Bit64 ctrl = (0x1 << 20) | ((ctx - &g_HwBreakControl[0]) << 16) | (0xF << 5) | 1;

                int idealCore;
                Bit64 affnity;
                nn::os::GetThreadCoreMask(&idealCore, &affnity, nn::os::GetCurrentThread());
                Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
                while (coreMask != 0)
                {
                    int coreNo = 63 - __builtin_clzll(coreMask);
                    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNo, 1ull << coreNo);
                    NN_ABORT_UNLESS(nn::svc::SetHardwareBreakPoint(
                                static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_I0 + (brk - &g_HwBreakControl[0])),
                                ctrl, addr).IsSuccess());
                    coreMask &= ~(1ull << coreNo);
                }
                nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), idealCore, affnity);

                brk->ref++;
                ctx->ref++;
            }
        }
        ShowBreakPoint();
    }

    void SetWatchPoint(Bit64 pid, uint64_t addr, HwWatchType type)
    {
        HwBreakControl* ctx = AssignContexId(pid);
        if (ctx != nullptr)
        {
            HwWatchControl* watch = nullptr;
            for (size_t i = 0; i < NumHwWatch; i++)
            {
                if (g_HwWatchControl[i].ref == 0)
                {
                    watch = &g_HwWatchControl[i];
                    break;
                }
            }
            if (watch != nullptr)
            {
                watch->pid = pid;
                watch->type = type;
                watch->addr = addr;

                uint32_t mask;
                uint32_t bas;
                uint32_t lsc;
                size_t len = 1;
                if (len < 8)
                {
                    mask = 0;
                    bas = (((1 << len) - 1) << (addr & 7));
                }
                else
                {
                    mask = 31 - __builtin_clz(len);
                    bas = 0xFF;
                }
                if (type == HwWatchType_Write)
                {
                    lsc = 2;
                }
                else if (type == HwWatchType_Read)
                {
                    lsc = 1;
                }
                else
                {
                    lsc = 3;
                }
                Bit64 ctrl = (mask << 24) | ((ctx - &g_HwBreakControl[0]) << 16) | (bas << 5) | (lsc << 3) | 1;
                int idealCore;
                Bit64 affnity;
                nn::os::GetThreadCoreMask(&idealCore, &affnity, nn::os::GetCurrentThread());
                Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
                while (coreMask != 0)
                {
                    int coreNo = 63 - __builtin_clzll(coreMask);
                    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNo, 1ull << coreNo);
                    NN_ABORT_UNLESS(nn::svc::SetHardwareBreakPoint(
                                static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_D0 + (watch - &g_HwWatchControl[0])),
                                ctrl, addr & ~0x7ul).IsSuccess());
                    coreMask &= ~(1ull << coreNo);
                }
                nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), idealCore, affnity);

                watch->ref++;
                ctx->ref++;
            }
        }
        ShowBreakPoint();
    }

    void DeleteBreakPoint(size_t id)
    {
        if (id < NumHwBreak)
        {
            HwBreakControl* brk =  &g_HwBreakControl[id];
            if (brk->ref > 0 && brk->type == HwBreakType_InstructionAddressMatch)
            {
                HwBreakControl* ctx = AssignContexId(brk->pid);
                NN_ABORT_UNLESS(ctx != nullptr);
                NN_ABORT_UNLESS(ctx->ref > 0);

                if (--brk->ref == 0)
                {
                    int idealCore;
                    Bit64 affnity;
                    nn::os::GetThreadCoreMask(&idealCore, &affnity, nn::os::GetCurrentThread());
                    Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
                    while (coreMask != 0)
                    {
                        int coreNo = 63 - __builtin_clzll(coreMask);
                        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNo, 1ull << coreNo);
                        nn::svc::SetHardwareBreakPoint(
                                static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_I0 + (brk - &g_HwBreakControl[0])), 0, 0);

                        coreMask &= ~(1ull << coreNo);
                    }
                    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), idealCore, affnity);
                }

                if (--ctx->ref == 0)
                {
                    int idealCore;
                    Bit64 affnity;
                    nn::os::GetThreadCoreMask(&idealCore, &affnity, nn::os::GetCurrentThread());
                    Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
                    while (coreMask != 0)
                    {
                        int coreNo = 63 - __builtin_clzll(coreMask);
                        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNo, 1ull << coreNo);

                        nn::svc::SetHardwareBreakPoint(
                                static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_I0 + (ctx - &g_HwBreakControl[0])), 0, 0);
                        coreMask &= ~(1ull << coreNo);
                    }
                    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), idealCore, affnity);
                }
            }
        }
        else if (id < NumHwBreak + NumHwWatch)
        {
            id -= NumHwBreak;
            HwWatchControl* watch =  &g_HwWatchControl[id];
            if (watch->ref > 0)
            {
                HwBreakControl* ctx = AssignContexId(watch->pid);
                NN_ABORT_UNLESS(ctx != nullptr);
                NN_ABORT_UNLESS(ctx->ref > 0);

                if (--watch->ref == 0)
                {
                    nn::svc::SetHardwareBreakPoint(
                            static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_D0 + (watch - &g_HwWatchControl[0])), 0, 0);
                }
                if (--ctx->ref == 0)
                {
                    nn::svc::SetHardwareBreakPoint(
                            static_cast<nn::svc::HardwareBreakPointRegisterName>(nn::svc::HardwareBreakPointRegisterName_I0 + (ctx - &g_HwBreakControl[0])), 0, 0);
                }
            }
        }
        ShowBreakPoint();
    }
#endif
}

ProcessManagementExecutor::ProcessManagementExecutor() NN_NOEXCEPT
{
}

ProcessManagementExecutor::~ProcessManagementExecutor() NN_NOEXCEPT
{
}

void ProcessManagementExecutor::GetCommandDefinitions(const CommandDefinition** ppDefinitions, int *pCount) NN_NOEXCEPT
{
    static const CommandDefinition Definitions[] = {
        {"LaunchProgram", 1, MaxCommandArgumentCount - 1,
            "L", "LaunchProgram filename"},
        {"TerminateProcess", 1, 1,
            "T", "TerminateProcess processId"},
        {"gdbserver", 2, MaxCommandArgumentCount - 1,
            "", "gdbserver :port filename"},
        {"AttachProcess", 2, 2,
            "A", "AttachProcess processid port"},
        {"PrintPort", 0, 1,
            "P", "PrintPort [pid]"},
        {"Dump Thread Info", 0, 1,
            "D", "DumpThread"},
        {"Dump Callstack", 0, 1,
            "C", "DumpCallstack"},
        {"Dump Kernel Object", 0, 0,
            "K", "DumpKernelObject"},
        {"Dump Handle", 0, 1,
            "Z", "DumpHandle"},
        {"Info Memory", 0, 1,
            "M", "InfoMemory"},
        {"Dump PageTable", 0, 1,
            "Y", "DumpPageTable"},
        {"CpuUtilization", 0, 2,
            "X", "CpuUtilization [pid(-1: all process, -2: kernel)] [times]"},

        {"DumpProcess",           0, 1, "DP", "DumpProcess [pid(-1: all process)]"},
        {"SetTraceTypeFilter",    1, 1, "TF", "SetTraceTypeFilter mask"},
        {"SetTraceProcessFilter", 2, 2, "PF", "SetTraceProcessFilter index mask"},
        {"StartTrace",            0, 0, "ST", "StartTrace"},
        {"StopTrace",             0, 0, "ET", "StopTrace"},
        {"InfoTrace",             0, 0, "IT", "InfoTrace"},

        {"SuspendProcess",        1, 1, "SP", "SuspendProcess pid"},
        {"ResumeProcess",         1, 1, "RP", "ResumeProcess pid"},

        {"AccessBreak",           2, 2, "AB", "AccessBreak pid addr"},
        {"WriteBreak",            2, 2, "WB", "WriteBreak pid addr"},
        {"ReadBreak",             2, 2, "RB", "ReadBreak pid addr"},
        {"ExecBreak",             2, 2, "EB", "ExecBreak pid addr"},
        {"ShowBreak",             0, 0, "SB", "ShowBreak"},
        {"DeleteBreak",           1, 1, "DB", "DeleteBreak id"},

    };
    *ppDefinitions = Definitions;
    *pCount = sizeof(Definitions) / sizeof(Definitions[0]);
}

int64_t ProcessManagementExecutor::LaunchProgram(const char* path, size_t length, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
{
    os::ProcessId processId;
    Result result;

    ns::ProgramLaunchProperty launchProperty;
    result = PrepareToLaunchProgram(&launchProperty, path, length, pArgument, argumentSize);
    if( result.IsFailure() )
    {
        return -1;
    }

    result = nn::ns::LaunchProgram(&processId, launchProperty, ns::LaunchProgramFlags_None);
    FlushProgramArgument(launchProperty.programId);
    if( result.IsFailure() )
    {
        NN_SDK_LOG("[shell] ns::LaunchProgram failed (result=%08x)\n", result.GetInnerValueForDebug());
        return -1;
    }

    return processId.value;
}

int64_t ProcessManagementExecutor::GdbServerProgram(const CommandExecuteContext* pContext, const char* path, size_t length, uint32_t port, const void* pArgument, size_t argumentSize) NN_NOEXCEPT
{
    nn::Result result;

    {
        std::lock_guard<nn::os::Mutex> lock(g_GdbServerLock);

        if( !g_GdbServerInitialized )
        {
            result = nn::dmnt0::Initialize();
            NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            g_GdbServerInitialized = true;
        }
    }

    ns::ProgramLaunchProperty launchProperty;
    result = PrepareToLaunchProgram(&launchProperty, path, length, pArgument, argumentSize);
    if( result.IsFailure() )
    {
        return -1;
    }

    nn::svc::Handle eventHandle;
    nn::os::ProcessId processId;

    result = nn::dmnt0::DebugNewProcess(&eventHandle, &processId, launchProperty.programId, port);
    FlushProgramArgument(launchProperty.programId);
    if( result.IsFailure() )
    {
        NN_SDK_LOG("[shell] dmnt0::DebugNewProcess failed (result=%08x)\n", result.GetInnerValueForDebug());
        return -1;
    }

    pContext->pConsole->TPrintf("Process %s created; pid = %lld\nListening on port %d\n", path, processId, port);

    nn::os::SystemEvent event(eventHandle.operator nnHandle().value, false,
                                nn::os::InvalidNativeHandle, false, nn::os::EventClearMode_ManualClear);
    event.Wait();
    nn::svc::CloseHandle(eventHandle);

    return 1;
}

int64_t ProcessManagementExecutor::TerminateProcess(Bit64 processId) NN_NOEXCEPT
{
    if (shell::TerminateProcess(processId))
    {
        return 0;
    }
    return -1;
}

int64_t ProcessManagementExecutor::AttachProcess(Bit64 processId, uint32_t port) NN_NOEXCEPT
{
    nn::Result result;

    {
        std::lock_guard<nn::os::Mutex> lock(g_GdbServerLock);

        if( !g_GdbServerInitialized )
        {
            result = nn::dmnt0::Initialize();
            NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            g_GdbServerInitialized = true;
        }
    }

    nn::svc::Handle eventHandle;
    os::ProcessId pid = { processId };

    result = nn::dmnt0::AttachProcess(&eventHandle, pid, port);
    if( result.IsFailure() )
    {
        return -1;
    }

    nn::svc::CloseHandle(eventHandle);

    return 0;
}


int64_t ProcessManagementExecutor::Execute(const CommandExecuteContext* pContext, int commandIndex, int argc, const char* const* argv) NN_NOEXCEPT
{
    int64_t ret = -1;
    NN_UNUSED(argc);

    switch (commandIndex)
    {
    case CommandIndexLaunchProgram:
        {
            const char* path = argv[1];
            size_t len = std::strlen(path);

            const char* pArgument = pContext->pArgumentPositions[1];
            size_t argumentSize = pContext->commandLineBytes - (pArgument - pContext->pCommandLine);
            ret = LaunchProgram(path, len, pArgument, argumentSize);
        }
        break;
    case CommandIndexGdbServerProgram:
        {
            const char* path = argv[2];
            size_t len = std::strlen(path);
            char* endptr = NULL;
            if (argv[1][0] == ':')
            {
                uint32_t port = std::strtoul(&argv[1][1], &endptr, 0);
                if (endptr != &argv[1][1])
                {
                    const char* pArgument = pContext->pArgumentPositions[2];
                    size_t argumentSize = pContext->commandLineBytes - (pArgument - pContext->pCommandLine);
                    ret = GdbServerProgram(pContext, path, len, port, pArgument, argumentSize);
                }
            }
        }
        break;
    case CommandIndexTerminateProcess:
        {
            char* endptr = NULL;
            Bit64 processId = std::strtoul(argv[1], &endptr, 0);
            if (endptr != argv[1])
            {
                ret = ProcessManagementExecutor::TerminateProcess(processId);
            }
        }
        break;
    case CommandIndexAttachProcess:
        {
            char* endptr = NULL;
            Bit64 processId = std::strtoul(argv[1], &endptr, 0);
            if (endptr != argv[1])
            {
                endptr = NULL;
                uint32_t port = std::strtoul(argv[2], &endptr, 0);
                if (endptr != argv[2])
                {
                    ret = AttachProcess(processId, port);
                }
            }
        }
        break;
    case CommandIndexPrintPort:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_Port, pid, 0, 0);
            ret = 0;
        }
        break;
    case CommandIndexDumpThread:
        {
            Bit64 tid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                tid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_Thread, tid, 0, 0);
            ret = 0;
        }
        break;
    case CommandIndexDumpCallStack:
        {
            Bit64 tid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                tid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_ThreadCallstack, tid, 0, 0);
            ret = 0;
        }
        break;
    case CommandIndexDumpKernelObject:
        {
            nn::svc::KernelDebug(nn::svc::KernelDebugType_KernelObject, 0, 0, 0);
            ret = 0;
        }
        break;
    case CommandIndexDumpHandle:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_Handle, pid, 0, 0);
            ret = 0;
        }
        break;
    case CommandIndexDumpMemory:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_Memory, pid, 0, 0);
            ret = 0;
        }
        break;

    case CommandIndexDumpPageTable:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_PageTable, pid, 0, 0);
            ret = 0;
        }
        break;

    case CommandIndexCpuUtilization:
        {
            int num = 1;
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 3)
            {
                char* endptr = NULL;
                num = std::strtoul(argv[2], &endptr, 0);
            }
            if (argc >= 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            for (int i = 0; i < num; i++)
            {
                nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, pid, 0, 0);
            }
            ret = 0;
        }
        break;

    case CommandIndexDumpProcess:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
            }
            nn::svc::KernelDebug(nn::svc::KernelDebugType_Process, pid, 0, 0);
            ret = 0;
        }
        break;

    case CommandIndexSetTraceTypeFilter:
        {
            if (argc == 2)
            {
                Bit64 mask;
                char* endptr = NULL;
                mask = std::strtoul(argv[1], &endptr, 0);
                nn::svc::KernelDebug(nn::svc::KernelDebugType_KTraceSetTypeFilter, mask, 0, 0);
                ret = 0;
            }
        }
        break;

    case CommandIndexSetTraceProcessFilter:
        {
            if (argc == 3)
            {
                char* endptr = NULL;
                Bit64 index = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    endptr = NULL;
                    Bit64 mask = std::strtoul(argv[2], &endptr, 0);
                    if (endptr != argv[2])
                    {
                        if (index == 0)
                        {
                            nn::svc::KernelDebug(nn::svc::KernelDebugType_KTraceSetProcessFilter0, mask, 0, 0);
                            ret = 0;
                        }
                        else if (index == 1)
                        {
                            nn::svc::KernelDebug(nn::svc::KernelDebugType_KTraceSetProcessFilter1, mask, 0, 0);
                            ret = 0;
                        }

                    }
                }
            }
        }
        break;

    case CommandIndexStartTrace:
        {
            nn::svc::ChangeKernelTraceState(nn::svc::KernelTraceState_Running);
            ret = 0;
        }
        break;

    case CommandIndexStopTrace:
        {
            nn::svc::ChangeKernelTraceState(nn::svc::KernelTraceState_Paused);
            ret = 0;
        }
        break;

    case CommandIndexInfoTrace:
        {
            nn::svc::KernelDebug(nn::svc::KernelDebugType_KTraceInfo, 0, 0, 0);
            ret = 0;
        }
        break;

    case CommandIndexSuspendProcess:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
                nn::svc::KernelDebug(nn::svc::KernelDebugType_SuspendProcess, pid, 0, 0);
            }
            ret = 0;
        }
        break;

    case CommandIndexResumeProcess:
        {
            Bit64 pid = static_cast<Bit64>(-1ll);
            if (argc == 2)
            {
                char* endptr = NULL;
                pid = std::strtoul(argv[1], &endptr, 0);
                nn::svc::KernelDebug(nn::svc::KernelDebugType_ResumeProcess, pid, 0, 0);
            }
            ret = 0;
        }
        break;

    case CommandIndexAccessBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            if (argc == 3)
            {
                char* endptr = nullptr;
                Bit64 pid = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    endptr = nullptr;
                    Bit64 addr = std::strtoul(argv[2], &endptr, 0);
                    if (endptr != argv[2])
                    {
                        SetWatchPoint(pid, addr, HwWatchType_Access);
                    }
                }
            }
#endif
            ret = 0;
        }
        break;

    case CommandIndexWriteBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            if (argc == 3)
            {
                char* endptr = nullptr;
                Bit64 pid = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    endptr = nullptr;
                    Bit64 addr = std::strtoul(argv[2], &endptr, 0);
                    if (endptr != argv[2])
                    {
                        SetWatchPoint(pid, addr, HwWatchType_Write);
                        ret = 0;
                    }
                }
            }
#endif
            ret = 0;
        }
        break;

    case CommandIndexReadBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            if (argc == 3)
            {
                char* endptr = nullptr;
                Bit64 pid = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    endptr = nullptr;
                    Bit64 addr = std::strtoul(argv[2], &endptr, 0);
                    if (endptr != argv[2])
                    {
                        SetWatchPoint(pid, addr, HwWatchType_Read);
                    }
                }
            }
#endif
            ret = 0;
        }
        break;

    case CommandIndexExecBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            if (argc == 3)
            {
                char* endptr = nullptr;
                Bit64 pid = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    endptr = nullptr;
                    Bit64 addr = std::strtoul(argv[2], &endptr, 0);
                    if (endptr != argv[2])
                    {
                        SetBreakPoint(pid, addr);
                        ret = 0;
                    }
                }
            }
#endif
            ret = 0;
        }
        break;

    case CommandIndexShowBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            ShowBreakPoint();
#endif
            ret = 0;
        }
        break;

    case CommandIndexDeleteBreak:
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            if (argc == 2)
            {
                char* endptr = nullptr;
                Bit64 id = std::strtoul(argv[1], &endptr, 0);
                if (endptr != argv[1])
                {
                    DeleteBreakPoint(id);
                }
            }
#endif
            ret = 0;
        }
        break;

    default:
        {
            NN_ABORT("invalid commandIndex %d", commandIndex);
        }
        break;
    }
    return ret;
} // NOLINT(impl/function_size)

}}  // namespace nn { namespace shell {
