﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <cstring>
#include <cctype>
#include "shell_Console.h"
#include "shell_CommandExecutorSet.h"

// コマンドの追加
//
// 1. CommandExecutor の派生クラスを作る
// 2. 1. の型のメンバ変数を CommandExecutorSet に 追加する
// 3. 2.で追加したメンバ変数を CommandExecutorSet::Register する
//
// CommandExecutorSet は接続（UART、Telnet）ごとに生成される。
// コマンドの実行は排他制御される（全体で同時に1つだけ実行される）。


namespace nn { namespace shell {
namespace {
    const int CommandIndexHelp = 0;
}

CommandExecutorSet::CommandExecutorSet() NN_NOEXCEPT
    : m_ExecuteInfoCount(0)
{
    Register(this);
    Register(&m_ProcessManagementExecutor);
}

CommandExecutorSet::~CommandExecutorSet() NN_NOEXCEPT
{
}

int64_t CommandExecutorSet::Execute(Console* pConsole, int argc, const char* const* argv, const char* pCommandLine, size_t commandLineBytes, const char** pArgumentPositions) NN_NOEXCEPT
{
    const char* pCommandName = argv[0];
    const CommandExecuteInfo* pExecuteInfo = GetExecuteInfo(pCommandName);
    if (pExecuteInfo == NULL)
    {
        pConsole->TPrintf("Command %s not found.\n", argv[0]);
        return -1;
    }

    if (argc - 1 < pExecuteInfo->pDefinition->minArgCount || pExecuteInfo->pDefinition->maxArgCount < argc - 1)
    {
        pConsole->TPrintf(
            "Invalid number of arguments for %s: %d to %d (incl.) required.\n",
            pExecuteInfo->pDefinition->pCommandName,
            pExecuteInfo->pDefinition->minArgCount,
            pExecuteInfo->pDefinition->maxArgCount);
        return -1;
    }

    CommandExecuteContext context;
    context.pConsole = pConsole;
    context.pCommandLine = pCommandLine;
    context.commandLineBytes = commandLineBytes;
    context.argumentCount = argc;
    context.pArgumentPositions = pArgumentPositions;
    return pExecuteInfo->pExecutor->Execute(&context, pExecuteInfo->commandIndexInExecutor, argc, argv);
}

void CommandExecutorSet::Register(CommandExecutor* pExecutor) NN_NOEXCEPT
{
    const CommandDefinition* pDefinitions;
    int count;
    pExecutor->GetCommandDefinitions(&pDefinitions, &count);

    for (int i = 0; i < count; i ++)
    {
        Register(pExecutor, i, &pDefinitions[i]);
    }
}

void CommandExecutorSet::Register(CommandExecutor* pExecutor, int commandIndexInExecutor, const CommandDefinition* pDefinition) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ExecuteInfoCount + 1 < MaxCommandCount);

    m_ExecuteInfo[m_ExecuteInfoCount].pDefinition = pDefinition;
    m_ExecuteInfo[m_ExecuteInfoCount].pExecutor = pExecutor;
    m_ExecuteInfo[m_ExecuteInfoCount].commandIndexInExecutor = commandIndexInExecutor;
    m_ExecuteInfoCount++;
}

const CommandExecutorSet::CommandExecuteInfo* CommandExecutorSet::GetExecuteInfo(const char* pCommandName) const NN_NOEXCEPT
{
    for (int i = 0; i < m_ExecuteInfoCount; i ++)
    {
        if (CommandExecuteInfoMatches(&m_ExecuteInfo[i], pCommandName))
        {
            return &m_ExecuteInfo[i];
        }
    }

    return NULL;
}

void CommandExecutorSet::GetCommandDefinitions(const CommandDefinition** ppDefinitions, int *pCount) NN_NOEXCEPT
{
    static const CommandDefinition Definitions[] = {
        {"help", 0, 0, "H", "help"},
    };
    *ppDefinitions = Definitions;
    *pCount = sizeof(Definitions) / sizeof(Definitions[0]);
}

int64_t CommandExecutorSet::Execute(const CommandExecuteContext* pContext, int commandIndex, int argc, const char* const* argv) NN_NOEXCEPT
{
    NN_UNUSED(argc);
    NN_UNUSED(argv);

    switch (commandIndex)
    {
    case CommandIndexHelp:
        PrintCommandList(pContext);
        return 0;
    default:
        NN_ABORT("invalid commandIndex %d", commandIndex);
        return -1;
    }
}

void CommandExecutorSet::PrintCommandList(const CommandExecuteContext* pContext) const NN_NOEXCEPT
{
    for (int i = 0; i < m_ExecuteInfoCount; i ++)
    {
        pContext->pConsole->TPrintf(
            " %1s / %s\n",
            m_ExecuteInfo[i].pDefinition->pShortCommandName,
            m_ExecuteInfo[i].pDefinition->pUsage);
    }
}

bool CommandExecutorSet::CommandExecuteInfoMatches(const CommandExecuteInfo* pExecuteInfo, const char* pCommandName) const NN_NOEXCEPT
{
    return std::strcmp(pExecuteInfo->pDefinition->pCommandName, pCommandName) == 0
           || std::strcmp(pExecuteInfo->pDefinition->pShortCommandName, pCommandName) == 0;
}

}}  // namespace nn { namespace shell {
