﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace scs {

    struct CommandHeader
    {
        int64_t     id;
        int32_t     command;
        uint32_t    dataSize;
    };

    struct ResponseHeader
    {
        int64_t     id;
        int32_t     reponse;
        uint32_t    dataSize;
    };

    class CommandProcessor
    {
    public:
        void Initialize();
        virtual bool ProcessCommand(const CommandHeader& header, const Bit8* pBody, int socket);

    protected:
        enum Command
        {
            Command_Null,
            Command_LaunchApplication,
            Command_TerminateProcess,
            Command_GetFirmwareVersion, // cs のみサポート
            Command_Reboot, // cs のみサポート
            Command_SetSafeMode, // cs のみサポート
            Command_RegisterTenvDefinitionFilePath,
            Command_TerminateApplication,
            Command_Shutdown, // cs のみサポート
            Command_SubscribeProcessEvent,
            Command_GetTitleName, // scs はスタブ実装
            Command_ControlVirtualTemperature,  // cs のみサポート
            Command_LaunchInstalledApplication,
            Command_LaunchGameCardApplication,
            Command_LaunchInstalledSystemProcess,
            Command_TakeScreenShot, // cs のみサポート
        };
        enum Response
        {
            Response_Null,
            Response_Success,
            Response_Error,
            Response_ProgramExited,
            Response_FirmwareVersion,
            Response_JitDebug,
            Response_ProgramLaunched,
            Response_TitleName,
            Response_ScreenShot,
        };

        static void SendErrorResult(int socket, const CommandHeader& command, Result result);
        static void SendSuccess(int socket, const CommandHeader& command);

    private:
        struct CommandLaunchApplicationBody
        {
            Bit64       programId;
            uint32_t    pathSize;
            uint32_t    argumentSize;
            Bit32       flags;
            uint32_t    envPathSize;
        };
        struct ResponseError
        {
            ResponseHeader  header;
            Bit32           result;
        };
        struct CommandRegisterTenvDefinitionFilePath
        {
            Bit64       processId;
            uint32_t    pathSize;
            uint32_t    reserved;
        };
        struct CommandSubscribeProcessEvent
        {
            bool        isRegister;
            uint8_t     reserved[3];
        };
        struct ResponseProgramLaunched
        {
            ResponseHeader  header;
            Bit64           processIndex;
        };
        struct ResponseProgramExited
        {
            ResponseHeader  header;
            Bit64           processIndex;
        };
        struct CommandGetTitleName
        {
            Bit64           processIndex;
        };
        struct CommandLaunchInstalledApplicationBody
        {
            Bit64       programId;
            uint32_t    argumentSize;
            Bit32       flags;
            uint32_t    envPathSize;
            uint32_t    reserved;
        };
        struct CommandLaunchGameCardApplicationBody
        {
            Bit64       programId;
            uint32_t    argumentSize;
            Bit32       flags;
            uint32_t    envPathSize;
            uint32_t    reserved;
        };
        struct CommandLaunchInstalledSystemProcessBody
        {
            Bit64       programId;
            uint32_t    argumentSize;
            uint32_t    reserved;
        };

        static void SendErrorResult(int socket, int64_t commandId, Result result);
        static void SendExited(int socket, int64_t commandId, Bit64 processIndex);
        static void SendJitDebug(int socket, int64_t commandId);
        static void SendLaunched(int socket, int64_t commandId, Bit64 processIndex);
        static void SendTitleName(const CommandHeader& header, const void* pBody, int socket);

        static void OnProcessExit(int64_t commandId, int socket, os::ProcessId processId);
        static void OnProcessRaiseJitDebug(int64_t commandId, int socket, os::ProcessId processId);
        static void OnProcessStart(int64_t commandId, int socket, os::ProcessId processId);

        bool IsAllowedSystemProcess(ncm::ProgramId programId);

        void ProcessCommandLaunchApplication(
            const CommandHeader& header,
            const void* pBody,
            int socket );
        void ProcessCommandLaunchInstalledApplication(
            const CommandHeader& header,
            const void* pBody,
            int socket );
        void ProcessCommandLaunchGameCardApplication(
            const CommandHeader& header,
            const void* pBody,
            int socket );
        void ProcessCommandLaunchInstalledSystemProcess(
            const CommandHeader& header,
            const void* pBody,
            int socket);
        void ProcessCommandTerminateProcess(
            const CommandHeader& header,
            int socket );
        void ProcessCommandRegisterTenvDefinitionFilePath(
            const CommandHeader& header,
            const void* pBody,
            int socket );
        void ProcessCommandTerminateApplication(
            const CommandHeader& header,
            int socket );
        void ProcessCommandSubscribeProcessEvent(
            const CommandHeader& header,
            const void* pBody,
            int socket,
            int64_t commandId );
        void ProcessCommandSetSafeMode(
            const CommandHeader& header,
            int socket );
        void ProcessCommandShutdown(
            const CommandHeader& header,
            int socket );
        void ProcessCommandReboot(
            const CommandHeader& header,
            int socket );

    };
    nn::os::SdkRecursiveMutexType* GetHtcsSendMutex();
}}  // namespace nn::scs
