﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/bpc/bpc.h>
#include <nn/settings/system/settings_SystemApplication.h>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>

#include "safemode_Display.h"

namespace nn { namespace safemode {

const int WaitAnimationWidth = 32;
const int WaitAnimationHeight = 32;
Bit8 g_WaitAnimationData[WaitAnimationWidth * WaitAnimationHeight * 4];

const int ProgressBarWidth = 640;
const int ProgressBarHeight = 8;
Bit8 g_ProgressBarData[ProgressBarWidth * ProgressBarHeight * 4];

int g_Progress = 0;
volatile bool g_AnimationExitFlag = false;
os::ThreadType g_AnimationThread;

const int Position[][2] = {
    {0, 0},
    {1, 0},
    {2, 0},
    {2, 1},
    {2, 2},
    {1, 2},
    {0, 2},
    {0, 1},
};

void ProgressAnimation() NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK char g_AnimationThreadStack[16 * 1024];
    os::CreateThread(&g_AnimationThread, [](void*)
    {
        int waitAnimIndex = 0;

        std::memset(g_WaitAnimationData, 0, sizeof(g_WaitAnimationData));
        std::memset(g_ProgressBarData, 64, sizeof(g_ProgressBarData));

        nn::safemode::SetupDisplay();

        while (g_AnimationExitFlag == false)
        {
            // 待ちアニメーション
            for (int i = 0; i < 8; ++i)
            {
                int bright = 112;
                if (waitAnimIndex == i)
                {
                    bright = 255;
                }

                for (int y = 0; y < 8; y++)
                {
                    std::memset(g_WaitAnimationData + (Position[i][1] * 12 + y) * WaitAnimationWidth * 4 + Position[i][0] * 12 * 4, bright, 8 * 4);
                }
            }

            waitAnimIndex = (waitAnimIndex + 1) % 8;

            // 待ちアニメーション表示
            nn::safemode::ShowDisplay(
                //280, 356, //左
                624, 308, // 上
                WaitAnimationWidth, WaitAnimationHeight,
                reinterpret_cast<const Bit32*>(g_WaitAnimationData), sizeof(g_WaitAnimationData));

            // プログレスバー表示
            for (int y = 0; y < ProgressBarHeight; y++)
            {
                std::memset(g_ProgressBarData + y * ProgressBarWidth * 4, 192, g_Progress * 4);
            }
            nn::safemode::ShowDisplay(
                328, 368,
                ProgressBarWidth, ProgressBarHeight,
                reinterpret_cast<const Bit32*>(g_ProgressBarData), sizeof(g_ProgressBarData));

            // Vsync を待つのは辛いのでスリープで代用
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(119) );
        }

    }, nullptr, g_AnimationThreadStack, sizeof(g_AnimationThreadStack), 16);
    os::StartThread(&g_AnimationThread);
}

Result StartNup()
{
    NN_SDK_LOG("[SafeMode] Start Safe mode NUP.\n");

    // デバッグ用
    //nn::svc::KernelDebug(nn::svc::KernelDebugType_Port, static_cast<nn::Bit64>(-1ll), 0, 0);
    //nn::svc::KernelDebug(nn::svc::KernelDebugType_ThreadCallstack, static_cast<Bit64>(-1ll), 0, 0);

    nn::ns::Initialize();
    nn::nifm::Initialize();

    // ネットワーク接続を待つ
    nn::nifm::SubmitNetworkRequestAndWait();

    bool networkFlag = nn::nifm::IsNetworkAvailable();
    NN_SDK_LOG("[SafeMode] network available = %d\n", networkFlag);
    if (networkFlag == false)
    {
        nn::bpc::RebootSystem();
    }

    // NUP 開始
    nn::ns::SystemUpdateControl systemUpdateControl;
    NN_RESULT_DO(systemUpdateControl.Occupy());

    nn::ns::AsyncResult asyncResult;
    NN_RESULT_DO(systemUpdateControl.RequestDownloadLatestUpdate(&asyncResult));

    while(!asyncResult.TryWait())
    {
        auto systemUpdateProgress = systemUpdateControl.GetDownloadProgress();
        NN_UNUSED(systemUpdateProgress);
        NN_SDK_LOG("[SafeMode] SystemUpdateProgress %d / %d\n", systemUpdateProgress.loaded, systemUpdateProgress.total);

        if (systemUpdateProgress.total > 0)
        {
            g_Progress = systemUpdateProgress.loaded * ProgressBarWidth / systemUpdateProgress.total;
        }

        // 1秒おきにプログレスバーを更新
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    if(asyncResult.Get().IsFailure())
    {
        // エラーが返った場合リセット（既に最新の場合もここに来る）
        nn::bpc::RebootSystem();
    }

    g_Progress = ProgressBarWidth;
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    // NUP の変更を適用
    systemUpdateControl.ApplyDownloadedUpdate();

    g_AnimationExitFlag = true;

    nn::os::WaitThread(&g_AnimationThread);

    NN_SDK_LOG("[SafeMode] Network update done.");

    NN_RESULT_SUCCESS;
}

}}

extern "C" void nninitStartup()
{
}
extern "C" void nndiagStartup()
{
}
extern "C" void nnMain()
{
    // リブート用
    nn::bpc::InitializeBoardPowerControl();

    // 引っ越し中断中は NUP しないでリセット
    nn::settings::system::AppletLaunchFlagSet appletLaunchFlag;
    nn::settings::system::GetAppletLaunchFlags(&appletLaunchFlag);
    if (appletLaunchFlag.Test<nn::settings::system::AppletLaunchFlag::Migration>())
    {
        nn::bpc::RebootSystem();
    }

    // アニメスレッド開始
    nn::safemode::ProgressAnimation();

    // NUP 開始
    nn::safemode::StartNup();

    // リブート
    nn::bpc::RebootSystem();
}
