﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/dd.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>

#include <t21x/arapbpm.h>
#include <t21x/arclk_rst.h>
#include <t21x/ardsi.h>
#include "reg.h"

#include "safemode_Display.h"

#define WRITE_REGISTERS(base, data)     WriterRegisters(base, data, (sizeof(data) / sizeof(*(data))))


namespace nn { namespace safemode {

    namespace
    {
        const uintptr_t FrameBufferAddress = 0xc0000000;

        const size_t DeviceAddressSpaceBlockSize    = (1 << 22);
        const size_t FrameBufferSize                = 768 * 1280 * 4;

        uintptr_t   NN_BLSM_REG_DISP1;

        dd::DeviceAddressSpaceType  g_DeviceAddressSpace;
        Bit32*                      g_pFrameBuffer = nullptr;

        Bit8                        g_FrameBuffer[FrameBufferSize + DeviceAddressSpaceBlockSize];

        struct RegWrite
        {
            Bit32 offset;
            Bit32 value;
        };

        const RegWrite Dc04[] =
        {
            { 0x0042, 0x00000040 },
            { 0x0700, 0 },
            { 0x0042, 0x00000020 },
            { 0x0700, 0 },
            { 0x0042, 0x00000010 }, /* window A, 0x20 for B, 0x40 for C, 0x80 for D */
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0703, 0x0000000d }, /* R8G8B8A8 */
            { 0x0700, 0 },
            { 0x0700, 0 },
            { 0x0704, 0 },
            { 0x0707, 0 },
            { 0x0708, 0 },
            { 0x0706, 0x05000b40 },
            { 0x0709, 0x10001000 },
            { 0x0705, 0x050002d0 },
            { 0x070a, 0x06000c00 },
            { 0x0702, 0 },
            { 0x080b, 0 },
            { 0x800,  FrameBufferAddress },
            { 0x0806, 0 },
            { 0x0808, 0 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0 },
            { 0x0402, 0x20000000 },
            { 0x0700, 0x40000000 },
            { 0x0032, 0x00000020 },
            { 0x0041, 0x00000300 },
            { 0x0041, 0x00000003 },
        };

        void WriterRegisters(uintptr_t baseAddress, const RegWrite* pWrites, int numWrites)
        {
            for( int i = 0; i < numWrites; ++i )
            {
                reg::Write(baseAddress + (pWrites[i].offset * sizeof(Bit32)), pWrites[i].value);
            }
        }

        void BitBlt(int offsetX, int offsetY, int width, int height, const Bit32* buffer, size_t size) NN_NOEXCEPT
        {
            // const int DisplayWidth = 720;
            const int DisplayHeight = 1280;
            const int DisplayStride = 768;

            NN_SDK_ASSERT(width * height * sizeof(Bit32) <= size);

            for( int y = 0; y < height; ++y )
            {
                for( int x = 0; x < width; ++x )
                {
                    int dispX = offsetY + y;
                    int dispY = DisplayHeight - (offsetX + x);
                    int frameBufferOffset = dispY * DisplayStride + dispX;
                    int bufferOffset = y * width + x;
                    g_pFrameBuffer[frameBufferOffset] = buffer[bufferOffset];
                }
            }
            nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);
        }

        void SetupFrameBuffer() NN_NOEXCEPT
        {
            if (!g_pFrameBuffer) // 一度だけ実行
            {
                const uintptr_t frameBufferAddress = reinterpret_cast<uintptr_t>(g_FrameBuffer);
                const uintptr_t alingedFrameBufferAddress = util::align_up(frameBufferAddress, DeviceAddressSpaceBlockSize);

                g_pFrameBuffer = reinterpret_cast<Bit32*>(alingedFrameBufferAddress);

                // ゼロクリアされていなかった場合に表示直後ノイズが乗る可能性があるため、明示的にクリアしておく (IAAA-3561)
                memset(g_pFrameBuffer, 0, FrameBufferSize);
                nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::CreateDeviceAddressSpace(&g_DeviceAddressSpace, 0, (1ull << 32)) );

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::AttachDeviceAddressSpace(&g_DeviceAddressSpace, dd::DeviceName_Dc) );

                NN_ABORT_UNLESS_RESULT_SUCCESS(
                    dd::MapDeviceAddressSpaceAligned(
                        &g_DeviceAddressSpace,
                        dd::GetCurrentProcessHandle(),
                        alingedFrameBufferAddress,
                        FrameBufferSize,
                        FrameBufferAddress,
                        dd::MemoryPermission_ReadOnly ));
            }
            else
            {
                memset(g_pFrameBuffer, 0, FrameBufferSize);
                nn::dd::FlushDataCache(g_pFrameBuffer, FrameBufferSize);
            }
        }
    }
    // anonymous namespace


    void SetupDisplay() NN_NOEXCEPT
    {
        NN_BLSM_REG_DISP1 = dd::QueryIoMappingAddress(0x54200000,  0x3000);

        SetupFrameBuffer();

        //
        // ディスプレイの設定は Dc02 を書き込むまで
        // boot プロセスで準備してからここに来る想定。
        // Clock And Reset の利用を boot に閉じるため。
        //

        WRITE_REGISTERS(NN_BLSM_REG_DISP1, Dc04);

        // #17 wait 35 ms
        os::SleepThread(TimeSpan::FromMilliSeconds(35));
    }

    void ShowDisplay(int offsetX, int offsetY, int width, int height, const Bit32* buffer, size_t size) NN_NOEXCEPT
    {
        BitBlt(offsetX, offsetY, width, height, buffer, size);

        // バックライトも boot でつけた状態でここに来る想定
    }

}} // namespace nn::safemode
