﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_Types.h>
#include <nn/ro/ro_Result.h>
#include <nn/ro/ro_ResultPrivate.h>
#include <nn/ro/ro_Types.h>
#include <nn/ro/ro_DebugMonitorTypes.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/ro/detail/ro_NroHeader.h>
#include <nn/ro/detail/ro_NrrHeader.h>
#include <algorithm>

namespace nn { namespace ro {
    struct NrrHash
    {
        Bit8 data[ro::detail::NrrHeader::HashSize];

        bool operator <(const NrrHash& rhs) const
        {
            return std::memcmp(this, &rhs, sizeof(rhs)) < 0;
        }
        bool operator ==(const NrrHash& rhs) const
        {
            return std::memcmp(this, &rhs, sizeof(rhs)) == 0;
        }
        bool operator !=(const NrrHash& rhs) const
        {
            return ! operator ==(rhs);
        }
    };

    struct NroInfo
    {
        uint64_t                moduleAddress;
        uint64_t                imageAddress;
        uint64_t                imageSize;
        uint64_t                bufferAddress;
        uint64_t                bufferSize;
        uint64_t                codeSize;
        uint64_t                dataSize;
        ro::detail::ModuleId    moduleId;
    };

    struct NrrInfo
    {
        const ro::detail::NrrHeader*        pHeader;
        uint64_t                            imageAddress;
        uint64_t                            imageSize;
        uint64_t                            aliasCodeAddress;
    };

    class RoServer :
        public nn::util::IntrusiveListBaseNode<RoServer>
    {
    public:
        RoServer() NN_NOEXCEPT
        {
            for (size_t i = 0; i < NN_ARRAY_SIZE(m_NroIsUsed); i++)
            {
                m_NroIsUsed[i] = false;
            }
            for (size_t i = 0; i < NN_ARRAY_SIZE(m_NrrIsUsed); i++)
            {
                m_NrrIsUsed[i] = false;
            }
        }
        ~RoServer() NN_NOEXCEPT;

        nn::Result RegisterModuleInfo(
                std::uint64_t                   processId,
                std::uint64_t                   imageAddress,
                std::uint64_t                   imageSize) NN_NOEXCEPT
        {
            if (!IsValidProcessId(processId))
            {
                return ro::ResultInvalidProcess();
            }
            return RegisterModuleInfo(imageAddress, imageSize);
        }

        nn::Result UnregisterModuleInfo(
                std::uint64_t                   processId,
                std::uint64_t                   imageAddress) NN_NOEXCEPT
        {
            if (!IsValidProcessId(processId))
            {
                return ro::ResultInvalidProcess();
            }
            return UnregisterModuleInfo(imageAddress);
        }

        nn::Result MapManualLoadModuleMemory(
                sf::Out<uint64_t>               pOutAddress,
                std::uint64_t                   processId,
                std::uint64_t                   imageAddress,
                std::uint64_t                   imageSize,
                std::uint64_t                   bufferAddress,
                std::uint64_t                   bufferSize) NN_NOEXCEPT
        {
            if (!IsValidProcessId(processId))
            {
                return ro::ResultInvalidProcess();
            }
            return MapManualLoadModuleMemory(pOutAddress.GetPointer(), imageAddress, imageSize, bufferAddress, bufferSize);
        }

        nn::Result UnmapManualLoadModuleMemory(
                std::uint64_t                   processId,
                std::uint64_t                   baseAddress) NN_NOEXCEPT
        {
            if (!IsValidProcessId(processId))
            {
                return ro::ResultInvalidProcess();
            }
            return UnmapManualLoadModuleMemory(baseAddress);
        }

        nn::Result RegisterProcessHandle(
                std::uint64_t                   processId,
                sf::NativeHandle&&              process) NN_NOEXCEPT;

        static nn::Result GetProcessModuleInfo(
                int*                            pOutCount,
                ModuleInfo*                     pBuffer,
                int                             num,
                os::ProcessId                   processId) NN_NOEXCEPT;

        static void SetDevelopmentHardware(bool isDevelopmentHardware) NN_NOEXCEPT
        {
            g_IsDevelopmentHardware = isDevelopmentHardware;
        }
        static bool IsDevelopmentHardware() { return g_IsDevelopmentHardware; }

        static void SetDevelopmentFunctionEnabled(bool isDevelopmentFunctionEnabled) NN_NOEXCEPT
        {
            g_IsDevelopmentFunctionEnabled = isDevelopmentFunctionEnabled;
        }
        static bool IsDevelopmentFunctionEnabled() { return g_IsDevelopmentFunctionEnabled; }
    private:
        nn::Result RegisterModuleInfo(
                uint64_t                        imageAddress,
                uint64_t                        imageSize) NN_NOEXCEPT;
        nn::Result UnregisterModuleInfo(
                uint64_t                        imageAddress) NN_NOEXCEPT;
        nn::Result MapManualLoadModuleMemory(
                uint64_t*                       pOutAddress,
                uint64_t                        imageAddress,
                uint64_t                        imageSize,
                uint64_t                        bufferAddress,
                uint64_t                        bufferSize) NN_NOEXCEPT;
        nn::Result UnmapManualLoadModuleMemory(
                uint64_t                        baseAddress) NN_NOEXCEPT;

        bool IsValidProcessId(std::uint64_t processId)
        {
            if (m_ProcessHandle.GetOsHandle() == os::InvalidNativeHandle)
            {
                return false;
            }
            os::ProcessId targetProcessId = { processId };
            if (m_ProcessId != targetProcessId)
            {
                return false;
            }
            return true;
        }

        os::ProcessId GetProcessId() const { return m_ProcessId; }

        Result VerifyNroAndGetNroMapSize(
                ro::detail::ModuleId* pOutModuleId,
                uint64_t* pOutTextSize,
                uint64_t* pOutRoSize,
                uint64_t* pOutDataSize,
                uint64_t baseAddress,
                uint64_t imageSize,
                uint64_t bufferSize) NN_NOEXCEPT;

        bool VerifyNroHash(const NrrHash* nroHash) NN_NOEXCEPT
        {
            for (size_t i = 0; i < NN_ARRAY_SIZE(m_NrrInfo); i++)
            {
                if (m_NrrIsUsed[i])
                {
                    auto pHeader = m_NrrInfo[i].pHeader;
                    const NrrHash* pHashList = reinterpret_cast<const NrrHash*>(pHeader->GetHashList());
                    if (std::binary_search(pHashList, pHashList + pHeader->GetNumHash(), *nroHash))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        bool HasNro(const ro::detail::ModuleId* pModuleId) NN_NOEXCEPT
        {
            for (size_t i = 0; i < NN_ARRAY_SIZE(m_NroInfo); i++)
            {
                if (m_NroIsUsed[i])
                {
                    if (std::memcmp(&m_NroInfo[i].moduleId, pModuleId, sizeof(ro::detail::ModuleId)) == 0)
                    {
                        return true;
                    }
                }
            }

            return false;
        }


    private:
        enum {
            NroCount  = 64,
        };

        bool             m_NroIsUsed[NroCount];
        bool             m_NrrIsUsed[NroCount];
        NroInfo          m_NroInfo[NroCount];
        NrrInfo          m_NrrInfo[NroCount];
        sf::NativeHandle m_ProcessHandle;
        os::ProcessId    m_ProcessId;
    private:
        typedef nn::util::IntrusiveList<RoServer, nn::util::IntrusiveListBaseNodeTraits<RoServer>> RoServerList;
        static RoServerList    g_List;
        static bool g_IsDevelopmentHardware;
        static bool g_IsDevelopmentFunctionEnabled;
    };

}}  // namespace nn::ro
