﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "ro_RoServer.h"
#include "ro_AutoClose.h"
#include "ro_Random.h"

// nrr 検証用公開鍵
extern nn::Bit8 DevKey_begin[];
extern nn::Bit8 DevKey_end[];
extern nn::Bit8 ProductKey_begin[];
extern nn::Bit8 ProductKey_end[];

namespace nn { namespace ro {
namespace {
const int RetrySearchCount = 512;

class MappedCodeMemory
{
public:
    MappedCodeMemory() NN_NOEXCEPT
        : m_Handle(svc::INVALID_HANDLE_VALUE)
        , m_Result(ro::ResultInternalError())
        , m_ToAddress(0)
        , m_FromAddress(0)
        , m_Size(0)
        {
        }
    MappedCodeMemory(svc::Handle handle, uint64_t toAddress, uint64_t fromAddress, uint64_t size) NN_NOEXCEPT
        : m_Handle(handle)
        , m_ToAddress(toAddress)
        , m_FromAddress(fromAddress)
        , m_Size(size)
        {
            m_Result = svc::MapProcessCodeMemory(m_Handle, m_ToAddress, m_FromAddress, m_Size);
        }
    ~MappedCodeMemory() NN_NOEXCEPT
    {
        if( m_Handle != svc::INVALID_HANDLE_VALUE && m_Size > 0 && m_Result.IsSuccess() )
        {
            Result result = svc::UnmapProcessCodeMemory(m_Handle, m_ToAddress, m_FromAddress, m_Size);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    Result GetResult() NN_NOEXCEPT
    {
        return m_Result;
    }
    bool IsFailure() NN_NOEXCEPT
    {
        return m_Result.IsFailure();
    }
    void Cancel() NN_NOEXCEPT
    {
        m_Handle = svc::INVALID_HANDLE_VALUE;
    }
    MappedCodeMemory& operator=(MappedCodeMemory&& other) NN_NOEXCEPT
    {
        m_Handle = other.m_Handle;
        m_Result = other.m_Result;
        m_ToAddress = other.m_ToAddress;
        m_FromAddress = other.m_FromAddress;
        m_Size = other.m_Size;
        other.m_Handle = svc::INVALID_HANDLE_VALUE;
        return *this;
    }

private:
    svc::Handle m_Handle;
    Result      m_Result;
    uint64_t    m_ToAddress;
    uint64_t    m_FromAddress;
    uint64_t    m_Size;
};

Result SearchFreeRegion(uintptr_t* pOut, uint64_t mappingSize) NN_NOEXCEPT
{
    uintptr_t heapBegin;
    uintptr_t heapSize;
    uintptr_t rsvBegin;
    uintptr_t rsvSize;
    uintptr_t aslrBegin;
    uintptr_t aslrSize;
    Bit64 value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    heapBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    heapSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    rsvBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    rsvSize = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionAddress, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    aslrBegin = value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionSize, svc::PSEUDO_HANDLE_CURRENT_PROCESS, 0));
    aslrSize = value;

    uintptr_t addr = aslrBegin;
    for (;;)
    {
        nn::svc::MemoryInfo  memoryInfo;
        nn::svc::PageInfo   pageInfo;

        if (addr + mappingSize <= addr)
        {
            return ro::ResultOutOfAddressSpace();
        }

        if (heapSize && !(addr + mappingSize - 1 < heapBegin || heapBegin + heapSize - 1 < addr))
        {
            if (heapBegin + heapSize <= addr)
            {
                return ro::ResultOutOfAddressSpace();
            }
            addr = heapBegin + heapSize;
            continue;
        }

        if (rsvSize && !(addr + mappingSize - 1 < rsvBegin || rsvBegin + rsvSize - 1 < addr))
        {
            if (rsvBegin + rsvSize <= addr)
            {
                return ro::ResultOutOfAddressSpace();
            }
            addr = rsvBegin + rsvSize;
            continue;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::QueryMemory(&memoryInfo, &pageInfo, addr));
        if (memoryInfo.state == nn::svc::MemoryState_Free && ((memoryInfo.baseAddress + memoryInfo.size) - addr) >= mappingSize)
        {
            *pOut = addr;
            return nn::ResultSuccess();
        }
        if (memoryInfo.baseAddress + memoryInfo.size <= addr)
        {
            return ro::ResultOutOfAddressSpace();
        }
        if (memoryInfo.baseAddress + memoryInfo.size - 1 >= aslrBegin + aslrSize - 1)
        {
            return ro::ResultOutOfAddressSpace();
        }
        addr = memoryInfo.baseAddress + memoryInfo.size;
    }
}

class ProcessRegionInfo
{
public:
    NN_IMPLICIT ProcessRegionInfo(nn::svc::Handle process) NN_NOEXCEPT
    {
        Bit64 value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionAddress, process, 0));
        m_HeapBegin = value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_HeapRegionSize, process, 0));
        m_HeapSize = value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionAddress, process, 0));
        m_RsvBegin = value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_ReservedRegionSize, process, 0));
        m_RsvSize = value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionAddress, process, 0));
        m_AslrBegin = value;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&value, nn::svc::InfoType_AslrRegionSize, process, 0));
        m_AslrSize = value;
    }

    uint64_t GetAslrRegion(uint64_t mappingSize) NN_NOEXCEPT
    {
        if (mappingSize > m_AslrSize)
        {
            return 0;
        }
        for (int i = 0; i < RetrySearchCount; i++)
        {
            uint64_t addr = m_AslrBegin + nn::ro::random::GetValue((m_AslrSize - mappingSize) / os::MemoryPageSize) * os::MemoryPageSize;

            if (m_HeapSize && !(addr + mappingSize - 1 < m_HeapBegin || m_HeapBegin + m_HeapSize - 1 < addr))
            {
                continue;
            }

            if (m_RsvSize && !(addr + mappingSize - 1 < m_RsvBegin || m_RsvBegin + m_RsvSize - 1 < addr))
            {
                continue;
            }
            return addr;
        }

        return 0;
    }

    bool CanEmplaceGuardSpaces(nn::svc::Handle process, uint64_t address, uint64_t size)
    {
        // 隣接している 16KiB をスタックガードとして使われていても問題ないように、そこが Free であるかチェックする。
        nn::svc::MemoryInfo memoryInfo;
        nn::svc::PageInfo   pageInfo;

        auto result = nn::svc::QueryProcessMemory(&memoryInfo, &pageInfo, process, address - 1);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
        if (memoryInfo.state == svc::MemoryState_Free &&
            address - StackGuardSize >= memoryInfo.baseAddress)
        {
            result = nn::svc::QueryProcessMemory(&memoryInfo, &pageInfo, process, address + size);
            NN_SDK_ASSERT(result.IsSuccess());
            NN_UNUSED(result);
            if (memoryInfo.state == svc::MemoryState_Free &&
                address + size + StackGuardSize <= memoryInfo.baseAddress + memoryInfo.size)
            {
                return true;
            }
        }

        return false;
    }

private:
    uint64_t m_HeapBegin;
    uint64_t m_HeapSize;
    uint64_t m_RsvBegin;
    uint64_t m_RsvSize;
    uint64_t m_AslrBegin;
    uint64_t m_AslrSize;
    const size_t StackGuardSize = os::MemoryPageSize * 4; // OS が確保するスタックガードのサイズ
};

Result VerifyCertification(const ro::detail::NrrHeader* pHeader) NN_NOEXCEPT
{
    const Bit8* publicKey = (RoServer::IsDevelopmentHardware() ? DevKey_begin : ProductKey_begin);
    const size_t publicKeySize = (RoServer::IsDevelopmentHardware() ?
            DevKey_end - DevKey_begin :
            ProductKey_end - ProductKey_begin);
    const Bit8 PublicExponent[] = { 0x01, 0x00, 0x01 };

    if (!crypto::VerifyRsa2048PssSha256(
                pHeader->GetCertificationSign(),
                ro::detail::NrrCertification::RsaKeySize,
                publicKey,
                publicKeySize,
                PublicExponent,
                sizeof(PublicExponent),
                pHeader->GetCertificationSignArea(),
                ro::detail::NrrCertification::SignAreaSize
                ))
    {
        return ro::ResultNotAuthorized();
    }

    if (!pHeader->CheckProgramId())
    {
        return ro::ResultNotAuthorized();
    }

    return ResultSuccess();
}

Result VerifyNrrSign(const ro::detail::NrrHeader* pHeader) NN_NOEXCEPT
{
    const Bit8 PublicExponent[] = { 0x01, 0x00, 0x01 };

    if (!crypto::VerifyRsa2048PssSha256(
                pHeader->GetSign(),
                ro::detail::NrrHeader::RsaKeySize,
                pHeader->GetCertificationPublicKey(),
                ro::detail::NrrHeader::RsaKeySize,
                PublicExponent,
                sizeof(PublicExponent),
                pHeader->GetSignArea(),
                pHeader->GetSignAreaSize()
                ))
    {
        return ro::ResultNotAuthorized();
    }
    return ResultSuccess();
}

Result MapNrr(
        const ro::detail::NrrHeader** ppHeader,
        uint64_t* pAliasCodeAddress,
        nn::svc::Handle process,
        uint64_t imageAddress, uint64_t imageSize) NN_NOEXCEPT
{
    uint64_t baseAddress = 0;
    int i;

    MappedCodeMemory imageMap;
    ProcessRegionInfo regionInfo(process);
    for (i = 0; i < RetrySearchCount; i++)
    {
        baseAddress = regionInfo.GetAslrRegion(imageSize);
        if (baseAddress == 0)
        {
            return ro::ResultOutOfAddressSpace();
        }

        MappedCodeMemory tmpImageMap(process, baseAddress, imageAddress, imageSize);
        if (tmpImageMap.GetResult() <= svc::ResultInvalidCurrentMemory())
        {
            continue;
        }
        if (tmpImageMap.IsFailure())
        {
            return tmpImageMap.GetResult();
        }

        // マップした領域がスタックガードを含んでしまっていないかをチェックする
        if (!regionInfo.CanEmplaceGuardSpaces(process, baseAddress, imageSize))
        {
            continue;
        }

        imageMap = std::move(tmpImageMap);
        break;
    }
    if (i == RetrySearchCount)
    {
        NN_SDK_LOG("[ro] Failed to map image (imageAddress=0x%llx, imageSize=0x%llx\n)", imageAddress, imageSize);
        return ro::ResultOutOfAddressSpace();
    }

    uintptr_t nrrAddress;
    auto result = SearchFreeRegion(&nrrAddress, imageSize);
    if (result.IsFailure())
    {
        return ro::ResultOutOfAddressSpace();
    }
    AutoCloseMap nrrMap(nrrAddress, process, baseAddress, imageSize);

    const ro::detail::NrrHeader* pHeader = reinterpret_cast<ro::detail::NrrHeader*>(nrrAddress);

    if (!pHeader->CheckSignature())
    {
        return ro::ResultInvalidNrrImage();
    }

    if (pHeader->GetSize() != imageSize)
    {
        return ro::ResultInvalidSize();
    }

    // 開発機で fwdbg に ease nro restriction が設定されていれば、nrr の検証失敗、ApplicationId の認証失敗はエラーにしないようにする
    bool isEaseNroRestriction = false;
    bool easeNroRestrictionValue;
    const size_t readEaseNroRestrictionSize = nn::settings::fwdbg::GetSettingsItemValue(&easeNroRestrictionValue, sizeof(easeNroRestrictionValue), "ro", "ease_nro_restriction");
    // 指定された条件が整っている場合のみ、フラグに true が入るようにする。
    if (readEaseNroRestrictionSize == sizeof(isEaseNroRestriction) && RoServer::IsDevelopmentFunctionEnabled())
    {
        isEaseNroRestriction = easeNroRestrictionValue;
    }

    // 証明書の検証
    result = VerifyCertification(pHeader);
    if (result.IsFailure())
    {
        if (!isEaseNroRestriction)
        {
            return result;
        }
    }

    // 署名の検証
    result = VerifyNrrSign(pHeader);
    if (result.IsFailure())
    {
        if (!isEaseNroRestriction)
        {
            return result;
        }
    }

    Bit64 programId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetInfo(&programId, nn::svc::InfoType_ProgramId, process, 0));

    if (pHeader->GetProgramId() != programId)
    {
        if (!isEaseNroRestriction)
        {
            return ro::ResultInvalidNrrImage();
        }
    }


    imageMap.Cancel();
    nrrMap.Cancel();

    *ppHeader = pHeader;
    *pAliasCodeAddress = baseAddress;

    return ResultSuccess();
} // NOLINT(impl/function_size)

Result UnmapNrr(
        nn::svc::Handle process,
        const ro::detail::NrrHeader* pHeader,
        uint64_t imageAddress, uint64_t imageSize,
        uint64_t aliasCodeAddress) NN_NOEXCEPT
{
    Result result = nn::svc::UnmapProcessMemory(
            reinterpret_cast<uintptr_t>(pHeader),
            process,
            aliasCodeAddress,
            imageSize);
    if (result.IsFailure())
    {
        return result;
    }

    return nn::svc::UnmapProcessCodeMemory(
            process,
            aliasCodeAddress,
            imageAddress,
            imageSize);
}

Result MapNro(
        uint64_t* pModuleAddress,
        nn::svc::Handle process,
        uint64_t imageAddress, uint64_t imageSize,
        uint64_t bufferAddress, uint64_t bufferSize) NN_NOEXCEPT
{
    uint64_t memSize = imageSize + bufferSize;
    uint64_t baseAddress = 0;
    int i;

    MappedCodeMemory nroMap;
    MappedCodeMemory bssMap;
    ProcessRegionInfo regionInfo(process);

    for (i = 0; i < RetrySearchCount; i++)
    {
        baseAddress = regionInfo.GetAslrRegion(memSize);
        if (baseAddress == 0)
        {
            return ro::ResultOutOfAddressSpace();
        }

        MappedCodeMemory tmpNroMap(process, baseAddress, imageAddress, imageSize);
        if (tmpNroMap.GetResult() <= svc::ResultInvalidCurrentMemory())
        {
            continue;
        }
        if (tmpNroMap.IsFailure())
        {
            return tmpNroMap.GetResult();
        }

        if (bufferSize > 0)
        {
            MappedCodeMemory tmpBssMap(process, baseAddress + imageSize, bufferAddress, bufferSize);
            if (tmpBssMap.GetResult() <= svc::ResultInvalidCurrentMemory())
            {
                continue;
            }
            if (tmpBssMap.IsFailure())
            {
                return tmpBssMap.GetResult();
            }

            // マップした領域がスタックガードを含んでしまっていないかをチェックする
            if (!regionInfo.CanEmplaceGuardSpaces(process, baseAddress, imageSize + bufferSize))
            {
                continue;
            }

            bssMap = std::move(tmpBssMap);
        }
        else
        {
            // マップした領域がスタックガードを含んでしまっていないかをチェックする
            if (!regionInfo.CanEmplaceGuardSpaces(process, baseAddress, imageSize))
            {
                continue;
            }
        }
        nroMap = std::move(tmpNroMap);
        break;
    }
    if (i == RetrySearchCount)
    {
        NN_SDK_LOG("[ro] Failed to map image (imageAddress=0x%llx, imageSize=0x%llx, bufferAddress=0x%llx, bufferSize=0x%llx)\n", imageAddress, imageSize, bufferAddress, bufferSize);
        return ro::ResultOutOfAddressSpace();
    }

    nroMap.Cancel();
    bssMap.Cancel();

    *pModuleAddress = baseAddress;

    return ResultSuccess();
}

Result UnmapNro(
        nn::svc::Handle process,
        uint64_t moduleAddress,
        uint64_t imageAddress,
        uint64_t bufferAddress, uint64_t bufferSize,
        uint64_t codeSize, uint64_t dataSize) NN_NOEXCEPT
{
    const uint64_t rwOffset = codeSize;
    const uint64_t rwAddress = moduleAddress + rwOffset;

    if (bufferSize > 0)
    {
        auto result = nn::svc::UnmapProcessCodeMemory(
                process, rwAddress + dataSize, bufferAddress, bufferSize);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ro] Failed to unmap buffer memory area (base=0x%llx, buffer=0x%llx, size=0x%llx)\n", rwAddress + dataSize, bufferAddress, bufferSize);
            return result;
        }
    }

    if (dataSize > 0)
    {
        auto result = nn::svc::UnmapProcessCodeMemory(
                process, rwAddress, imageAddress + rwOffset, dataSize);
        if (result.IsFailure())
        {
            NN_SDK_LOG("[ro] Failed to unmap data memory area (base=0x%llx, addr=0x%llx, size=0x%llx)\n",
                    rwAddress, imageAddress + rwOffset, dataSize);
            return result;
        }
    }

    auto result = nn::svc::UnmapProcessCodeMemory(
            process, moduleAddress, imageAddress, codeSize);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[ro] Failed to unmap text and ro memory area (base=0x%llx, addr=0x%llx, size=0x%llx)\n", moduleAddress, imageAddress, codeSize);
        return result;
    }

    return ResultSuccess();
}

Result SetNroPermission(
        nn::svc::Handle process,
        uint64_t baseAddress,
        uint64_t textMapSize, uint64_t roMapSize, uint64_t dataMapSize) NN_NOEXCEPT
{
    const uint64_t textOffset = 0;
    const uint64_t roOffset = textOffset + textMapSize;
    const uint64_t dataOffset = roOffset + roMapSize;
    Result result;

    result = nn::svc::SetProcessMemoryPermission(
            process, baseAddress + textOffset, textMapSize,
            nn::svc::MemoryPermission_ReadExecute);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[ro] Failed to change permission of text area (addr=0x%llx, size=0x%llx)\n", baseAddress, textMapSize);
        return result;
    }

    result = nn::svc::SetProcessMemoryPermission(
            process, baseAddress + roOffset, roMapSize,
            nn::svc::MemoryPermission_Read);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[ro] Failed to change permission of ro data area (addr=0x%llx, size=0x%llx)\n", baseAddress + roOffset, roMapSize);
        return result;
    }

    result = nn::svc::SetProcessMemoryPermission(
            process, baseAddress + dataOffset, dataMapSize,
            nn::svc::MemoryPermission_ReadWrite);
    if (result.IsFailure())
    {
        NN_SDK_LOG("[ro] Failed to change permission of rw data area (addr=0x%llx, size=0x%llx)\n", baseAddress + dataOffset, dataMapSize);
        return result;
    }

    return ResultSuccess();
}

}


RoServer::RoServerList RoServer::g_List;
bool RoServer::g_IsDevelopmentHardware;
bool RoServer::g_IsDevelopmentFunctionEnabled;

RoServer::~RoServer() NN_NOEXCEPT
{
    svc::Handle processHandle(m_ProcessHandle.GetOsHandle());

    if (m_ProcessHandle.GetOsHandle() != os::InvalidNativeHandle)
    {
        RoServerList::iterator it = g_List.iterator_to(*this);
        g_List.erase(it);

        svc::Handle process(m_ProcessHandle.GetOsHandle());

        for (size_t i = 0; i < NN_ARRAY_SIZE(m_NrrIsUsed); i++)
        {
            if (m_NrrIsUsed[i])
            {
                UnmapNrr(process,
                        m_NrrInfo[i].pHeader,
                        m_NrrInfo[i].imageAddress,
                        m_NrrInfo[i].imageSize,
                        m_NrrInfo[i].aliasCodeAddress);
            }
        }
    }
}

nn::Result RoServer::RegisterModuleInfo(uint64_t imageAddress, uint64_t imageSize) NN_NOEXCEPT
{
    if ((imageAddress & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidAddress();
    }

    if (imageSize == 0)
    {
        return ro::ResultInvalidSize();
    }

    if ((imageSize & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidSize();
    }

    if (!(imageAddress < imageAddress + imageSize))
    {
        return ro::ResultInvalidSize();
    }

    size_t i = 0;
    for (i = 0; i < NN_ARRAY_SIZE(m_NrrIsUsed); i++)
    {
        if (!m_NrrIsUsed[i])
        {
            break;
        }
    }
    if (i == NN_ARRAY_SIZE(m_NrrIsUsed))
    {
        return ro::ResultMaxRegistration();
    }

    const ro::detail::NrrHeader* pHeader = nullptr;
    uint64_t aliasCodeAddress = 0;
    svc::Handle process(m_ProcessHandle.GetOsHandle());
    Result result = MapNrr(&pHeader, &aliasCodeAddress, process, imageAddress, imageSize);
    if (result.IsFailure())
    {
        return result;
    }
    m_NrrInfo[i].pHeader = pHeader;
    m_NrrInfo[i].imageAddress = imageAddress;
    m_NrrInfo[i].imageSize = imageSize;
    m_NrrInfo[i].aliasCodeAddress = aliasCodeAddress;
    m_NrrIsUsed[i] = true;

    return ResultSuccess();
}

nn::Result RoServer::UnregisterModuleInfo(uint64_t imageAddress) NN_NOEXCEPT
{
    if ((imageAddress & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidAddress();
    }

    size_t i = 0;
    for (i = 0; i < NN_ARRAY_SIZE(m_NrrInfo); i++)
    {
        if (!m_NrrIsUsed[i])
        {
            continue;
        }

        if (m_NrrInfo[i].imageAddress == imageAddress)
        {
            break;
        }
    }
    if (i == NN_ARRAY_SIZE(m_NrrInfo))
    {
        return ro::ResultNotRegistered();
    }

    // アンマップに失敗した場合、マップが残り続ける
    // セッションクローズ時にもう一度アンマップするチャンスはあるが、失敗する可能性が高いのでハンドリングしない
    m_NrrIsUsed[i] = false;

    svc::Handle process(m_ProcessHandle.GetOsHandle());
    return UnmapNrr(process,
            m_NrrInfo[i].pHeader,
            m_NrrInfo[i].imageAddress,
            m_NrrInfo[i].imageSize,
            m_NrrInfo[i].aliasCodeAddress);
}

Result RoServer::VerifyNroAndGetNroMapSize(
        ro::detail::ModuleId* pOutModuleId,
        uint64_t* pOutTextSize,
        uint64_t* pOutRoSize,
        uint64_t* pOutDataSize,
        uint64_t baseAddress,
        uint64_t imageSize,
        uint64_t bufferSize) NN_NOEXCEPT
{
    uintptr_t moduleAddress = 0;
    svc::Handle process(m_ProcessHandle.GetOsHandle());
    auto result = SearchFreeRegion(&moduleAddress, imageSize);
    if (result.IsFailure())
    {
        return ro::ResultOutOfAddressSpace();
    }

    AutoCloseMap imageMap(moduleAddress, process, baseAddress, imageSize);
    if (imageMap.IsFailure())
    {
        return imageMap.GetResult();
    }

    const ro::detail::NroHeader* pHeader = reinterpret_cast<const ro::detail::NroHeader*>(moduleAddress);
    if (!pHeader->CheckSignature())
    {
        return ro::ResultInvalidNroImage();
    }

    if ((pHeader->GetTextSize() & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidNroImage();
    }

    if ((pHeader->GetRoSize() & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidNroImage();
    }

    if ((pHeader->GetDataSize() & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidNroImage();
    }

    if ((pHeader->GetBssSize() & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetSize() != imageSize)
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetBssSize() != bufferSize)
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetTextOffsetAddress() > pHeader->GetRoOffsetAddress())
    {
        return ro::ResultInvalidNroImage();
    }
    if (pHeader->GetRoOffsetAddress() > pHeader->GetDataOffsetAddress())
    {
        return ro::ResultInvalidNroImage();
    }

    // 各セグメントの間に隙間がないことを確認
    if (pHeader->GetTextOffsetAddress() != 0)
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetTextSize() != pHeader->GetRoOffsetAddress())
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetRoOffsetAddress() + pHeader->GetRoSize() != pHeader->GetDataOffsetAddress())
    {
        return ro::ResultInvalidNroImage();
    }

    if (pHeader->GetDataOffsetAddress() + pHeader->GetDataSize() != pHeader->GetSize())
    {
        return ro::ResultInvalidNroImage();
    }

    NrrHash hash;
    crypto::GenerateSha256Hash(
            &hash,
            sizeof(hash),
            reinterpret_cast<void*>(moduleAddress),
            pHeader->GetSize());

    if (!VerifyNroHash(&hash))
    {
        return ro::ResultNotAuthorized();
    }

    // 既に読み込まれているモジュールか調べる
    const ro::detail::ModuleId* pModuleId = pHeader->GetModuleId();
    if (HasNro(pModuleId))
    {
        return ro::ResultNroAlreadyLoaded();
    }

    *pOutModuleId = *pModuleId;
    *pOutTextSize = pHeader->GetTextSize();
    *pOutRoSize = pHeader->GetRoSize();
    *pOutDataSize = pHeader->GetDataSize();

    return ResultSuccess();
}


nn::Result RoServer::MapManualLoadModuleMemory(
        uint64_t*                       pOutAddress,
        uint64_t                        imageAddress,
        uint64_t                        imageSize,
        uint64_t                        bufferAddress,
        uint64_t                        bufferSize) NN_NOEXCEPT
{
    if ((imageAddress & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidAddress();
    }
    if ((imageSize & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidSize();
    }
    if ((bufferAddress & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidAddress();
    }
    if ((bufferSize & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidSize();
    }

    if (imageSize == 0)
    {
        return ro::ResultInvalidSize();
    }
    if (imageAddress >= imageAddress + imageSize)
    {
        return ro::ResultInvalidSize();
    }
    if (bufferSize > 0 && bufferAddress >= bufferAddress + bufferSize)
    {
        return ro::ResultInvalidSize();
    }

    uint64_t memSize = imageSize + bufferSize;
    if (memSize < imageSize || memSize < bufferSize)
    {
        return ro::ResultInvalidSize();
    }

    size_t i = 0;
    for (i = 0; i < NN_ARRAY_SIZE(m_NroIsUsed); i++)
    {
        if (!m_NroIsUsed[i])
        {
            break;
        }
    }
    if (i == NN_ARRAY_SIZE(m_NroIsUsed))
    {
        return ro::ResultMaxModule();
    }

    uint64_t                moduleAddress = 0;
    uint64_t                textSize = 0;
    uint64_t                roSize = 0;
    uint64_t                dataSize = 0;
    ro::detail::ModuleId    moduleId = {};
    svc::Handle process(m_ProcessHandle.GetOsHandle());

    Result result;

    result = MapNro(
            &moduleAddress,
            process,
            imageAddress, imageSize,
            bufferAddress, bufferSize);
    if (result.IsFailure())
    {
        return result;
    }

    result = VerifyNroAndGetNroMapSize(
            &moduleId,
            &textSize, &roSize, &dataSize,
            moduleAddress, imageSize, bufferSize);
    if (result.IsFailure())
    {
        UnmapNro(
                process,
                moduleAddress,
                imageAddress,
                bufferAddress, bufferSize,
                imageSize, 0);

        return result;
    }

    result = SetNroPermission(
            process,
            moduleAddress,
            textSize, roSize, dataSize + bufferSize);

    if (result.IsFailure())
    {
        UnmapNro(
                process,
                moduleAddress,
                imageAddress,
                bufferAddress, bufferSize,
                textSize + roSize, dataSize);

        return result;
    }

    m_NroIsUsed[i] = true;

    m_NroInfo[i].moduleAddress = moduleAddress;
    m_NroInfo[i].imageAddress = imageAddress;
    m_NroInfo[i].imageSize = imageSize;
    m_NroInfo[i].bufferAddress = bufferAddress;
    m_NroInfo[i].bufferSize = bufferSize;
    m_NroInfo[i].codeSize = textSize + roSize;
    m_NroInfo[i].dataSize = dataSize;
    m_NroInfo[i].moduleId = moduleId;

    *pOutAddress = moduleAddress;

    return result;
} // NOLINT(impl/function_size)

nn::Result RoServer::UnmapManualLoadModuleMemory(uint64_t baseAddress) NN_NOEXCEPT
{
    if ((baseAddress & (os::MemoryPageSize - 1)) != 0)
    {
        return ro::ResultInvalidAddress();
    }

    size_t i = 0;
    for (i = 0; i < NN_ARRAY_SIZE(m_NroInfo); i++)
    {
        if (!m_NroIsUsed[i])
        {
            continue;
        }

        if (m_NroInfo[i].moduleAddress == baseAddress)
        {
            break;
        }
    }
    if (i == NN_ARRAY_SIZE(m_NroInfo))
    {
        return ro::ResultNotLoaded();
    }

    m_NroIsUsed[i] = false;

    svc::Handle process(m_ProcessHandle.GetOsHandle());
    return UnmapNro(
            process,
            m_NroInfo[i].moduleAddress,
            m_NroInfo[i].imageAddress,
            m_NroInfo[i].bufferAddress, m_NroInfo[i].bufferSize,
            m_NroInfo[i].codeSize, m_NroInfo[i].dataSize);
}

nn::Result RoServer::RegisterProcessHandle(std::uint64_t processId, sf::NativeHandle&& process) NN_NOEXCEPT
{
    Bit64 targetProcessIdValue;
    svc::Handle processHandle(process.GetOsHandle());
    auto result = svc::GetProcessId(&targetProcessIdValue, processHandle);
    if (result.IsFailure())
    {
        return ro::ResultInvalidProcess();
    }

    os::ProcessId targetProcessId = { targetProcessIdValue };
    os::ProcessId receivedProcessId = { processId };
    if (receivedProcessId != targetProcessId)
    {
        return ro::ResultInvalidProcess();
    }

    // 登録済みプロセスかどうかチェックする
    for (RoServerList::const_iterator it = g_List.begin(); it != g_List.end(); it++)
    {
        if (it->GetProcessId() == receivedProcessId)
        {
            return ro::ResultInvalidSession();
        }
    }

    g_List.push_back(*this);

    m_ProcessHandle = std::move(process);
    m_ProcessId = receivedProcessId;
    return nn::ResultSuccess();
}

nn::Result RoServer::GetProcessModuleInfo(
        int* pOutCount,
        ModuleInfo* pBuffer,
        int num,
        os::ProcessId processId) NN_NOEXCEPT
{
    int count = 0;
    for (RoServerList::const_iterator it = g_List.begin(); it != g_List.end(); it++)
    {
        if (it->GetProcessId() == processId)
        {
            const RoServer* pServer = &*it;

            for (size_t i = 0; i < NN_ARRAY_SIZE(m_NroInfo) && count < num; i++)
            {
                if (!pServer->m_NroIsUsed[i])
                {
                    continue;
                }

                ModuleInfo* pInfo = &pBuffer[count++];
                NN_STATIC_ASSERT(sizeof(pInfo->moduleId) == sizeof(pServer->m_NroInfo[i].moduleId));
                std::memcpy(pInfo->moduleId, &pServer->m_NroInfo[i].moduleId, sizeof(pInfo->moduleId));
                pInfo->address = pServer->m_NroInfo[i].moduleAddress;
                pInfo->size = pServer->m_NroInfo[i].imageSize + pServer->m_NroInfo[i].bufferSize;
            }

            break;
        }
    }

    *pOutCount = count;
    return ResultSuccess();
}

}}  // nn::ro
