﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/ts.h>
#include <nn/ts/detail/ts_Log.h>
#include <nn/ts/impl/ts.h>
#include <nn/ts/server/ts_HipcServiceName.h>
#include <nn/ts/server/ts_IMeasurementServer.autogen.h>
#include <nn/ts/server/ts_MeasurementServer.h>
#include <nn/ts/ts_ShimInternal.h>

namespace nn { namespace ptm {

namespace {

const auto MaxSessionForPort = 3; // nvdbgsvc, application + 1(margin)
const auto SessionCountMax = MaxSessionForPort;
const auto PortCountMax = 1; // ts

::nn::sf::UnmanagedServiceObject<::nn::ts::server::IMeasurementServer, ::nn::ts::server::MeasurementServer> g_MeasurementServer;

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MeasurementServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax>
{
};

MeasurementServerManager g_MeasurementServerManager;


const size_t MeasurementServerStackSize = 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_MeasurementServerStack[MeasurementServerStackSize];
nn::os::ThreadType g_MeasurementServerThread;
}


void InitializeThermalSensorServer() NN_NOEXCEPT
{
    // 準備ができていないサービスに対して接続中に、サービスの登録を行おうとするとデッドロックする場合があるので
    // ポート登録は何よりも先に行うのが安全
    g_MeasurementServerManager.RegisterObjectForPort( g_MeasurementServer.GetShared(),
                                                MaxSessionForPort,
                                                ::nn::ts::server::MeasurementServiceName);

    // ライブラリ実装部(impl)の初期化。
    // TORIAEZU : デバイスの初期化もこの中で行うため、失敗する可能性がある。
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::ts::impl::Initialize());
    ::nn::ts::InitializeWith(g_MeasurementServer.GetShared());

    // Thermal sensor 用サーバの初期化。
    g_MeasurementServerManager.Start();

    // Thermal sensor 用サーバ処理スレッドの作成。
    auto result = ::nn::os::CreateThread(&g_MeasurementServerThread,
                                         [] (void*) { g_MeasurementServerManager.LoopAuto(); },
                                         nullptr,
                                         g_MeasurementServerStack,
                                         sizeof(g_MeasurementServerStack),
                                         NN_SYSTEM_THREAD_PRIORITY(ts, IpcServer));

    // スレッドの作成に失敗した場合は ABORT する。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    ::nn::os::SetThreadNamePointer(&g_MeasurementServerThread, NN_SYSTEM_THREAD_NAME(ts, IpcServer));

    ::nn::os::StartThread(&g_MeasurementServerThread);

    NN_DETAIL_TS_INFO("Start Thermal Sensor.\n");
}


void RequestStopThermalSensorServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す。
    g_MeasurementServerManager.RequestStop();
}

void WaitAndFinalizeThermalSensorServer() NN_NOEXCEPT
{
    // サーバ処理スレッド終了待機と破棄。
    ::nn::os::WaitThread(&g_MeasurementServerThread);
    ::nn::os::DestroyThread(&g_MeasurementServerThread);

    ::nn::ts::impl::Finalize();
}

}}  // namespace nn::ptm
