﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/psm/driver/psm.h>
#include <nn/psm/psm_ServiceName.h>
#include <nn/psm/psm_IPsmServer.sfdl.h>
#include <nn/psm/psm_PsmServer.h>
#include <nn/psm/psm_Internal.h>

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
#include <nn/psm/psm_IPsmManufactureServer.sfdl.h>
#include <nn/psm/psm_ManufactureInternal.h>
#include <nn/psm/psm_ManufactureServiceName.h>
#include <nn/psm/psm_PsmManufactureServer.h>
#endif

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace ptm {

namespace {

// セッションの最大数
const int MaxSessionForPort = 12; // am, ppc, hid, glue, eclct, qlaunch, overlayDisp, wlan, fatal, LibraryAppletController, application + 1(margin)
const int EventSessionCountMax = 7;
#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
const int ManufactureMaxSessionForPort = 1;
const int SessionCountMax = MaxSessionForPort + EventSessionCountMax + ManufactureMaxSessionForPort;
#else
const int SessionCountMax = MaxSessionForPort + EventSessionCountMax;
#endif

// ポートの最大数
#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
const int PortCountMax = 2;
#else
const int PortCountMax = 1;
#endif

// IPC 待ち受けスレッドのスタックサイズ
const size_t ServerStackSize = 4096;

// サーバマネージャを扱うためのクラス
class ServerManager : public
::nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax>
{};

// サーバマネージャ
ServerManager g_ServerManager;

// IPC 待ち受けスレッド
nn::os::ThreadType g_ServerThread;

// IPC 待ち受けスレッドのスタック
NN_ALIGNAS(::nn::os::ThreadStackAlignment)
uint8_t g_ServerStack[ServerStackSize];

} // namespace

void InitializePsmServer() NN_NOEXCEPT
{
    static ::nn::sf::UnmanagedServiceObject<
        ::nn::psm::IPsmServer, ::nn::psm::PsmServer> s_Server;

    // 準備ができていないサービスに対して接続中に、サービスの登録を行おうとするとデッドロックする場合があるので
    // ポート登録は何よりも先に行うのが安全
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.RegisterObjectForPort(
            s_Server.GetShared(),
            MaxSessionForPort,
            ::nn::psm::ServiceName));

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    static ::nn::sf::UnmanagedServiceObject<
        ::nn::psm::IPsmManufactureServer, ::nn::psm::PsmManufactureServer> s_ManufactureServer;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.RegisterObjectForPort(
            s_ManufactureServer.GetShared(),
            ManufactureMaxSessionForPort,
            ::nn::psm::ManufactureServiceName));
#endif

    ::nn::psm::driver::Initialize();
    ::nn::psm::InitializeWith(s_Server.GetShared());

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    ::nn::psm::InitializeForManufactureWith(s_ManufactureServer.GetShared());
#endif

    g_ServerManager.Start();

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::CreateThread(
            &g_ServerThread,
            [] (void*) { g_ServerManager.LoopAuto(); },
            nullptr,
            g_ServerStack,
            sizeof(g_ServerStack),
            NN_SYSTEM_THREAD_PRIORITY(psm, IpcServer)));

    ::nn::os::StartThread(&g_ServerThread);

    ::nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(psm, IpcServer));

    NN_DETAIL_PSM_INFO("Start PSM Server.\n");
}

void FinalizePsmServer() NN_NOEXCEPT
{
    g_ServerManager.RequestStop();
    ::nn::os::WaitThread(&g_ServerThread);
    ::nn::os::DestroyThread(&g_ServerThread);

    ::nn::psm::driver::Finalize();
}

}} // namespace nn::ptm
