﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/vi/profiler/vi_Profiler.h>
NN_PRAGMA_POP_WARNINGS

#include "profiler_Defines.h"
#include "profiler_Vsync.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_SamplingThread.h"
#include "profiler_TargetApplication.h"
#include "profiler_ThreadPriorities.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    nn::os::ThreadType thread;

    const size_t VsyncThreadStackSize = 8 * 1024;
    uint8_t* stackMemory;

    void VsyncThread_ThreadFunc(void *arg)
    {
        NN_UNUSED(arg);
        nn::os::SystemEventType vsyncEvent;
        nn::Result result;

        result = nn::vi::profiler::GetVsyncEvent(&vsyncEvent);

        if (result.IsFailure())
        {
            ERROR_LOG("Error registering vsync event!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            return;
        }

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitSystemEvent(&vsyncEvent);
            nn::os::ClearSystemEvent(&vsyncEvent);
            RecordVsync();
        }

        nn::os::DestroySystemEvent(&vsyncEvent);
    }
}

void InitializeVsyncSupport()
{
    nn::Result result;

    result = nn::vi::profiler::Initialize();
    if (result.IsFailure())
    {
        ERROR_LOG("Could not initialize profiler vsync hook!\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        return;
    }

    stackMemory = reinterpret_cast<uint8_t*>(
        Memory::GetInstance()->Allocate(VsyncThreadStackSize, nn::os::GuardedStackAlignment));

    result = nn::os::CreateThread(
        &thread,
        &VsyncThread_ThreadFunc,
        nullptr,
        stackMemory,
        VsyncThreadStackSize,
        ThreadPriority_Vsync,
        ProfilerPrimaryCore);

    if (result.IsFailure())
    {
        ERROR_LOG("Error starting vsync threads!\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT();
    }

    nn::os::SetThreadName(&thread, "[profiler] Vsync");
    nn::os::StartThread(&thread);
}

void ChangeVsyncThreadAffinity(int core)
{
    NN_SDK_ASSERT(core >= 0 && core < static_cast<int>(SupportedCoreCount));
    nn::os::SetThreadCoreMask(&thread, core, (1 << core));
}

} // profiler
} // nn
