﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <type_traits>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
NN_PRAGMA_POP_WARNINGS

#include <nn/profiler/profiler_Api.h>
#include <nn/profiler/profiler_Control.private.h>
#include <nn/profiler/profiler_Result.h>

#include "pccom/profiler_Pccom.h"
#include "profiler_Comms.h"
#include "profiler_CommsIpc.h"
#include "profiler_CommMessages.h"
#include "profiler_Core.h"
#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_IProfiler.sfdl.h"
#include "profiler_IpcEvent.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Main.h"
#include "profiler_Memory.h"
#include "profiler_ResultPrivate.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_ThreadPriorities.h"
#include "profiler_Workarea.h"

namespace nn { namespace profiler {

    const char ProfilerServiceName[] = "banana";
    nn::os::SystemEventType g_ProfilerProcessEvent;

    volatile ProfilerServerStatus g_ProfilerStatus = ProfilerServerStatus_Uninitialized;

    ProfilerServerStatus GetProfilerServerStatus()
    {
        return g_ProfilerStatus;
    }


    class ProfilerImpl
    {
    private:
        nn::os::SharedMemoryType m_sharedMem[SampleBufferIndex_MAX];

    private:
        void CloseBuffer(SampleBufferIndex sbi, size_t filledSize)
        {
            if (m_sharedMem[sbi]._state != nn::os::SharedMemoryType::State_NotInitialized)
            {
                nn::os::DestroySharedMemory(&m_sharedMem[sbi]);
            }

            auto currentBuffer = nn::profiler::SampleBuffers::GetInstance()->GetBuffer(sbi);
            if (currentBuffer->listCount > 0)
            {
                auto listItem = currentBuffer->list[currentBuffer->listCount - 1];
                uint8_t* writePtr = static_cast<uint8_t*>(listItem.memory) + filledSize;
                nn::profiler::SampleBuffers::GetInstance()->CloseCurrent(writePtr, sbi);
            }
        }

    public:
        ProfilerImpl() NN_NOEXCEPT
        {
            g_ProfilerStatus = ProfilerServerStatus_Uninitialized;
        }

        ~ProfilerImpl() NN_NOEXCEPT
        {
            if (g_ProfilerStatus != ProfilerServerStatus_Uninitialized)
            {
                LibraryFinalize(TargetApplication::GetCurrent()->GetLibraryProcessId());
            }
        }

        nn::Result GetSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOut) NN_NOEXCEPT
        {
            if (g_ProfilerStatus == ProfilerServerStatus_Uninitialized)
            {
                return nn::profiler::ResultInvalidProfilerStatus();
            }
            *pOut = sf::NativeHandle(nn::os::GetReadableHandleOfSystemEvent(&g_ProfilerProcessEvent), false);
            return nn::ResultSuccess();
        }

        nn::Result StartSignalingEvent(::std::int64_t ns) NN_NOEXCEPT
        {
            NN_UNUSED(ns);
            return nn::profiler::ResultIpcDecprecated();
        }

        nn::Result StopSignalingEvent() NN_NOEXCEPT
        {
            return nn::profiler::ResultIpcDecprecated();
        }

        nn::Result LibraryInitialize(
            uint32_t sdkVersion,
            uint64_t processId) NN_NOEXCEPT
        {
            nn::Result result;

            g_ProfilerStatus = ProfilerServerStatus_Initialized;

            result = TargetApplication::RegisterLibrary(processId);
            if (result.IsFailure())
            {
                DumpResultInformation(LOG_AS_WARNING, result);
            }

            TargetApplication::GetCurrent()->SetSdkVersion(sdkVersion);

            auto eventQueue = GetIpcEventQueue();
            eventQueue->Clear();

            return nn::ResultSuccess();
        }

        nn::Result LibraryFinalize(uint64_t processId) NN_NOEXCEPT
        {
            nn::Result result;

            nn::os::SignalEvent(GetStopEvent());

            CommReleaseTransferMemory();

            result = TargetApplication::UnregisterLibrary(processId);
            if (result.IsFailure())
            {
                DumpResultInformation(LOG_AS_WARNING, result);
            }

            g_ProfilerStatus = ProfilerServerStatus_Uninitialized;
            return nn::ResultSuccess();
        }

        nn::Result GetProfilerStatus(nn::sf::Out<int32_t> pOutStatus) NN_NOEXCEPT
        {
            *pOutStatus = static_cast<int32_t>(::nn::profiler::GetProfilerStatus());
            return nn::ResultSuccess();
        }

        nn::Result GetIpcEvent(nn::sf::OutBuffer buffer) NN_NOEXCEPT
        {
            auto eventQueue = GetIpcEventQueue();
            const IpcEventInfo* info = eventQueue->GetFront();
            if (info == nullptr)
            {
                return nn::profiler::ResultNoEvent();
            }

            NN_SDK_ASSERT(sizeof(*info) <= buffer.GetSize());
            memcpy(buffer.GetPointerUnsafe(), info, sizeof(*info));
            eventQueue->Pop();
            return nn::ResultSuccess();
        }

        nn::Result CloseAndGetNextBuffer(nn::sf::Out<nn::sf::NativeHandle> pOut, uint64_t filledSize, uint32_t index) NN_NOEXCEPT
        {
            NN_UNUSED(pOut);
            NN_UNUSED(filledSize);
            NN_UNUSED(index);
            return nn::os::ResultNotImplemented();
            //if (index >= SampleBufferIndex_MAX)
            //{
            //    return nn::profiler::ResultInvalidArgument();
            //}

            //SampleBufferIndex sbi = SampleBufferIndex(index);
            //CloseBuffer(sbi, filledSize);

            //if (::nn::profiler::GetProfilerStatus() != ::nn::profiler::ProfilerStatus_Profiling)
            //{
            //    return nn::profiler::ResultInvalidProfilerStatus();
            //}

            //bool success = nn::profiler::ObtainAndSetupSampleBuffer(sbi);
            //if (!success)
            //{
            //    return nn::profiler::ResultMemoryAllocationFailure();
            //}

            //auto currentBuffer = nn::profiler::SampleBuffers::GetInstance()->GetBuffer(sbi);
            //{
            //    auto listItem = currentBuffer->list[currentBuffer->listCount - 1];
            //    uint8_t* writePtr = static_cast<uint8_t*>(listItem.memory);
            //    nn::Result result = nn::os::CreateSharedMemory(
            //        &m_sharedMem[sbi],
            //        SampleMemoryBlockSize,
            //        nn::os::MemoryPermission_ReadWrite,
            //        nn::os::MemoryPermission_ReadWrite);
            //    if (result.IsFailure())
            //    {
            //        return result;
            //    }

            //    nn::os::MapSharedMemory(&m_sharedMem[sbi])
            //    auto handle = nn::os::DetachTransferMemory(&m_sharedMem[sbi]);
            //    *pOut = nn::sf::NativeHandle(handle, false);
            //}

            //return nn::ResultSuccess();
        }

        nn::Result CloseAndFinalizeBuffer(uint64_t filledSize, uint32_t index) NN_NOEXCEPT
        {
            if (index >= SampleBufferIndex_MAX)
            {
                return nn::profiler::ResultInvalidArgument();
            }

            SampleBufferIndex sbi = SampleBufferIndex(index);
            CloseBuffer(sbi, filledSize);
            nn::profiler::SampleBuffers::GetInstance()->FinalizeBuffer(sbi);

            return nn::ResultSuccess();
        }

        nn::Result SendMessageToPcInBuffer(uint32_t id, nn::sf::InBuffer buffer, bool callback) NN_NOEXCEPT
        {
            return CommSendMessage(id, buffer.GetPointerUnsafe(), buffer.GetSize(), 0, callback);
        }

        nn::Result SendMessageToPcTransferMem(uint32_t id, uint64_t address, uint64_t size, bool callback) NN_NOEXCEPT
        {
            return CommSendMessage(id, nullptr, size, address, callback);
        }

        nn::Result StartMultipartMessageToPc(
            uint32_t id,
            uint64_t totalSize,
            uint32_t pieces,
            bool callback) NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(sizeof(uint64_t) >= sizeof(size_t));
            return CommStartMultipart(id, totalSize, pieces, callback);
        }

        nn::Result SendMultipartMessageToPcInBuffer(nn::sf::InBuffer buffer, uint32_t remaining) NN_NOEXCEPT
        {
            return CommSendMultipart(buffer.GetPointerUnsafe(), buffer.GetSize(), 0, remaining);
        }

        nn::Result SendMultipartMessageToPcTransferMem(uint64_t address, uint64_t size, uint32_t remaining) NN_NOEXCEPT
        {
            return CommSendMultipart(nullptr, size, address, remaining);
        }

        nn::Result GetProfileSettings(nn::sf::OutBuffer buffer) NN_NOEXCEPT
        {
            auto settings = GetProfilerSettingsPointer();
            NN_SDK_ASSERT(sizeof(*settings) == buffer.GetSize());
            if (sizeof(*settings) != buffer.GetSize())
            {
                return nn::profiler::ResultInvalidArgument();
            }
            memcpy(buffer.GetPointerUnsafe(), settings, buffer.GetSize());
            return nn::ResultSuccess();
        }

        nn::Result SetProfileSettings(
            uint32_t affinityMask,
            uint32_t flags,
            uint32_t performanceCounterGroup,
            uint32_t sampleRate) NN_NOEXCEPT
        {
            return nn::profiler::SetProfileSettings(
                affinityMask,
                flags,
                static_cast<nn::profiler::PerformanceCounterGroup>(performanceCounterGroup),
                static_cast<nn::profiler::SampleRate>(sampleRate));
        }

        nn::Result TransferSampleBuffers(nn::sf::NativeHandle handle, uint64_t originalAddress, uint64_t size) NN_NOEXCEPT
        {
            auto result = nn::profiler::CommSetTransferMemory(handle.GetOsHandle(), originalAddress, size, handle.IsManaged());
            handle.Detach();
            return result;
        }

        nn::Result SendSampleBufferAsMultipart(uint32_t blockId, uint64_t size, uint32_t remaining) NN_NOEXCEPT
        {
            return nn::profiler::CommSendSampleBufferAsMultipart(blockId, size, remaining);
        }

        nn::Result IsPcSynced(nn::sf::Out<bool> pOutIsSynced) NN_NOEXCEPT
        {
            pOutIsSynced.Set(nn::profiler::IsPCConnected());
            return nn::ResultSuccess();
        }

        nn::Result SendBuffer(nn::sf::InBuffer buffer, uint32_t type) NN_NOEXCEPT
        {
            switch (SendBufferType(type))
            {
            case SendBufferType_MarkerNames:
                {
                    auto markerNames = GetMarkerNamesStringTable();
                    markerNames->OverwriteWithCopy(buffer.GetPointerUnsafe(), buffer.GetSize());
                }
                return nn::ResultSuccess();

            case SendBufferType_NvnTable:
                return CommCreateNvnFunctionTable(buffer.GetPointerUnsafe(), buffer.GetSize());

            default:
                return nn::profiler::ResultInvalidArgument();
            }
        }

        nn::Result SendIpcMessage(uint32_t id, uint64_t data) NN_NOEXCEPT
        {
            NN_UNUSED(data);

            DUMP_CURRENT_LINE();

            switch (ProfilerIpcMessage(id))
            {
            case ProfilerIpcMessage_AllDataSent:
                SignalAllDataReceived();
                return nn::ResultSuccess();

            case ProfilerIpcMessage_CoreMask:
                TargetApplication::GetCurrent()->SetCoreMask(static_cast<uint32_t>(data));
                SendCoreAvailable(TargetApplication::GetCurrent()->GetCoreMask());
                return nn::ResultSuccess();

            case ProfilerIpcMessage_StartProfiling:
                return StartProfiling();

            case ProfilerIpcMessage_StopProfiling:
                return StopProfiling();

            case ProfilerIpcMessage_ForceAttachToInProcess:
                {
                    bool success = DynamicMemorySetup(false);
                    if (!success) { return nn::profiler::ResultMemoryAllocationFailure(); }
                    auto app = TargetApplication::GetCurrent();
                    if (app == nullptr) { return nn::profiler::ResultNullArgument(); }
                    if (!app->IsLibraryInitialized()) { return nn::profiler::ResultNotInitialized(); }
                    if (app->IsAttached()) { return nn::ResultSuccess(); }
                    return TargetApplication::Attach(app->GetLibraryProcessId());
                }

            default:
                return nn::profiler::ResultInvalidArgument();
            }
        }
    };


    namespace {

        class ProfilerManager : public nn::sf::HipcSimpleAllInOneServerManager<1, 1>
        {
        protected:
            Result OnNeedsToAccept(int acceptIndex, AcceptTarget *pAcceptTarget) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_SDK_ASSERT(acceptIndex == 0);
                NN_UNUSED(acceptIndex); // for Release builds
                typedef nn::sf::ObjectFactory<nn::sf::impl::StaticOneAllocationPolicy>  Factory;
                return AcceptImpl(pAcceptTarget, Factory::CreateSharedEmplaced<IProfiler, ProfilerImpl>());
            }
        };

        std::aligned_storage<sizeof(ProfilerManager), NN_ALIGNOF(ProfilerManager)>::type g_ProfilerManagerStorage;
        ProfilerManager* g_pProfilerManager;

    } // namespace


    // Initialize the server
    void InitializeProfilerServer() NN_NOEXCEPT
    {
        nn::Result result;

        NN_SDK_ASSERT(!g_pProfilerManager);

        result = nn::os::CreateSystemEvent(&g_ProfilerProcessEvent, nn::os::EventClearMode_ManualClear, true);
        if (result.IsFailure())
        {
            DumpResultInformation(LOG_AS_ERROR, result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        g_pProfilerManager = new (&g_ProfilerManagerStorage) ProfilerManager;

        // CreateProfilerByDfc() で得られるオブジェクトを、サービス名で関連付けて、システムに登録
        auto sessionCountMax = 1;
        result = g_pProfilerManager->InitializePort(0, sessionCountMax, ProfilerServiceName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // Start server manager
        g_pProfilerManager->Start();

    }

    void LoopProfilerServer() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_pProfilerManager);
        g_pProfilerManager->LoopAuto();
    }

    void SignalIpcSystemEvent() NN_NOEXCEPT
    {
        if (g_ProfilerStatus != ProfilerServerStatus_Uninitialized)
        {
            nn::os::SignalSystemEvent(&g_ProfilerProcessEvent);
        }
    }
}}

namespace {
    const size_t MemoryBufferSize = 128 * 1024;
    NN_ALIGNAS(4096) nn::Bit8 g_MemoryBuffer[MemoryBufferSize];
}

extern "C" void nninitStartup()
{
}
extern "C" void nndiagStartup()
{
}
extern "C" void nnMain()
{
    // Set up the minimal heap needed
    {
        nn::Bit8* heapMem = g_MemoryBuffer;
        size_t heapSize = MemoryBufferSize;
        bool success = nn::profiler::Memory::GetInstance()->Initialize(heapMem, heapSize);
        NN_ABORT_UNLESS(success, "[profiler] Failed to initialize memory system\n");
    }

    nn::profiler::DynamicMemorySetup(true);

    nn::os::SetThreadName(nn::os::GetCurrentThread(), "[profiler] Looper");
    DumpThreadInformation();

    nn::profiler::GetIpcEventQueue()->Initialize();

    nn::profiler::StartProfilerCore();

    DEBUG_LOG("Profiler process has started!\n");

    nn::profiler::InitializeProfilerServer();
    nn::profiler::LoopProfilerServer();
}
