﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>

#include "profiler_CommsIpc.h"
#include "profiler_IpcEventQueue.h"

namespace nn { namespace profiler {

    // Constructor sets default values.
    IpcEventQueue::IpcEventQueue()
        : m_current(0)
        , m_elements(0)
    {
        memset(&m_mutex, 0, sizeof(m_mutex));
    }



    void IpcEventQueue::Initialize()
    {
        if (m_mutex._state == nn::os::MutexType::State_NotInitialized)
        {
            nn::os::InitializeMutex(&m_mutex, false, 0);
        }
    }



    void IpcEventQueue::Finalize()
    {
        if (m_mutex._state == nn::os::MutexType::State_Initialized)
        {
            ////////////////////////////////////////////
            // Critical Section
            nn::os::LockMutex(&m_mutex);

            // "Clear" the queue
            m_current = 0;
            m_elements = 0;

            nn::os::UnlockMutex(&m_mutex);
            //
            ////////////////////////////////////////////

            nn::os::FinalizeMutex(&m_mutex);
            memset(&m_mutex, 0, sizeof(m_mutex));
        }
    }



    IpcEventInfo const * IpcEventQueue::GetFront()
    {
        IpcEventInfo *front = nullptr;

        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);
        if (m_elements > 0)
        {
            front = &m_queue[m_current];
        }
        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////

        return front;
    }



    // Adds a task to the queue.
    bool IpcEventQueue::Push(IpcEventInfo* task)
    {
        bool pushed = false;

        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);

        // Make sure we aren't full.
        if (m_elements < QueueSize)
        {
            uint32_t next = m_current + m_elements;
            if (next >= QueueSize)
                next -= QueueSize;

            // Copy the new task.
            m_queue[next] = *task;

            ++m_elements;

            pushed = true;
        }

        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////

        if (pushed)
        {
            SignalIpcSystemEvent();
        }

        return pushed;
    }



    // Removes a task from the queue.
    void IpcEventQueue::Pop()
    {
        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);

        // Make sure we aren't empty.
        if (m_elements > 0)
        {
            uint32_t next = m_current + 1;
            if (next >= QueueSize)
                next = 0;

            m_current = next;

            --m_elements;
        }

        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////
    }



    void IpcEventQueue::Clear()
    {
        ////////////////////////////////////////////
        // Critical Section
        nn::os::LockMutex(&m_mutex);

        m_current = 0;
        m_elements = 0;

        nn::os::UnlockMutex(&m_mutex);
        //
        ////////////////////////////////////////////
    }



    uint32_t IpcEventQueue::CurrentCount()
    {
        return m_elements;
    }

}} // nn::profiler
