﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Types.h>

#include "profiler_CommMessages.h"
#include "profiler_Comms.h"
#include "profiler_CommsIpc.h"
#include "profiler_Defines.h"
#include "profiler_ExceptionHandling.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_RecordMethods.h"
#include "profiler_ResultPrivate.h"
#include "profiler_TargetApplication.h"
#include "profiler_Workarea.h"
#include "profiler_WriteToBuffer.h"

namespace nn { namespace profiler {

namespace /*anonymous*/ {

    const char* gExceptionNames[] NN_IS_UNUSED_MEMBER = {
        "Undefined Instruction",
        "Access Violation: Instruction",
        "Access Violation: Data",
        "Data Type Misaligned",
        "Attach Break",
        "Breakpoint",
        "User Break",
        "Debugger Break",
        "Undefined System Call",
        "Memory System Error",
    };

    const char* gUserBreakReasons[] NN_IS_UNUSED_MEMBER = {
        "Panic",
        "Assert",
        "User",
        "Pre-Load DLL",
        "Post-Load DLL",
        "Pre-Unload DLL",
        "Post-Unload DLL",
        "C++ Exception",
    };

    struct ProcState
    {
        union
        {
            struct
            {
                uint32_t N   : 1;
                uint32_t Z   : 1;
                uint32_t C   : 1;
                uint32_t V   : 1;
                uint32_t D   : 1;
                uint32_t A   : 1;
                uint32_t I   : 1;
                uint32_t F   : 1;
                uint32_t SS  : 1;
                uint32_t IL  : 1;
                uint32_t EL  : 2;
                uint32_t nRW : 1;
                uint32_t SP  : 1;
                uint32_t Q   : 1;
                uint32_t GE  : 4;
                uint32_t IT  : 8;
                uint32_t J   : 1;
                uint32_t T   : 1;
                uint32_t E   : 1;
                uint32_t M   : 5;
            };
            uint32_t pstate;
        };
    };

    enum CrashReportSection : uint32_t
    {
        CrashReportSection_ExceptionStruct      = 0xFFFFFF01,
        CrashReportSection_ThreadDetails        = 0xFFFFFF02,
        CrashReportSection_Stack                = 0xFFFFFF03,
        CrashReportSection_End                  = 0xFFFFFFFF,
    };

    void SendReport(nn::svc::DebugInfoException* exception, nn::os::ThreadId threadId, nn::svc::ThreadContext* context);

    typedef nn::Result (*ExceptionHandler)(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);

    nn::Result UndefinedInstruction(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result AccessViolationInstruction(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result AccessViolationData(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result DataTypeMisaligned(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result AttachBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result Breakpoint(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result UserBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result DebuggerBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result UndefinedSystemCall(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);
    nn::Result MemorySystemError(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context);

    ExceptionHandler gExceptions[] = {
        UndefinedInstruction,
        AccessViolationInstruction,
        AccessViolationData,
        DataTypeMisaligned,
        AttachBreak,
        Breakpoint,
        UserBreak,
        DebuggerBreak,
        UndefinedSystemCall,
        MemorySystemError,
    };


#if 0
    inline void AdvanceProgramCounter(nn::svc::ThreadContext context)
    {
        context.pc += ((context.pstate & 0x20) ? 2 : 4);
    }
#endif


    void SendReport(
        nn::svc::DebugInfoException* exception,
        nn::os::ThreadId threadId,
        nn::svc::ThreadContext* context)
    {
        WorkArea work;
        memset(&work, 0, sizeof(WorkArea));

        const size_t allocationSize = sizeof(*exception) + sizeof(*context) + SentinelSize * 2;
        uint8_t* temp = static_cast<uint8_t*>(Memory::GetInstance()->Allocate(allocationSize, 32));

        work.startPtr = temp;
        work.curPtr = work.startPtr;
        work.sentinel = work.startPtr + (allocationSize - SentinelSize);
        work.context.fp = context->fp;
        work.context.sp = context->sp;
        work.context.pc = context->pc;
        work.context.lr = context->lr;
        work.thread_to_profile = threadId;
        work.fields.SetBit(WorkArea::Is64Bit, TargetApplication::GetCurrent()->Is64Bit());
        work.tempStack = reinterpret_cast<uintptr_t>(Memory::GetInstance()->Allocate(ReadStackStorageSize));

        // Fill in some crash report details
        temp = WriteToBuffer(temp, static_cast<uint32_t>(CrashReportSection_ExceptionStruct));
        temp = WriteToBuffer(temp, static_cast<uint32_t>(sizeof(*exception)));
        memcpy(temp, exception, sizeof(*exception));
        temp += sizeof(*exception);

        auto threadList = TargetApplication::GetCurrent()->GetThreadList();
        auto threadDetails = threadList->GetByKey(threadId);
        UpdateSingleThreadInfo(threadDetails);

        const size_t threadDetailsSize =
            sizeof(*context) +
            sizeof(uint64_t) * 2 +
            sizeof(uint32_t) * 3 +
            nn::os::ThreadNameLengthMax;

        temp = WriteToBuffer(temp, static_cast<uint32_t>(CrashReportSection_ThreadDetails));
        temp = WriteToBuffer(temp, static_cast<uint32_t>(threadDetailsSize));
        temp = WriteToBuffer(temp, static_cast<uint64_t>(threadId));
        temp = WriteToBuffer(temp, static_cast<uint64_t>(threadDetails->stackBase));
        temp = WriteToBuffer(temp, static_cast<uint32_t>(threadDetails->priority));
        temp = WriteToBuffer(temp, static_cast<uint32_t>(threadDetails->coreMask));
        temp = WriteToBuffer(temp, static_cast<uint32_t>(threadDetails->idealCore));
        memcpy(temp, threadDetails->threadName, nn::os::ThreadNameLengthMax);
        temp += nn::os::ThreadNameLengthMax;
        memcpy(temp, context, sizeof(*context));
        temp += sizeof(*context);

        temp = WriteToBuffer(temp, static_cast<uint32_t>(CrashReportSection_Stack));

        bool is64bit = TargetApplication::GetCurrent()->Is64Bit();
        temp = WriteToBuffer(temp, static_cast<uint32_t>(is64bit));
        if (is64bit)
        {
            temp = WriteToBuffer(temp, static_cast<uint64_t>(context->pc));
        }
        else
        {
            temp = WriteToBuffer(temp, static_cast<uint32_t>(context->pc));
        }
        work.curPtr = temp;

        // Send the stack
        RecordStack(&work);

        temp = work.curPtr;
        temp = WriteToBuffer(temp, static_cast<uint32_t>(CrashReportSection_End));
        work.curPtr = temp;

        NN_SDK_ASSERT(work.curPtr >= work.startPtr);
        size_t usedSize = static_cast<size_t>(work.curPtr - work.startPtr);
        NN_SDK_ASSERT(usedSize <= allocationSize);

        nn::Result result = CommSendMessage(ProfilerCommMessage_CrashReport, work.startPtr, usedSize, 0, false);
        if (result.IsFailure())
        {
            DumpResultInformation(LOG_AS_ERROR, result);
        }

        Memory::GetInstance()->Free(reinterpret_cast<void*>(work.tempStack));
    }


    nn::Result UndefinedInstruction(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        ERROR_LOG("Instruction: 0x%08x\n", exception->detail.undefinedInstruction.code);

        SendReport(exception, eventInfo->threadId, context);

        return nn::profiler::ResultKillApplication();
    }


    nn::Result AccessViolationInstruction(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        return AccessViolationData(eventInfo, context);
    }


    nn::Result AccessViolationData(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        ERROR_LOG("Fault Address: 0x%p\n", exception->detail.accessViolationData.faultAddress);

        SendReport(exception, eventInfo->threadId, context);

        return nn::profiler::ResultKillApplication();
    }


    nn::Result DataTypeMisaligned(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        ERROR_LOG("Fault Address: %p\n", exception->detail.dataTypeMissaligned.faultAddress);

        SendReport(exception, eventInfo->threadId, context);

        return nn::profiler::ResultKillApplication();
    }


    nn::Result AttachBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        // Do nothing
        NN_UNUSED(eventInfo);
        NN_UNUSED(context);
        return nn::profiler::ResultContinueFromException();
    }


    nn::Result Breakpoint(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        NN_UNUSED(context);

        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        ERROR_LOG("Type: %d\n", exception->detail.breakPoint.type);
        ERROR_LOG("Address: 0x%p\n", exception->detail.breakPoint.dataAddress);

        return nn::profiler::ResultContinueFromException();
    }


    nn::Result UserBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        nn::svc::BreakReason reason = exception->detail.userBreak.reason;
        nn::svc::BreakReason noNotification =
            nn::svc::BreakReason(reason & ~nn::svc::BreakReason_NotificationOnlyFlag);

        if (noNotification == nn::svc::BreakReason_PreLoadDll ||
            noNotification == nn::svc::BreakReason_PreUnloadDll)
        {
            ERROR_LOG("Reason: %s(%d)\n", gUserBreakReasons[noNotification], exception->detail.userBreak.reason);
            ERROR_LOG("Data: 0x%p\n", exception->detail.userBreak.data);
            ERROR_LOG("Size: %lld\n", exception->detail.userBreak.size);
            return nn::profiler::ResultContinueFromException();
        }
        else if (noNotification == nn::svc::BreakReason_PostLoadDll)
        {
            ERROR_LOG("Reason: %s(%d)\n", gUserBreakReasons[noNotification], exception->detail.userBreak.reason);
            ERROR_LOG("Data: 0x%p\n", exception->detail.userBreak.data);
            ERROR_LOG("Size: %lld\n", exception->detail.userBreak.size);
            return nn::profiler::ResultContinueFromException();
        }
        else if (noNotification == nn::svc::BreakReason_PostUnloadDll)
        {
            ERROR_LOG("Reason: %s(%d)\n", gUserBreakReasons[noNotification], exception->detail.userBreak.reason);
            ERROR_LOG("Data: 0x%p\n", exception->detail.userBreak.data);
            ERROR_LOG("Size: %lld\n", exception->detail.userBreak.size);
            return nn::profiler::ResultContinueFromException();
        }
        else
        {
            ERROR_LOG("Reason: %s(%d)\n", gUserBreakReasons[noNotification], exception->detail.userBreak.reason);
            ERROR_LOG("Data: 0x%p\n", exception->detail.userBreak.data);
            ERROR_LOG("Size: %lld\n", exception->detail.userBreak.size);

            if (reason == noNotification)
            {
                SendReport(exception, eventInfo->threadId, context);

                return nn::profiler::ResultKillApplication();
            }
            else
            {
                return nn::profiler::ResultContinueFromException();
            }
        }
    }


    nn::Result DebuggerBreak(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        // Do nothing
        NN_UNUSED(eventInfo);
        NN_UNUSED(context);
        return nn::profiler::ResultContinueFromException();
    }


    nn::Result UndefinedSystemCall(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        ERROR_LOG("SVC ID: %d\n", exception->detail.undefinedSystemCall.svcId);

        SendReport(exception, eventInfo->threadId, context);

        return nn::profiler::ResultKillApplication();
    }


    nn::Result MemorySystemError(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
    {
        nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

        SendReport(exception, eventInfo->threadId, context);

        return nn::profiler::ResultKillApplication();
    }

} // <anonymous>


nn::Result HandleException(nn::svc::DebugEventInfo* eventInfo, nn::svc::ThreadContext* context)
{
    nn::Result result;

    nn::svc::DebugInfoException* exception __attribute__((unused)) = &eventInfo->info.exception;

    auto threadList NN_IS_UNUSED_MEMBER = TargetApplication::GetCurrent()->GetThreadList();
    auto threadInfo NN_IS_UNUSED_MEMBER = threadList->GetByKey(eventInfo->threadId);

    ERROR_LOG("Exception\n");
    ERROR_LOG("--\n");
    ERROR_LOG("Type: %s (%d)\n", gExceptionNames[exception->exceptionCode], exception->exceptionCode);
    ERROR_LOG("Address: 0x%p\n", exception->exceptionAddress);
    ERROR_LOG("Thread: %lld (%s)\n", eventInfo->threadId, threadInfo->threadName);

    result = gExceptions[eventInfo->info.exception.exceptionCode](eventInfo, context);

    ERROR_LOG("--\n");
    ERROR_LOG("Exception handled\n");

    return result;
}


void DumpContext(nn::svc::ThreadContext* context)
{
    for (uint32_t i = 0; i < (sizeof(context->r) / sizeof(context->r[0])); ++i)
    {
        ERROR_LOG("r[%02d]: 0x%016lx\n", i, context->r[i]);
    }
    ERROR_LOG("sp: %p\n", context->sp);
    ERROR_LOG("lr: %p\n", context->lr);
    ERROR_LOG("pc: %p\n", context->pc);
    ERROR_LOG("fpsr: %08x\n", context->fpsr);
    ERROR_LOG("fpcr: %08x\n", context->fpcr);
    ERROR_LOG("pstate: %08x\n", context->pstate);

    ProcState p;
    p.pstate = context->pstate;
    const char* mode = nullptr;
    switch (p.M)
    {
    case 0x10: mode = ("_usr"); break;
    case 0x11: mode = ("_fiq"); break;
    case 0x12: mode = ("_irq"); break;
    case 0x13: mode = ("_svc"); break;
    case 0x17: mode = ("_abt"); break;
    case 0x1b: mode = ("_und"); break;
    case 0x1f: mode = ("_sys"); break;
    default: mode = ("_unknown"); break;
    }

    //         N Z C V D A I F SSILELRWSPQ GEITJ T E M
    ERROR_LOG("%c%c%c%c%c%c%c%c%c%c%d%d%c%c%d%d%c%c%c%s\n",
        ((p.N)  ? 'N' : '-'),
        ((p.Z)  ? 'Z' : '-'),
        ((p.C)  ? 'C' : '-'),
        ((p.V)  ? 'V' : '-'),
        ((p.D)  ? 'D' : '-'),
        ((p.A)  ? 'A' : '-'),
        ((p.I)  ? 'I' : '-'),
        ((p.F)  ? 'F' : '-'),
        ((p.SS) ? 'S' : '-'),
        ((p.IL) ? 'i' : '-'),
        p.EL,
        p.nRW,
        p.SP,
        ((p.Q) ? 'Q' : '-'),
        p.GE,
        p.IT,
        ((p.J) ? 'J' : '-'),
        ((p.T) ? 'T' : '-'),
        ((p.E) ? 'E' : '-'),
        mode);
}


}} // nn::profiler
