﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdio>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/util/util_StringUtil.h>
NN_PRAGMA_POP_WARNINGS

#include <nn/profiler/profiler_Api.h>

#include "pccom/profiler_Pccom.h"
#include "profiler_Comms.h"
#include "profiler_CommsIpc.h"
#include "profiler_CommMessages.h"
#include "profiler_Core.h"
#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_ForceSend.h"
#include "profiler_GnuBuildId.h"
#include "profiler_HeaderWriter.h"
#include "profiler_LibModule.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_Messages.h"
#include "profiler_RecordMethods.h"
#include "profiler_ResultPrivate.h"
#include "profiler_SamplingThread.h"
#include "profiler_Scenarios.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_Workarea.h"
#include "profiler_WriteToBuffer.h"



namespace nn { namespace profiler {


//-------------------------------------------------------------------
// Globals
//-------------------------------------------------------------------

const uint16_t ThreadListVersion = 2;
const uint32_t InstrumentedBufferVersion = 1;

// Size of buffer to store strings that are sent in messages
const size_t TEMP_BUFFER_SIZE = 1024;

const size_t SIZE_OF_COMM_UNIT_HEAP = (TEMP_BUFFER_SIZE * 8) + sizeof(::nn::lmem::HeapCommonHead) + 256;

namespace // anonymous
{
    // The following value is not const because PCCOM does not support the
    //  sending of a uint32_t independently
    uint32_t ProfilerVersion = ProfilerRuntimeVersion;

    // Event signaled to being profiling
    nn::os::EventType s_beginEvent;

    // Event signaling that a transfer has completed
    nn::os::EventType s_xferCompleteEvent;

    // Event signaled to stop profiling
    nn::os::EventType s_stopEvent;

    nn::os::LightEventType s_waitForData;

    nn::profiler::pccom::TwoWayCommunication* s_commLayer;
    IpcEventQueue s_ipcEventQueue;

    // Pointer to where received settings should be stored
    SettingsFromThePcGui* gSettings = nullptr;

    void* s_commBufferMemory = nullptr;
    nn::lmem::HeapHandle s_commBuffers;

    nn::os::TransferMemoryType s_transferMemory;
    void* s_transferMappedMemory;
    uintptr_t s_transferOriginalAddress;

    bool firstReadySinceConnect = false;
    bool pcSynced = false;


    struct CommThreadList
    {
        uint32_t numThreads;
        uint32_t padding; // since 64-bit values are 8-byte aligned
        nn::os::ThreadId threadIDs[ThreadIdListSize];
        uint32_t coreIDs[ThreadIdListSize];
    };
    CommThreadList* threadInfo;


    char* s_threadNameSection = nullptr;
    const int ThreadNameSectionSize = 16 + ThreadIdListSize * (nn::os::ThreadNameLengthMax + sizeof(nn::os::ThreadId) + sizeof(int) + sizeof(int));



    char* s_channelInfo = nullptr;
    char* s_nvnPointerTable = nullptr;
    size_t s_nvnPointerTableSize = 0;


    struct ModuleData
    {
        uint64_t baseAddress;
        uint64_t size;
        uint64_t loadTime;
        uint64_t unloadTime;
        char name[MaximumFilePathLength];
        GnuBuildId buildId;
        uint32_t padding; // GnuBuildId size is not 8-byte multiple
    };

    struct ApplicationInfo
    {
        uint32_t version;
        uint32_t coreMask;
        uint32_t moduleCount;
        uint32_t padding;
        ModuleData moduleData[];
    };
    ApplicationInfo* s_appInfo = nullptr;
    size_t s_appInfoSize;


    void FillThreadList();
    void FillThreadNameBuffer(char* buffer, size_t bufferSize);
    char* AllocateCommMsgMem();
    void MessageSentCallbackToFreeMem(uint32_t msg, nn::Result result, void* data);


    void OnMessage(void* info, size_t length, uint32_t message);
    void StateChangeHandler(pccom::PccomState state);
    void TransferRequest(
        void **destination,
        size_t size,
        uint32_t message,
        nn::profiler::pccom::RecvCompleteCallback *callback);
    void OnSendDataComplete(uint32_t message, nn::Result result);
    void ClearPccomTemporaryMemory(uint32_t message, nn::Result result);
    void OnIpcSendMessageComplete(uint32_t message, nn::Result result);

    void SendAvailableProfileModes();


    inline bool IsInTransferMemory(uintptr_t addr, size_t size)
    {
        return ((s_transferOriginalAddress != 0) &&
            (s_transferOriginalAddress <= addr) &&
            ((addr + size) < (s_transferOriginalAddress + s_transferMemory._size)));
    }



    void FillThreadList()
    {
        TargetApplication *app = TargetApplication::GetCurrent();
        ThreadList* threadList = app->GetThreadList();
        threadList->ReadLock();
        int listCount = std::min(threadList->Count(), int(ThreadIdListSize));
        threadInfo->numThreads = uint32_t(listCount);
        for (int i = 0; i < listCount; ++i)
        {
            auto item = threadList->GetByIndex(i);
            threadInfo->threadIDs[i] = item->id;
            threadInfo->coreIDs[i] = static_cast<uint32_t>(item->coreMask);
        }
        threadList->ReadUnlock();
    }



    void FillThreadNameBuffer(char* buffer, size_t bufferSize)
    {
        NN_SDK_ASSERT(bufferSize >= ThreadNameSectionSize);

        char* tnsWritePtr = buffer;
        char* endPtr = tnsWritePtr + bufferSize;

        memset(buffer, 0, bufferSize);

        // Write the header
        {
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint32_t>(HeaderSpecialValues_ThreadHeaderBegin));
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint32_t>(bufferSize - 8));
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(0xFFFF)); // version denotation
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(ThreadListVersion));
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(ThreadIdListSize));
            tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(nn::os::ThreadNameLengthMax));
        }

        // Write thread names
        DEBUG_LOG("Filling Thread Name Buffer\n");
        {
            TargetApplication *app = TargetApplication::GetCurrent();
            ThreadList* threadList = app->GetThreadList();
            threadList->ReadLock();
            int listCount = std::min(threadList->Count(), int(ThreadIdListSize));
            threadInfo->numThreads = uint32_t(listCount);
            DEBUG_LOG("  Count = %d (%d)\n", listCount, threadList->Count());
            for (int i = 0; i < listCount; ++i)
            {
                auto item = threadList->GetByIndex(i);
                DEBUG_LOG("  %d: %p\n", i, item);
                if (item != nullptr)
                {
                    DEBUG_LOG("      %lld, %d, %08x, %s\n", item->id, item->priority, item->coreMask, item->threadName);
                    tnsWritePtr = WriteToBuffer(tnsWritePtr, item->id);
                    tnsWritePtr = WriteToBuffer(tnsWritePtr, item->priority);
                    tnsWritePtr = WriteToBuffer(tnsWritePtr, item->coreMask);
                    memcpy(tnsWritePtr, item->threadName, nn::os::ThreadNameLengthMax);
                    tnsWritePtr += nn::os::ThreadNameLengthMax;
                }
                else
                {
                    const size_t writeSize = sizeof(item->id) + sizeof(item->priority) + sizeof(item->coreMask) + nn::os::ThreadNameLengthMax;
                    memset(tnsWritePtr, 0, writeSize);
                    tnsWritePtr += writeSize;
                }
            }
            threadList->ReadUnlock();
        }

        NN_SDK_ASSERT(tnsWritePtr <= endPtr);
        NN_UNUSED(endPtr);
    }



    char* AllocateCommMsgMem()
    {
        void* mem = nullptr;
        while (mem == nullptr)
        {
            mem = nn::lmem::AllocateFromUnitHeap(s_commBuffers);
            if (mem == nullptr)
            {
                nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(10));
            }
        }
        return reinterpret_cast<char*>(mem);
    }



    void MessageSentCallbackToFreeMem(
        uint32_t msg,
        nn::Result result,
        void* data)
    {
        NN_UNUSED(msg);
        NN_UNUSED(result);

        nn::lmem::FreeToUnitHeap(s_commBuffers, data);
    }



    void FillApplicationInfo()
    {
        if (s_appInfo != nullptr)
        {
            Memory::GetInstance()->Free(s_appInfo);
            s_appInfo = nullptr;
        }

        int moduleCount = gPastModules.size();
        if (moduleCount <= 0)
        {
            TargetApplication::GetCurrent()->FindCodeRegions();
        }
        moduleCount = gPastModules.size();
        NN_SDK_ASSERT(moduleCount > 0);
        if (moduleCount < 0) { moduleCount = 0; }

        s_appInfoSize = sizeof(ApplicationInfo) + (sizeof(ModuleData) * size_t(moduleCount));
        s_appInfo = reinterpret_cast<ApplicationInfo*>(Memory::GetInstance()->Allocate(s_appInfoSize, 32));

        memset(s_appInfo, 0, s_appInfoSize);

        s_appInfo->version = ProfilerApplicationInfoVersion;
        s_appInfo->coreMask = TargetApplication::GetCurrent()->GetCoreMask();
        s_appInfo->moduleCount = static_cast<uint32_t>(moduleCount);
        s_appInfo->padding = 0xaaaaaaaa;

        if (moduleCount <= 0) { return; }

        int index = 0;
        for (auto& x : gPastModules)
        {
            auto& data = s_appInfo->moduleData[index];
            data.baseAddress = x.Address;
            data.size = x.Size;
            data.loadTime = x.LoadTime;
            data.unloadTime = x.UnloadTime;
            memcpy(data.name, x.Name, MaximumFilePathLength);
            memcpy(&data.buildId, &x.BuildId, sizeof(data.buildId));
            ++index;
        }
        NN_SDK_ASSERT(index == moduleCount);
    }



    void OnMessage(
        void* info,
        size_t length,
        uint32_t message)
    {
        DEBUG_LOG("== Message ==\n");
        switch (message)
        {
        case ProfilerCommMessage_Query:
            {
                MessageQuery query = MessageQuery_Invalid;
                if (info != nullptr && length != 0)
                {
                    NN_SDK_ASSERT(length >= 4);
                    query = *static_cast<MessageQuery*>(info);
                }

                switch (query)
                {
                case MessageQuery_IsLibraryInitialized:
                    {
                        char data[sizeof(uint32_t) * 2];
                        char* pData = data;
                        bool isLibInit = TargetApplication::GetCurrent()->IsLibraryInitialized();

                        pData = WriteToBuffer(pData, static_cast<uint32_t>(MessageQuery_IsLibraryInitialized));
                        pData = WriteToBuffer(pData, static_cast<uint32_t>(isLibInit ? 1 : 0));
                        s_commLayer->Send(
                            ProfilerCommMessage_Query,
                            data,
                            sizeof(data),
                            nullptr);
                    }
                    break;

                case MessageQuery_Dump:
                    FORCE_LOG("MessageQuery_Dump\n");
                    DumpTargetApplicationInfo();
                    DumpSamplingThreadInfo();
                    break;

                case MessageQuery_SupportsSystemProcess:
                    {
                        char data[sizeof(uint32_t) * 2];
                        char* pData = data;
                        uint32_t supported = 0;
#if defined(NN_PROFILER_SUPPORT_SYSTEM_PROCESS)
                        supported = 1;
#endif
                        pData = WriteToBuffer(pData, static_cast<uint32_t>(MessageQuery_SupportsSystemProcess));
                        pData = WriteToBuffer(pData, supported);
                        s_commLayer->Send(ProfilerCommMessage_Query, data, sizeof(data), nullptr);
                    }
                    break;

                case MessageQuery_Invalid:
                default:
                    // do nothing
                    break;
                }
            }
            break;

        case ProfilerCommMessage_AvailableProfilingModes:
            SendAvailableProfileModes();
            break;

        case ProfilerCommMessage_Argv:
            {
                if (TargetApplication::GetCurrent()->IsLibraryInitialized())
                {
                    DUMP_CURRENT_LINE();

                    IpcEventInfo info;
                    info.event = IpcEvent_PcMessageForward;
                    info.info.pcMessageForward.message = message;
                    info.info.pcMessageForward.size = 0;
                    info.info.pcMessageForward.smallPayload = 0;

                    auto queue = GetIpcEventQueue();
                    queue->Push(&info);
                }
                //else if (TargetApplication::GetCurrent()->IsAttached())
                //{
                //    int index = 0;
                //    if (info != nullptr && length != 0)
                //    {
                //        index = *reinterpret_cast<int*>(info);
                //    }

                //    int argc = nn::os::GetHostArgc();
                //    char** argv = nn::os::GetHostArgv();
                //    char* msgMem = AllocateCommMsgMem();
                //    char* writePtr = msgMem;
                //    memset(msgMem, 0, TEMP_BUFFER_SIZE);
                //    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(argc));

                //    const char* appName = nullptr;
                //    size_t appNameLength;
                //    if (index == 0)
                //    {
                //        TargetApplication::GetCurrent()->GetApplicationName(appName, appNameLength);
                //    }
                //    if (index == 0 && appName != nullptr)
                //    {
                //        writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(index));
                //        size_t copyLength = std::min(appNameLength, TEMP_BUFFER_SIZE - sizeof(int32_t) * 2);
                //        memcpy(writePtr, appName, copyLength);
                //    }
                //    else if (0 <= index && index < argc)
                //    {
                //        writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(index));
                //        nn::util::Strlcpy(writePtr, argv[index], TEMP_BUFFER_SIZE - sizeof(int32_t) * 2);
                //    }
                //    else
                //    {
                //        writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(-1));
                //    }

                //    s_commLayer->Send(ProfilerCommMessage_Argv, msgMem, TEMP_BUFFER_SIZE, nullptr);
                //    MessageSentCallbackToFreeMem(ProfilerCommMessage_Argv, nn::ResultSuccess(), msgMem);
                //}
                else
                {
                    char* msgMem = AllocateCommMsgMem();
                    char* writePtr = msgMem;
                    memset(msgMem, 0, TEMP_BUFFER_SIZE);

                    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(0));
                    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(-1));

                    s_commLayer->Send(ProfilerCommMessage_Argv, msgMem, TEMP_BUFFER_SIZE, nullptr);
                    MessageSentCallbackToFreeMem(ProfilerCommMessage_Argv, nn::ResultSuccess(), msgMem);
                }
            }
            break;

        case ProfilerCommMessage_Version:
            s_commLayer->Send(
                ProfilerCommMessage_Version,
                &ProfilerVersion,
                sizeof(ProfilerVersion),
                nullptr);
            break;

        case ProfilerCommMessage_SetScenario:
            DEBUG_LOG("Setting Scenario\n");
            if (info != nullptr && length >= sizeof(int))
            {
                nn::Result result = nn::ResultSuccess();
                Scenarios scenario = Scenarios(*static_cast<int32_t*>(info));

                if (scenario == Scenarios_InProcess)
                {
                    // Nothing to do, no out-of-process profiling is supported
                    SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
                }
                else if (scenario == Scenarios_Launch)
                {
                    result = TargetApplication::Launch();
                    if (result.IsFailure())
                    {
                        WARNING_LOG("Failed to launch application.\n");
                        DumpResultInformation(LOG_AS_WARNING, result);
                    }
                }
                else if (scenario == Scenarios_AttachOnLaunch)
                {
                    nn::ncm::ProgramId programId;
                    programId.value = *reinterpret_cast<uint64_t*>(static_cast<int32_t*>(info) + 1);
                    result = TargetApplication::ScheduleAttachOnLaunch(programId);
                    if (result.IsFailure())
                    {
                        WARNING_LOG("Failed to attach on launch.\n");
                        DumpResultInformation(LOG_AS_WARNING, result);
                    }
                }
                else if (scenario == Scenarios_AttachToProgramId)
                {
                    nn::ncm::ProgramId programId;
                    programId.value = *reinterpret_cast<uint64_t*>(static_cast<int32_t*>(info) + 1);
                    result = TargetApplication::Attach(programId);
                    if (result.IsFailure())
                    {
                        WARNING_LOG("Failed to attach.\n");
                        DumpResultInformation(LOG_AS_WARNING, result);
                    }
                    SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
                }
                else if (scenario == Scenarios_AttachToProcessId)
                {
                    nn::Bit64 processId;
                    processId = *reinterpret_cast<nn::Bit64*>(static_cast<int32_t*>(info) + 1);
                    result = TargetApplication::Attach(processId);
                    if (result.IsFailure())
                    {
                        WARNING_LOG("Failed to attach.\n");
                        DumpResultInformation(LOG_AS_WARNING, result);
                    }
                    SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
                }
                else if (scenario == Scenarios_AttachToInProcess)
                {
                    auto app = TargetApplication::GetCurrent();
                    if (app->IsLibraryInitialized())
                    {
                        result = TargetApplication::Attach(app->GetLibraryProcessId());
                    }
                    else
                    {
                        result = nn::profiler::ResultInvalidProfilerStatus();
                    }

                    if (result.IsFailure())
                    {
                        WARNING_LOG("Failed to attach.\n");
                        DumpResultInformation(LOG_AS_WARNING, result);
                    }
                    SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
                }
                else
                {
                    NN_ABORT("Invalid Scenario!\n");
                }

                if (result.IsFailure())
                {
                    if (nn::profiler::ResultDebuggerAttached::Includes(result))
                    {
                        SendNotificationToPC(NotificationWarningDebuggerAttached, result);
                    }
                    else /*if (nn::profiler::ResultCouldNotAttach::Includes(result))*/
                    {
                        SendNotificationToPC(NotificationErrorCouldNotAttach, result);
                    }
                    // TODO: Send general warnings when launch/attach fails?
                }
            }
            else
            {
                WARNING_LOG("Invalid parameters set for scenario!\n");
            }
            break;

        case ProfilerCommMessage_ApplicationInfo:
            DEBUG_LOG("APPLICATION INFO\n");
            {
                if (TargetApplication::GetCurrent()->IsAttached())
                {
                    FillApplicationInfo();
                    s_commLayer->Send(
                        ProfilerCommMessage_ApplicationInfo,
                        s_appInfo,
                        s_appInfoSize,
                        nullptr);
                }
                else if (TargetApplication::GetCurrent()->IsLibraryInitialized())
                {
                    DUMP_CURRENT_LINE();

                    IpcEventInfo info;
                    info.event = IpcEvent_PcMessageForward;
                    info.info.pcMessageForward.message = message;
                    info.info.pcMessageForward.size = 0;
                    info.info.pcMessageForward.smallPayload = 0;

                    auto queue = GetIpcEventQueue();
                    queue->Push(&info);
                }
                else
                {
                    ERROR_LOG("No profiling client available!\n");
                    DumpResultInformation(LOG_AS_ERROR, nn::profiler::ResultInvalidProfilerStatus());
                    SendNotificationToPC(
                        NotificationErrorNoProfilingClient,
                        nn::profiler::ResultInvalidProfilerStatus());
                }
            }
            break;

        case ProfilerCommMessage_Start:
            if (gSettings != nullptr)
            {
                DEBUG_LOG("PROFILE START\n");
                if (info != nullptr && length != 0)
                {
                    if (length != sizeof(SettingsFromThePcGui))
                    {
                        ERROR_LOG("PCCOM: Settings may be too big [%lu] != [%lu]\n",
                            length,
                            sizeof(SettingsFromThePcGui));
                    }
                    DUMP_CURRENT_LINE();
                    memcpy(gSettings, info, sizeof(SettingsFromThePcGui));
                    DUMP_CURRENT_LINE();

                    DUMP_CURRENT_LINE();
                    SignalBeginEvent();
                }
            }
            break;

        case ProfilerCommMessage_Stop:
            DEBUG_LOG("STOP PROFILING\n");
            nn::os::SignalEvent(GetStopEvent());
            break;

        case ProfilerCommMessage_TransferThreads:
            {
                FillThreadList();

                // Send a message back to the PC with the list of the threads
                //  Channel size is numThreads + 1 as the numThread variable is also
                //  being sent
                s_commLayer->Send(
                    ProfilerCommMessage_TransferThreads,
                    threadInfo,
                    sizeof(CommThreadList),
                    nullptr);
            }
            break;

        case ProfilerCommMessage_ReadyToProfile:
            {
                ProfilerStatus prevStatus = GetProfilerStatus();
                if (prevStatus == ProfilerStatus_Profiling)
                {
                    SendProfilingHasBegun();
                }
                else if (prevStatus == ProfilerStatus_Active)
                {
                    if (firstReadySinceConnect)
                    {
                        firstReadySinceConnect = false;
                        if (TargetApplication::GetCurrent()->IsAttached() &&
                            SampleBuffers::GetInstance()->HasData())
                        {
                            SetProfilerStatus(ProfilerStatus_Transferring);
                            nn::os::EventType* sendEvent = SendDataToPC();
                            if (sendEvent != nullptr)
                            {
                                nn::os::WaitEvent(sendEvent);
                                SampleBuffers::GetInstance()->Reset();
                            }
                            SetProfilerStatus(prevStatus);
                        }
                        else if (TargetApplication::GetCurrent()->IsLibraryInitialized())
                        {
                            IpcEventInfo info;
                            info.event = IpcEvent_PcMessageForward;
                            info.info.pcMessageForward.message = message;
                            info.info.pcMessageForward.size = 0;
                            info.info.pcMessageForward.smallPayload = 0;

                            auto ipcQueue = GetIpcEventQueue();
                            ipcQueue->Push(&info);
                        }
                    }
                }
                else if (prevStatus == ProfilerStatus_Transferring)
                {
                    SetProfilerStatus(ProfilerStatus_Active);
                }
                pcSynced = true;
            }
            break;


        default:
            DEBUG_LOG("PCCOM: Unhandled Message: %d\n", message);
            break;
        }
    } // NOLINT(impl/function_size)



    void StateChangeHandler(pccom::PccomState state)
    {
        switch (state)
        {
            case pccom::PccomState_Disconnected:
                TargetApplication::GetCurrent()->Kill();
                pcSynced = false;
                firstReadySinceConnect = false;
                break;

            case pccom::PccomState_Connected:
                DynamicMemorySetup(false);
                SendAvailableProfileModes();
                pcSynced = false;
                firstReadySinceConnect = true;
                break;

            case pccom::PccomState_Waiting:
            default:
                // do nothing
                break;
        }
    }



    void TransferRequest(
        void **destination,
        size_t size,
        uint32_t message,
        nn::profiler::pccom::RecvCompleteCallback *callback)
    {
        NN_UNUSED(size);

        DEBUG_LOG("Transfer Request:\n  Size: %lu\n  Message %d\n", size, message);

        switch (message)
        {
        case ProfilerCommMessage_Start:
            NN_SDK_ASSERT(size <= sizeof(nn::profiler::SettingsFromThePcGui));
            *destination = gSettings;
            break;

        default:
            NN_SDK_ASSERT(size <= TEMP_BUFFER_SIZE);
            *destination = s_channelInfo;
            break;
        }

        *callback = OnMessage;
    }



    void OnSendDataComplete(
        uint32_t message,
        nn::Result result)
    {
        DEBUG_LOG("== Send Data Complete ==\n");
        if (result.IsSuccess())
        {
            nn::os::SignalEvent(&s_xferCompleteEvent);
        }
        else
        {
            ERROR_LOG("Error on data send: %p\n", result.GetInnerValueForDebug());
        }

        if (s_transferMappedMemory != nullptr)
        {
            OnIpcSendMessageComplete(message, result);
        }

        switch (message)
        {
        case ProfilerCommMessage_TransferData:
            Memory::GetInstance()->Free(s_threadNameSection);
            s_threadNameSection = nullptr;
            Memory::GetInstance()->Free(s_nvnPointerTable);
            s_nvnPointerTable = nullptr;
            break;

        default:
            break;
        }
    }



    void ClearPccomTemporaryMemory(
        uint32_t message,
        nn::Result result)
    {
        NN_UNUSED(message);
        NN_UNUSED(result);

        s_commLayer->DestroyTemporaryHeap();
    }



    void OnIpcSendMessageComplete(
        uint32_t message,
        nn::Result result)
    {
        nn::os::SignalEvent(&s_xferCompleteEvent);

        if (message == ProfilerCommMessage_TransferData)
        {
            ClearPccomTemporaryMemory(message, result);
            CommReleaseTransferMemory();
        }

        IpcEventInfo eventInfo;
        eventInfo.event = IpcEvent_MessageCallback;
        eventInfo.info.messageCallback.id = message;
        eventInfo.info.messageCallback.resultCode = result.GetInnerValueForDebug();

        auto eventQueue = GetIpcEventQueue();
        eventQueue->Push(&eventInfo);
    }



    void SendAvailableProfileModes()
    {
        ProfilingMode modes = TargetApplication::GetCurrent()->GetAllAvailableProfilingModes();
        INFO_LOG("CM Sending Available Profiling Modes: %d\n", modes);
        s_commLayer->Send(
            ProfilerCommMessage_AvailableProfilingModes,
            &modes,
            sizeof(modes),
            nullptr);
    }
} // anonymous namespace


//-------------------------------------------------------------------
//
//-------------------------------------------------------------------
void InitializeCommunicationsLayer()
{
    nn::Result result;

    Memory* memHandler = Memory::GetInstance();

    memset(&s_transferMemory, 0, sizeof(s_transferMemory));
    s_transferOriginalAddress = 0;
    s_transferMappedMemory = nullptr;

    pcSynced = false;

    result = nn::profiler::pccom::Initialize();
    if (result.IsFailure())
    {
        ERROR_LOG("Error initializing pccom.\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        ProfilerPanic();
    }

    {
        threadInfo = memHandler->Allocate<CommThreadList>();
        if (threadInfo == nullptr)
        {
            ERROR_LOG("Error getting memory for communication layer.\n");
            ProfilerPanic();
        }
        new (threadInfo) CommThreadList;
    }

    {
        s_commLayer = memHandler->Allocate<pccom::TwoWayCommunication>();
        if (s_commLayer == nullptr)
        {
            ERROR_LOG("Error getting memory from communication layer.\n");
            ProfilerPanic();
        }

        new (s_commLayer) pccom::TwoWayCommunication;
        result = s_commLayer->Initialize(TransferRequest, StateChangeHandler);
        if (result.IsFailure())
        {
            ERROR_LOG("Error initializing communication layer.\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            ProfilerPanic();
        }
    }

    s_commBufferMemory = memHandler->Allocate(SIZE_OF_COMM_UNIT_HEAP);
    if (s_commBufferMemory == nullptr)
    {
        ERROR_LOG("Error allocating memory for comm buffer memory.\n");
        ProfilerPanic();
    }

    s_commBuffers = nn::lmem::CreateUnitHeap(
        s_commBufferMemory,
        SIZE_OF_COMM_UNIT_HEAP,
        TEMP_BUFFER_SIZE,
        nn::lmem::CreationOption_ThreadSafe,
        32,
        nn::lmem::InfoPlacement_Head);
    if (s_commBuffers == nullptr)
    {
        ERROR_LOG("Error creating unit heap.\n");
        ProfilerPanic();
    }
    s_channelInfo = static_cast<char*>(nn::lmem::AllocateFromUnitHeap(s_commBuffers));

    // Initialize all events here
    nn::os::InitializeEvent(&s_beginEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&s_xferCompleteEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&s_stopEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeLightEvent(&s_waitForData, false, nn::os::EventClearMode_AutoClear);
}



nn::os::EventType* CheckForBegin(SettingsFromThePcGui *settings)
{
    // Maintain a copy of the settings pointer
    // The PCCOM Begin request will write the settings to this location
    gSettings = settings;

    return &s_beginEvent;
}



nn::os::EventType* SendDataToPC()
{
    DEBUG_LOG("Attempting to send data to PC\n");

    DUMP_CURRENT_LINE();

    // Make sure that the event is cleared before going into this
    // Users of the event can also clear the event
    nn::os::ClearEvent(&s_xferCompleteEvent);

    if (IsPCConnected() == false)
    {
        return nullptr;
    }

    // Request all necessary data from library, assuming one is connected
    if (TargetApplication::GetCurrent()->IsLibraryInitialized())
    {
        IpcEventInfo eventInfo;
        eventInfo.event = IpcEvent_SendData;

        auto queue = GetIpcEventQueue();
        queue->Push(&eventInfo);

        DUMP_CURRENT_LINE();

        bool success = nn::os::TimedWaitLightEvent(&s_waitForData, nn::TimeSpan::FromSeconds(1));
        if (!success)
        {
            WARNING_LOG("Failed to receive data from target application in a timely manner.\n");
        }
    }

    DUMP_CURRENT_LINE();

    nn::Result result;

    size_t totalSize = 0;
    uint32_t pieces = 0;

    const int SectionOffsetCount = SupportedCoreCount + 5;
    const int NsoOffsetIndex = SupportedCoreCount + 0;
    const int ThreadNameOffsetIndex = SupportedCoreCount + 1;
    const int UserMarkerNamesIndex = SupportedCoreCount + 2;
    const int NvnFunctionNamesIndex = SupportedCoreCount + 3;
    const int InstrumentationBufferIndex = SupportedCoreCount + 4;

    uint32_t sectionOffsets[SectionOffsetCount] = { 0 };
    uint32_t sectionCount = 0;

    // Add in master header
    Header* masterHeader = GetMasterHeader();
    totalSize += masterHeader->GetSize();
    pieces += 1;

    DUMP_CURRENT_LINE();

    size_t coreBufferSizes[SupportedCoreCount] = { 0 };
    for (uint32_t core = 0; core < SupportedCoreCount; ++core)
    {
        auto coreBuffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex(core));
        size_t coreBufferTotalSize = coreBuffer->totalSize;
        if (coreBufferTotalSize > 0)
        {
            ++sectionCount;

            // Add in pieces for sample buffers
            sectionOffsets[core] = static_cast<uint32_t>(totalSize);
            totalSize += coreBufferTotalSize;
            pieces += coreBuffer->listCount;
            coreBufferSizes[core] = coreBufferTotalSize;

            // Add in core header
            Header *coreHeader = GetCoreHeader(core);
            totalSize += coreHeader->GetSize();
            pieces += 1;
        }
    }

    DUMP_CURRENT_LINE();

    {
        FillApplicationInfo();
        ++sectionCount;
        sectionOffsets[NsoOffsetIndex] = static_cast<uint32_t>(totalSize);
        totalSize += s_appInfoSize;
        pieces += 1;
    }

    DUMP_CURRENT_LINE();

    {
        s_threadNameSection = reinterpret_cast<char*>(Memory::GetInstance()->Allocate(ThreadNameSectionSize));
        FillThreadNameBuffer(s_threadNameSection, ThreadNameSectionSize);
        ++sectionCount;
        sectionOffsets[ThreadNameOffsetIndex] = static_cast<uint32_t>(totalSize);
        totalSize += ThreadNameSectionSize;
        pieces += 1;
    }

    DUMP_CURRENT_LINE();

    StringTable* markerNames = GetMarkerNamesStringTable();
    {
        ++sectionCount;
        sectionOffsets[UserMarkerNamesIndex] = static_cast<uint32_t>(totalSize);
        totalSize += markerNames->GetTotalSize();
        pieces += 1;
    }

    DUMP_CURRENT_LINE();

    const char* nvnFuncs = s_nvnPointerTable;
    size_t nvnFuncSize = s_nvnPointerTableSize;
    NN_SDK_ASSERT((nvnFuncs != nullptr && nvnFuncSize > 0) || nvnFuncs == nullptr);
    bool nvnFuncsFound = (nvnFuncs != nullptr);
    if (nvnFuncsFound == false)
    {
        nvnFuncs = "\1\0\0\0\0\0\0\0\0";
        nvnFuncSize = sizeof(uint32_t) * 2;
    }
    {
        ++sectionCount;
        sectionOffsets[NvnFunctionNamesIndex] = static_cast<uint32_t>(totalSize);
        totalSize += nvnFuncSize;
        pieces += 1;
    }

    // Check for instrumented code/data
    size_t libInstrumentedSize = 0;
    size_t localInstrumentedSize = 0;
    size_t totalInstrumentedSize = 0;
    {
        if (s_transferMappedMemory != nullptr)
        {
            if (s_transferMemory._size > 0)
            {
                DEBUG_LOG("Transfer Mem for Instrumented Buffer: %lld\n", s_transferMemory._size);
                libInstrumentedSize += s_transferMemory._size;
                totalInstrumentedSize += libInstrumentedSize;
                pieces += 1;
            }
        }

        auto instBuffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex_Instrumentation);
        DEBUG_LOG("Local Instrumentation Buffer Size: %lld\n", instBuffer->totalSize);
        if (instBuffer->totalSize > 0)
        {
            NN_SDK_ASSERT(instBuffer->listCount > 0);
            localInstrumentedSize += instBuffer->totalSize;
            totalInstrumentedSize += localInstrumentedSize;
            pieces += instBuffer->listCount;
        }

        if (totalInstrumentedSize > 0)
        {
            ++sectionCount;
            sectionOffsets[InstrumentationBufferIndex] = static_cast<uint32_t>(totalSize);
            totalSize += totalInstrumentedSize;
        }
    }

    DUMP_CURRENT_LINE();

    // Provide offsets after the master header to each core's data
    {
        uint32_t oldMasterHeaderSize = static_cast<uint32_t>(masterHeader->GetSize());

        for (uint32_t core = 0; core < SupportedCoreCount; ++core)
        {
            if (sectionOffsets[core] > 0)
            {
                masterHeader->Write(
                    HeaderSpecialValues_CoreOffset,
                    core,
                    sectionOffsets[core] - oldMasterHeaderSize);
            }
        }

        DUMP_CURRENT_LINE();

        masterHeader->Write(
            HeaderSpecialValues_NsoRegionOffset,
            sectionOffsets[NsoOffsetIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialValues_ThreadListOffset,
            sectionOffsets[ThreadNameOffsetIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialValues_UserMarkerNames,
            sectionOffsets[UserMarkerNamesIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialvalues_UserMarkerNamesBase,
            reinterpret_cast<uint64_t>(markerNames->GetTableAddress()));

        masterHeader->Write(
            HeaderSpecialValues_NvnFuncNamesOffset,
            sectionOffsets[NvnFunctionNamesIndex] - oldMasterHeaderSize);

        if (totalInstrumentedSize > 0)
        {
            NN_SDK_ASSERT(totalInstrumentedSize <= UINT32_MAX);
            masterHeader->Write(
                HeaderSpecialValues_InstrumentedOffset,
                sectionOffsets[InstrumentationBufferIndex] - oldMasterHeaderSize,
                InstrumentedBufferVersion,
                static_cast<uint32_t>(totalInstrumentedSize),
                gSettings->perf_counter_cycle);
        }

        masterHeader->WriteControlValueOnly(HeaderSpecialValues_HeaderEnd);

        totalSize -= oldMasterHeaderSize;
        totalSize += masterHeader->GetSize();
    }

    DUMP_CURRENT_LINE();

    DEBUG_LOG("Starting multi-part send size = %lu; pieces = %d\n", totalSize, pieces);
    FORCE_SEND(s_commLayer->StartMultipartSend(
        ProfilerCommMessage_TransferData,
        totalSize,
        pieces,
        OnSendDataComplete));

    DUMP_CURRENT_LINE();

    // Write master header
    FORCE_SEND(s_commLayer->SendMultipart(
        masterHeader->GetBuffer(),
        masterHeader->GetSize(),
        pieces - 1,
        false));
    --pieces;

    DUMP_CURRENT_LINE();

    for (uint32_t core = 0; core < SupportedCoreCount; ++core)
    {
        size_t coreBufferSize = coreBufferSizes[core];
        auto coreBuffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex(core));
        if (coreBufferSize > 0)
        {
            // Write core header
            Header *coreHeader = GetCoreHeader(core);
            FORCE_SEND(s_commLayer->SendMultipart(
                coreHeader->GetBuffer(),
                coreHeader->GetSize(),
                pieces - 1,
                false));
            --pieces;

            for (uint32_t listIdx = 0; listIdx < coreBuffer->listCount; ++listIdx)
            {
                auto data = coreBuffer->list[listIdx];
                size_t writeSize = std::min(data.size, coreBufferSize);
                coreBufferSize -= writeSize;
                FORCE_SEND(s_commLayer->SendMultipart(data.memory, writeSize, pieces - 1, false));
                --pieces;
            }
        }
    }

    DUMP_CURRENT_LINE();

    FORCE_SEND(s_commLayer->SendMultipart(s_appInfo, s_appInfoSize, pieces - 1, false));
    --pieces;

    DUMP_CURRENT_LINE();

    FORCE_SEND(s_commLayer->SendMultipart(s_threadNameSection, ThreadNameSectionSize, pieces - 1, false));
    --pieces;

    DUMP_CURRENT_LINE();

    FORCE_SEND(s_commLayer->SendMultipart(
        markerNames->GetTableAddress(),
        markerNames->GetTotalSize(),
        pieces - 1,
        false));
    --pieces;

    DUMP_CURRENT_LINE();

    FORCE_SEND(s_commLayer->SendMultipart(nvnFuncs, nvnFuncSize, pieces - 1, false));
    --pieces;

    if (totalInstrumentedSize > 0)
    {
        if (libInstrumentedSize > 0)
        {
            size_t writeSize = libInstrumentedSize;
            FORCE_SEND(s_commLayer->SendMultipart(s_transferMappedMemory, writeSize, pieces - 1, false));
            --pieces;
        }

        if (localInstrumentedSize > 0)
        {
            auto buffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex_Instrumentation);
            for (uint32_t listIdx = 0; listIdx < buffer->listCount; ++listIdx)
            {
                auto data = buffer->list[listIdx];
                size_t writeSize = std::min(data.size, localInstrumentedSize);
                localInstrumentedSize -= writeSize;
                FORCE_SEND(s_commLayer->SendMultipart(data.memory, writeSize, pieces - 1, false));
                --pieces;
            }
        }
    }

    DUMP_CURRENT_LINE();

    NN_SDK_ASSERT(pieces == 0u);

    return &s_xferCompleteEvent;
} // NOLINT(impl/function_size)



nn::os::EventType* GetTransferCompleteEvent()
{
    return &s_xferCompleteEvent;
}



void SignalBeginEvent()
{
    nn::os::ClearEvent(GetStopEvent());
    nn::os::SignalEvent(&s_beginEvent);
}



nn::os::EventType* GetStopEvent()
{
    return &s_stopEvent;
}


void SendNotificationToPC(
    uint32_t code,
    nn::Result resultToSend)
{
    char* buffer = AllocateCommMsgMem();
    memset(buffer, 0, TEMP_BUFFER_SIZE);

    char* writePtr = buffer;
    writePtr = WriteToBuffer(writePtr, code);
    writePtr = WriteToBuffer(writePtr, resultToSend.GetInnerValueForDebug());

    FORCE_SEND(s_commLayer->Send(
        ProfilerCommMessage_Error,
        buffer,
        TEMP_BUFFER_SIZE,
        nullptr));
    MessageSentCallbackToFreeMem(code, nn::ResultSuccess(), buffer);
}



void SendReadyToProfile(size_t maxBuffSize)
{
    char* buffer = AllocateCommMsgMem();
    uint32_t* tempBuffer = reinterpret_cast<uint32_t*>(buffer);
    size_t bufferSize = 0;

    tempBuffer[0] = static_cast<uint32_t>(maxBuffSize);
    bufferSize += sizeof(*tempBuffer);

    INFO_LOG("Ready To Profile\n");

    s_commLayer->Send(
        ProfilerCommMessage_ReadyToProfile,
        buffer,
        bufferSize,
        nullptr);
    MessageSentCallbackToFreeMem(ProfilerCommMessage_ReadyToProfile, nn::ResultSuccess(), buffer);
}



void SendProfilingHasBegun()
{
    s_commLayer->Send(ProfilerCommMessage_ProfilingStarted, nullptr, 0, nullptr);
}



bool IsPCConnected()
{
    return pcSynced && s_commLayer->IsPCConnected();
}


void SendCoreAvailable(uint32_t coreMask)
{
    char* buffer = AllocateCommMsgMem();
    char* tempBuffer = buffer;
    size_t bufferSize = 0;

    tempBuffer = WriteToBuffer(tempBuffer, coreMask);
    bufferSize = static_cast<size_t>(tempBuffer - buffer);

    s_commLayer->Send(
        ProfilerCommMessage_CoreAvailable,
        buffer,
        bufferSize,
        nullptr);
    MessageSentCallbackToFreeMem(ProfilerCommMessage_CoreAvailable, nn::ResultSuccess(), buffer);
}


void SendPcDebugOutput(const char* message, ...)
{
    char buffer[TEMP_BUFFER_SIZE];

    va_list args;
    va_start(args, message);
    vsnprintf(buffer, TEMP_BUFFER_SIZE, message, args);
    va_end(args);

    size_t msgLen = strnlen(buffer, TEMP_BUFFER_SIZE);
    FORCE_SEND(s_commLayer->Send(
        ProfilerCommMessage_Output,
        buffer,
        msgLen,
        nullptr));
}


nn::Result CommSendMessage(uint32_t id, const void* buffer, size_t size, uintptr_t originalAddress, bool callback)
{
    nn::profiler::pccom::SendCallback sc = nullptr;
    if (callback)
    {
        sc = &OnIpcSendMessageComplete;
    }

    bool copy = true;
    if (IsInTransferMemory(originalAddress, size))
    {
        copy = false;
        buffer = reinterpret_cast<const void*>(
            static_cast<char*>(s_transferMappedMemory) + originalAddress - s_transferOriginalAddress);
    }
    NN_SDK_ASSERT((buffer != nullptr && size > 0) || (buffer == nullptr && size == 0));

    //return s_commLayer->Send(id, buffer, size, sc, copy);
    return s_commLayer->SendImmediate(id, buffer, size, sc);
}


nn::Result CommStartMultipart(uint32_t id, size_t totalSize, uint32_t pieces, bool callback)
{
    nn::profiler::pccom::SendCallback sc = nullptr;
    if (callback)
    {
        sc = &OnIpcSendMessageComplete;
    }

    if (id == ProfilerCommMessage_TransferData)
    {
        s_commLayer->CreateTemporaryHeap();

        if (sc == nullptr)
        {
            sc = &ClearPccomTemporaryMemory;
        }
    }
    DEBUG_LOG("Attempting to start multipart message: %d, %ld, %d, %p\n", id, totalSize, pieces, sc);
    //return s_commLayer->StartMultipartSend(id, totalSize, pieces, sc);
    return s_commLayer->StartMultipartSendImmediate(id, totalSize, pieces, sc);
}


nn::Result CommSendMultipart(const void* buffer, size_t size, uintptr_t originalAddress, uint32_t remaining)
{
    bool copy = true;
    if (IsInTransferMemory(originalAddress, size))
    {
        copy = false;
        buffer = reinterpret_cast<const void*>(
            static_cast<char*>(s_transferMappedMemory) + originalAddress - s_transferOriginalAddress);
    }
    NN_SDK_ASSERT((buffer != nullptr && size > 0) || (buffer == nullptr && size == 0));

    //return s_commLayer->SendMultipart(buffer, size, remaining, copy);
    return s_commLayer->SendMultipartImmediate(buffer, size, remaining);
}


nn::Result CommSetTransferMemory(nn::os::NativeHandle handle, uintptr_t origAddr, size_t size, bool managed)
{
    nn::Result result;

    NN_SDK_ASSERT(s_transferMappedMemory == nullptr);

    DEBUG_LOG("Original Transfer Memory Address %p\n", origAddr);

    s_transferOriginalAddress = origAddr;

    nn::os::AttachTransferMemory(&s_transferMemory, size, handle, managed);
    result = nn::os::MapTransferMemory(&s_transferMappedMemory, &s_transferMemory, nn::os::MemoryPermission_None);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        return result;
    }

    return nn::ResultSuccess();
}


nn::Result CommReleaseTransferMemory()
{
    DUMP_CURRENT_LINE();
    if (s_transferMappedMemory != nullptr &&
        s_transferMemory._state != s_transferMemory.State_Uninitialized)
    {
        nn::os::DestroyTransferMemory(&s_transferMemory);
        memset(&s_transferMemory, 0, sizeof(s_transferMemory));
        s_transferMappedMemory = nullptr;
        s_transferOriginalAddress = 0;
    }
    return nn::ResultSuccess();
}


nn::Result CommSendSampleBufferAsMultipart(uint32_t blockId, size_t size, uint32_t remaining)
{
    void *buffer = static_cast<char*>(s_transferMappedMemory) + (blockId * SampleMemoryBlockSize);
    return CommSendMultipart(buffer, size, 0, remaining);
}


IpcEventQueue* GetIpcEventQueue()
{
    return &s_ipcEventQueue;
}


nn::Result CommCreateNvnFunctionTable(const void* buffer, size_t size)
{
    if (size > 0)
    {
        NN_SDK_ASSERT(s_nvnPointerTable == nullptr);
        s_nvnPointerTable = static_cast<char*>(Memory::GetInstance()->Allocate(size));
        if (s_nvnPointerTable == nullptr)
        {
            return nn::profiler::ResultMemoryAllocationFailure();
        }
        memcpy(s_nvnPointerTable, buffer, size);
    }
    s_nvnPointerTableSize = size;
    return nn::ResultSuccess();
}


void SignalAllDataReceived()
{
    nn::os::SignalLightEvent(&s_waitForData);
}


} // profiler
} // nn
