﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_BUILD_CONFIG_HTC_ENABLED

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/os/os_Thread.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/htcs.h>
#include <nn/socket/socket_Constants.h>
NN_PRAGMA_POP_WARNINGS

#include <nn/profiler/profiler_Result.h>

#include "profiler_Pccom.h"
#include "../profiler_Logging.h"
#include "../profiler_ThreadPriorities.h"

#include "profiler_IProfiler.sfdl.h"
#include "profiler_Comms.h"
#include "profiler_Defines.h"
#include "profiler_Memory.h"
#include "profiler_ResultPrivate.h"

#define PCCOM_HIO_DOWNTIME nn::TimeSpan::FromMilliSeconds(100)

//#define HIJACK_SAMPLE_BUFFERS_FOR_TEMP
//#define CLOSE_HTCS_SERVICE_SOCKET_ON_CONNECT

namespace nn { namespace profiler { namespace pccom {

    namespace /*anonymous*/
    {
        const char* const HtcsPortName = "NintendoCpuProfiler:out";

        const size_t StackSize = 8 * 1024;
        const size_t MaximumSendSize = 0x8000;

        bool gPcConnected = false;

        uint8_t *g_sendStack;
        uint8_t *g_recvStack;

        nn::os::BarrierType g_SendRecvSynchronization;
        nn::os::MutexType g_SendImmediateMutex;

#if defined(HIJACK_SAMPLE_BUFFERS_FOR_TEMP)
        nn::lmem::HeapHandle g_temporaryHeap = nullptr;
#endif

        void* Allocate(size_t size)
        {
            void *memory = nn::profiler::Memory::GetInstance()->Allocate(size);
            DEBUG_LOG("Allocated %lu bytes to address %p\n", size, memory);
            return memory;
        }

        void Deallocate(void* p, size_t size)
        {
            NN_UNUSED(size);
            DEBUG_LOG("Freeing address %p\n", p);
            nn::profiler::Memory::GetInstance()->Free(p);
        }

        void* AllocateTemp(size_t size)
        {
#if defined(HIJACK_SAMPLE_BUFFERS_FOR_TEMP)
            if (g_temporaryHeap != nullptr)
            {
                return nn::lmem::AllocateFromExpHeap(g_temporaryHeap, size);
            }
#endif
            return Allocate(size);
        }


        void FreeTemp(void* mem)
        {
#if defined(HIJACK_SAMPLE_BUFFERS_FOR_TEMP)
            if (g_temporaryHeap != nullptr)
            {
                return nn::lmem::FreeToExpHeap(g_temporaryHeap, mem);
            }
#endif
            return Deallocate(mem, 0);
        }


    } // anonymous



    nn::Result Initialize()
    {
        nn::htcs::Initialize(&Allocate, &Deallocate);
        nn::os::InitializeBarrier(&g_SendRecvSynchronization, 2);
        nn::os::InitializeMutex(&g_SendImmediateMutex, false, 0);
        return nn::ResultSuccess();
    }



    TwoWayCommunication::TwoWayCommunication() :
        m_socket(nn::socket::InvalidSocket),
        m_serviceSocket(nn::socket::InvalidSocket),
        m_recvCallback(nullptr),
        m_stateChangeCallback(nullptr),
        m_socketIsClosed(true),
        m_multipartPartsRemaining(0),
        m_multipartMessage(0),
        m_multipartTaskCallback(nullptr)
    {
    }


    nn::Result TwoWayCommunication::Initialize(
        RecvCallback recv_callback,
        StateChangeCallback state_callback)
    {
        nn::Result result;

        INFO_LOG("Initializing Two Way Communication\n");

        m_socketIsClosed = false;
        m_recvCallback = recv_callback;
        m_stateChangeCallback = state_callback;
        m_queue.Initialize();

        g_sendStack = reinterpret_cast<uint8_t*>(
            Memory::GetInstance()->Allocate(StackSize, nn::os::GuardedStackAlignment));
        g_recvStack = reinterpret_cast<uint8_t*>(
            Memory::GetInstance()->Allocate(StackSize, nn::os::GuardedStackAlignment));

        INFO_LOG("Starting Wait for connection thread\n");
        result = nn::os::CreateThread(
            &m_recvThread,
            TwoWayWaitForConnection,
            this,
            g_recvStack,
            StackSize,
            ThreadPriority_PccomWait,
            ProfilerPrimaryCore);
        if (result.IsFailure())
        {
            ERROR_LOG("Error creating comm layer Recv thread!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            ProfilerPanic();
        }
        nn::os::SetThreadName(&m_recvThread, "[profiler] PCCOM Wait");
        nn::os::StartThread(&m_recvThread);

        INFO_LOG("Starting Send thread\n");
        result = nn::os::CreateThread(
            &m_sendThread,
            TwoWaySendThreadProc,
            this,
            g_sendStack,
            StackSize,
            ThreadPriority_PccomSend,
            ProfilerPrimaryCore);
        if (result.IsFailure())
        {
            ERROR_LOG("Error creating comm layer Send thread!\n");
            DumpResultInformation(LOG_AS_ERROR, result);
            ProfilerPanic();
        }
        nn::os::SetThreadName(&m_sendThread, "[profiler] PCCOM Send");
        nn::os::StartThread(&m_sendThread);

        nn::os::YieldThread();

        return nn::ResultSuccess();
    }



    void TwoWayCommunication::CreateTemporaryHeap()
    {
#if defined(HIJACK_SAMPLE_BUFFERS_FOR_TEMP)
        void* tempMemory = SampleBuffers::GetInstance()->GetStartAddress();
        size_t tempSize = SampleBuffers::GetInstance()->GetSize();
        g_temporaryHeap = nn::lmem::CreateExpHeap(tempMemory, tempSize, nn::lmem::CreationOption_ThreadSafe);
#endif
    }



    void TwoWayCommunication::DestroyTemporaryHeap()
    {
#if defined(HIJACK_SAMPLE_BUFFERS_FOR_TEMP)
        nn::lmem::DestroyExpHeap(g_temporaryHeap);
        g_temporaryHeap = nullptr;
#endif
    }



    nn::Result TwoWayCommunication::Send(
        uint32_t message,
        const void* data,
        size_t data_size,
        SendCallback callback,
        bool copy)
    {
        nn::Result result;

        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }

        TaskItem item;
        item.m_message = message;
        item.m_dataSize = static_cast<uint32_t>(data_size);
        item.m_taskCallback = callback;
        item.m_partsRemaining = 0;
        item.m_needsFree = copy;

        if (copy)
        {
            void *mem = AllocateTemp(data_size);
            if (mem == nullptr) { return nn::profiler::ResultMemoryAllocationFailure(); }
            memcpy(mem, data, data_size);
            item.m_data = mem;
        }
        else
        {
            item.m_data = data;
        }

        bool pushed = m_queue.Push(&item);
        if (!pushed)
        {
            if (item.m_needsFree)
            {
                FreeTemp(const_cast<void*>(item.m_data));
            }
            return nn::profiler::ResultPccomQueueFull();
        }
        else { return nn::ResultSuccess(); }
    }



    nn::Result TwoWayCommunication::SendImmediate(
        uint32_t message,
        const void* data,
        size_t data_size,
        SendCallback callback)
    {
        nn::Result result;

        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }
        if (m_queue.CurrentCount() > 0) { return nn::profiler::ResultPccomQueueFull(); }

        TaskItem item;
        item.m_message = message;
        item.m_dataSize = static_cast<uint32_t>(data_size);
        item.m_taskCallback = callback;
        item.m_partsRemaining = 0;
        item.m_needsFree = false;
        item.m_data = data;

        return SendImmediate(&item);
    }



    nn::Result TwoWayCommunication::StartMultipartSend(
        uint32_t message,
        size_t total_transfer_size,
        uint32_t total_pieces,
        SendCallback callback)
    {
        if (m_multipartPartsRemaining != 0)
        {
            return nn::profiler::ResultPccomQueueMultipartInProgress();
        }

        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }

        TaskItem item;
        item.m_message = message;
        item.m_data = nullptr;
        item.m_dataSize = static_cast<uint32_t>(total_transfer_size);
        item.m_partsRemaining = total_pieces;
        item.m_taskCallback = callback;

        m_multipartPartsRemaining = total_pieces;
        m_multipartMessage = message;
        m_multipartTaskCallback = callback;

        bool pushed = m_queue.Push(&item);
        if (!pushed) { return nn::profiler::ResultPccomQueueFull(); }
        else { return nn::ResultSuccess(); }
    }



    nn::Result TwoWayCommunication::StartMultipartSendImmediate(
        uint32_t message,
        size_t total_transfer_size,
        uint32_t total_pieces,
        SendCallback callback)
    {
        if (m_multipartPartsRemaining != 0)
        {
            return nn::profiler::ResultPccomQueueMultipartInProgress();
        }

        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }
        if (m_queue.CurrentCount() > 0) { return nn::profiler::ResultPccomQueueFull(); }

        TaskItem item;
        item.m_message = message;
        item.m_data = nullptr;
        item.m_dataSize = static_cast<uint32_t>(total_transfer_size);
        item.m_partsRemaining = total_pieces;
        item.m_taskCallback = callback;
        item.m_needsFree = false;

        m_multipartPartsRemaining = total_pieces;
        m_multipartMessage = message;
        m_multipartTaskCallback = callback;

        return SendImmediate(&item);
    }



    nn::Result TwoWayCommunication::SendMultipart(
        const void* data,
        size_t data_size,
        uint32_t remaining,
        bool copy)
    {
        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }

        TaskItem item;
        item.m_message = m_multipartMessage;
        item.m_dataSize = static_cast<uint32_t>(data_size);
        item.m_partsRemaining = remaining;
        item.m_taskCallback = m_multipartTaskCallback;
        item.m_needsFree = copy;

        if (copy)
        {
            void *mem = AllocateTemp(data_size);
            if (mem == nullptr) { return nn::profiler::ResultMemoryAllocationFailure(); }
            memcpy(mem, data, data_size);
            item.m_data = mem;
        }
        else
        {
            item.m_data = data;
        }

        m_multipartPartsRemaining = remaining;

        bool pushed = m_queue.Push(&item);
        if (!pushed)
        {
            if (item.m_needsFree)
            {
                FreeTemp(const_cast<void*>(item.m_data));
            }
            return nn::profiler::ResultPccomQueueFull();
        }
        else { return nn::ResultSuccess(); }
    }



    nn::Result TwoWayCommunication::SendMultipartImmediate(
        const void* data,
        size_t data_size,
        uint32_t remaining)
    {
        if (!IsPCConnected()) { return nn::profiler::ResultPccomNoPcConnected(); }
        if (m_queue.CurrentCount() > 0) { return nn::profiler::ResultPccomQueueFull(); }

        TaskItem item;
        item.m_message = m_multipartMessage;
        item.m_dataSize = static_cast<uint32_t>(data_size);
        item.m_partsRemaining = remaining;
        item.m_taskCallback = m_multipartTaskCallback;
        item.m_needsFree = false;
        item.m_data = data;

        m_multipartPartsRemaining = remaining;

        return SendImmediate(&item);
    }



    nn::Result TwoWayCommunication::SendImmediate(const TaskItem * const item)
    {

        MessageHeader header;
        nn::Result writeSuccessful = nn::ResultSuccess();

        DEBUG_LOG(
            "PCCOM Sending: [%d], %p, %d bytes, %d remaining, free: %d\n",
            item->m_message,
            item->m_data,
            item->m_dataSize,
            item->m_partsRemaining,
            item->m_needsFree);

        header.size = item->m_dataSize;
        header.message = item->m_message;
        header.partsRemaining = item->m_partsRemaining;

        nn::os::LockMutex(&g_SendImmediateMutex);
        nn::htcs::ssize_t writeAmt = 0;
        writeAmt = nn::htcs::Send(m_socket, &header, sizeof(header), 0);
        if (writeAmt < 0)
        {
            ERROR_LOG("Error writing header. HTCS error: %d\n", nn::htcs::GetLastError());
            writeSuccessful = nn::profiler::ResultPccomSendError();
        }
        else
        {
            if (item->m_dataSize > 0 && item->m_data)
            {
                DEBUG_LOG(
                    "WRITING Data: %p @ %d bytes\n",
                    item->m_data,
                    item->m_dataSize);

                size_t remainingSize = item->m_dataSize;
                const uint8_t *writePtr = reinterpret_cast<const uint8_t*>(item->m_data);
                while (writeAmt > 0 && remainingSize > 0)
                {
                    size_t writeSize = std::min(remainingSize, MaximumSendSize);
                    writeAmt = nn::htcs::Send(m_socket, writePtr, writeSize, 0);
                    if (writeAmt < 0)
                    {
                        ERROR_LOG("Error writing data. HTCS error: %d\n", nn::htcs::GetLastError());
                        writeSuccessful = nn::profiler::ResultPccomSendError();
                        break;
                    }
                    remainingSize -= static_cast<size_t>(writeAmt);
                    writePtr += writeAmt;
                }
            }
        }
        nn::os::UnlockMutex(&g_SendImmediateMutex);

        if (item->m_taskCallback != nullptr && item->m_partsRemaining == 0)
        {
            DEBUG_LOG("Attempting to call callback: %p\n", item->m_taskCallback);
            item->m_taskCallback(item->m_message, writeSuccessful);
        }

        // We are freeing the memory, and will not use it after this, should be safe to remove const.
        if (item->m_needsFree)
        {
            FreeTemp(const_cast<void*>(item->m_data));
        }


        return writeSuccessful;
    }



    void TwoWayCommunication::WaitForConnection()
    {
        nn::Result result;
        int htcsResult;

        nn::os::SetThreadName(&m_recvThread, "[profiler] PCCOM Wait");

        NN_SDK_ASSERT(IsPCConnected() == false, "Wait started while the PC was still connected!\n");

        DumpThreadInformation();

        if (m_stateChangeCallback != nullptr)
        {
            m_stateChangeCallback(PccomState_Waiting);
        }

#if !defined(CLOSE_HTCS_SERVICE_SOCKET_ON_CONNECT)
        if (m_socketIsClosed)
        {
            if (m_serviceSocket >= 0)
            {
                nn::htcs::Close(m_serviceSocket);
                m_serviceSocket = nn::socket::InvalidSocket;
            }
        }
#endif

        m_socketIsClosed = false;
        m_socket = nn::socket::InvalidSocket;

        // Open the HIO channel
        INFO_LOG("Opening HIO channel\n");
        nn::htcs::SockAddrHtcs serviceAddr;
        serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
        serviceAddr.peerName = nn::htcs::GetPeerNameAny();
        std::strncpy(serviceAddr.portName.name, HtcsPortName, nn::htcs::PortNameBufferLength);
        serviceAddr.portName.name[nn::htcs::PortNameBufferLength - 1] = 0;

        while (m_socket < 0)
        {
            if (m_serviceSocket >= 0)
            {
                nn::htcs::Close(m_serviceSocket);
                m_serviceSocket = nn::socket::InvalidSocket;
            }

            while ((m_serviceSocket = nn::htcs::Socket()) < 0)
            {
                int error = nn::htcs::GetLastError();
                if (error != nn::htcs::HTCS_ENETDOWN &&
                    error != nn::htcs::HTCS_ENONE)
                {
                    ERROR_LOG("Error getting HTCS socket. HTCS error: %d\n", error);
                    NN_SDK_ASSERT(
                        error == nn::htcs::HTCS_ENONE,
                        "Error getting HTCS socket. HTCS error: %d\n",
                        error);
                }

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            htcsResult = nn::htcs::Bind(m_serviceSocket, &serviceAddr);
            if (htcsResult < 0)
            {
                ERROR_LOG("Error binding HTCS service socket. HTCS error: %d\n", nn::htcs::GetLastError());
                NN_SDK_ASSERT(
                    htcsResult >= 0,
                    "Error binding HTCS service socket. HTCS error: %d\n",
                    nn::htcs::GetLastError());
                continue;
            }

            htcsResult = nn::htcs::Listen(m_serviceSocket, 1);
            if (htcsResult < 0)
            {
                ERROR_LOG("Error listening on HTCS service socket. HTCS error: %d\n", nn::htcs::GetLastError());
                NN_SDK_ASSERT(
                    htcsResult >= 0,
                    "Error listening on HTCS service socket. HTCS error: %d\n",
                    nn::htcs::GetLastError());
                continue;
            }

            INFO_LOG("Waiting for PC to connect...\n");
            // Wait for a connection from the PC
            m_socket = nn::htcs::Accept(m_serviceSocket, nullptr);
            if (m_socket < 0)
            {
                ERROR_LOG("Could not accept HTCS connection. HTCS Error: %d\n", nn::htcs::GetLastError());
                continue;
            }
        }
        INFO_LOG("PC Connected!\n");

#if defined(CLOSE_HTCS_SERVICE_SOCKET_ON_CONNECT)
        if (m_serviceSocket >= 0)
        {
            int closeRetVal = nn::htcs::Close(m_serviceSocket);
            if (closeRetVal < 0)
            {
                ERROR_LOG("Error closing service socket. HTCS Error: %d\n", nn::htcs::GetLastError());
            }
            m_serviceSocket = nn::socket::InvalidSocket;
        }
#endif

        gPcConnected = true;

        if (m_stateChangeCallback != nullptr)
        {
            m_stateChangeCallback(PccomState_Connected);
        }
    }



    bool TwoWayCommunication::IsPCConnected() const
    {
        return gPcConnected;
    }



    void TwoWayCommunication::SendThreadProc()
    {
        nn::Result result;

        NN_SDK_ASSERT(IsPCConnected() == true, "Send started before Wait finished!\n");

        DumpThreadInformation();

        while (!m_socketIsClosed)
        {
            DEBUG_LOG("SEND Loop Start\n");

            m_queue.WaitForItem();

            DEBUG_LOG("Checking SEND queue: %d items\n", m_queue.CurrentCount());

            while (m_queue.CurrentCount() > 0)
            {
                const TaskItem * const item = m_queue.GetFront();

                SendImmediate(item);

                DEBUG_LOG("Send complete\n");

                m_queue.Pop();

                DEBUG_LOG("Send queue popped\n");

                nn::os::YieldThread();
            }
        }

        DEBUG_LOG("Exited SEND Loop\n");
        gPcConnected = false;

        int closeResult = nn::htcs::Close(m_socket);
        if (closeResult < 0)
        {
            ERROR_LOG("Error closing HTCS socket. HTCS error: %d\n", nn::htcs::GetLastError());
        }

        INFO_LOG("Exiting Send thread\n");
    } // NOLINT(impl/function_size)


    void TwoWayCommunication::FinalizeSendProc()
    {
        ClearSendQueue();

        if (m_stateChangeCallback != nullptr)
        {
            m_stateChangeCallback(PccomState_Disconnected);
        }
    }


    void TwoWayCommunication::RecvThreadProc()
    {
        nn::Result result;

        nn::os::SetThreadName(&m_recvThread, "[profiler] PCCOM Recv");

        NN_SDK_ASSERT(IsPCConnected() == true, "Recv started before Wait finished!\n");

        DumpThreadInformation();

        while (!m_socketIsClosed)
        {
            DEBUG_LOG("PCCOM - Waiting to receive header...\n");

            MessageHeader header;
            if (!ReadOnePiece(&header, sizeof(header)))
            {
                INFO_LOG("PCCOM - Failed to read part of header\n");
                continue;
            }

            DEBUG_LOG("PCCOM - Waiting to receive %d bytes...\n", header.size);

            void* destination = nullptr;
            RecvCompleteCallback callback = nullptr;
            m_recvCallback(&destination, static_cast<size_t>(header.size), header.message, &callback);
            if (destination != nullptr)
            {
                if (header.size > 0)
                {
                    if (!ReadOnePiece(destination, static_cast<size_t>(header.size)))
                    {
                        // We failed for some reason, don't try to do anything else with this data.
                        INFO_LOG("PCCOM - Failed to read piece.\n");
                        continue;
                    }
                }

                if (callback != nullptr)
                {
                    DEBUG_LOG("PCCOM - Calling callback %p\n", callback);
                    callback(destination, static_cast<size_t>(header.size), header.message);
                }
            }
            else
            {
                DEBUG_LOG("PCCOM - Ignoring message as there was no place to store the data.\n");

                // Ignore the message as there is no place to put it
                uint32_t buffer[1];
                while (header.size != 0)
                {
                    bool success = ReadOnePiece(buffer, 1);
                    if (!success)
                    {
                        INFO_LOG("PCCOM - Failed to clear buffer of sent data.\n");
                        break;
                    }
                    --header.size;
                }
            }

            nn::os::YieldThread();
        }

        INFO_LOG("Exiting Receive thread\n");
        m_queue.ForceReleaseWait();
    }



    bool TwoWayCommunication::ReadOnePiece(
        void* destination,
        size_t size)
    {
        nn::htcs::ssize_t readAmt;
        nn::htcs::ssize_t targetTotal = static_cast<nn::htcs::ssize_t>(size);
        nn::htcs::ssize_t offset = 0;
        const int MAX_LOOP_COUNT = 3;
        int loopCount = MAX_LOOP_COUNT;

        // Loop to ensure we get everything (to resolve the problem that led to the peek)
        while (targetTotal > 0)
        {
            uint8_t* target = reinterpret_cast<uint8_t*>(destination) + offset;
            readAmt = nn::htcs::Recv(m_socket, target, static_cast<size_t>(targetTotal), 0);
            if (readAmt < 0)
            {
                int errValue = nn::htcs::GetLastError();
                if (loopCount > 0 &&
                    (errValue == nn::htcs::HTCS_EAGAIN ||
                     errValue == nn::htcs::HTCS_EWOULDBLOCK ||
                     errValue == nn::htcs::HTCS_EINPROGRESS ||
                     errValue == nn::htcs::HTCS_EINTR))
                {
                    // Attempt to perform the receive again in a bit
                    nn::os::SleepThread(PCCOM_HIO_DOWNTIME);
                    --loopCount;
                    continue;
                }

                if (errValue != nn::htcs::HTCS_ENONE &&
                    errValue != nn::htcs::HTCS_ECONNRESET)
                {
                    ERROR_LOG("PCCOM Read Error. HTCS error: %d\n", errValue);
                }
                m_socketIsClosed = true;
                return false;
            }
            else if (readAmt == 0)
            {
                INFO_LOG("Host transmission has ended!!\n");
                m_socketIsClosed = true;
                return false;
            }
            else if (readAmt < targetTotal)
            {
                INFO_LOG("PCCOM Read Incomplete: [%lld] < [%lld]\n", readAmt, size);
            }
            offset += readAmt;
            targetTotal -= readAmt;
        }

        return loopCount > 0;
    }



    void TwoWayCommunication::ClearSendQueue()
    {
        // It was requested that we exit while there were still items in the send queue.
        // Clear it out.
        while (m_queue.CurrentCount() > 0)
        {
            const TaskItem* const item = m_queue.GetFront();
            nn::Result writeSuccessful = nn::profiler::ResultPccomSendError();
            m_queue.Pop();
            if (item->m_taskCallback != nullptr && item->m_partsRemaining == 0)
            {
                item->m_taskCallback(item->m_message, writeSuccessful);
            }
            Memory::GetInstance()->Free(const_cast<void*>(item->m_data));
        }
    }



    void TwoWayWaitForConnection(void* that)
    {
        NN_SDK_ASSERT_NOT_NULL(that);

        auto ptr = reinterpret_cast<TwoWayCommunication*>(that);

        nn::os::AwaitBarrier(&g_SendRecvSynchronization);

        while (NN_STATIC_CONDITION(true))
        {
            ptr->WaitForConnection();
            nn::os::AwaitBarrier(&g_SendRecvSynchronization);

            ptr->RecvThreadProc();
            nn::os::AwaitBarrier(&g_SendRecvSynchronization);
        }
    }



    void TwoWaySendThreadProc(void* that)
    {
        NN_SDK_ASSERT_NOT_NULL(that);

        auto ptr = reinterpret_cast<TwoWayCommunication*>(that);

        nn::os::AwaitBarrier(&g_SendRecvSynchronization);

        while (NN_STATIC_CONDITION(true))
        {
            // We sleep here to try to ensure that the barrier will be set back to threadCount = 0
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
            nn::os::AwaitBarrier(&g_SendRecvSynchronization);

            ptr->SendThreadProc();
            nn::os::AwaitBarrier(&g_SendRecvSynchronization);
            ptr->FinalizeSendProc();
        }
    }

} // pccom
} // profiler
} // nn
