﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <cstring>
#include <string>
#include <algorithm>
#include <utility>

#include <nn/nn_Common.h>

#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_StringTable.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const size_t StringTableVersionLocation = 0;
    const size_t StringTableCountLocation = StringTableVersionLocation + 4;
    const size_t StringTableStartLocation = StringTableCountLocation + 4;

    inline size_t GetStringLength(const char* str) NN_NOEXCEPT
    {
        size_t len = strlen(str);
        if (len > StringTable::MaxStrLen) { len = StringTable::MaxStrLen; }
        return len;
    }

    inline size_t HashString(const char* str, const char* strEnd) NN_NOEXCEPT
    {
        size_t hash = std::__do_string_hash(str, strEnd);
        return hash;
    }
} // anonymous

#define TABLE_VERSION(x) (*reinterpret_cast<int*>((x) + StringTableVersionLocation))
#define TABLE_COUNT(x) (*reinterpret_cast<std::atomic<int>*>((x) + StringTableCountLocation))
#define TABLE_STRINGS(x) ((x) + StringTableStartLocation)


int StringTable::GetVersion() NN_NOEXCEPT
{
    return 2;
}


StringTable::StringTable() NN_NOEXCEPT
{
    m_table = static_cast<char*>(Memory::GetInstance()->Allocate(TableSize));
    m_size = TableSize;
    this->Clear();

    INFO_LOG("Table Address: %p\n", m_table);
}


StringTable::~StringTable() NN_NOEXCEPT
{
    auto mem = Memory::GetInstance();

    mem->Free(m_table);
    m_table = nullptr;
    m_current = nullptr;
}


uintptr_t StringTable::AddString(const char* str) NN_NOEXCEPT
{
    NN_SDK_ASSERT(str != nullptr);
    NN_STATIC_ASSERT(sizeof(uintptr_t) >= sizeof(size_t));

    size_t len = GetStringLength(str);
    if (len == 0) { return UINTPTR_MAX; }

    const char* strEnd = str + len;
    size_t hash = HashString(str, strEnd);

    uintptr_t result = FindString(str, len, hash);
    if (result == UINTPTR_MAX)
    {
        if (static_cast<size_t>(m_current - m_table) + sizeof(size_t) + 1 + len > m_size)
        {
            ERROR_LOG("Ran out of memory in StringTable!\n");
            return UINTPTR_MAX;
        }
        TABLE_COUNT(m_table) += 1;
        DEBUG_LOG("Adding %s\n", str);
        char* writeTo = m_current.fetch_add(static_cast<ptrdiff_t>(sizeof(size_t) + 1 + len));
        result = reinterpret_cast<uintptr_t>(writeTo);
        *reinterpret_cast<size_t*>(writeTo) = hash;
        writeTo += sizeof(hash);
        *writeTo = static_cast<char>(len);
        ++writeTo;
        std::memcpy(writeTo, str, len);
        writeTo += len;
    }
    else
    {
        VERBOSE_LOG("String already in table: %s\n", str);
    }

    return hash;
}


uintptr_t StringTable::FindString(const char* str) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(str != nullptr);

    size_t len = GetStringLength(str);
    if (len == 0) { return UINTPTR_MAX; }

    const char* strEnd = str + len;
    size_t hash = HashString(str, strEnd);

    uintptr_t result = FindString(str, len, hash);
    if (result == UINTPTR_MAX)
    {
        DEBUG_LOG("String not in table!\n");
    }

    return result;
}


uintptr_t StringTable::FindString(const char* str, size_t len, size_t hash) const NN_NOEXCEPT
{
    NN_UNUSED(str);
    NN_UNUSED(len);

    char* walk = TABLE_STRINGS(m_table);
    char* current = m_current.load();
    while (walk < current)
    {
        size_t h = *reinterpret_cast<size_t*>(walk);
        if (h == hash) { return reinterpret_cast<uintptr_t>(walk); }

        walk += sizeof(size_t);
        char curStrLen = *walk;
        walk += (curStrLen + 1);

    }
    return UINTPTR_MAX;
}


void StringTable::Clear() NN_NOEXCEPT
{
    memset(m_table, 0, m_size);
    TABLE_VERSION(m_table) = GetVersion();
    TABLE_COUNT(m_table) = 0;
    m_current = TABLE_STRINGS(m_table);
}


int StringTable::GetStringCount() const NN_NOEXCEPT
{
    return TABLE_COUNT(m_table);
}


}} // nn::profiler
