﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include "profiler_Defines.h"
#include "profiler_HeaderWriter.h"
#include "profiler_Memory.h"
#include "profiler_WriteToBuffer.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const size_t HeaderCount = 1 + SupportedCoreCount;

    Header g_Headers[HeaderCount];
    char * g_MemoryBuffers[HeaderCount];
} // anonymous



void InitializeHeaders()
{
    for (size_t i = 0; i < HeaderCount; ++i)
    {
        g_MemoryBuffers[i] = reinterpret_cast<char*>(Memory::GetInstance()->Allocate(1024, 1024));
    }
}



void FinalizeHeaders()
{
    for (size_t i = 0; i < HeaderCount; ++i)
    {
        Memory::GetInstance()->Free(g_MemoryBuffers[i]);
    }
}



Header* GetMasterHeader()
{
    return &g_Headers[0];
}



Header* GetCoreHeader(uint32_t core)
{
    NN_SDK_ASSERT(core < SupportedCoreCount);
    return &g_Headers[1 + core];
}



Header::Header() : m_startPtr(nullptr), m_curPtr(nullptr)
{
    // Nothing to do
}



void Header::Initialize()
{
    m_startPtr = nullptr;
    for (uint32_t i = 0; i < 1 + SupportedCoreCount; ++i)
    {
        if (this == &g_Headers[i])
        {
            m_startPtr = g_MemoryBuffers[i];
        }
    }
    m_curPtr = m_startPtr;
}



void Header::Finalize()
{
}



void* Header::GetBuffer()
{
    return m_startPtr;
}



size_t Header::GetSize()
{
    return static_cast<size_t>(reinterpret_cast<uint8_t*>(m_curPtr) - reinterpret_cast<uint8_t*>(m_startPtr));
}



void Header::WriteControlValueOnly(uint32_t controlValue)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
}



void Header::Write(uint32_t controlValue)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 0); // size
}



void Header::Write(uint32_t controlValue, uint32_t data1)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 4); // size
    m_curPtr = WriteToBuffer(m_curPtr, data1);
}



void Header::Write(uint32_t controlValue, uint64_t data1)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 8); // size
    m_curPtr = WriteToBuffer(m_curPtr, data1);
}



void Header::Write(uint32_t controlValue, uint32_t data1, uint32_t data2)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 8); // size
    m_curPtr = WriteToBuffer(m_curPtr, data1);
    m_curPtr = WriteToBuffer(m_curPtr, data2);
}



void Header::Write(uint32_t controlValue, uint32_t data1, uint32_t data2, uint32_t data3)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 12); // size
    m_curPtr = WriteToBuffer(m_curPtr, data1);
    m_curPtr = WriteToBuffer(m_curPtr, data2);
    m_curPtr = WriteToBuffer(m_curPtr, data3);
}



void Header::Write(
    uint32_t controlValue,
    uint32_t data1,
    uint32_t data2,
    uint32_t data3,
    uint32_t data4)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, 16); // size
    m_curPtr = WriteToBuffer(m_curPtr, data1);
    m_curPtr = WriteToBuffer(m_curPtr, data2);
    m_curPtr = WriteToBuffer(m_curPtr, data3);
    m_curPtr = WriteToBuffer(m_curPtr, data4);
}



void Header::Write(uint32_t controlValue, uint16_t *dataArray, uint32_t count, uint32_t data2)
{
    m_curPtr = WriteToBuffer(m_curPtr, controlValue);
    m_curPtr = WriteToBuffer(m_curPtr, static_cast<uint32_t>(sizeof(*dataArray) * count + 4)); // size
    for (uint32_t i = 0; i < count; ++i)
    {
        m_curPtr = WriteToBuffer(m_curPtr, dataArray[i]);
    }
    m_curPtr = WriteToBuffer(m_curPtr, data2);
}



} // profiler
} // nn
